<?php

/**********************************************************************
* OpenSRS - Domains Pro WHMCS module

* Copyright (C) 2019 by Tucows Inc. and OpenSRS.
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
**********************************************************************/

if (!class_exists('openSRS_base')) {
    require_once dirname(__FILE__).DIRECTORY_SEPARATOR."opensrs".DIRECTORY_SEPARATOR."openSRS_loader.php";
}

# Set Laravel class alias
use Illuminate\Database\Capsule\Manager as Capsule;

# Set global variables
global $osrsError;
global $osrsLogError;

global $transferStatus;
global $transferResponse;
global $transferReason;
global $requestAddress;
global $displayTransferStatus;

define("TEST_FWD_IP", "216.40.33.60");
define("PROD_FWD_IP", "64.99.64.37");

/**************************** PRIMARY FUNCTIONS ************************************/

/**
* OpenSRS Domains Pro Module Config
* @return array $configarray: Module config settings
**/
function opensrspro_GetConfigArray() {
   $configarray = array(
        "TestUsername"            => array("Type" => "text", "Size" => "20", "Description" => "Enter your test reseller username here",),
        "TestAPIKey"              => array("Type" => "password", "Size" => "32", "Description" => "Enter your test API Key here",),
        "ProdUsername"            => array("Type" => "text", "Size" => "20", "Description" => "Enter your production reseller username here",),
        "ProdAPIKey"              => array("Type" => "password", "Size" => "32", "Description" => "Enter your production API Key here",),
        "HashKey"                 => array("Type" => "password", "Size" => "32", "Description" => "Enter your Hash Key here",),
        "TestMode"                => array("Type" => "yesno",),
        "ForceForwardingIp"       => array("Type" => "yesno", "Description" => "If a customer adds a URL forward should a A record to the forwarding server be created?"),
        "ChangeLockedDomains"     => array("Type" => "yesno", "Description" => "If this option is selected, locked domains will be unlocked and the change will be made. Once the changes are complete, the domain will be locked again."),
        "LockTechContact"         => array("Type" => "yesno", "Description" => "Use the default Tech contact address set in your OpenSRS reseller account."),
        "GeneralError"            => array("Type" => "text", "Size" => "50", "Description" => "A general error to be displayed to the end user.",),
        "DisableTemplatesChanges" => array("Type" => "yesno", "Description" => "Check to disable changes in client area templates"),
        "Version"                 => array("Description" => "3.7.4"),
    );
	
    return $configarray;
}

/**
 * Get Name servers
 * @param array $params::Data from WHMCS
 * @return mixed::Operation status or error message
 */
function opensrspro_GetNameservers($params) {
    # Set request data
    $request_data = array(
        'func' => 'lookupGetDomain',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'type' => "nameservers"
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute get nameservers
    $openSRSHandler = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $openSRSHandler->resultFullRaw, '', $params);

    # Handle response
    if ($openSRSHandler->resultFullRaw["is_success"] == True) {
        # Set return nameservers
        foreach ($openSRSHandler->resultFullRaw["attributes"]["nameserver_list"] as $nameserver) {
            $sortOrder = $nameserver["sortorder"];
            $values["ns" . $sortOrder] = $nameserver["name"];
        }
        return $values;
    } else {
        $errors = $openSRSHandler->resultFullRaw["response_text"];
        $values["error"] = 'Unable to get nameservers.<br/>ERROR: '.$errors;
        return $values;
    }
}

/**
 * Save Nameservers
 * @param array $params::Data from WHMCS
 * @return mixed::Operation status or error message
 */
function opensrspro_SaveNameservers($params) {
    # Get domain lock state from OpenSRS
    $lockstate = opensrspro_GetRegistrarLock($params);
    # Return error if GetRegistrarLock returns an error ["error" => "message"] 
    if(is_array($lockstate)) {
        return $lockstate;
    }
    # Set user IP address
    $user_ip = $_SERVER['REMOTE_ADDR'];

    # Unlock domain if locked
    $tempunlock = "off";
    if (strcmp($lockstate, "locked") == 0 && strcmp($params["ChangeLockedDomains"], "on") == 0) {
        $tempunlock = "on";
        opensrspro_ModifyRegistrarLock($params, False, $user_ip);
    }

    # Initialize variable
    $nameserver_list = "";

    # Pulls nameservers from the Params and puts them into csv format
    for ($i = 1; $i <= 5; $i++) {
        if (array_key_exists("ns" . $i, $params)) {
            if (strcmp($params["ns" . $i], "") != 0) {
                $nameserver_list .= $params["ns" . $i];
                $nameserver_list .= ",";
            }
        }
    }

    # Set nameserver list
    $nameserver_list = substr($nameserver_list, 0, -1);

    # Set request data
    $request_data = array(
        'func' => 'nsAdvancedUpdt',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'op_type' => 'assign',
            'assign_ns' => $nameserver_list
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute nameserver update
    $openSRSHandler = processOpensrs("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $openSRSHandler->resultFullRaw, '', $params);

    # Lock domain if unlocked
    if (strcmp($tempunlock, "on") == 0) {
        $tempunlock = "off";
        opensrspro_ModifyRegistrarLock($params, True, $user_ip);
    }

    # Handle response
    if ($openSRSHandler->resultFullRaw["is_success"] == True) {
        # Return success
        return True;
    } else {
        # Return error
        $errors = $openSRSHandler->resultFullRaw["response_text"];
        $values['error'] = 'ERROR: '.$errors;
        return $values;
    }
}

/**
 * Get Registrar Lock Status
 * @param array $params::WHMCS input data
 * @return array $lock_status::Registrar lock status
 */
function opensrspro_GetRegistrarLock($params) {
    # Set user IP address
    $user_ip = $_SERVER['REMOTE_ADDR'];

    # Set request data
    $request_data = array(
        'func' => 'lookupGetDomain',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'type' => "status",
            'registrant_ip' => $user_ip
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute request
    $response = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, '', $params);

    # Initialize variable
    $lock_status = 'locked';

    # Handle response
    if ($response->resultFullRaw["is_success"] == 1) {
        # Set registrar lock status
        if ($response->resultFullRaw["attributes"]["lock_state"] == True) {
            $lock_status = "locked";
        } else {
            $lock_status = "unlocked";
        }
    } else {
        return array("error" => "Failed! Unable to get registrar lock status. Please contact support for 
        assistance.");
    }

    # Return status
    return $lock_status;
}

/**
 * Save Registrar Lock Status
 * @param array $params::WHMCS input data
 * @return array::Registrar lock update response
 */
function opensrspro_SaveRegistrarLock($params) {
    # Set user IP address
    $user_ip = $_SERVER['REMOTE_ADDR'];

    # Set lock state
    if (strcmp($params["lockenabled"], "locked") == 0) {
        $lock_status = "1";
    } else {
        $lock_status = "0";
    }

    # Set request data
    $request_data = array(
        'func' => 'provModify',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'data' => "status",
            'lock_state' => $lock_status,
            'registrant_ip' => $user_ip
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute request
    $response = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, '', []);

    # Handle response
    if($response->resultFullRaw['is_success'] == 1) {
        return True;
    } else {
        return array("error" => "Failed! Unable to updated registrar lock. Please contact support for assistance.");
    }
}

/**
 * Domain Renewal
 * @param array $params::Data from WHMCS
 * @return array $values::Renewal status
 */
function opensrspro_RenewDomain($params) {

    # Set global variables
    global $osrsLogError;
    global $osrsError;

    # Set domain data
    $tld = $params["tld"];
    $sld = $params["sld"];
    $regperiod = $params["regperiod"];

    # Get domain's current expiration date from OpenSRS
    $expirationYear = getExpirationYear($params["domain_punycode"], $params);

    # Set domain's current expiration year
    $currentYear = intval(date("Y"));

    # Calculate maximum renewal period allowed
    if (intval($expirationYear) > $currentYear) {
        $regMax = intval(date("Y")) + 10;
    } else {
        $regMax = intval($expirationYear) + 10;
    }

    # Calculate max renewal period requested
    $renewedUntil = intval($expirationYear) + intval($regperiod);

    # If max renewal period is less than requested renewal period...
    if ($regMax < $renewedUntil) {
        # Output error message
        $osrsError .= "Domain can only be renewed to a maximum of 10 years. <br />";
        $osrsLogError .= "Renewal attempt for " . $sld . "." . $tld . " failed, renewal exceeded 10 year limit. \n";
    } else {
        # else, set call data
        $callArray = array(
            'func' => 'provRenew',
            'data' => array(
                'domain' =>   $params["domain_punycode"],
                'handle' => "process",
                'period' => $regperiod,
                'auto_renew' => "0",
                'currentexpirationyear' => $expirationYear
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        # Sets a user-defined error handler function
        set_error_handler("osrsError", E_USER_WARNING);
        # Process request
        $openSRSHandler = processOpenSRS("array", $callArray);
        # Restores the previous error handler function
        restore_error_handler();

        # Module log
        opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $openSRSHandler->resultFullRaw, $params);

        # If API called is successful...
        if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") != 0) {
            # Set API reponse
            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    # If error log is NOT empty...
    if (!empty($osrsLogError)) {
        # If API called failed and no error message was set, output default error message.
        if (empty($osrsError))
            # Set default error message
            $osrsError = $params["GeneralError"];
    }

    # Set output error message
    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);

    # Return status
    return $values;
}

/**
 * Get Contact Info
 * @param array $params::Data from WHMCS
 * @return array $contact_sets::Domain contact info
 */
function opensrspro_GetContactDetails($params) {
    # Initialize variables
    $lockTech = $params["LockTechContact"];
    $contacts = array();

    # Set user IP address
    $user_ip = $_SERVER['REMOTE_ADDR'];

    # Set request data
    $request_data = array(
        'func' => 'lookupGetDomain',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'type' => "all_info",
            'registrant_ip' => $user_ip
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute contact set request
    $contacts_request = processOpenSRS("array", $request_data);

    # Set OpenSRS keys for contact sets
    $owner = 'owner';
    $admin = 'admin';
    $billing = 'billing';
    $tech = 'tech';

    # Set contact sets from response
    $contact_sets = $contacts_request->resultFullRaw["attributes"]["contact_set"];

    # Set Owner contact
    $contacts['Registrant'] = array(
        'First Name' => $contact_sets[$owner]['first_name'],
        'Last Name' => $contact_sets[$owner]['last_name'],
        'Organization Name' => $contact_sets[$owner]['org_name'],
        'Address 1' => $contact_sets[$owner]['address1'],
        'Address 2' => $contact_sets[$owner]['address2'],
        'Address 3' => $contact_sets[$owner]['address3'],
        'City' => $contact_sets[$owner]['city'],
        'State' => $contact_sets[$owner]['state'],
        'Postal Code' => $contact_sets[$owner]['postal_code'],
        'Country' => $contact_sets[$owner]['country'],
        'Phone' => $contact_sets[$owner]['phone'],
        'Fax' => $contact_sets[$owner]['fax'],
        'Email' => $contact_sets[$owner]['email']
    );

    # Set Admin contact
    $contacts['Admin'] = array(
        'First Name' => $contact_sets[$admin]['first_name'],
        'Last Name' => $contact_sets[$admin]['last_name'],
        'Organization Name' => $contact_sets[$admin]['org_name'],
        'Address 1' => $contact_sets[$admin]['address1'],
        'Address 2' => $contact_sets[$admin]['address2'],
        'Address 3' => $contact_sets[$admin]['address3'],
        'City' => $contact_sets[$admin]['city'],
        'State' => $contact_sets[$admin]['state'],
        'Postal Code' => $contact_sets[$admin]['postal_code'],
        'Country' => $contact_sets[$admin]['country'],
        'Phone' => $contact_sets[$admin]['phone'],
        'Fax' => $contact_sets[$admin]['fax'],
        'Email' => $contact_sets[$admin]['email']
    );

    # Set Billing contact
    $contacts['Billing'] = array(
        'First Name' => $contact_sets[$billing]['first_name'],
        'Last Name' => $contact_sets[$billing]['last_name'],
        'Organization Name' => $contact_sets[$billing]['org_name'],
        'Address 1' => $contact_sets[$billing]['address1'],
        'Address 2' => $contact_sets[$billing]['address2'],
        'Address 3' => $contact_sets[$billing]['address3'],
        'City' => $contact_sets[$billing]['city'],
        'State' => $contact_sets[$billing]['state'],
        'Postal Code' => $contact_sets[$billing]['postal_code'],
        'Country' => $contact_sets[$billing]['country'],
        'Phone' => $contact_sets[$billing]['phone'],
        'Fax' => $contact_sets[$billing]['fax'],
        'Email' => $contact_sets[$billing]['email']
    );

    # Handle Tech contact lock
    if(strcmp($lockTech, "on") != 0) {
        # Set Tech contact
        $contacts['Tech'] = array(
            'First Name' => $contact_sets[$tech]['first_name'],
            'Last Name' => $contact_sets[$tech]['last_name'],
            'Organization Name' => $contact_sets[$tech]['org_name'],
            'Address 1' => $contact_sets[$tech]['address1'],
            'Address 2' => $contact_sets[$tech]['address2'],
            'Address 3' => $contact_sets[$tech]['address3'],
            'City' => $contact_sets[$tech]['city'],
            'State' => $contact_sets[$tech]['state'],
            'Postal Code' => $contact_sets[$tech]['postal_code'],
            'Country' => $contact_sets[$tech]['country'],
            'Phone' => $contact_sets[$tech]['phone'],
            'Fax' => $contact_sets[$tech]['fax'],
            'Email' => $contact_sets[$tech]['email']
        );
    }

    # Handle specific TLD requirements
    switch ($params["tld"]) {
        # .NL
        case 'nl':
            # Remove non-supported fields
            unset($contacts['Registrant']['State']);
            unset($contacts['Admin']['State']);
            unset($contacts['Tech']['State']);
            unset($contacts['Billing']);
            break;
    }

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $contacts_request->resultFullRaw, $contact_sets,
        $params);

    # Return data
    return $contacts;
}

/**
 * Save Contact Details
 * @param array $params::Data from WHMCS
 * @return array::Operation status
 */
function opensrspro_SaveContactDetails($params) {
    # Get tech lock setting
    $lock_tech = $params["LockTechContact"];
    # Get TLD
    $tld = $params["tld"];
    # Get contact sets from WHMCS
    $contact_sets = $params["contactdetails"];

    # Set user IP address
    $user_ip = $_SERVER['REMOTE_ADDR'];

    # Get registrar lock state
    $domain_locked = opensrspro_GetRegistrarLock($params);

    # Set initial update status
    $can_update = False;

    # GC::Domain locked?
    if($domain_locked == "locked") {
        # GC::Unlock domain setting enabled in module config?
        if($params["ChangeLockedDomains"] == "on") {
            # Unlock domain
            $response = opensrspro_SaveRegistrarLock($params);

            # Domain unlocked?
            if($response == True) {
                # Set flag
                $can_update = True;
            } else {
                $can_update = False;
            }
        } else {
            $can_update = False;
        }
    } else {
        $can_update = True;
    }

    # GC::Update contacts?
    if($can_update == True) {
        # Set request data
        $request_data = array(
            'func' => 'lookupGetDomain',
            'data' => array(
                'domain' =>  $params["domain_punycode"],
                'type' => 'owner',
                'registrant_ip' => $user_ip
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        # Execute request
        $response = processOpenSRS("array", $request_data);

        # Initialize variables
        $trade_required = False;
        $process_update = False;
        $approval_required = True;

        # Set trade conditions
        $trade_triggers = array('first_name', 'last_name', 'org_name', 'email');

        # GC::Determine if trade is required on owner contact set
        foreach($response->resultFullRaw['attributes']['contact_set']['owner'] as $field => $value) {
            if(in_array($field, $trade_triggers)) {
                if ($field == 'first_name' && $value != $contact_sets['Registrant']['First Name']) {
                    $trade_required = True;
                    break;
                }
                if ($field == 'last_name' && $value != $contact_sets['Registrant']['Last Name']) {
                    $trade_required = True;
                    break;
                }
                if ($field == 'org_name' && $value != $contact_sets['Registrant']['Organization Name']) {
                    $trade_required = True;
                    break;
                }
                if ($field == 'email' && $value != $contact_sets['Registrant']['Email']) {
                    $trade_required = True;
                    break;
                }
            }
        }

        # Process trade if required
        if($trade_required == True) {
            # Get DA status for current registrant
            $da_status = opensrspro_GetDAStatus($params);

            # GC::DA enabled for current registrant?
            if($da_status == True) {
                # Enable DA for new registrant
                $enable_da = opensrspro_EnableDA($params, $user_ip);

                # Set process flags
                if($enable_da == True) {
                    $process_update = True;
                    $approval_required = False;
                } else {
                    $process_update = True;
                    $approval_required = True;
                }
            } else {
                # Enable DA for current registrant
                $modify_da = opensrspro_ModifyDAState($params, $user_ip);

                # GC::DA enabled for current registrant?
                if ($modify_da == True) {
                    # Enable DA for new registrant
                    $enable_da = opensrspro_EnableDA($params, $user_ip);

                    # Set process flags
                    if($enable_da == True) {
                        $process_update = True;
                        $approval_required = False;
                    } else {
                        $process_update = True;
                        $approval_required = True;
                    }
                } else {
                    # Set process flags
                    $process_update = True;
                    $approval_required = True;
                }
            }
        } else {
            $process_update = True;
        }

        # Process contact update if enabled
        if($process_update == True) {
            # Set default contacts to update
            $update_types = array("owner", "admin", "billing", "tech");

            # Handle tech contact lock
            if ($lock_tech == "on") {
                # Exclude tech contact set
                unset($update_types[3]);
            }

            # Handle specific TLD requirements
            switch ($tld) {
                # .NL
                case 'nl':
                    # Exclude billing contact set
                    unset($update_types[2]);
                    break;

                # .CA
                case 'ca':
                    # Exclude billing contact set
                    unset($update_types[2]);
                    break;
            }

            # Set final update contact types
            $contact_types = implode(",", $update_types);

            # Set request data
            $request_data = array(
                'func' => 'provUpdateContacts',
                'data' => array(
                    'domain' =>  $params["domain_punycode"],
                    'types' => $contact_types,
                    'registrant_ip' => $user_ip
                ),
                'connect' => opensrspro_generateConnectData($params)
            );

            # Set WHMCS keys for contact sets
            $owner = 'Registrant';
            $admin = 'Admin';
            $billing = 'Billing';
            $tech = 'Tech';

            # Set Owner contact set
            $request_data['personal'] = array(
                'first_name' => $contact_sets[$owner]['First Name'],
                'last_name' => $contact_sets[$owner]['Last Name'],
                'address1' => $contact_sets[$owner]['Address 1'],
                'address2' => $contact_sets[$owner]['Address 2'],
                'city' => $contact_sets[$owner]['City'],
                'state' => $contact_sets[$owner]['State'],
                'country' => $contact_sets[$owner]['Country'],
                'phone' => $contact_sets[$owner]['Phone'],
                'fax' => $contact_sets[$owner]['Fax'],
                'email' => $contact_sets[$owner]['Email'],
                'lang_pref' => 'EN',
            );

            # Handle contact set array created by WHMCS "Use Existing Contact" feature
            # Org Name
            if(!empty($contact_sets[$owner]['Organization Name'])){
                $request_data['personal']['org_name'] =$contact_sets[$owner]['Organization Name'];
            } else {
                $request_data['personal']['org_name'] = $contact_sets[$owner]['Company Name'];
            }
            # Address line 3
            if(!empty($contact_sets[$owner]['Address 3'])){
                $request_data['personal']['address3'] = $contact_sets[$owner]['Address 3'];
            }
            # Postal code
            if(!empty($contact_sets[$owner]['Postal Code'])){
                $request_data['personal']['postal_code'] = $contact_sets[$owner]['Postal Code'];
            } else {
                $request_data['personal']['postal_code'] = $contact_sets[$owner]['Postcode'];
            }


            # Set Admin contact set
            $request_data['admin'] = array(
                'first_name' => $contact_sets[$admin]['First Name'],
                'last_name' => $contact_sets[$admin]['Last Name'],
                'address1' => $contact_sets[$admin]['Address 1'],
                'address2' => $contact_sets[$admin]['Address 2'],
                'city' => $contact_sets[$admin]['City'],
                'state' => $contact_sets[$admin]['State'],
                'country' => $contact_sets[$admin]['Country'],
                'email' => $contact_sets[$admin]['Email'],
                'phone' => $contact_sets[$admin]['Phone'],
                'fax' => $contact_sets[$admin]['Fax'],
                'lang_pref' => 'EN',
            );

            # Handle contact set array created by WHMCS "Use Existing Contact" feature
            # Org Name
            if(!empty($contact_sets[$admin]['Organization Name'])){
                $request_data['admin']['org_name'] = $contact_sets[$admin]['Organization Name'];
            } else {
                $request_data['admin']['org_name'] = $contact_sets[$admin]['Company Name'];
            }
            # Address line 3
            if(!empty($contact_sets[$admin]['Address 3'])){
                $request_data['admin']['address3'] = $contact_sets[$admin]['Address 3'];
            }
            # Postal code
            if(!empty($contact_sets[$admin]['Postal Code'])){
                $request_data['admin']['postal_code'] = $contact_sets[$admin]['Postal Code'];
            } else {
                $request_data['admin']['postal_code'] = $contact_sets[$admin]['Postcode'];
            }

            # Set Billing contact set
            $request_data['billing'] = array(
                'first_name' => $contact_sets[$billing]['First Name'],
                'last_name' => $contact_sets[$billing]['Last Name'],
                'address1' => $contact_sets[$billing]['Address 1'],
                'address2' => $contact_sets[$billing]['Address 2'],
                'city' => $contact_sets[$billing]['City'],
                'state' => $contact_sets[$billing]['State'],
                'country' => $contact_sets[$billing]['Country'],
                'email' => $contact_sets[$billing]['Email'],
                'phone' => $contact_sets[$billing]['Phone'],
                'fax' => $contact_sets[$billing]['Fax'],
                'lang_pref' => 'EN',
            );

            # Handle contact set array created by WHMCS "Use Existing Contact" feature
            # Org Name
            if(!empty($contact_sets[$billing]['Organization Name'])){
                $request_data['billing']['org_name'] = $contact_sets[$billing]['Organization Name'];
            } else {
                $request_data['billing']['org_name'] = $contact_sets[$billing]['Company Name'];
            }
            # Address line 3
            if(!empty($contact_sets[$billing]['Address 3'])){
                $request_data['billing']['address3'] = $contact_sets[$billing]['Address 3'];
            }
            # Postal code
            if(!empty($contact_sets[$billing]['Postal Code'])){
                $request_data['billing']['postal_code'] = $contact_sets[$billing]['Postal Code'];
            } else {
                $request_data['billing']['postal_code'] = $contact_sets[$billing]['Postcode'];
            }


            # Handle tech contact lock
            if ($lock_tech != "on"){
                # Set Tech contact set
                $request_data['tech'] = array(
                    'first_name' => $contact_sets[$tech]['First Name'],
                    'last_name' => $contact_sets[$tech]['Last Name'],
                    'address1' => $contact_sets[$tech]['Address 1'],
                    'address2' => $contact_sets[$tech]['Address 2'],
                    'city' => $contact_sets[$tech]['City'],
                    'state' => $contact_sets[$tech]['State'],
                    'country' => $contact_sets[$tech]['Country'],
                    'email' => $contact_sets[$tech]['Email'],
                    'phone' => $contact_sets[$tech]['Phone'],
                    'fax' => $contact_sets[$tech]['Fax'],
                    'lang_pref' => 'EN',
                );

                # Handle contact set array created by WHMCS "Use Existing Contact" feature
                # Org Name
                if(!empty($contact_sets[$tech]['Organization Name'])){
                    $request_data['tech']['org_name'] = $contact_sets[$tech]['Organization Name'];
                } else {
                    $request_data['tech']['org_name'] = $contact_sets[$tech]['Company Name'];
                }
                # Address line 3
                if(!empty($contact_sets[$tech]['Address 3'])){
                    $request_data['tech']['address3'] = $contact_sets[$tech]['Address 3'];
                }
                # Postal code
                if(!empty($contact_sets[$tech]['Postal Code'])){
                    $request_data['tech']['postal_code'] = $contact_sets[$tech]['Postal Code'];
                } else {
                    $request_data['tech']['postal_code'] = $contact_sets[$tech]['Postcode'];
                }
            }

            # Handle Specific TLD
            # .CA
            if($tld == 'ca') {
                $additional_data = addCCTLDFields($params, $request_data);
                $request_data['personal'] = $additional_data['owner'];
                $request_data['admin'] = $additional_data['admin'];
                $request_data['tech'] = $additional_data['tech'];
            }

            # .AE
            if($tld == 'ae') {
                # Set request data
                $domain_trade_data = array(
                    'func' => 'provTradeDomain',
                    'data' => array(
                        'domain' =>  $params["domain_punycode"],
                        'first_name' => $request_data['personal']['first_name'],
                        'last_name' => $request_data['personal']['last_name'],
                        'org_name' => $request_data['personal']['org_name'],
                        'phone' => $request_data['personal']['phone'],
                        'fax' => $request_data['personal']['fax'],
                        'address1' => $request_data['personal']['address1'],
                        'city' => $request_data['personal']['city'],
                        'state' => $request_data['personal']['state'],
                        'postal_code' => $request_data['personal']['postal_code'],
                        'country' => $request_data['personal']['country'],
                        'email' => $request_data['personal']['email'],
                    ),
                    'connect' => opensrspro_generateConnectData($params)
                );

                # Send contact update for "Owner"
                try {
                    $response = processOpenSRS("array", $domain_trade_data);
                } catch (Exception $e) {
                    return array('error' => $e->getMessage());
                }

                # Module log
                opensrspro_logModuleCall(__FUNCTION__.'_Owner', $domain_trade_data, $response->resultFullRaw, $params,
                    $params);

                # Handle response
                if($response->resultFullRaw['is_success'] == true) {
                    # Send Update for "Owner" only
                    $request_data['data']['types'] = 'admin';

                    # Send update for "Tech"
                    try {
                        $response = processOpenSRS("array", $request_data);
                    } catch (Exception $e) {
                        return array('error' => $e->getMessage());
                    }

                    # Module log
                    opensrspro_logModuleCall(__FUNCTION__.'_Admin', $request_data, $response->resultFullRaw, $params,
                        $params);

                    # Handle response
                    if($response->resultFullRaw['is_success'] == true) {
                        # Send Update for "Owner" only
                        $request_data['data']['types'] = 'tech';

                        # Send update for "Tech"
                        try {
                            $response = processOpenSRS("array", $request_data);
                        } catch (Exception $e) {
                            return array('error' => $e->getMessage());
                        }

                        # Module log
                        opensrspro_logModuleCall(__FUNCTION__.'_Tech', $request_data, $response->resultFullRaw,
                            $params,
                            $params);

                        # Handle response
                        if($response->resultFullRaw['is_success'] == true) {
                            return array('success' => true);
                        } else {
                            # Return error
                            return array("error" => 'Error::'. $response->resultFullRaw['response_text']);
                        }
                    } else {
                        # Return error
                        return array("error" => 'Error::'. $response->resultFullRaw['response_text']);
                    }
                } else {
                    # Return error
                    return array("error" => 'Error::'. $response->resultFullRaw['response_text']);
                }


            }

            # Execute contact update
            try {
                $response = processOpenSRS("array", $request_data);
            } catch (Exception $e) {
                return array('error' => $e->getMessage());
            }

            # Module log
            opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, $params, $params);

            # Opt-out of 60 transfer lock after contact update
            if($response->resultFullRaw['is_success'] == 1) {
                opensrspro_Modify60DayLock($params, 0, $user_ip);
            }

            # Disable registrar lock if required
            if($domain_locked == "locked") {
                opensrspro_ModifyRegistrarLock($params, True, $user_ip);
            }

            # Handle response
            if($response->resultFullRaw['is_success'] == true) {
                if($approval_required == True) {
                    return array("success" => "Contact update submitted, however approval is required. Please follow
                    the instructions in the email you will receive shortly.");
                } else {
                    return array("success" => True);
                }
            } else {
                # Return error
                return array("error" => 'Error::'. $response->resultFullRaw['attributes']['details'][  $params["domainname"]]['response_text']);
            }
        }
    } else {
        return array("error" => "Unable to update contact details. Please unlock the domain first.");
    }
}

/**
 * Domain Sync
 * @param $params::Data from WHMCS
 * @return array::Return domain sync status
 */
function opensrspro_Sync($params) {
    # Get domain name
    $domain =   $params["domain_punycode"];

    # Get domain's expiry date from OpenSRS
    try {
        # Construct API call
        $request = array(
            'func' => 'lookupGetDomain',
            'data' => array(
                'type' => 'all_info',
                'domain' => $domain
            ),
            'connect' => opensrspro_generateConnectData($params)
        );
        # Send API call
        $results = processOpenSRS("array", $request);

        # Get expiry date
        if($results->resultFullRaw["is_success"] == True) {
            # Set expiry date
            $expiry_date = date('Y-m-d', strtotime($results->resultFullRaw['attributes']['expiredate']));
        }
    } catch (Exception $e) {
        return 'Unable to connect to OpenSRS API server. | Error:'. $e->getMessage();
    }

    # Get today's date
    $date = date('Y-m-d');

    # Calculate date value in seconds
    $date_value = strtotime($expiry_date) - strtotime($date);

    # Log module call
    opensrspro_logModuleCall(__FUNCTION__, $request, $results->resultFullRaw, "", []);

    # Check if domain expired
    if ($date_value <= 0){
        return array(
            'expired' => true,
            'expirydate' => $expiry_date
        );
    } else {
        return array(
            'active' => true,
            'expirydate' => $expiry_date
        );
    }
}

/**
 * Transfer Sync
 * @param array $params: Data from WHMCS
 * @return array $values: Transfer status
 **/
function opensrspro_TransferSync($params) {
    # Set domain name
    $domain =   $params["domain_punycode"];

    # Connect to API Server and run check_transfer command
    $check_transfer = array(
        'func' => 'transCheck',
        'data' => array(
            'domain' => $domain,
            'check_status' => "1",
            'get_request_address' => "1",
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Set OpenSRS response
    $check_transfer_response = processOpenSRS("array", $check_transfer);

    # Set returned values
    $status = $check_transfer_response->resultFullRaw["attributes"]["status"];
    $response = $check_transfer_response->resultFullRaw["response_text"];
    $reason = $check_transfer_response->resultFullRaw["attributes"]["reason"];
    $isSuccess = $check_transfer_response->resultFullRaw["is_success"];
    $requestAddress = $check_transfer_response->resultFullRaw["attributes"]["request_address"];

    # Set return array
    $values = array();

    # If transfer of domain completed successfully...
    if ($status == "completed") {
        # Connect to OpenSRS and get new expiration date
        $get_exp = array(
            'func' => 'lookupGetDomain',
            'data' => array(
                'domain' => $domain,
                'type' => "all_info",
                'limit' => "10",
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        # Set OpenSRS response
        $response_expiration = processOpenSRS("array", $get_exp);
        # Get expiration date
        $expiredate = $response_expiration->resultFullRaw["attributes"]["expiredate"];

        # Set domain transfer status to completed
        $values['completed'] = true;
        # Set new expiration date
        $values['expirydate'] = $expiredate;
    }

    # If domain transfer order has been cancelled, update status to failed...
    if  ($status == "cancelled") {
        # Set transfer status to failed
        $values['failed'] = true;
        # Set reason
        $values['reason'] = $status;
    }

    # Handle pending transfer status
    if ($status == "pending_owner" || $status == "pending_registry") {
        # Output error message
        $values['error'] = "<b>Current Status: </b>".$status ."<b> Reason: </b>".$reason ."<b> Approval Email: </b>" .$requestAddress;
    }

    # Construct request/response for module log
    $request_log = array(
        "check_transfer_request" => $check_transfer,
        "get_expiration_request" => $get_exp
    );

    $reponse_log = array(
        "check_transfer_response" => $check_transfer_response->resultFullRaw,
        "get_expiration_reponse" => $response_expiration->resultFullRaw,
        "return_values" => $values
    );

    # Module Log
    opensrspro_logModuleCall(__FUNCTION__, $request_log, $reponse_log, "", []);

    # Return response
    return $values;
}

/**
 * Register Domain Name
 * @param array $params::Data from WHMCS
 * @return mixed
 */
function opensrspro_RegisterDomain($params) {

    global $osrsLogError;
    global $osrsError;

    $osrsLogError = "";
    $osrsError = "";

    $hashKey = $params["HashKey"];
    $lockTech = $params["LockTechContact"];

    $sld = $params["sld"];
    $tld = $params["tld"];

    $regperiod = $params["regperiod"];
    $privacy = $params ['idprotection'];

    $puny_sld = str_replace("xn--","", $params["sld_punycode"]);
    $domainUser = getDomainUser($tld, $puny_sld);
    $domainPass = getDomainPass($tld, $sld, $hashKey);

    $callArray = array(
        'func' => 'provSWregister',
        'data' => array(
            'domain' =>  $params["domain_punycode"],
            'custom_nameservers' => "1",
            'f_lock_domain' => "1",
            'reg_username' => $domainUser,
            'reg_password' => $domainPass,
            'period' => $regperiod,
            'reg_type' => "new",
            'encoding_type' => $params["idnLanguage"],
            'handle' => "process"
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    if (strcmp($lockTech, "on") == 0)
        $callArray["data"]["custom_tech_contact"] = "0";
    else
        $callArray["data"]["custom_tech_contact"] = "1";

    if ($privacy)
        $callArray["data"]["f_whois_privacy"] = "1";
    else
        $callArray["data"]["f_whois_privacy"] = "0";


    // Pulls contact details from the params.  The phone number has to be set
    // to 1.xxxxx on the form and we just add a + at the beginning to put
    // it into good format for OSRS.

    $contactType = 'personal';
    $contactSet = $params;
    $contactValueType = "";

    $callArray[$contactType]["first_name"] = $contactSet[$contactValueType . "firstname"];
    $callArray[$contactType]["last_name"] = $contactSet[$contactValueType . "lastname"];

    //check if org_name is blank
    if (empty($contactSet[$contactValueType . "companyname"]))
        $callArray[$contactType]["org_name"] = $contactSet[$contactValueType . "firstname"] . " " . $contactSet[$contactValueType . "lastname"];
    else
        $callArray[$contactType]["org_name"] = $contactSet[$contactValueType . "companyname"];

    $callArray[$contactType]["address1"] = $contactSet[$contactValueType . "address1"];
    $callArray[$contactType]["address2"] = $contactSet[$contactValueType . "address2"];
    $callArray[$contactType]["address3"] = $contactSet[$contactValueType . "address3"];
    $callArray[$contactType]["city"] = $contactSet[$contactValueType . "city"];
    $callArray[$contactType]["state"] = $contactSet[$contactValueType . "state"];
    $callArray[$contactType]["postal_code"] = $contactSet[$contactValueType . "postcode"];
    $callArray[$contactType]["country"] = $contactSet[$contactValueType . "country"];
    $callArray[$contactType]["email"] = $contactSet[$contactValueType . "email"];

    if($tld == 'ca')
    {
        $callArray[$contactType]["phone"] = $contactSet[$contactValueType . "phonenumberformatted"];
    }
    else
    {
        $callArray[$contactType]["phone"] = $contactSet[$contactValueType . "phonenumberformatted"];
    }
    opensrspro_logModuleCall('Phone_TEST', $callArray[$contactType]["phone"], $params, $params, $params);

    if ($tld == 'de') {
        $callArray[$contactType]["fax"] = $params['additionalfields']['Fax'];
    }
    else
    {
        $callArray[$contactType]["fax"] = $contactSet[$contactValueType . "fax"];
    }

    $callArray[$contactType]["lang_pref"] = "EN";


    // Tech and Billing contacts are copies of the Admin, no separate contacts
    // for WHMCS just yet.
    //Adding Nameservers, putting them into CSV strings for the call
    for ($i = 1; $i <= 5; $i++) {
        if (strcmp($params["ns" . $i], "") != 0) {
            $callArray['data']['name' . $i] = $params["ns" . $i];
            $callArray['data']['sortorder' . $i] = $i;
        }
    }

    set_error_handler("osrsError", E_USER_WARNING);

    // Function that grabs CCTLD fields and checks on things like
    // province format.  Returns any errors as E_USER_WARNING like the
    // toolkit does.

    $callArray = addCCTLDFields($params, $callArray);

    $openSRSHandler = processOpenSRS("array", $callArray);

    restore_error_handler();

    // Checks for errors from OSRS call, specifically errors for the Telephone
    // number.
    if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") != 0) {

        if (strcmp($openSRSHandler->resultFullRaw["response_code"], "465") == 0) {

            if (strpos($openSRSHandler->resultFullRaw["attributes"]["error"], "phone") != false)
                str_replace("+", "", $openSRSHandler->resultFullRaw["attributes"]["error"]);
            $osrsError .= $openSRSHandler->resultFullRaw["attributes"]["error"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["attributes"]["error"] . "\n";
        } else {
            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    if (!empty($osrsLogError)) {
        if (empty($osrsError))
            $osrsError = $params["GeneralError"];
    }

    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);

    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $openSRSHandler->resultFullRaw, $params);

    return $values;
}

/**
 * Admin Domains Tab Fields
 * @param array $param::Data from WHMCS
 * @return array::Admin domain tab fields
 */
function opensrspro_AdminDomainsTabFields($params){

    # Get registrant verification status
    $results = opensrspro_GetRegistrantVerificationStatus($params);

    # Set verification status
    if($results['is_success'] == true) {
        $verify_status = '
        <table cellspacing="1" cellpadding="3" width="100%" border="0" class="datatable">
            <thead>
                <tr>
                    <th>'. ucwords($results['attributes']['registrant_verification_status']) .'</th>
                </tr>
            </thead> 
        </table>
        ';
    } else {
        $verify_status = '
        <table cellspacing="1" cellpadding="3" width="100%" border="0" class="datatable">
            <thead>
                <tr>
                    <th>'. $results['response_text'] .'</th>
                </tr>
            </thead> 
        </table>
        ';
    }

    # Get domain notes
    $notes_modal = opensrspro_ViewDomainNotes($params);

    # Construct domain notes button
    $notes_button = '
    <div style="text-align: center">
        <button type="button" class="btn btn-success btn-block"  data-toggle="modal" data-target="#view_notes">View Notes</button>
    </div>';

    # Log module call
    opensrspro_logModuleCall(__FUNCTION__, $params, $results, "", []);

    # Return output
    return array(
        'Registrant Verification Status' => '<b>'.$verify_status.'</b>',
        'Domain Notes' => $notes_button.$notes_modal,
    );
}

/**
 * Admin Custom Buttons
 * @param array $params::Data from WHMCS
 * @return array $buttons::Admin buttons
 */
function opensrspro_AdminCustomButtonArray($params) {
    # Set domain name
    $domain =   $params["domain_punycode"];

    # Get domain data
    $domain_data = Capsule::table('tbldomains')->where('domain', $domain)->first();

    # If domain is not active, return
    if($domain_data->status != 'Active' && $domain_data->status  != 'Pending Transfer')
        return;

    # Show check transfer button if domain is still pending transfer
    if($domain_data->status  == 'Pending Transfer' && $domain_data->status  != 'Active'){
        # Set button
        $buttons['Check Transfer Status'] = 'TransferCheck';
    };

    # Check domain's verification status
    $verify_status = opensrspro_GetRegistrantVerificationStatus($params);

    # Create button to resend verification email if domain is in verifying status
    if ($verify_status['attributes']['registrant_verification_status'] == 'verifying'){
        # Set button
        $buttons['Resend Verification Email'] = 'SendVerificationEmail';
    }

    # Log module call
    opensrspro_logModuleCall(__FUNCTION__, $domain_data->status, $buttons, "", $params);

    # Return button
    return $buttons;
}




function opensrspro_IDProtectToggle($params) {

    global $osrsLogError;
    global $osrsError;

    if ($params["protectenable"]) {
        $whois_state = "enable";
    } else {
        $whois_state = "disable";
    }

    $callArray = array(
        'func' => 'provModify',
        'data' => array(
            'domain_name'    =>   $params["domain_punycode"],
            'domain'         =>   $params["domain_punycode"],
            'affect_domains' => "0",
            'data'           => "whois_privacy_state",
            'state'          => $whois_state,
            'bypass'         =>   $params["domain_punycode"],
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    set_error_handler("osrsError", E_USER_WARNING);
    $openSRSHandler = processOpenSRS("array", $callArray);
    restore_error_handler();

    if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") != 0) {
        $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
        $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
    }

    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);
    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $values, $params);

    return $values;
}

function opensrspro_GetDNS($params) {

    global $osrsLogError;
    global $osrsError;

    $osrsLogError = "";
    $osrsError = "";

    $hashKey = $params["HashKey"];
    $tld = $params["tld"];
    $sld = $params["sld"];
    $hostrecords = array();

    // Grab the DNS records first
    $callArray = array(
        'func' => 'dnsGet',
        'data' => array(
            'domain' =>   $params["domain_punycode"]
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    set_error_handler("osrsError", E_USER_WARNING);

    $openSRSHandler = processOpenSRS("array", $callArray);

    restore_error_handler();
    
    if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") == 0) {

        // Pull records from OSRS return format and shape them into the format
        // WHMC sees.  WHMCS uses: hostname type address, for all types of
        // records.
        $aRecords = $openSRSHandler->resultFullRaw["attributes"]["records"]["A"];
        if(!$aRecords) {
            $aRecords = [];
        }
        foreach ($aRecords as $aRecord) {
            $hostrecords[] = array(
                "hostname" => $aRecord["subdomain"],
                "type" => "A",
                "address" => $aRecord["ip_address"]
            );
        }

        $cnameRecords = $openSRSHandler->resultFullRaw["attributes"]["records"]["CNAME"];
        if(!$cnameRecords) {
            $cnameRecords = [];
        }
        foreach ($cnameRecords as $cnameRecord) {
            $hostrecords[] = array(
                "hostname" => $cnameRecord["subdomain"],
                "type" => "CNAME",
                "address" => $cnameRecord["hostname"]
            );
        }

        $mxRecords = $openSRSHandler->resultFullRaw["attributes"]["records"]["MX"];
        if(!$mxRecords) {
            $mxRecords = [];
        }    
        /* Added by BC : NG : 10-9-2014 : To sort domain based on priority which is inputted manually */  
        usort($mxRecords,'sort_by_priority');
        /* END : To sort domain based on priority which is inputted manually */  
        foreach ($mxRecords as $mxRecord) {
            $hostrecords[] = array(
                "hostname" => $mxRecord["subdomain"],
                "type" => "MX",
                "address" => $mxRecord["hostname"],
                "priority" => $mxRecord["priority"]
            );
        }

        $txtRecords = $openSRSHandler->resultFullRaw["attributes"]["records"]["TXT"];
        foreach ($txtRecords as $txtRecord) {
            $hostrecords[] = array(
                "hostname" => $txtRecord["subdomain"],
                "type" => "TXT",
                "address" => $txtRecord["text"]
            );
        }
        
        /*Added by BC : NG : 4-9-2014 : To add SRV Records to DNS management page */   
        $srvRecords = $openSRSHandler->resultFullRaw["attributes"]["records"]["SRV"];
        /* Added by BC : NG : 10-9-2014 : To sort domain based on priority which is inputted manually */  
        if(!$srvRecords) {
            $srvRecords = [];
        }
        usort($srvRecords,'sort_by_priority');
        /* END : To sort domain based on priority which is inputted manually */  
        foreach ($srvRecords as $srvRecord) {
            $hostrecords[] = array(
                "hostname" => $srvRecord["subdomain"],
                "type" => "SRV",
                "address" => $srvRecord["hostname"],
                "priority" => $srvRecord["priority"],
                "weight" => $srvRecord["weight"],
                "port" => $srvRecord["port"]
            );
        }
        /* END : To add SRV Records to DNS management page */   
        
    } else {
        $values = [];
        $values["error"] .= $openSRSHandler->resultFullRaw["response_text"] . " ";
    }

    $cookieBypass = true;
    $puny_sld = str_replace("xn--","", $params["sld_punycode"]);
    $domainUser = getDomainUser($tld, $puny_sld);
    $domainPass = getDomainPass($tld, $sld, $hashKey);

    if (!$cookieBypass)
        $cookie = getCookie($sld . "." . $tld, $domainUser, $domainPass, $params);
    else
        $cookie = false;

    if ($cookie !== false || $cookieBypass) {

        // Grab the forwarding records next, grabbing fwd records needs
        // a cookie or bypass.
        $callArray = array(
            'func' => 'fwdGet',
            'data' => array(
                'domain' =>   $params["domain_punycode"],
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        if ($cookieBypass)
            $callArray['data']['bypass'] = $sld . "." . $tld;
        else
            $callArray['data']['cookie'] = $cookie;

        set_error_handler("osrsError", E_USER_WARNING);

        $openSRSHandler = processOpenSRS("array", $callArray);

        restore_error_handler();
        
        if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") == 0) {

            // Pull forwarding records in WHMCS format: hostname type address
            // like with DNS.
            $fwds = $openSRSHandler->resultFullRaw["attributes"]["forwarding"];
            /* Added by BC : NG : 9-7-2014 : To resolve error "Domain forwarding not found for domain" */  
            $_SESSION['doamin_fwd'] = $fwds;
            /* End : To resolve error "Domain forwarding not found for domain" */
            foreach ($fwds as $fwd) {
                if (strcmp($fwd["masked"], "1") == 0) {
                    $hostrecords[] = array(
                        "hostname" => $fwd["subdomain"],
                        "type" => "FRAME",
                        "address" => $fwd["destination_url"]
                    );
                } else {
                    $hostrecords[] = array(
                        "hostname" => $fwd["subdomain"],
                        "type" => "URL",
                        "address" => $fwd["destination_url"]
                    );
                }
            }
        } else {
            /* Added by BC : NG : 9-7-2014 : To resolve error "Domain forwarding not found for domain" */  
            $_SESSION['doamin_fwd'] = "";
            /* End : To resolve error "Domain forwarding not found for domain" */  
            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $hostrecords, $params);

    return $hostrecords;
}

function opensrspro_SaveDNS($params) {

    // Set up forwarding server IP in case
    // force forwarding IP is turned on
    $testMode = $params["TestMode"];
    if (strcmp($testMode, "on") == 0)
        $fwdServerIP = TEST_FWD_IP;
    else
        $fwdServerIP = PROD_FWD_IP;

    $forceForwardingIP = false;

    if (strcmp($params["ForceForwardingIp"], "on") == 0)
        $forceForwardingIP = true;

    // Set up record CSV lines
    $cname_hostnames = "";
    $cname_subdomains = "";
    $a_ip_addresses = "";
    $a_subdomains = "";
    $fwd_addresses = "";
    $fwd_subdomains = "";
    $fwd_maskeds = "";
    $mx_subdomains = "";
    $mx_hostnames = "";
    $mx_priorities = "";
    $txt_texts = "";
    $txt_subdomains = "";
    $numSRV = [];
    $srv_hostnames = "";
    $srv_subdomains = "";
    $srv_priorities = "";
    $srv_weight = "";
    $srv_port = "";
    // Set up container arrays, numMX keeps a count on each MX record hostname
    // to set priority.  WHMCS doesn't allow for a priority field, so priority is
    // set by user order in the list for each hostname.

    $values = array();
    $numMX = array();
    $fwdSubDomains = array();
    $i = 1;

    /*Added by BC : NG : 4-9-2014 : To add SRV Records to DNS management page */
    $dnsrecordWeights = $_POST['dnsrecordweight'];
    $dnsrecordPorts   = $_POST['dnsrecordport'];
    /* End : To add SRV Records to DNS management page */

    // Push records in OSRS toolkit format.  OSRS toolkit takes csv strings
    // of the different parameters.
    foreach ($params["dnsrecords"] AS $dnsKey => $dnsValue) {

        // Resets the MX record counter if it's empty for this hostname
        if (strcmp($dnsValue["type"], "MX") == 0 && $numMX[$dnsValue["hostname"]] == null)
            $numMX[$dnsValue["hostname"]] = 0;

        // If the address is blank, the record will be omitted.  This allows
        // the end user to delete records by wiping the address or just
        // providing whitespace.
        $addressBlank = strcmp(trim($dnsValue["address"]), "");

        if ($addressBlank != 0) {
            // Keeps track of the forward records for forcing the forwarding IP.
            // if a forwarding ip A record for this subdomain already exists,
            // it will skip that forwarding record.
            if (strcmp($dnsValue["type"], "A") == 0 && strcmp($dnsValue["address"], $fwdServerIP) == 0) {
                $fwdSubDomains[$i] = $dnsValue["hostname"];
                $i++;
            }

            switch ($dnsValue["type"]) {
                case "CNAME":
                    $cname_hostnames .= $dnsValue["address"] . ",";
                    $cname_subdomains .= $dnsValue["hostname"] . ",";
                    break;

                case "A":
                    $a_ip_addresses .= $dnsValue["address"] . ",";
                    $a_subdomains .= $dnsValue["hostname"] . ",";
                    break;

                case "MX":
                case "MXE":
                    $numMX[$dnsValue["hostname"]]++;
                    $mx_hostnames .= $dnsValue["address"] . ",";
                    $mx_subdomains .= $dnsValue["hostname"] . ",";
                    /* Changed by BC : NG : 10-9-2014 : To set priority which is inputted manually */  
                    /*$mx_priorities .= 10 * $numMX[$dnsValue["hostname"]] . ",";*/
                    $mx_priorities .= $dnsValue['priority'] . ",";
                    /* END : To set priority which is inputted manually */  
                    break;

                case "TXT":
                    $txt_texts .= $dnsValue["address"] . ",";
                    $txt_subdomains .= $dnsValue["hostname"] . ",";
                    break;

                case "URL":
                    $fwd_addresses .= $dnsValue["address"] . ",";
                    $fwd_subdomains .= $dnsValue["hostname"] . ",";
                    $fwd_maskeds .= "0,";

                    $searchResult = array_search($dnsValue["hostname"], $fwdSubDomains);

                    if ($searchResult == false && $forceForwardingIP == true) {
                        $a_ip_addresses .= $fwdServerIP . ",";
                        $a_subdomains .= $dnsValue["hostname"] . ",";
                    }

                    break;

                case "FRAME":
                    $fwd_addresses .= $dnsValue["address"] . ",";
                    $fwd_subdomains .= $dnsValue["hostname"] . ",";
                    $fwd_maskeds .= "1,";

                    $searchResult = array_search($dnsValue["hostname"], $fwdSubDomains);

                    if ($searchResult == false && $forceForwardingIP == true) {
                        $a_ip_addresses .= $fwdServerIP . ",";
                        $a_subdomains .= $dnsValue["hostname"] . ",";
                    }
                    break;
                
                /*Added by BC : NG : 4-9-2014 : To add SRV Records to DNS management page */   
               case "SRV":
                    $numSRV[$dnsValue["hostname"]]++;
                    $srv_hostnames .= $dnsValue["address"] . ",";
                    $srv_subdomains .= $dnsValue["hostname"] . ",";
                    /* Changed by BC : NG : 10-9-2014 : To set priority which is inputted manually */  
                    /*$srv_priorities .= 10 * $numSRV[$dnsValue["hostname"]] . ",";*/
                    $srv_priorities .= $dnsValue['priority'] . ",";
                    /* END : To set priority which is inputted manually */ 
                    $srv_weight .= $dnsrecordWeights[$dnsKey] . ",";
                    $srv_port .= $dnsrecordPorts[$dnsKey] . ",";
                    break;
              /* End : To add SRV Records to DNS management page */   
              
            }
        }
    }

    // Scrub the last character in the csv, so there is no trailing comma
    $cname_hostnames = substr($cname_hostnames, 0, -1);
    $cname_subdomains = substr($cname_subdomains, 0, -1);
    $a_ip_addresses = substr($a_ip_addresses, 0, -1);
    $a_subdomains = substr($a_subdomains, 0, -1);
    $mx_hostnames = substr($mx_hostnames, 0, -1);
    $mx_subdomains = substr($mx_subdomains, 0, -1);
    $mx_priorities = substr($mx_priorities, 0, -1);
    $txt_texts = substr($txt_texts, 0, -1);
    $txt_subdomains = substr($txt_subdomains, 0, -1);
    $fwd_addresses = substr($fwd_addresses, 0, -1);
    $fwd_subdomains = substr($fwd_subdomains, 0, -1);
    $fwd_maskeds = substr($fwd_maskeds, 0, -1);
    
    // SRV
    /*Added by BC : NG : 4-9-2014 : To add SRV Records to DNS management page */   
    $srv_hostnames = substr($srv_hostnames, 0, -1);
    $srv_subdomains = substr($srv_subdomains, 0, -1);
    $srv_priorities = substr($srv_priorities, 0, -1);
    $srv_weight = substr($srv_weight, 0, -1);
    $srv_port = substr($srv_port, 0, -1);
    /*End : To add SRV Records to DNS management page */

    // Create the DNS service on this domain if it doesn't already exist
    $createArray = array(
        'func' => 'dnsCreate',
        'data' => array(
            'domain' =>   $params["domain_punycode"]
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    $createHandler = processOpenSRS("array", $createArray);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__.'_CreateHandler', $createArray, $createHandler->resultFullRaw, '', $params);

    // Call the set function to set the records in OSRS
    $callArray = array(
        'func' => 'dnsSet',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'a_ip_addresses' => $a_ip_addresses,
            'a_subdomains' => $a_subdomains,
            'cname_hostnames' => $cname_hostnames,
            'cname_subdomains' => $cname_subdomains,
            'mx_hostnames' => $mx_hostnames,
            'mx_subdomains' => $mx_subdomains,
            'mx_priorities' => $mx_priorities,
            'srv_hostname' => $srv_hostnames,
            'srv_subdomain' => $srv_subdomains,
            'srv_priority' => $srv_priorities,
            'srv_weight' => $srv_weight,
            'srv_port' => $srv_port,
            'txt_texts' => $txt_texts,
            'txt_subdomains' => $txt_subdomains
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    $openSRSHandler = processOpenSRS("array", $callArray);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__.'_SetDNS', $callArray, $openSRSHandler->resultFullRaw, '', $params);
    if ($openSRSHandler->resultFullRaw["is_success"] == "0") {
        $value['error'] = $openSRSHandler->resultFullRaw["response_text"];
        return $value;
    }

    /* Start : To delete domain forwarding */
    $cookieBypass = true;
    if (empty($fwd_subdomains)) {
        if(!empty($_SESSION['doamin_fwd'])){
            $callArray = array(
                    'func' => 'fwdDelete',
                    'data' => array(
                        'domain' =>   $params["domain_punycode"],
                    ),
                    'connect' => opensrspro_generateConnectData($params)
                );

                $openSRSHandler = processOpenSRS("array", $callArray);

                # Module log
                opensrspro_logModuleCall(__FUNCTION__.'_ForwardDelete', $callArray, $openSRSHandler->resultFullRaw, '', $params);

                if ($openSRSHandler->resultFullRaw["is_success"] == "0") {
                    $value['error'] = $openSRSHandler->resultFullRaw["response_text"];
                    return $value;
                }
                # Return status
                return True;
        }
    } else {
        // Create the forwarding service for this domain name.  A cookie is
        // required, no bypass, for this function.  Using getDomainCredentials
        // to grab the username and password if bypass is on.
        $createArray = array(
            'func' => 'fwdCreate',
            'data' => array(
                'domain' =>   $params["domain_punycode"]
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        // Checks to see if there was an error grabbing the cookie or if bypass
        // is on
        $createHandler = processOpenSRS("array", $createArray);

        # Module log
        opensrspro_logModuleCall(__FUNCTION__.'_ForwardCreate', $createArray, $createHandler->resultFullRaw, '', $params);

        // Once the create function is called the forwarding set function is
        // called to set the URL forwards.
        $callArray = array(
            'func' => 'fwdSet',
            'data' => array(
                'domain' =>   $params["domain_punycode"],
                'subdomain' => $fwd_subdomains,
                'destination_urls' => $fwd_addresses,
                'maskeds' => $fwd_maskeds,
                'cookie' => $cookie,
                'bypass' => $cookieBypass,
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        $openSRSHandler = processOpenSRS("array", $callArray);

        # Module log
        opensrspro_logModuleCall(__FUNCTION__.'_ForwardSet', $callArray, $openSRSHandler->resultFullRaw, '', $params);

        if ($openSRSHandler->resultFullRaw["is_success"] == "0") {
            $value['error'] = $openSRSHandler->resultFullRaw["response_text"];
            return $value;
        }
    }
    /* End : To delete domain forwarding */

}

function opensrspro_TransferDomain($params) {

    global $osrsLogError;
    global $osrsError;

    $osrsLogError = "";
    $osrsError = "";

    $hashKey = $params["HashKey"];
    $lockTech = $params["LockTechContact"];

    $sld = $params["sld"];
    $tld = $params["tld"];

    $regperiod = $params["regperiod"];
    $transfersecret = $params["transfersecret"];
    $privacy = $params ['idprotection'];

    $puny_sld = str_replace("xn--","", $params["sld_punycode"]);
    $domainUser = getDomainUser($tld, $puny_sld);
    $domainPass = getDomainPass($tld, $sld, $hashKey);

    $callArray = array(
        'func' => 'provSWregister',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'f_lock_domain' => "1",
            'custom_nameservers' => "1",
            'reg_username' => $domainUser,
            'reg_password' => $domainPass,
            'period' => $regperiod,
            'handle' => "process",
            'reg_type' => "transfer"
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    if (strcmp($lockTech, "on") == 0)
        $callArray["data"]["custom_tech_contact"] = "0";
    else
        $callArray["data"]["custom_tech_contact"] = "1";

    if ($privacy)
        $callArray["data"]["f_whois_privacy"] = "1";
    else
        $callArray["data"]["f_whois_privacy"] = "0";

    //$contactTypes = array("owner", "admin", "tech", "billing");

    $contactType = 'personal';

    $contactSet = $params;
    $contactValueType = "";

//        if (strcmp($contactType, "owner") == 0)
//            $contactValueType = "";
//        else
//            $contactValueType = $contactType;

    $callArray[$contactType]["first_name"] = $contactSet[$contactValueType . "firstname"];
    $callArray[$contactType]["last_name"] = $contactSet[$contactValueType . "lastname"];

    //check if org_name is blank
    if (empty($contactSet[$contactValueType . "companyname"]))
        $callArray[$contactType]["org_name"] = $contactSet[$contactValueType . "firstname"] . " " . $contactSet[$contactValueType . "lastname"];
    else
        $callArray[$contactType]["org_name"] = $contactSet[$contactValueType . "companyname"];

    $callArray[$contactType]["address1"] = $contactSet[$contactValueType . "address1"];
    $callArray[$contactType]["address2"] = $contactSet[$contactValueType . "address2"];
    $callArray[$contactType]["address3"] = $contactSet[$contactValueType . "address3"];
    $callArray[$contactType]["city"] = $contactSet[$contactValueType . "city"];
    $callArray[$contactType]["state"] = $contactSet[$contactValueType . "state"];
    $callArray[$contactType]["postal_code"] = $contactSet[$contactValueType . "postcode"];
    $callArray[$contactType]["country"] = $contactSet[$contactValueType . "country"];
    $callArray[$contactType]["email"] = $contactSet[$contactValueType . "email"];
    $callArray[$contactType]["phone"] = $contactSet[$contactValueType . "fullphonenumber"];
    $callArray[$contactType]["fax"] = $contactSet[$contactValueType . "fax"];
    $callArray[$contactType]["lang_pref"] = "EN";

    // Add in the auth code if supplied
    if ( !empty($transfersecret) ){
        $callArray["data"]["auth_info"] = $transfersecret;
    }
    
    // Tech and Billing contacts are copies of the Admin, no seperate contacts
    // for WHMCS just yet.
    $callArray["tech"] = $callArray["admin"];
    $callArray["billing"] = $callArray["admin"];

    //Adding Nameservers
    for ($i = 1; $i <= 5; $i++) {
        if (strcmp($params["ns" . $i], "") != 0) {
            $callArray['data']['name' . $i] = $params["ns" . $i];
            $callArray['data']['sortorder' . $i] = $i;
        }
    }

    set_error_handler("osrsError", E_USER_WARNING);

    $callArray = addCCTLDFields($params, $callArray);

    $openSRSHandler = processOpenSRS("array", $callArray);
    
    restore_error_handler();

    if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") != 0) {

        if (strcmp($openSRSHandler->resultFullRaw["response_code"], "465") == 0) {

            if (strpos($openSRSHandler->resultFullRaw["attributes"]["error"], "phone") != false)
                str_replace("+", "", $openSRSHandler->resultFullRaw["attributes"]["error"]);

            $osrsError .= $openSRSHandler->resultFullRaw["attributes"]["error"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["attributes"]["error"] . "\n";
        } else {
            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    if (!empty($osrsLogError)) {
        if (empty($osrsError))
            $osrsError = $params["GeneralError"];
    }

    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);

    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $openSRSHandler->resultFullRaw, $params);

    return $values;
}

function opensrspro_GetEPPCode($params) {

    global $osrsLogError;
    global $osrsError;

    $osrsLogError = "";
    $osrsError = "";

    $hashKey = $params["HashKey"];
    $tld = $params["tld"];
    $sld = $params["sld"];
    $cookieBypass = true;


    if (!$cookieBypass) {
        $domainUser = getDomainUser($tld, $params["sld_punycode"]);
        $domainPass = getDomainPass($tld, $sld, $hashKey);
        $cookie = getCookie($sld . "." . $tld, $domainUser, $domainPass, $params);
    } else
        $cookie = false;

    // Checks to see if there was an error grabbing the cookie
    if ($cookie !== false || $cookieBypass) {

        $callArray = array(
            'func' => 'lookupGetDomain',
            'data' => array(
                'domain' =>  $params["domain_punycode"],
                'cookie' => $cookie,
                'type' => "domain_auth_info"
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        set_error_handler("osrsError", E_USER_WARNING);

        $openSRSHandler = processOpenSRS("array", $callArray);

        restore_error_handler();

        if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") == 0) {
            $eppcode = $openSRSHandler->resultFullRaw["attributes"]["domain_auth_info"];
        } else {

            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    $values["eppcode"] = $eppcode;

    if (!empty($osrsLogError)) {
        if (empty($osrsError))
            $osrsError = $params["GeneralError"];
    }

    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);

    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $values, $params);

    return $values;
}

function opensrspro_RegisterNameserver($params) {

    global $osrsLogError;
    global $osrsError;

    $osrsLogError = "";
    $osrsError = "";

    $hashKey = $params["HashKey"];
    $values = array();

    $tld = $params["tld"];
    $sld = $params["sld"];

    /*
      if(strcmp($params['CookieBypass'],"on")==0)
      $cookieBypass = true;
      else
      $cookieBypass = false;
     */
    $cookieBypass = true;


    if (!$cookieBypass) {
        $puny_sld = str_replace("xn--","", $params["sld_punycode"]);
        $domainUser = getDomainUser($tld, $puny_sld);
        $domainPass = getDomainPass($tld, $sld, $hashKey);
        $cookie = getCookie($sld . "." . $tld, $domainUser, $domainPass, $params);
    } else
        $cookie = false;

    // Checks to see if there was an error grabbing the cookie
    if ($cookie !== false || $cookieBypass) {

        $callArray = array(
            'func' => 'nsCreate',
            'data' => array(
                'name' => $params["nameserver"],
                'ipaddress' => $params["ipaddress"],
                'add_to_all_registry' => "1"
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        if ($cookieBypass)
            $callArray['data']['bypass'] = $sld . "." . $tld;
        else
            $callArray['data']['cookie'] = $cookie;

        set_error_handler("osrsError", E_USER_WARNING);

        $openSRSHandler = processOpenSRS("array", $callArray);

        restore_error_handler();

        if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") != 0) {
            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    if (!empty($osrsLogError)) {
        if (empty($osrsError))
            $osrsError = $params["GeneralError"];
    }

    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);

    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $values, $params);

    return $values;
}

function opensrspro_ModifyNameserver($params) {
    //require_once("./opensrs/openSRS_loader.php");

    global $osrsLogError;
    global $osrsError;

    $osrsLogError = "";
    $osrsError = "";

    $username = $params["Username"];
    $key = $params["APIKey"];
    $hashKey = $params["HashKey"];
    $testMode = $params["TestMode"];
    $values = array();

    $tld = $params["tld"];
    $sld = $params["sld"];
    $puny_sld = str_replace("xn--","", $params["sld_punycode"]);
    $domainUser = getDomainUser($tld, $puny_sld);
    $domainPass = getDomainPass($tld, $sld, $hashKey);

    /*
      if(strcmp($params['CookieBypass'],"on")==0)
      $cookieBypass = true;
      else
      $cookieBypass = false;
     */
    $cookieBypass = true;

    if (!$cookieBypass)
        $cookie = getCookie($sld . "." . $tld, $domainUser, $domainPass, $params);
    else
        $cookie = false;

    // Checks to see if there was an error grabbing the cookie
    if ($cookie !== false || $cookieBypass) {

        $callArray = array(
            'func' => 'nsModify',
            'data' => array(
                'name' => $params["nameserver"],
                'ipaddress' => $params["newipaddress"]
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        if ($cookieBypass)
            $callArray['data']['bypass'] = $sld . "." . $tld;
        else
            $callArray['data']['cookie'] = $cookie;

        set_error_handler("osrsError", E_USER_WARNING);

        $openSRSHandler = processOpenSRS("array", $callArray);

        restore_error_handler();

        if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") != 0) {
            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    if (!empty($osrsLogError)) {
        if (empty($osrsError))
            $osrsError = $params["GeneralError"];
    }

    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);

    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $values, $params);

    return $values;
}

function opensrspro_DeleteNameserver($params) {
    //require_once("./opensrs/openSRS_loader.php");

    global $osrsLogError;
    global $osrsError;

    $osrsLogError = "";
    $osrsError = "";

    $username = $params["Username"];
    $key = $params["APIKey"];
    $hashKey = $params["HashKey"];
    $testMode = $params["TestMode"];
    $values = array();

    $tld = $params["tld"];
    $sld = $params["sld"];

    $puny_sld = str_replace("xn--","", $params["sld_punycode"]);
    $domainUser = getDomainUser($tld, $puny_sld);
    $domainPass = getDomainPass($tld, $sld, $hashKey);

    /*
      if(strcmp($params['CookieBypass'],"on")==0)
      $cookieBypass = true;
      else
      $cookieBypass = false;
     */
    $cookieBypass = true;

    if (!$cookieBypass)
        $cookie = getCookie($sld . "." . $tld, $domainUser, $domainPass, $params);
    else
        $cookie = false;

    // Checks to see if there was an error grabbing the cookie
    if ($cookie !== false || $cookieBypass) {

        $callArray = array(
            'func' => 'nsDelete',
            'data' => array(
                'name' => $params["nameserver"],
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        if ($cookieBypass)
            $callArray['data']['bypass'] = $sld . "." . $tld;
        else
            $callArray['data']['cookie'] = $cookie;

        set_error_handler("osrsError", E_USER_WARNING);

        $openSRSHandler = processOpenSRS("array", $callArray);

        restore_error_handler();

        if (strcmp($openSRSHandler->resultFullRaw["is_success"], "1") != 0) {
            $osrsError .= $openSRSHandler->resultFullRaw["response_text"] . "<br />";
            $osrsLogError .= $openSRSHandler->resultFullRaw["response_text"] . "\n";
        }
    }

    if (!empty($osrsLogError)) {
        if (empty($osrsError))
            $osrsError = $params["GeneralError"];
    }

    $values["error"] = filterForResellerError($osrsError, $params["GeneralError"]);

    opensrspro_logModuleCall(__FUNCTION__, $callArray, $openSRSHandler->resultFullRaw, $values, $params);

    return $values;
}

/**************************** HELPER FUNCTIONS ************************************/
/**
 * Generate OpenSRS Connection Data
 * @param array $params::Data from WHMCS
 * @return array $connectData::Connection data
 */
function opensrspro_GenerateConnectData($params) {
    if (strcmp($params["TestMode"], "on") == 0 || $params["TestMode"] == 1) {
        $connectData["osrs_username"] = $params["TestUsername"];
        $connectData["osrs_password"] = "placeholder";
        $connectData["osrs_key"] = $params["TestAPIKey"];
        $connectData["osrs_environment"] = "TEST";
        $connectData["osrs_host"] = "horizon.opensrs.net";
        $connectData["osrs_port"] = "55000";
        $connectData["osrs_sslPort"] = "55443";
    } else {
        $connectData["osrs_username"] = $params["ProdUsername"];
        $connectData["osrs_password"] = "placeholder";
        $connectData["osrs_key"] = $params["ProdAPIKey"];
        $connectData["osrs_environment"] = "PROD";
        $connectData["osrs_host"] = "rr-n1-tor.opensrs.net";
        $connectData["osrs_port"] = "55000";
        $connectData["osrs_sslPort"] = "55443";
    }

    $connectData["osrs_protocol"] = "XCP";
    $connectData["osrs_baseClassVersion"] = "2.8.0";
    $connectData["osrs_version"] = "XML:0.1";

    return $connectData;
}

/**
 * Get Module Config Data
 * @return array $params::Module config data
 */
function opensrspro_GetConfigurationParamsData() {
    # Get module data from database
    $result = Capsule::table('tblregistrars')->where('registrar', 'opensrspro')->get();

    # Decrypt data
    foreach($result as $item) {
        $params[$item->setting] = decrypt($item->value);
    }

    # Return config data
    return $params;
}

/**
 * Modify Registrar Lock
 * @param array $params::Data from WHMCS
 * @param boolean $lock_state::Domain registrar lock state
 * @param string $user_ip::Registrant's IP address
 * @return boolean::Operation status
 */
function opensrspro_ModifyRegistrarLock($params, $lock_state, $user_ip) {
    # Convert lock state to OpenSRS API format
    if ($lock_state == True) {
        $state = '1';
    } else {
        $state = '0';
    }
    # Set request data
    $request_data = array(
        'func' => 'provModify',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'data' => "status",
            'lock_state' => $state,
            'registrant_ip' => $user_ip
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute request
    $response = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, '', $params);

    # Handle response
    if ($response->resultFullRaw["is_success"] == True) {
        # Return error
        return True;
    } else {
        return False;
    }
}

/**
 * Modify DA State for Old Registrant
 * @param array $params::Data from WHMCS
 * @param $user_ip
 * @return boolean::Operation Status
 */
function opensrspro_ModifyDAState($params, $user_ip) {
    # Set request data
    $request_data = array(
        'func' => 'modifyTradeDAStatus',
        'data' => array(
            'domain' =>  $params["domain_punycode"],
            'enable' => 1,
            'ip_address' => $user_ip
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute modify DA for current registrant
    $response = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, '', $params);

    # Return response
    if($response->resultFullRaw['is_success'] == True) {
        return True;
    } else {
        return False;
    }
}

/**
 * Modify 60 Day Transfer Lock
 * @param array @params::Data from WHMCS
 * @return boolean::Operation Status
 */
function opensrspro_Modify60DayLock($params, $state, $user_ip) {
    # Set request data
    $request_data = array(
        'func' => 'modifyTradeLock',
        'data' => array(
            'domain' =>   $params["domain_punycode"],
            'enable' => $state,
            'ip_address' => $user_ip
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute modify transfer lock
    $response = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, '', $params);

    # Handle response
    if($response->resultFullRaw['is_success'] == True) {
        return True;
    } else {
        return False;
    }
}

/**
 * Enable DA for New Contact Set
 * @param array $params::Data from WHMCS
 * @param string $user_ip::Registrant IP address
 * @return boolean::Operation status
 */
function opensrspro_EnableDA($params, $user_ip) {
    # Set request data
    $request_data = array(
        'func' => 'enableDATrade',
        'data' => array(
            'first_name' => $params['contactdetails']['Registrant']['First Name'],
            'last_name' => $params['contactdetails']['Registrant']['Last Name'],
            'email' => $params['contactdetails']['Registrant']['Email'],
            'enable' => 1,
            'ip_address' => $user_ip
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Handle "Use Existing Contact" contact set array
    if(!empty($params['contactdetails']['Registrant']['Organization Name'])){
        $request_data['data']['org_name'] = $params['contactdetails']['Registrant']['Organization Name'];
    } else {
        $request_data['data']['org_name'] = $params['contactdetails']['Registrant']['Company Name'];
    }

    # Execute enable DA for new registrant
    $response = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, '', $params);

    # Handle response
    if($response->resultFullRaw['is_success'] == True) {
        return True;
    } else {
        return False;
    }
}

/**
 * Get DA Status for Old Registrant
 * @param array $params::Data from WHMCS
 * @return boolean::Da status
 */
function opensrspro_GetDAStatus($params) {
    # Set request data
    $request_data = array(
        'func' => 'getTradeDAStatus',
        'data' => array(
            'domain' =>  $params["domain_punycode"],
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Execute get DA status for current registrant
    $response = processOpenSRS("array", $request_data);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request_data, $response->resultFullRaw, '', $params);

    # Handle response
    if($response->resultFullRaw['attributes']['allow_designated_agent'] == True) {
        return True;
    } else {
        return False;
    }
}

/**
 * Validate Contact Sets
 * @param array $contact_sets::Contact sets from WHMCS
 * @return string::Errors in HTML list
 */
function opensrspro_ValidateContactSets($contact_sets)
{
    # Define required fields
    $required_fields = array(
        'First Name',
        'Last Name',
        'Organization Name',
        'Address 1',
        'City',
        'State',
        'Postal Code',
        'Country',
        'Phone',
        'Email'
    );

    # Initialize variables
    $empty_field = False;

    # Validate contact sets
    foreach ($contact_sets as $contact_type => $type) {
        # Initialize variables
        $missing_fields = array();

        # Check for missing fields
        foreach ($type as $field => $value) {
            # Check for valid required fields
            if (in_array($field, $required_fields)) {
                # Check for empty field
                if (empty($value)) {
                    $empty_field = True;
                    $missing_fields[$contact_type][] = $field;
                }
            }
        }

        # Remove owner errors if empty
        if (!empty($missing_fields['Registrant'])) {
            # Initialize variable
            $owner_fields = '';
            foreach ($missing_fields['Registrant'] as $field) {
                $owner_fields .= '<li> ' . $field . ' </li>';
            }

            $owner = '
                <tr style="font-weight: 600;">
                    <td> Registrant Contact </td>
                </tr>
                <tr>
                    <td>
                    ' . $owner_fields . '
                    </td>
                </tr>
            ';
        }

        # Remove admin errors if empty
        if (!empty($missing_fields['Admin'])) {
            # Initialize variable
            $admin_fields = '';
            foreach ($missing_fields['Admin'] as $field) {
                $admin_fields .= '<li> ' . $field . ' </li>';
            }

            $admin = '
                <tr style="font-weight: 600;">
                    <td> Admin Contact </td>
                </tr>
                <tr>
                    <td>
                    ' . $admin_fields . '
                    </td>
                </tr>
            ';
        }

        # Remove billing errors if empty
        if (!empty($missing_fields['Billing'])) {
            # Initialize variable
            $billing_fields = '';
            foreach ($missing_fields['Billing'] as $field) {
                $billing_fields .= '<li> ' . $field . ' </li>';
            }

            $billing = '
                <tr style="font-weight: 600;">
                    <td> Billing Contact </td>
                </tr>
                <tr>
                    <td>
                    ' . $billing_fields . '
                    </td>
                </tr>
            ';
        }

        # Remove tech errors if empty
        if (!empty($missing_fields['Tech'])) {
            # Initialize variable
            $tech_fields = '';
            foreach ($missing_fields['Tech'] as $field) {
                $tech_fields .= '<li> ' . $field . ' </li>';
            }

            $tech = '
            <tr style="font-weight: 600;">
                <td> Tech Contact </td>
            </tr>
            <tr>
                <td>
                ' . $tech_fields . '
                </td>
            </tr>
            ';
        }
    }

    # Set final output
    $results = '
    <table style="text-align: left;">
        ' . $owner . '
        ' . $admin . '
        ' . $billing . '
        ' . $tech . '
    </table>
    ';

    # Initialize variables
    $invalid_country = False;
    $contact_type = '';

    # Check country fields for two character code
    foreach ($contact_sets as $type => $item) {
        foreach ($item as $field => $value) {
            if ($field == 'Country') {
                if (strlen($value) > 2) {
                    $invalid_country = True;
                    $contact_type .= $type .', ';
                }
            }
        }
    }

    # Return error if country code is invalid
    if ($invalid_country == True) {
        return array(
            'error' => 'Country code invalid for ' . $contact_type . ' </br> Please provide a two character country code. (Example: US)'
        );
    }

    # Return if empty fields are found
    if ($empty_field == True) {
        return array(
            'status' => False,
            'error' => '<p class="text-left"> There are fields missing in your submission. Please correct the following: </br></br>' . $results . '</p>'
        );
    } else {
        return array(
            'status' => True
        );
    }
}

/**
 * Custom Module logs
 * @param string $function::Function name
 * @param array $request_string::Request data
 * @param array $response_data::Response data
 * @param $processed_data::Post processed data
 * @param array $params::Data from WHMCS
 */
function opensrspro_logModuleCall($function, $callArray, $resultFullRaw, $return, $params) {
    $module = 'OpenSRSPro';
    $action = substr($function, 0, 11) == 'opensrspro_' ? substr($function, 11) : $function;
    $requeststring = $callArray;
    $responsedata = $resultFullRaw;
    //$processeddata = $return;
    $replacevars = array($params["TestUsername"], $params["TestAPIKey"], $params["ProdUsername"], $params["ProdAPIKey"]);
    logModuleCall($module, $action, $requeststring, $responsedata, $return, $replacevars);
}

/**
 * Get Registrant Verification Status
 * @param $params::Data from WHMCS
 * @return array $results::API response
 */
function opensrspro_GetRegistrantVerificationStatus($params){
    # Set domain name
    $domain =  $params["sld_punycode"] . "." .  $params["tld_punycode"];
    # Get module config data
    $params = array_merge($params, opensrspro_GetConfigurationParamsData());

    # Construct API call
    $request = array(
        'func' => 'lookupGetRegistrantVerificationStatus',
        'data' => array(
            'domain' => $domain,
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Send API call
    $results = processOpenSRS("array", $request);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__,$request,$results->resultFullRaw,$results->resultFullRaw,$params);

    # Return results
    return $results->resultFullRaw;
}

/**
 * Get Domain Notes
 * @param array $params::Data from WHMCS
 * @param string $results::Domain notes
 */
function opensrspro_ViewDomainNotes($params){
    # Set domain name
    $domain =   $params["domain_punycode"];

    # Get module config data
    $params = array_merge($params,opensrspro_GetConfigurationParamsData());

    # Construct API call
    $request = array(
        'func' => 'lookupDomainNotes',
        'data' => array(
            'domain' => $domain,
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Send API call
    $openSRSHandler = processOpenSRS("array", $request);

    $table = '<!--Table-->
    <table cellspacing="1" cellpadding="3" width="100%" border="0" class="datatable">
        <thead>
        <tr>
            <th>Notes</th>
            <th>Date</th>
        </tr>
        </thead>        
        <tbody>        
    ';

    foreach($openSRSHandler->resultFullRaw['attributes']['notes'] as $notesData) {
        $table .= '            
        <tr>
            <td>'.$notesData['note'].'</td>
            <td>'.$notesData['timestamp'].'</td>
        </tr>
        ';
    }

    $table .= '
        </tbody>
    </table>
    ';

    $output = '<!-- Modal -->
    <div class="modal fade" id="view_notes" tabindex="-1" role="dialog" aria-labelledby="view-notes" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-body"> '. $table .' </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    ';

    # Return domain notes
    return $output;
}

/**
 * Send Verification Email
 * @param array $params::Data from WHMCS
 * @return string::Operation status
 */
function opensrspro_SendVerificationEmail($params){
    # Set domain name
    $domain =   $params["domain_punycode"];

    # Get module config data
    $params = array_merge($params,opensrspro_GetConfigurationParamsData());

    # Construct API call
    $request = array(
        'func' => 'provSendVerificationEmail',
        'data' => array(
            'domain' => $domain,
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Send API call
    $response = processOpenSRS("array", $request);

    # Module log
    opensrspro_logModuleCall(__FUNCTION__,$request,$response->resultFullRaw,$response->resultFullRaw,$params);

    # Return operation status
    if($response->resultFullRaw["is_success"] == true) {
        $status = 'Verification email sent.';
    } else {
        $status = $response->resultFullRaw["response_text"];
    }

    # Output status to browser
    echo '
    <table cellspacing="1" cellpadding="3" width="100%" border="0" class="datatable">
        <thead>
            <tr>
                <th>'. $status .'</th>
            </tr>
        </thead> 
    </table>
    ';
}

/**
 * Check Transfer status
 * @param array $params::Data from WHMCS
 * @return string::Operation status
 */
function opensrspro_TransferCheck($params) {

    # Set domain name
    $domain =   $params["domain_punycode"];

    # Construct API call
    $request = array(
        'func' => 'transCheck',
        'data' => array(
            'domain' => $domain,
            'check_status' => "1",
            'get_request_address' => "1",
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    # Send API call
    $response = processOpenSRS("array", $request);

    # Set response status
    if($response->resultFullRaw["attributes"]["status"] != null){
        $transfer_status = $response->resultFullRaw["attributes"]["status"];
    } else {
        $transfer_status = 'N/A';
    }
    # Set transfer message
    $transfer_msg = $response->resultFullRaw["attributes"]["reason"];
    # Set approval email
    $approval_email = $response->resultFullRaw["attributes"]["request_address"];

    # GC::Check if API call was successful
    if($response->resultFullRaw["is_success"] == true) {
        # Set success output
        $status = 'Current status: '. $transfer_status .' | Approval email: '. $approval_email .' | Message: '.$transfer_msg;
    } else {
        # Set error output
        $status = 'Transfer check failed. Error: '. $response->resultFullRaw["response_text"];
    }

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $request, $response->resultFullRaw, $response->resultFullRaw, $params);

    # Output status to browser
    echo '
    <table cellspacing="1" cellpadding="3" width="100%" border="0" class="datatable">
        <thead>
            <tr>
                <th>'. $status .'</th>
            </tr>
        </thead> 
    </table>
    ';

}



/* Added by BC : NG : 10-9-2014 : To sort domain based on priority which is inputted manually */
function sort_by_priority($a, $b) {
    return $a["priority"] - $b["priority"];
}
/* END : To sort domain based on priority which is inputted manually */

function opensrspro_getTransfersAway($params, $from = false, $to = false) {

    global $osrsLogError;
    $osrsLogError = "";

    $page = 1;
    $domains = array();

    $call = array(
        'func' => 'transGetAway',
        'data' => array(
            'limit' => '40',
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    if ($from)
        $call['data']['from'] = $from;
    if ($to)
        $call['data']['to'] = $to;

    do {
        $call['data']['page'] = $page;

        set_error_handler("osrsError", E_USER_WARNING);
        $return = processOpenSRS("array", $call);
        restore_error_handler();



        if (strcmp($return->resultFullRaw["is_success"], "1") != 0) {
            $osrsLogError .= $return->resultFullRaw["response_text"] . "\n";
            opensrspro_logModuleCall(__FUNCTION__, $call, $return->resultFullRaw, false, $params);
            return false;
        } else {
            if (isset($return->resultFullRaw["attributes"]['transfers']))
                $domains = array_merge($return->resultFullRaw["attributes"]['transfers'], $domains);
            $page++;
        }
    }while ($return->resultFullRaw["attributes"]['total'] > count($domains));

    $result = array();
    foreach ($domains as $dom) {
        $result[$dom['domain']] = $dom;
    }

    opensrspro_logModuleCall(__FUNCTION__, $call, $return->resultFullRaw, $result, $params);

    return $result;
}

function opensrspro_getTransfersIn($params, $from = false, $to = false) {

    global $osrsLogError;
    $osrsLogError = "";

    $page = 1;
    $domains = array();

    $call = array(
        'func' => 'transGetIn',
        'data' => array(
            'limit' => '40',
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    if ($from)
        $call['data']['from'] = $from;
    if ($to)
        $call['data']['to'] = $to;

    do {
        $call['data']['page'] = $page;

        set_error_handler("osrsError", E_USER_WARNING);
        $return = processOpenSRS("array", $call);
        restore_error_handler();

        if (strcmp($return->resultFullRaw["is_success"], "1") != 0) {
            $osrsLogError .= $return->resultFullRaw["response_text"] . "\n";
            opensrspro_logModuleCall(__FUNCTION__, $call, $return->resultFullRaw, false, $params);
            return false;
        } else {
            if (isset($return->resultFullRaw["attributes"]['transfers']))
                $domains = array_merge($return->resultFullRaw["attributes"]['transfers'], $domains);
            $page++;
        }
    }while ($return->resultFullRaw["attributes"]['total'] > count($domains));

    $result = array();
    foreach ($domains as $dom) {
        $result[$dom['domain']] = $dom;
    }
    opensrspro_logModuleCall(__FUNCTION__, $call, $return->resultFullRaw, $result, $params);
    return $result;
}

function opensrspro_getDomainsByExpiry($params, $from = '1970-01-01', $to = '2038-01-01') {

    global $osrsLogError;
    $osrsLogError = "";

    $page = 1;
    $domains = array();
    do {
        $call = array(
            'func' => 'lookupGetDomainsByExpiry',
            'data' => array(
                'limit' => '40',
                'page' => $page,
                'exp_from' => $from,
                'exp_to' => $to,
            ),
            'connect' => opensrspro_generateConnectData($params)
        );

        set_error_handler("osrsError", E_USER_WARNING);
        $return = processOpenSRS("array", $call);
        restore_error_handler();

        if (strcmp($return->resultFullRaw["is_success"], "1") != 0) {
            $osrsLogError .= $return->resultFullRaw["response_text"] . "\n";
            opensrspro_logModuleCall(__FUNCTION__, $call, $return->resultFullRaw, false, $params);
            return false;
        } else {
            if (isset($return->resultFullRaw["attributes"]['exp_domains']))
                $domains = array_merge($return->resultFullRaw["attributes"]['exp_domains'], $domains);
            $page++;
        }
    }while ($return->resultFullRaw["attributes"]['remainder']);

    $result = array();
    foreach ($domains as $dom) {
        $result[$dom['name']] = $dom;
    }
    opensrspro_logModuleCall(__FUNCTION__, $call, $return->resultFullRaw, $result, $params);
    return $result;
}

// Generates the User name based on domain name
function getDomainUser($tld, $sld) {
    /* Added by BC : RA : 5-7-2014 : To set request string at logModuleCall */
    $callArray = array('tld' => $tld, 'sld' => $sld);
    /* End : To set request string at logModuleCall */
    $sld = str_replace("xn--","", $sld);
    $domainUser = $sld . $tld;
    $domainUser = str_replace("-", "", $domainUser);
    $domainUser = str_replace(".", "", $domainUser);

    if (strlen($domainUser) > 20) {
        $domainUser = substr($domainUser, 0, 19);
    }
    
    /* Added by BC : NG : 27-6-2014 : For add log in getDomainUser function */ 
    opensrspro_logModuleCall(__FUNCTION__, $callArray, $domainUser, $domainUser, []);
    /* End : For add log in getDomainUser function */

    return $domainUser;
}

// Generates a secure password based on the domain name and a admin
// provided hash key.
function getDomainPass($tld, $sld, $hashKey) {
    /* Added by BC : RA : 5-7-2014 : To set request string at logModuleCall */
    $callArray = array('tld' => $tld, 'sld' => $sld, 'hashKey' => $hashKey);
    /* End : To set request string at logModuleCall */
    $domainPass = sha1(sha1($tld . $sld . $hashKey) . $hashKey);
    $domainPass = substr($domainPass, 0, 19);
    
     /* Added by BC : NG : 27-6-2014 : For add log in getDomainPass function */ 
    opensrspro_logModuleCall(__FUNCTION__, $callArray, $domainPass, $domainPass, []);
    /* End : For add log in getDomainPass function */

    return $domainPass;
}

// Grabs the cookie from OSRS, sends any errors back via the global error variables
function getCookie($domain, $domainUser, $domainPass, $params) {

    global $osrsLogError;
    global $osrsError;
    $cookie = false;


    $cookieCall = array(
        'func' => 'cookieSet',
        'data' => array(
            'domain' => $domain,
            'reg_username' => $domainUser,
            'reg_password' => $domainPass
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    set_error_handler("osrsError", E_USER_WARNING);

    $cookieReturn = processOpenSRS("array", $cookieCall);

    restore_error_handler();

    if (strcmp($cookieReturn->resultFullRaw["is_success"], "1") == 0) {
        $cookie = $cookieReturn->resultFullRaw["attributes"]["cookie"];
    } else {
        $osrsLogError = $cookieReturn->resultFullRaw["response_text"];
    }

    opensrspro_logModuleCall(__FUNCTION__, $cookieCall, $cookieReturn->resultFullRaw, $cookie, $params);

    return $cookie;
}

// Grabs the expiration year and sends back errors via the global error variables
function getExpirationYear($domain, $params) {

    global $osrsLogError;

    $expirationYear = false;

    $expirationCall = array(
        'func' => 'lookupGetDomain',
        'data' => array(
            'domain' => $domain,
            'type' => "all_info",
            'bypass' => $domain,
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    set_error_handler("osrsError", E_USER_WARNING);

    $expiryReturn = processOpenSRS("array", $expirationCall);

    restore_error_handler();

    if (strcmp($expiryReturn->resultFullRaw["is_success"], "1") == 0) {
        $expirationDate = $expiryReturn->resultFullRaw["attributes"]["expiredate"] ? $expiryReturn->resultFullRaw["attributes"]["expiredate"] : $expiryReturn->resultFullRaw["attributes"]["registry_expiredate"];
        $expirationDateArray = explode("-", $expirationDate);
        $expirationYear = $expirationDateArray[0];
    } else {
        $osrsLogError .= $expiryReturn->resultFullRaw["response_text"] . "\n";
    }

    opensrspro_logModuleCall(__FUNCTION__, $expirationCall, $expiryReturn->resultFullRaw, $expirationYear, $params);

    return $expirationYear;
}

// Takes any OSRS errors and
function osrsError($errno, $errstr, $errfile, $errline) {

    global $osrsError;
    global $osrsLogError;
    
    /* Added by BC : RA : 5-7-2014 : To set request string at logModuleCall */
    $callArray = array('Error No.' => $errno, 'Error String' => $errstr, 'Error File' => $errfile, 'Error Line' => $errline);
    /* End : To set request string at logModuleCall */


    // Error to be logged, includes file and error line.
    $osrsLogError .=$errstr . " " . " File: " . $errfile . " Line: " . $errline;

    // Error to be displayed to end user, only the error string itself.
    $osrsError.= $errstr . "<br />";
    
    /* Added by BC : RA : 5-7-2014 : To set request string at logModuleCall */ 
    $responseArray = array('osrsLogError' => $osrsLogError, 'osrsError' => $osrsError);
    opensrspro_logModuleCall(__FUNCTION__, $callArray, $responseArray, $responseArray, []);
    /* End : To set request string at logModuleCall */
}

// Checks call array for specific CCTLD requirements
function addCCTLDFields($params, $callArray) {
    
    /* Added by BC : RA : 5-7-2014 : To set request string at logModuleCall */
    $call = array('Params' => $params, 'Call Array' => $callArray);
    /* End : To set request string at logModuleCall */

    $tld = $params["tld"];

    // Puts eu and be language into call
    $lang = isset($params ['additionalfields'] ['Language']) ? $params ['additionalfields'] ['Language'] : 'en';

    if ($tld == 'eu' || $tld == 'be') {
        $callArray["owner"]["lang"] = $lang;
        $callArray["tech"]["lang"] = $lang;
        $callArray["admin"]["lang"] = $lang;
        $callArray["billing"]["lang"] = $lang;
        $callArray["data"]["lang"] = $lang;
        
        /* Changed by BC : NG : 11-2-2015 : To resolve issue for tld .EU domain registration */
        if($tld == 'eu'){
            $callArray["data"]["eu_country"] = $callArray["personal"]["country"];
        }
        else
        {
            $callArray["data"]["country"] = $callArray["owner"]["country"];                      
        }
        /* End : To resolve issue for tld .EU domain registration */
    }

    //.PRO
    if ($tld == "pro") {
        $callArray["professional_data"]["profession"] = $params["additionalfields"]["Profession"];
    }

    //.AU
    if ($tld == "com.au" || $tld == "net.au" || $tld == "org.au" || $tld == "asn.au" || $tld == "id.au") {
        $callArray["au_registrant_info"]["registrant_name"] = $params['additionalfields']['Registrant Name'];
        $callArray["au_registrant_info"]["registrant_id_type"] = $params['additionalfields']['Registrant ID Type'];
        $callArray["au_registrant_info"]["registrant_id"] = $params['additionalfields']['Registrant ID'];
        $callArray["au_registrant_info"]["eligibility_type"] = $params['additionalfields']['Eligibility Type'];
        $callArray["au_registrant_info"]["eligibility_name"] = $params['additionalfields']['Eligibility Name'];
        $callArray["au_registrant_info"]["eligibility_id_type"] = $params['additionalfields']['Eligibility ID Type'];
        $callArray["au_registrant_info"]["eligibility_id"] = $params['additionalfields']['Eligibility ID'];
        $callArray["au_registrant_info"]["policy_reason"] = $params['additionalfields']['Eligibility Reason'];
        $callArray["au_registrant_info"]["owner_confirm_address"] = $callArray["personal"]["email"];
    }

    // Pushes in owner confirm address for eu, be and de transfers
    if ($tld == "eu" || $tld == "be" || $tld == "de" || $tld == "it") { 
        if($tld == "de" || $tld == "eu" || $tld == "be")
        {
            $callArray["data"]["owner_confirm_address"] = $callArray["personal"]["email"];
        }
        else
        {
            $callArray["data"]["owner_confirm_address"] = $callArray["owner"]["email"];
        }
    }

    // Pushes Nexus information into call

    if ($tld == 'us') {

        $callArray["nexus"]["category"] = $params['additionalfields']['Nexus Category'];

        $usDomainPurpose = trim($params['additionalfields']['Application Purpose']);

        if (strtolower($usDomainPurpose) == strtolower('Business use for profit')) {
            $callArray["nexus"]["app_purpose"] = 'P1';
        } else if (strtolower($usDomainPurpose) == strtolower('Educational purposes')) {
            $callArray["nexus"]["app_purpose"] = 'P4';
        } else if (strtolower($usDomainPurpose) == strtolower('Personal Use')) {
            $callArray["nexus"]["app_purpose"] = 'P3';
        } else if (strtolower($usDomainPurpose) == strtolower('Government purposes')) {
            $callArray["nexus"]["app_purpose"] = 'P5';
        } else {
            $callArray["nexus"]["app_purpose"] = 'P2';
        }

        $callArray["nexus"]["validator"] = $params['additionalfields']['Nexus Country'];
    }

    //.DE
    if ($tld == 'de') {
        $callArray["data"]["zone_fax"] = $params['additionalfields']['Zone Contact Fax'];
    }

    //.COM.BR
    if ($tld == 'com.br') {
        $callArray["data"]["br_register_number"] = $params['additionalfields']['CPF/CNPJ number'];
    }

    // .asia
    if ($tld == 'asia') {

        $callArray["cedinfo"]["legal_entity_type"] = $params ['additionalfields'] ['Legal Type'];
        $callArray["cedinfo"]["id_type"] = $params ['additionalfields'] ['Identity Form'];
        $callArray["cedinfo"]["id_number"] = $params ['additionalfields'] ['Identity Number'];
        $callArray["cedinfo"]["contact_type"] = "owner";
         /* Changed by BC : NG : 9-9-2014 : To resolve issue of tld .asia registration error  */
        
        /*$callArray["cedinfo"]["locality_country"] = $callArray["owner"]["country"];*/
        $callArray["cedinfo"]["locality_country"] = $callArray["personal"]["country"];
    }
    
    //.SE & .NU
    /* Added by BC : NG : 7-10-2014 : To add support for tld .se and .nu in Domains Plugin */
    
    if($tld == 'se' || $tld == 'nu')
    {

        $callArray["registrant_extra_info"]["registrant_type"] = $params['additionalfields']['Legal Type'];
         switch ($params['additionalfields']['Legal Type']) {
            case "individual":
                $callArray["registrant_extra_info"]["id_card_number"] = $params['additionalfields']['Identification Number'];
                break;
            case "organization":
                $callArray["registrant_extra_info"]["registrant_vat_id"] = $params['additionalfields']['Vat Id'];
                $callArray["registrant_extra_info"]["registration_number"] = $params['additionalfields']['Identification Number'];
                break;
            default:
                break;
        }  
    }
    /* End : To add support for tld .se and .nu in Domains Plugin */

    //.IT
    if ($tld == 'it') {
        switch ($params['additionalfields'] ['Legal Type']) {
            case "Italian and foreign natural persons":
                $callArray["it_registrant_info"]["entity_type"] = "1";
                break;
            case "Companies/one man companies":
                $callArray["it_registrant_info"]["entity_type"] = "2";
                break;
            case "Freelance workers/professionals":
                $callArray["it_registrant_info"]["entity_type"] = "3";
                break;
            case "non-profit organizations":
                $callArray["it_registrant_info"]["entity_type"] = "4";
                break;
            case "public organizations":
                $callArray["it_registrant_info"]["entity_type"] = "5";
                break;
            case "other subjects":
                $callArray["it_registrant_info"]["entity_type"] = "6";
                break;
            case "non natural foreigners":
                $callArray["it_registrant_info"]["entity_type"] = "7";
                break;
        }
        $callArray["it_registrant_info"]["reg_code"] = $params ['additionalfields']['Tax ID'];
    }

    //.FR
    if ($tld == 'fr') {
        $callArray["data"]["registrant_type"] = strtolower($params['additionalfields']['Legal Type']);

        switch ($params['additionalfields']['Legal Type']) {
            case "Individual":
                $callArray["data"]["country_of_birth"] = $params['additionalfields']['Country of Birth'];
                $callArray["data"]["date_of_birth"] = $params['additionalfields']['Date of Birth'];
                $callArray["data"]["place_of_birth"] = $params['additionalfields']['Place of Birth'];
                $callArray["data"]["postal_code_of_birth"] = $params['additionalfields']['Postal Code of Birth'];
                break;
            case "Organization":
                $callArray["data"]["registrant_vat_id"] = $params['additionalfields']['VAT Number'];
                $callArray["data"]["trademark_number"] = $params['additionalfields']['Trademark Number'];
                $callArray["data"]["siren_siret"] = $params['additionalfields']['SIREN/SIRET Number'];
                break;
            default:
                break;
        }
    }
    // Pushes in all CA required information as well as checking to make sure
    // the province is in two letter format.
    if ($tld == 'ca') {

        $callArray["data"]["lang_pref"] = $lang;

        switch ($params['additionalfields'] ['Legal Type']) {
            case "Corporation":
                $callArray["data"]["legal_type"] = "CCO";
                break;
            case "Canadian Citizen":
                $callArray["data"]["legal_type"] = "CCT";
                break;
            case "Permanent Resident of Canada":
                $callArray["data"]["legal_type"] = "RES";
                break;
            case "Government":
                $callArray["data"]["legal_type"] = "GOV";
                break;
            case "Canadian Educational Institution":
                $callArray["data"]["legal_type"] = "EDU";
                break;
            case "Canadian Unincorporated Association":
                $callArray["data"]["legal_type"] = "ASS";
                break;
            case "Canadian Hospital":
                $callArray["data"]["legal_type"] = "HOP";
                break;
            case "Partnership Registered in Canada":
                $callArray["data"]["legal_type"] = "PRT";
                break;
            case "Trade-mark registered in Canada":
                $callArray["data"]["legal_type"] = "TDM";
                break;
            case "Canadian Trade Union":
                $callArray["data"]["legal_type"] = "TRD";
                break;
            case "Canadian Political Party":
                $callArray["data"]["legal_type"] = "PLT";
                break;
            case "Canadian Library Archive or Museum":
                $callArray["data"]["legal_type"] = "LAM";
                break;
            case "Trust established in Canada":
                $callArray["data"]["legal_type"] = "TRS";
                break;
            case "Aboriginal Peoples":
                $callArray["data"]["legal_type"] = "ABO";
                break;
            case "Legal Representative of a Canadian Citizen":
                $callArray["data"]["legal_type"] = "LGR";
                break;
            case "Official mark registered in Canada":
                $callArray["data"]["legal_type"] = "OMK";
                break;
            default:
                break;
        }

        if ($params['additionalfields'] ['Registrant Name'])
            $callArray["owner"]["org_name"] = $params['additionalfields'] ['Registrant Name'];
        $trademarkNum = $params['additionalfields'] ['Trademark Number'];

        $contacts = array("owner", "admin", "tech", "billing");

        // Checking for two letter format and reformatting if recognized
        foreach ($contacts as $contact) {
            if(!isset($callArray[$contact])){
                $callArray[$contact] = $callArray['personal'];
            }
            switch (strtolower($callArray[$contact]['state'])) {
                case "ontario":
                case "ont":
                case "ont.":
                case "on":
                    $callArray[$contact]['state'] = "ON";
                    break;
                case "alberta":
                case "alta":
                case "alta.":
                case "ab":
                    $callArray[$contact]['state'] = "AB";
                    break;
                case "quebec":
                case "que.":
                case "que":
                case "p.q.":
                case "qc":
                case "pq":
                    $callArray[$contact]['state'] = "QC";
                    break;
                case "nova scotia":
                case "n.s.":
                case "ns":
                    $callArray[$contact]['state'] = "NS";
                    break;
                case "new brunswick":
                case "n.b.":
                case "nb":
                    $callArray[$contact]['state'] = "NB";
                    break;
                case "manitoba":
                case "man":
                case "man.":
                case "mb":
                    $callArray[$contact]['state'] = "MB";
                    break;
                case "british columbia":
                case "b.c.":
                case "bc":
                    $callArray[$contact]['state'] = "BC";
                    break;

                case "prince edward island":
                case "p.e.i.":
                case "pei":
                case "pe":
                    $callArray[$contact]['state'] = "PE";
                    break;

                case "saskatchewan":
                case "sask":
                case "sask.":
                case "sk":
                    $callArray[$contact]['state'] = "SK";
                    break;

                case "newfoundland and labrador":
                case "newfoundland":
                case "labrador":
                case "nfld":
                case "nfld.":
                case "lab":
                case "lab.":
                case "nl":
                    $callArray[$contact]['state'] = "NL";
                    break;

                case "northwest territories":
                case "nt":
                    $callArray[$contact]['state'] = "NT";
                    break;

                case "yukon":
                case "yt":
                    $callArray[$contact]['state'] = "YT";
                    break;

                case "nunavut":
                case "nu":
                    $callArray[$contact]['state'] = "NU";
                    break;
                default:
                    trigger_error("Unable to recognize province for " . $contact . ", please provide in two letter format.", E_USER_WARNING);
                    break;
            }
        }

        $trademarkNum = trim($trademarkNum);

        if (empty($trademarkNum))
            $callArray["data"]["isa_trademark"] = "0";
        else
            $callArray["data"]["isa_trademark"] = "1";
    }

    # Module log
    opensrspro_logModuleCall(__FUNCTION__, $call, $callArray, $callArray, $params);

    return $callArray;
}

// Grabs username and password from OSRS if reseller account is allowed to do so.
function getDomainCredentials($domain, $params) {
    $domainCredentials = array();

    $credentialsCall = array(
        'func' => 'lookupGetUserAccessInfo',
        'data' => array(
            'domain' => $domain,
        ),
        'connect' => opensrspro_generateConnectData($params)
    );

    set_error_handler("osrsError", E_USER_WARNING);

    $credentialsReturn = processOpenSRS("array", $credentialsCall);

    restore_error_handler();

    if (strcmp($credentialsReturn->resultFullRaw["is_success"], "1") == 0) {
        $domainCredentials = $credentialsReturn->resultFullRaw["attributes"];
    } else {
        $osrsLogError = $credentialsReturn->resultFullRaw["response_text"];
    }

    opensrspro_logModuleCall(__FUNCTION__, $credentialsCall, $credentialsReturn->resultFullRaw, $domainCredentials, $params);

    return $domainCredentials;
}

// Checks to make sure the word reseller is not in an error message.  If it is,
// it will replace it with the general error.
function filterForResellerError($error, $generalError) {
    
     /* Added by BC : RA : 5-7-2014 : To set request string at logModuleCall */
    $callArray = array('error' => $error, 'generalError' => $generalError);
    /* End : To set request string at logModuleCall */

    $newError = "";

    if (preg_match("/\sreseller[\s\.,;\-:]/", $error) == 0)
        $newError = $error;
    else
        $newError = $generalError;
        
     /* Added by BC : NG : 27-6-2014 : For add log in filterForResellerError function */   
    if($error != "" && $generalError != "")
    {
        opensrspro_logModuleCall(__FUNCTION__, $callArray, $newError, $newError, []);
    }
    /* End : For add log in filterForResellerError function */

    return $newError;
}

?>
