#!/usr/bin/env node

/**
 * Admin Claim Tool - Set Firebase custom claims for role-based access control
 * 
 * Usage: node tools/setAdminClaim.ts <uid> <role>
 * Roles: owner, management, staff, analyst, user, pro
 * 
 * Environment variables required:
 * - FIREBASE_PROJECT_ID
 * - FIREBASE_CLIENT_EMAIL  
 * - FIREBASE_PRIVATE_KEY
 */

import admin from 'firebase-admin';
import type { Role } from '../shared/schema';

const VALID_ROLES: Role[] = ['owner', 'management', 'staff', 'analyst', 'user', 'pro'];

const ROLE_HIERARCHY: Record<Role, number> = {
  'owner': 4,
  'management': 3,
  'staff': 2,
  'analyst': 1,
  'pro': 0,
  'user': 0
};

async function setAdminClaim(uid: string, role: Role): Promise<void> {
  // Validate role
  if (!VALID_ROLES.includes(role)) {
    throw new Error(`Invalid role: ${role}. Valid roles: ${VALID_ROLES.join(', ')}`);
  }

  // Validate required environment variables
  const requiredEnvVars = ['FIREBASE_PROJECT_ID', 'FIREBASE_CLIENT_EMAIL', 'FIREBASE_PRIVATE_KEY'];
  const missingVars = requiredEnvVars.filter(varName => !process.env[varName]);
  
  if (missingVars.length > 0) {
    throw new Error(`Missing required environment variables: ${missingVars.join(', ')}`);
  }

  // Initialize Firebase Admin SDK
  if (!admin.apps.length) {
    const creds = {
      projectId: process.env.FIREBASE_PROJECT_ID!,
      clientEmail: process.env.FIREBASE_CLIENT_EMAIL!,
      privateKey: (process.env.FIREBASE_PRIVATE_KEY || "").replace(/\\n/g, "\n"),
    };

    admin.initializeApp({
      credential: admin.credential.cert(creds as any)
    });
  }

  try {
    // Get user to verify they exist
    const user = await admin.auth().getUser(uid);
    console.log(`Found user: ${user.email || user.uid}`);

    // Get current claims
    const currentClaims = user.customClaims || {};
    console.log(`Current claims:`, currentClaims);

    // Set the new role claim
    await admin.auth().setCustomUserClaims(uid, {
      ...currentClaims,
      role: role,
      roleLevel: ROLE_HIERARCHY[role],
      updatedAt: new Date().toISOString()
    });

    console.log(`✅ Successfully set role '${role}' (level ${ROLE_HIERARCHY[role]}) for user ${uid}`);
    
    // Verify the claim was set
    const updatedUser = await admin.auth().getUser(uid);
    console.log(`Verification - Updated claims:`, updatedUser.customClaims);
    
  } catch (error) {
    console.error(`❌ Failed to set role '${role}' for user ${uid}:`, error);
    throw error;
  }
}

async function getCurrentRole(uid: string): Promise<Role | null> {
  try {
    const user = await admin.auth().getUser(uid);
    const claims = user.customClaims || {};
    return claims.role as Role || null;
  } catch (error) {
    console.error(`Failed to get role for user ${uid}:`, error);
    return null;
  }
}

async function listUserRoles(): Promise<void> {
  console.log('Listing all users with roles...');
  
  let users: admin.auth.UserRecord[] = [];
  let nextPageToken: string | undefined;
  
  do {
    const result = await admin.auth().listUsers(1000, nextPageToken);
    users = users.concat(result.users);
    nextPageToken = result.pageToken;
  } while (nextPageToken);

  console.log(`Found ${users.length} total users`);
  
  const usersWithRoles = users
    .filter(user => user.customClaims?.role)
    .map(user => ({
      uid: user.uid,
      email: user.email || 'No email',
      role: user.customClaims?.role,
      roleLevel: user.customClaims?.roleLevel || ROLE_HIERARCHY[user.customClaims?.role as Role] || 0
    }))
    .sort((a, b) => b.roleLevel - a.roleLevel);

  if (usersWithRoles.length === 0) {
    console.log('No users with roles found.');
    return;
  }

  console.log('\nUsers with admin roles:');
  console.table(usersWithRoles);
}

// CLI Interface
async function main() {
  const args = process.argv.slice(2);
  
  if (args.length === 0 || args[0] === '--help' || args[0] === '-h') {
    console.log(`
Usage: 
  node tools/setAdminClaim.ts <uid> <role>     - Set role for user
  node tools/setAdminClaim.ts <uid>            - Get current role for user  
  node tools/setAdminClaim.ts --list           - List all users with roles

Roles (hierarchy level):
  owner (4)      - Full system access
  management (3) - Business management access  
  staff (2)      - Operations access
  analyst (1)    - Read-only analytics access
  user (0)       - Regular user (default)
  pro (0)        - Pro subscriber user

Examples:
  node tools/setAdminClaim.ts abc123 owner
  node tools/setAdminClaim.ts abc123 staff
  node tools/setAdminClaim.ts abc123
  node tools/setAdminClaim.ts --list
`);
    return;
  }

  if (args[0] === '--list') {
    await listUserRoles();
    return;
  }

  const uid = args[0];
  const role = args[1] as Role;

  if (!uid) {
    console.error('❌ User ID is required');
    process.exit(1);
  }

  if (role) {
    // Set role
    await setAdminClaim(uid, role);
  } else {
    // Get current role
    const currentRole = await getCurrentRole(uid);
    if (currentRole) {
      console.log(`User ${uid} current role: ${currentRole} (level ${ROLE_HIERARCHY[currentRole]})`);
    } else {
      console.log(`User ${uid} has no role set (defaults to 'user')`);
    }
  }
}

if (require.main === module) {
  main()
    .then(() => {
      console.log('\n✅ Operation completed');
      process.exit(0);
    })
    .catch((error) => {
      console.error('\n❌ Operation failed:', error.message);
      process.exit(1);
    });
}

export { setAdminClaim, getCurrentRole, listUserRoles };