// src/pages/BrandKit/BrandKitPage.tsx
import React, { useMemo, useState, useEffect, useRef } from "react";
import { withPro } from "@/lib/withPro";
import { ColorSwatch } from "@/components/ColorSwatch";
import {
  rollAll,
  rollOne,
  type Palette,
  type PaletteKey,
  type PalettePreset,
  applyPresetToPalette,
  rollWithinPreset,
} from "@/services/brand/palette";
import { FontPicker } from "@/components/FontPicker";
import RecraftGenerator from "@/components/RecraftGenerator";
import PerLetterColorText from "@/components/PerLetterColorText";
import {
  Download,
  Package,
  Wand2,
  MessageSquare,
  Copy,
  Check,
  ChevronRight,
  Palette as PaletteIcon,
  Monitor,
  CreditCard,
  AppWindow,
} from "lucide-react";
import { generate as generateSlogans, type SloganRequest } from "@/services/ai/slogan";
import { useToast } from "@/hooks/use-toast";
import { featureFlags } from "@/lib/featureFlags";
import JSZip from "jszip";

// ---------- Types ----------
type TextColorChoice = "black" | PaletteKey;
type LayoutOption = "side-by-side" | "top-bottom";
type WizardStep = 1 | 2 | 3 | 4;

// ---------- Seed data ----------
const POPULAR_INDUSTRIES = [
  "Restaurant", "Consulting", "Beauty", "Photography", "Fitness",
  "Real Estate", "Tech Startup", "E-commerce", "Education", "Healthcare",
  "Finance", "Non-profit", "Construction", "Marketing", "Coaching",
];

type StyleCard = { id: string; title: string; subtitle?: string; bg: string };
const STYLE_CARDS: StyleCard[] = [
  { id: "balanced",  title: "Balanced",      subtitle: "Geometric + Wordmark + Badge", bg: "bg-neutral-900" },
  { id: "minimal",   title: "Minimal",       subtitle: "Clean lines, lots of space",    bg: "bg-zinc-800" },
  { id: "bold",      title: "Bold",          subtitle: "Heavy weight, strong shapes",   bg: "bg-indigo-800" },
  { id: "monogram",  title: "Monogram",      subtitle: "Letter-focused marks",          bg: "bg-emerald-800" },
  { id: "badge",     title: "Badge",         subtitle: "Emblem styles",                 bg: "bg-sky-800" },
  { id: "playful",   title: "Playful",       subtitle: "Rounded, friendly",             bg: "bg-pink-700" },
  { id: "elegant",   title: "Elegant",       subtitle: "Serif, luxury vibes",           bg: "bg-stone-800" },
  { id: "tech",      title: "Techy",         subtitle: "Futuristic, angular",           bg: "bg-cyan-800" },
  { id: "organic",   title: "Organic",       subtitle: "Nature, hand-drawn",            bg: "bg-green-800" },
];

// LOOKA-style quick palette tiles (6 swatches per tile)
type PaletteTile = {
  id: string;
  name: string;
  swatches: [string, string, string, string, string, string]; // primary, secondary, accent, highlight, neutral, surface
  textLight?: string;
  textDark?: string;
};
const PALETTE_TILES: PaletteTile[] = [
  { id: "warm-1", name: "Warm 1", swatches: ["#D94841","#C15C2E","#F29F05","#F2C14E","#F3F4F6","#FFFFFF"], textLight:"#F8FAFC", textDark:"#111827" },
  { id: "cool-1", name: "Cool 1", swatches: ["#2563EB","#0EA5E9","#10B981","#A7F3D0","#F3F4F6","#FFFFFF"], textLight:"#F8FAFC", textDark:"#111827" },
  { id: "earth-1", name: "Earth", swatches: ["#6B7280","#92400E","#A16207","#65A30D","#E5E7EB","#FFFFFF"], textLight:"#F8FAFC", textDark:"#111827" },
  { id: "pastel-1", name: "Pastel", swatches: ["#A78BFA","#F472B6","#60A5FA","#86EFAC","#F3F4F6","#FFFFFF"], textLight:"#FDFDFE", textDark:"#111827" },
  { id: "mono-ink", name: "Mono Ink", swatches: ["#111827","#374151","#2563EB","#9CA3AF","#E5E7EB","#FFFFFF"], textLight:"#F8FAFC", textDark:"#0B1220" },
  { id: "neon-1", name: "Neon", swatches: ["#22D3EE","#22C55E","#EAB308","#A855F7","#111827","#0B1220"], textLight:"#F8FAFC", textDark:"#0B1220" },
];

// ---------- Helpers ----------
const tileToPalette = (tile: PaletteTile): Palette => ({
  primary: tile.swatches[0],
  secondary: tile.swatches[1],
  accent: tile.swatches[2],
  highlight: tile.swatches[3],
  neutral: tile.swatches[4],
  surface: tile.swatches[5],
  textLight: tile.textLight ?? "#F8FAFC",
  textDark: tile.textDark ?? "#1F2937",
});

// ---------- Page ----------
function BrandKitPage() {
  // Wizard
  const [step, setStep] = useState<WizardStep>(1);

  // Basics
  const [name, setName] = useState("");
  const canNextFrom1 = name.trim().length > 1;

  // Industry + Style inspo
  const [industry, setIndustry] = useState("");
  const [selectedStyles, setSelectedStyles] = useState<string[]>([]);
  const canNextFrom2 = industry.trim().length > 0;

  // Slogans
  const [slogans, setSlogans] = useState<string[]>([]);
  const [isGeneratingSlogans, setIsGeneratingSlogans] = useState(false);
  const [sloganError, setSloganError] = useState<string | null>(null);
  const { toast } = useToast();

  // Colors
  const [palette, setPalette] = useState<Palette>({
    primary: "#4274b9",
    secondary: "#b94c42",
    accent: "#42afb9",
    highlight: "#a78bfa",
    neutral: "#f3f4f6",
    surface: "#ffffff",
    textLight: "#f8fafc",
    textDark: "#1f2937",
  });
  const [locks, setLocks] = useState<Record<PaletteKey, boolean>>({
    primary: false,
    secondary: false,
    accent: false,
    highlight: false,
    neutral: true,
    surface: true,
    textLight: true,
    textDark: true,
  });
  const [preset, setPreset] = useState<PalettePreset>("warm");

  // Font + text color
  const [fontFamily, setFontFamily] = useState<string>(
    '"Inter", system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif'
  );
  const [textColor, setTextColor] = useState<TextColorChoice>("primary");
  const resolveTextColor = () => (textColor === "black" ? "#000000" : palette[textColor]);

  // Logo composition
  const [generatedSvgUrl, setGeneratedSvgUrl] = useState<string | null>(null);
  const [layout, setLayout] = useState<LayoutOption>("side-by-side");

  // Size controls
  const [textSize, setTextSize] = useState<number>(72);
  const [iconSize, setIconSize] = useState<number>(120);

  // Brand kit download
  const [isDownloadingKit, setIsDownloadingKit] = useState(false);

  const perLetterRef = useRef<any>(null);

  // Pre-fill from URL
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const businessName = urlParams.get("name");
    if (businessName) setName(decodeURIComponent(businessName));
    const ind = urlParams.get("industry");
    if (ind) setIndustry(decodeURIComponent(ind));
  }, []);

  // --- Slogans ---
  const handleGenerateSlogans = async () => {
    if (!name.trim()) {
      toast({
        title: "Brand name required",
        description: "Please enter a brand name first",
        variant: "destructive",
      });
      return;
    }
    setIsGeneratingSlogans(true);
    setSloganError(null);
    setSlogans([]);
    try {
      const request: SloganRequest = {
        brandName: name.trim(),
        tone: "Professional",
        maxResults: 3,
      };
      const response = await generateSlogans(request);
      setSlogans(response.suggestions);
    } catch (err) {
      console.error("Failed to generate slogans:", err);
      setSloganError("Failed to generate slogans. Please try again.");
    } finally {
      setIsGeneratingSlogans(false);
    }
  };
  const handleCopySlogan = async (slogan: string) => {
    try {
      await navigator.clipboard.writeText(slogan);
      toast({ title: "Copied!", description: "Slogan copied to clipboard" });
    } catch {
      toast({ title: "Failed to copy", description: "Could not copy to clipboard", variant: "destructive" });
    }
  };

  // --- Palette helpers ---
  const rollAllUnlocked = () => {
    const all = rollAll(palette);
    const next: Palette = { ...palette };
    (Object.keys(next) as PaletteKey[]).forEach((k) => {
      if (!locks[k]) next[k] = (all as any)[k];
    });
    setPalette(next);
  };
  const rollSingle = (key: PaletteKey) => { if (!locks[key]) setPalette((p) => rollOne(p, key)); };
  const toggleLock = (key: PaletteKey) => setLocks((l) => ({ ...l, [key]: !l[key] }));
  const lockAll = () => setLocks({
    primary: true, secondary: true, accent: true, highlight: true,
    neutral: true, surface: true, textLight: true, textDark: true,
  });
  const unlockAll = () => setLocks({
    primary: false, secondary: false, accent: false, highlight: false,
    neutral: false, surface: false, textLight: false, textDark: false,
  });
  const applyPreset = () => setPalette((p) => applyPresetToPalette(preset, p, locks));
  const rollPreset = () => setPalette((p) => rollWithinPreset(preset, p, locks));

  // --- Logo composition + brand kit ---
  const onLogoComplete = (svgUrl: string) => setGeneratedSvgUrl(svgUrl);

  const downloadComposedLogo = async () => {
    if (!generatedSvgUrl || !name.trim()) return;
    const canvas = document.createElement("canvas");
    const ctx = canvas.getContext("2d");
    if (!ctx) return;
    canvas.width = 1200; canvas.height = 800;
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    try {
      const svgImg = new Image();
      svgImg.crossOrigin = "anonymous";
      await new Promise((resolve, reject) => {
        svgImg.onload = resolve;
        svgImg.onerror = reject;
        svgImg.src = generatedSvgUrl;
      });
      const logoSize = iconSize;
      const padding = Math.max(40, Math.min(iconSize, textSize) * 0.5);
      ctx.fillStyle = resolveTextColor();
      ctx.font = `bold ${textSize}px ${fontFamily}`;
      ctx.textBaseline = layout === "side-by-side" ? "bottom" : "middle";
      const textMetrics = ctx.measureText(name);
      const textWidth = textMetrics.width;
      const textHeight = textSize;

      if (layout === "side-by-side") {
        const totalWidth = logoSize + textWidth + padding;
        const startX = Math.max(50, (canvas.width - totalWidth) / 2);
        const centerY = canvas.height / 2;
        ctx.drawImage(svgImg, startX, centerY - logoSize / 2, logoSize, logoSize);
        ctx.fillText(name, startX + logoSize + padding, centerY + logoSize / 2);
      } else {
        const totalHeight = logoSize + textHeight + padding;
        const startY = Math.max(50, (canvas.height - totalHeight) / 2);
        const centerX = canvas.width / 2;
        ctx.drawImage(svgImg, centerX - logoSize / 2, startY, logoSize, logoSize);
        ctx.textAlign = "center";
        ctx.fillText(name, centerX, startY + logoSize + padding + textHeight / 2);
      }

      const link = document.createElement("a");
      link.download = `${name.replace(/[^a-zA-Z0-9]/g, "-")}-composed-logo.png`;
      link.href = canvas.toDataURL("image/png");
      link.click();
    } catch (error) {
      console.error("Failed to create composed logo:", error);
    }
  };

  const downloadBrandKit = async () => {
    if (!name.trim()) { alert("Please enter a brand name first"); return; }
    setIsDownloadingKit(true);
    try {
      const zip = new JSZip();
      const brandFolderName = name.replace(/[^a-zA-Z0-9]/g, "-").toLowerCase();
      const brandInfo = {
        brandName: name,
        industry,
        styles: selectedStyles,
        createdAt: new Date().toISOString(),
        font: { family: fontFamily, textSize, textColor: resolveTextColor() },
        palette: { ...palette },
        settings: { preset, layout, iconSize },
      };
      zip.file(`${brandFolderName}/brand-info.json`, JSON.stringify(brandInfo, null, 2));

      const colorPalette = `
${name} - Color Palette
==============================
Primary:   ${palette.primary}
Secondary: ${palette.secondary}
Accent:    ${palette.accent}
Highlight: ${palette.highlight}
Neutral:   ${palette.neutral}
Surface:   ${palette.surface}
TextLight: ${palette.textLight}
TextDark:  ${palette.textDark}
`.trim();
      zip.file(`${brandFolderName}/color-palette.txt`, colorPalette);

      if (generatedSvgUrl) {
        try {
          const svgResponse = await fetch(generatedSvgUrl);
          const svgContent = await svgResponse.text();
          zip.file(`${brandFolderName}/icon.svg`, svgContent);
        } catch (e) { console.warn("Could not include SVG icon:", e); }
      }

      const zipBlob = await zip.generateAsync({ type: "blob" });
      const link = document.createElement("a");
      link.href = URL.createObjectURL(zipBlob);
      link.download = `${brandFolderName}-brand-kit.zip`;
      link.click();
      URL.revokeObjectURL(link.href);
    } catch (error) {
      console.error("Failed to create brand kit:", error);
      alert("Failed to create brand kit. Please try again.");
    } finally {
      setIsDownloadingKit(false);
    }
  };

  // ---------- UI helpers ----------
  const paletteEntries: { key: PaletteKey; label: string }[] = useMemo(
    () => [
      { key: "primary", label: "Primary" },
      { key: "secondary", label: "Secondary" },
      { key: "accent", label: "Accent" },
      { key: "highlight", label: "Highlight" },
      { key: "neutral", label: "Neutral" },
      { key: "surface", label: "Surface" },
      { key: "textLight", label: "Text (Light)" },
      { key: "textDark", label: "Text (Dark)" },
    ],
    []
  );

  const stepTitle = {
    1: "Let’s start with your brand name",
    2: "Pick your industry",
    3: "Pick some styles you like",
    4: "Build your brand kit",
  }[step];

  // ---------- Local components ----------
  const PaletteTileCard: React.FC<{ tile: PaletteTile; onApply: () => void }> = ({ tile, onApply }) => (
    <button
      onClick={onApply}
      className="group relative rounded-2xl border hover:border-neutral-300 transition p-3 text-left"
      title={`Apply ${tile.name}`}
    >
      <div className="grid grid-cols-6 gap-1 mb-2">
        {tile.swatches.map((hex, i) => (
          <div key={i} className="h-8 rounded" style={{ backgroundColor: hex }} />
        ))}
      </div>
      <div className="flex items-center gap-2 text-sm">
        <PaletteIcon className="w-4 h-4 text-neutral-500" />
        <span className="font-medium">{tile.name}</span>
      </div>
      <div className="absolute inset-0 rounded-2xl ring-0 group-hover:ring-1 ring-neutral-300 pointer-events-none" />
    </button>
  );

  const ComposedMark: React.FC<{ size?: "sm" | "md" }> = ({ size = "md" }) => (
    <div
      className={`flex ${layout === "side-by-side" ? "flex-row items-end gap-3" : "flex-col items-center gap-2"}`}
      style={{ color: resolveTextColor(), fontFamily, lineHeight: 1.1 }}
    >
      {generatedSvgUrl && (
        <img
          src={generatedSvgUrl}
          alt="Logo"
          className="object-contain"
          style={{ width: size === "sm" ? 48 : iconSize, height: size === "sm" ? 48 : iconSize }}
        />
      )}
      {name && (
        <span style={{ fontSize: size === "sm" ? 20 : textSize, fontWeight: 800, letterSpacing: "0.04em" }}>
          {name}
        </span>
      )}
    </div>
  );

  const Mockups: React.FC = () => (
    <div className="grid md:grid-cols-3 gap-4">
      {/* Business Card */}
      <div className="rounded-2xl border bg-white overflow-hidden">
        <div className="flex items-center gap-3 p-5" style={{ backgroundColor: palette.surface }}>
          <CreditCard className="w-4 h-4 text-neutral-400" />
          <div className="text-sm font-medium">Business Card</div>
        </div>
        <div className="p-6" style={{ background: `linear-gradient(180deg, ${palette.surface}, ${palette.neutral})` }}>
          <div className="rounded-xl border bg-white p-5 flex items-center justify-center">
            <ComposedMark size="sm" />
          </div>
          <div className="mt-4 grid grid-cols-2 gap-2 text-xs text-neutral-500">
            <div>Email</div><div>hello@brand.com</div>
            <div>Phone</div><div>(555) 555-5555</div>
          </div>
        </div>
      </div>

      {/* Website Header */}
      <div className="rounded-2xl border overflow-hidden">
        <div className="flex items-center gap-3 p-5 bg-white">
          <Monitor className="w-4 h-4 text-neutral-400" />
          <div className="text-sm font-medium">Website Header</div>
        </div>
        <div className="p-0">
          <div
            className="px-6 py-4 flex items-center justify-between"
            style={{ backgroundColor: palette.primary, color: palette.textLight }}
          >
            <div className="flex items-center gap-3">
              {generatedSvgUrl && <img src={generatedSvgUrl} alt="" className="w-7 h-7 object-contain" />}
              <span className="font-semibold">{name || "Brand"}</span>
            </div>
            <div className="flex gap-5 text-sm opacity-90">
              <span>Home</span><span>Products</span><span>Contact</span>
            </div>
          </div>
          <div className="p-6 bg-white text-sm text-neutral-600">
            Hero copy and CTA live here. Your brand colors and mark feel right at home.
          </div>
        </div>
      </div>

      {/* App Icon */}
      <div className="rounded-2xl border bg-white overflow-hidden">
        <div className="flex items-center gap-3 p-5" style={{ backgroundColor: palette.surface }}>
          <AppWindow className="w-4 h-4 text-neutral-400" />
          <div className="text-sm font-medium">App Icon</div>
        </div>
        <div className="p-6 flex items-center justify-center">
          <div
            className="rounded-2xl shadow-inner flex items-center justify-center"
            style={{ width: 96, height: 96, backgroundColor: palette.accent }}
          >
            {generatedSvgUrl ? (
              <img src={generatedSvgUrl} alt="" className="w-12 h-12 object-contain" />
            ) : (
              <span className="text-white font-bold">A</span>
            )}
          </div>
        </div>
      </div>
    </div>
  );

  // ---------- Render ----------
  return (
    <div className="max-w-6xl mx-auto p-4 md:p-6">
      <h1 className="text-2xl md:text-3xl font-semibold mb-4">Brand Kit Generator</h1>

      {/* Stepper */}
      <div className="flex items-center gap-3 mb-6">
        {[1, 2, 3, 4].map((n) => (
          <button
            key={n}
            className={`px-3 py-1.5 rounded-full text-sm border transition
              ${step === n ? "bg-neutral-900 text-white border-neutral-900" : "bg-white text-neutral-700 hover:bg-neutral-50"}`}
            onClick={() => setStep(n as WizardStep)}
          >
            Step {n}
          </button>
        ))}
      </div>

      {/* Step header */}
      <div className="flex items-start justify-between mb-2">
        <div>
          <h2 className="text-xl md:text-2xl font-semibold">{stepTitle}</h2>
          <p className="text-sm text-neutral-500 mt-1">
            {step === 1 && "We’ll use this name across your logo, fonts, and brand kit."}
            {step === 2 && "Knowing your industry helps suggest symbols, colors, and styles."}
            {step === 3 && "Choose 2–3 cards as inspiration. This guides the AI."}
            {step === 4 && "Tune colors, pick fonts, generate a logo, and compose your mark."}
          </p>
        </div>

        {/* Skip link for Step 3 */}
        {step === 3 && (
          <button
            onClick={() => setStep(4)}
            className="text-sm text-neutral-500 hover:text-neutral-800 underline"
          >
            Skip →
          </button>
        )}
      </div>

      {/* ---------- STEP 1: NAME ---------- */}
      {step === 1 && (
        <form
          className="space-y-6"
          onSubmit={(e) => { e.preventDefault(); if (canNextFrom1) setStep(2); }}
        >
          <input
            className="w-full px-4 py-3 rounded-md border text-lg"
            value={name}
            onChange={(e) => { setName(e.target.value); if (slogans.length) setSlogans([]); }}
            placeholder="Enter your brand name"
            autoFocus
          />

          {featureFlags.sloganGenerator && name.trim().length > 1 && (
            <div className="rounded-xl border bg-white p-4 space-y-4">
              <div className="flex items-center gap-2">
                <MessageSquare className="h-5 w-5 text-primary" />
                <h3 className="text-lg font-semibold">Generate Slogans</h3>
              </div>
              <p className="text-sm text-neutral-500">Quick ideas to shape your voice. (Saved into your kit.)</p>
              <button
                type="button"
                onClick={handleGenerateSlogans}
                disabled={isGeneratingSlogans || !name.trim()}
                className={`flex items-center gap-2 px-4 py-2 rounded-lg ${
                  isGeneratingSlogans ? "bg-neutral-200 text-neutral-500 cursor-not-allowed" : "bg-primary hover:bg-primary/90 text-white"
                }`}
              >
                {isGeneratingSlogans ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent" />
                    Generating...
                  </>
                ) : (
                  <>
                    <Wand2 className="h-4 w-4" />
                    Generate Slogans
                  </>
                )}
              </button>

              {sloganError && <div className="p-3 bg-red-50 border border-red-200 rounded-lg text-sm text-red-600">{sloganError}</div>}

              {slogans.length > 0 && (
                <div className="space-y-2">
                  {slogans.map((s, i) => (
                    <div key={i} className="flex items-center justify-between p-3 bg-neutral-50 rounded-lg border">
                      <span className="text-sm">“{s}”</span>
                      <button onClick={() => handleCopySlogan(s)} className="ml-3 p-1.5 text-neutral-500 hover:text-neutral-800">
                        <Copy className="h-4 w-4" />
                      </button>
                    </div>
                  ))}
                </div>
              )}
            </div>
          )}

          <div className="pt-2 flex justify-end">
            <button
              type="submit"
              disabled={!canNextFrom1}
              className={`px-4 py-2 rounded-lg flex items-center gap-2 ${
                canNextFrom1 ? "bg-indigo-600 hover:bg-indigo-700 text-white" : "bg-neutral-200 text-neutral-500 cursor-not-allowed"
              }`}
            >
              Continue <ChevronRight className="h-4 w-4" />
            </button>
          </div>
        </form>
      )}

      {/* ---------- STEP 2: INDUSTRY ---------- */}
      {step === 2 && (
        <form
          className="space-y-6"
          onSubmit={(e) => { e.preventDefault(); if (canNextFrom2) setStep(3); }}
        >
          <div>
            <input
              list="industries"
              className="w-full px-4 py-3 rounded-md border text-lg"
              value={industry}
              onChange={(e) => setIndustry(e.target.value)}
              placeholder="e.g., Consulting, Fitness, Photography..."
              autoFocus
            />
            <datalist id="industries">
              {POPULAR_INDUSTRIES.map((opt) => (
                <option key={opt} value={opt} />
              ))}
            </datalist>
            <p className="text-xs text-neutral-500 mt-2">
              Type to search or enter your own industry.
            </p>
          </div>

          <div className="flex items-center justify-between pt-2">
            <button type="button" onClick={() => setStep(1)} className="px-3 py-2 rounded-lg border hover:bg-neutral-50">
              ← Back
            </button>
            <button
              type="submit"
              className={`px-4 py-2 rounded-lg flex items-center gap-2 ${
                canNextFrom2 ? "bg-indigo-600 hover:bg-indigo-700 text-white" : "bg-neutral-200 text-neutral-500 cursor-not-allowed"
              }`}
              disabled={!canNextFrom2}
            >
              Continue <ChevronRight className="h-4 w-4" />
            </button>
          </div>
        </form>
      )}

      {/* ---------- STEP 3: STYLE INSPIRATION ---------- */}
      {step === 3 && (
        <div className="space-y-6">
          <div className="grid sm:grid-cols-2 md:grid-cols-3 gap-4">
            {STYLE_CARDS.map((card) => {
              const active = selectedStyles.includes(card.id);
              return (
                <button
                  key={card.id}
                  onClick={() =>
                    setSelectedStyles((prev) =>
                      prev.includes(card.id) ? prev.filter((id) => id !== card.id) : [...prev, card.id]
                    )
                  }
                  className={`relative rounded-2xl p-5 text-left border transition ${
                    active ? "ring-2 ring-indigo-500 border-indigo-500" : "border-neutral-200 hover:border-neutral-300"
                  }`}
                >
                  <div className={`h-28 rounded-xl ${card.bg} mb-4`} />
                  <div className="font-medium">{card.title}</div>
                  <div className="text-xs text-neutral-500">{card.subtitle}</div>
                  {active && (
                    <div className="absolute top-3 right-3 bg-white rounded-full p-1 border">
                      <Check className="w-4 h-4 text-indigo-600" />
                    </div>
                  )}
                </button>
              );
            })}
          </div>
          <p className="text-xs text-neutral-500">
            Tip: Picking <span className="font-medium">2–3</span> helps the generator aim your taste. Or{" "}
            <button className="underline" onClick={() => setSelectedStyles([])}>skip</button>.
          </p>

          <div className="flex items-center justify-between pt-2">
            <button onClick={() => setStep(2)} className="px-3 py-2 rounded-lg border hover:bg-neutral-50">← Back</button>
            <button
              onClick={() => setStep(4)}
              className="px-4 py-2 rounded-lg bg-indigo-600 hover:bg-indigo-700 text-white flex items-center gap-2"
            >
              Continue <ChevronRight className="h-4 w-4" />
            </button>
          </div>
        </div>
      )}

      {/* ---------- STEP 4: BUILD (Colors / Fonts / AI Logo / Compose / Mockups) ---------- */}
      {step === 4 && (
        <section className="space-y-6">
          {/* Quick Palettes */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="flex items-center gap-2">
              <PaletteIcon className="w-4 h-4 text-neutral-500" />
              <div className="text-sm font-semibold">Quick Palettes</div>
            </div>
            <div className="grid md:grid-cols-3 gap-3">
              {PALETTE_TILES.map((tile) => (
                <PaletteTileCard
                  key={tile.id}
                  tile={tile}
                  onApply={() => setPalette(tileToPalette(tile))}
                />
              ))}
            </div>
            <div className="text-xs text-neutral-500">Click a tile to apply instantly. You can refine below.</div>
          </div>

          {/* Advanced Palette Controls */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="flex items-center justify-between">
              <div className="text-sm font-semibold">Advanced Colors</div>
              <div className="flex flex-wrap items-center gap-2">
                <label className="text-sm text-neutral-700">Preset</label>
                <select
                  value={preset}
                  onChange={(e) => setPreset(e.target.value as PalettePreset)}
                  className="px-3 py-1.5 rounded-md border"
                >
                  <option value="warm">Warm</option>
                  <option value="cool">Cool</option>
                  <option value="earth">Earth Tones</option>
                  <option value="sepia">Sepia / Vintage</option>
                  <option value="pastel">Pastel</option>
                  <option value="neon">Neon</option>
                </select>
                <button onClick={applyPreset} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">Apply</button>
                <button onClick={rollPreset} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">🎲 Roll within</button>
                <span className="mx-2 h-5 w-px bg-neutral-300" />
                <button onClick={rollAllUnlocked} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">🎲 Roll all</button>
                <button onClick={lockAll} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">📌 Lock all</button>
                <button onClick={unlockAll} className="px-3 py-1.5 rounded-md border hover:bg-neutral-50">🔓 Unlock all</button>
              </div>
            </div>

            <div className="grid sm:grid-cols-2 md:grid-cols-3 gap-4">
              {paletteEntries.map(({ key, label }) => (
                <ColorSwatch
                  key={key}
                  label={label}
                  hex={palette[key]}
                  locked={locks[key]}
                  onChange={(v) => setPalette((p) => ({ ...p, [key]: v }))}
                  onRoll={() => rollSingle(key)}
                  onToggleLock={() => toggleLock(key)}
                />
              ))}
            </div>
            <div className="text-xs text-neutral-500">
              Neutral/Surface/Text are locked by default to keep accessibility solid.
            </div>
          </div>

          {/* Fonts */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Fonts</div>
            <FontPicker value={fontFamily} onChange={setFontFamily} sampleText={name || "Sample Text"} />
          </div>

          {/* Per-letter color editor */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Advanced: Per-letter color customization</div>
            <div className="text-xs text-neutral-500">Paint individual letters using your palette and export as SVG.</div>
            {name.trim().length > 0 ? (
              <PerLetterColorText
                ref={perLetterRef}
                key={`${name}-${textSize}`}
                text={name}
                fontFamily={fontFamily.replace(/["\']/g, "").split(",")[0].trim()}
                fontSize={textSize}
                palette={[
                  palette.primary, palette.secondary, palette.accent, palette.highlight,
                  palette.neutral, palette.surface, palette.textLight, palette.textDark,
                ]}
              />
            ) : (
              <div className="text-sm text-neutral-400 italic">Enter your brand name in Step 1 to use this editor.</div>
            )}
          </div>

          {/* AI Logo */}
          <div className="rounded-xl border bg-white p-4 space-y-2">
            <div className="text-sm font-semibold">Generate AI-powered SVG logo</div>
            <div className="text-xs text-neutral-500 mb-2">
              We’ll use your name{industry ? ` and ${industry}` : ""}{selectedStyles.length ? ` + ${selectedStyles.join(", ")}` : ""} as context.
            </div>
            <RecraftGenerator businessName={name || "Sample Business"} className="w-full" onComplete={onLogoComplete} />
          </div>

          {/* Compose */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Compose your logo</div>
            <div className="text-xs text-neutral-500">Combine your icon and wordmark. Adjust sizes below.</div>

            <div className="space-y-2">
              <div className="text-xs font-medium text-neutral-700">Layout</div>
              <div className="flex gap-2">
                <button
                  onClick={() => setLayout("side-by-side")}
                  className={`px-3 py-2 rounded-lg border text-sm ${layout === "side-by-side" ? "border-blue-500 bg-blue-50 text-blue-700" : "border-neutral-300 hover:bg-neutral-50"}`}
                >
                  Side by Side
                </button>
                <button
                  onClick={() => setLayout("top-bottom")}
                  className={`px-3 py-2 rounded-lg border text-sm ${layout === "top-bottom" ? "border-blue-500 bg-blue-50 text-blue-700" : "border-neutral-300 hover:bg-neutral-50"}`}
                >
                  Top/Bottom
                </button>
              </div>
            </div>

            <div
              className={`border rounded-lg p-6 min-h-32 flex justify-center ${layout === "side-by-side" ? "flex-row items-end" : "flex-col gap-4 items-center"}`}
              style={{ backgroundColor: palette.surface, gap: layout === "side-by-side" ? `${textSize * 0.1}px` : undefined }}
            >
              {generatedSvgUrl && (
                <div className="rounded-lg overflow-hidden flex-shrink-0" style={{ width: `${iconSize}px`, height: `${iconSize}px` }}>
                  <img src={generatedSvgUrl} alt="Generated Logo" className="w-full h-full object-contain" />
                </div>
              )}
              {name.trim().length > 0 && (
                <div className="font-bold" style={{ fontFamily, color: resolveTextColor(), fontSize: `${textSize}px`, lineHeight: 1.1, letterSpacing: "0.05em" }}>
                  {name}
                </div>
              )}
              {!generatedSvgUrl && name.trim().length === 0 && (
                <div className="text-neutral-400 text-sm">Generate a logo and enter your business name to see preview</div>
              )}
            </div>

            {/* Sliders */}
            <div className="space-y-4">
              <div>
                <div className="text-xs text-neutral-600 mb-2">Text size: {textSize}px</div>
                <input
                  type="range" min="24" max="200" step="2" value={textSize}
                  onChange={(e) => setTextSize(parseInt(e.target.value))}
                  className="w-full h-2 bg-neutral-200 rounded-lg appearance-none cursor-pointer"
                />
              </div>
              {generatedSvgUrl && (
                <div>
                  <div className="text-xs text-neutral-600 mb-2">Icon size: {iconSize}px</div>
                  <input
                    type="range" min="32" max="300" step="4" value={iconSize}
                    onChange={(e) => setIconSize(parseInt(e.target.value))}
                    className="w-full h-2 bg-neutral-200 rounded-lg appearance-none cursor-pointer"
                  />
                </div>
              )}
            </div>

            {/* Actions */}
            <div className="flex flex-wrap gap-3">
              {(generatedSvgUrl && name.trim().length > 0) && (
                <button
                  onClick={downloadComposedLogo}
                  className="flex items-center gap-2 px-4 py-2 rounded-lg bg-[#05445e] hover:bg-[#033a4a] text-white transition-colors"
                >
                  <Download className="w-4 h-4" />
                  Download Logo
                </button>
              )}
              {name.trim().length > 0 && (
                <button
                  onClick={downloadBrandKit}
                  disabled={isDownloadingKit}
                  className="flex items-center gap-2 px-4 py-2 rounded-lg bg-[#f99f1b] hover:bg-[#e88e06] text-black transition-colors disabled:opacity-50"
                >
                  <Package className="w-4 h-4" />
                  {isDownloadingKit ? "Creating Brand Kit..." : "Download Brand Kit"}
                </button>
              )}
            </div>
          </div>

          {/* Mockups */}
          <div className="rounded-xl border bg-white p-4 space-y-4">
            <div className="text-sm font-semibold">Brand Mockups</div>
            <Mockups />
          </div>

          {/* Back */}
          <div className="flex items-center gap-2">
            <button onClick={() => setStep(3)} className="px-3 py-2 rounded-lg border hover:bg-neutral-50">← Back</button>
          </div>
        </section>
      )}
    </div>
  );
}

export default withPro(BrandKitPage, {
  feature: "brand kit generation",
  description: "Generate complete brand kits with logos, colors, fonts, and slogans",
});
