import { sql } from "drizzle-orm";
import { pgTable, pgEnum, text, varchar, timestamp, jsonb, boolean, integer, numeric, unique, index, check } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// Role types for the admin system
export type Role = 'owner' | 'management' | 'manager' | 'creator' | 'staff' | 'analyst' | 'user' | 'pro';

// Cancellation reason constants to prevent string fragility
export const CANCELLATION_REASONS = {
  TOO_EXPENSIVE: 'Too expensive',
  DIDNT_GET_VALUE: 'Didn\'t get value', 
  MISSING_FEATURE: 'Missing feature',
  TECHNICAL_ISSUES: 'Technical issues',
  TEMPORARY_PAUSE: 'Temporary pause',
  OTHER: 'Other'
} as const;

export const CANCELLATION_REASON_VALUES = Object.values(CANCELLATION_REASONS);

// Domain credit status constants
export const DOMAIN_CREDIT_STATUSES = {
  AVAILABLE: 'available',
  USED: 'used',
  EXPIRED: 'expired',
  REVOKED: 'revoked'
} as const;

export const DOMAIN_CREDIT_STATUS_VALUES = Object.values(DOMAIN_CREDIT_STATUSES);

// Cover template category constants
export const COVER_TEMPLATE_CATEGORIES = {
  BUSINESS: 'business',
  CREATIVE: 'creative',
  MINIMAL: 'minimal',
  MODERN: 'modern',
  PROFESSIONAL: 'professional'
} as const;

export const COVER_TEMPLATE_CATEGORY_VALUES = Object.values(COVER_TEMPLATE_CATEGORIES);

// Cover template top tier industry constants
export const COVER_TEMPLATE_TOP_TIERS = {
  FIG: 'FIG',
  TMT: 'TMT',
  HEALTHCARE_PHARMA: 'Healthcare/Pharma',
  GENERAL: 'General'
} as const;

export const COVER_TEMPLATE_TOP_TIER_VALUES = Object.values(COVER_TEMPLATE_TOP_TIERS);

// Cover template approval status constants
export const COVER_TEMPLATE_APPROVAL_STATUSES = {
  PENDING: 'pending',
  APPROVED: 'approved',
  REJECTED: 'rejected'
} as const;

export const COVER_TEMPLATE_APPROVAL_STATUS_VALUES = Object.values(COVER_TEMPLATE_APPROVAL_STATUSES);

// Cover purchase status constants
export const COVER_PURCHASE_STATUSES = {
  PENDING: 'pending',
  PAID: 'paid',
  DELIVERED: 'delivered',
  FAILED: 'failed'
} as const;

export const COVER_PURCHASE_STATUS_VALUES = Object.values(COVER_PURCHASE_STATUSES);

// Creator Marketplace constants for data integrity - read from environment variables
export const CREATOR_MIN_PRICE_CENTS = 199; // $1.99 minimum
export const CREATOR_MAX_PRICE_CENTS = 4999; // $49.99 maximum

// Creator onboarding status constants
export const CREATOR_ONBOARDING_STATUSES = {
  PENDING: 'pending',
  IN_PROGRESS: 'in_progress', 
  COMPLETED: 'completed',
  REJECTED: 'rejected'
} as const;

export const CREATOR_ONBOARDING_STATUS_VALUES = Object.values(CREATOR_ONBOARDING_STATUSES);

// Creator asset approval status constants
export const CREATOR_APPROVAL_STATUSES = {
  PENDING: 'pending',
  APPROVED: 'approved',
  REJECTED: 'rejected',
  CANCELLED: 'cancelled'
} as const;

export const CREATOR_APPROVAL_STATUS_VALUES = Object.values(CREATOR_APPROVAL_STATUSES);

// Creator earning type constants
export const CREATOR_EARNING_TYPES = {
  SALE: 'sale',
  COMMISSION: 'commission',
  BONUS: 'bonus',
  PAYOUT: 'payout'
} as const;

export const CREATOR_EARNING_TYPE_VALUES = Object.values(CREATOR_EARNING_TYPES);

// Creator payout status constants
export const CREATOR_PAYOUT_STATUSES = {
  PENDING: 'pending',
  PROCESSING: 'processing',
  PAID: 'paid',
  FAILED: 'failed'
} as const;

export const CREATOR_PAYOUT_STATUS_VALUES = Object.values(CREATOR_PAYOUT_STATUSES);

// Database enums for Creator Marketplace - these enforce values at the DB level
export const creatorOnboardingStatusEnum = pgEnum('creator_onboarding_status', CREATOR_ONBOARDING_STATUS_VALUES as [string, ...string[]]);
export const creatorApprovalStatusEnum = pgEnum('creator_approval_status', CREATOR_APPROVAL_STATUS_VALUES as [string, ...string[]]);
export const creatorEarningTypeEnum = pgEnum('creator_earning_type', CREATOR_EARNING_TYPE_VALUES as [string, ...string[]]);
export const creatorPayoutStatusEnum = pgEnum('creator_payout_status', CREATOR_PAYOUT_STATUS_VALUES as [string, ...string[]]);

// Database enums for Cover Templates
export const coverTemplateCategoryEnum = pgEnum('cover_template_category', COVER_TEMPLATE_CATEGORY_VALUES as [string, ...string[]]);
export const coverTemplateTopTierEnum = pgEnum('cover_template_top_tier', COVER_TEMPLATE_TOP_TIER_VALUES as [string, ...string[]]);
export const coverTemplateApprovalStatusEnum = pgEnum('cover_template_approval_status', COVER_TEMPLATE_APPROVAL_STATUS_VALUES as [string, ...string[]]);
export const coverPurchaseStatusEnum = pgEnum('cover_purchase_status', COVER_PURCHASE_STATUS_VALUES as [string, ...string[]]);

export const users = pgTable("users", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  firebaseUid: text("firebase_uid").notNull().unique(),
  email: text("email").notNull().unique(),
  displayName: text("display_name").notNull(),
  company: text("company"),
  avatarUrl: text("avatar_url"),
  isPaid: boolean("is_paid").default(false),
  role: text("role").default("user"),
  createdAt: timestamp("created_at").defaultNow(),
  proActivatedAt: timestamp("pro_activated_at"),
  proWelcomeDismissed: boolean("pro_welcome_dismissed").default(false),
  subscriptionStatus: text("subscription_status"),
  pausedAt: timestamp("paused_at", { withTimezone: true }),
  resumeAt: timestamp("resume_at", { withTimezone: true }),
  // TOTP (Two-Factor Authentication) fields
  totpSecret: text("totp_secret"), // Encrypted TOTP secret key
  totpEnabled: boolean("totp_enabled").default(false), // Whether TOTP is enabled
  totpBackupCodes: text("totp_backup_codes").array(), // Array of backup recovery codes
  // Quota tracking fields for subscription download limits
  monthlyDownloadQuota: integer("monthly_download_quota").default(0), // Max downloads per month based on subscription
  currentMonthDownloads: integer("current_month_downloads").default(0), // Downloads used this month
  quotaResetDate: timestamp("quota_reset_date", { withTimezone: true }), // When quota resets (monthly cycle)
  subscriptionLookupKey: text("subscription_lookup_key"), // Current subscription's lookup_key for quota mapping
});

export const brandKits = pgTable("brand_kits", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  name: text("name").notNull(),
  businessName: text("business_name"),
  industry: text("industry"),
  personality: text("personality"),
  colors: jsonb("colors"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const businessNames = pgTable("business_names", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  name: text("name").notNull(),
  industry: text("industry"),
  keywords: text("keywords"),
  isSaved: boolean("is_saved").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const socialMediaKits = pgTable("social_media_kits", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  name: text("name").notNull(),
  platforms: jsonb("platforms"),
  templateCount: integer("template_count").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const templateCategories = pgTable("template_categories", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  description: text("description"),
  slug: text("slug").notNull().unique(),
  displayOrder: integer("display_order").default(0),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
});

export const templateConfigurations = pgTable("template_configurations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  templateId: varchar("template_id").notNull(),
  name: text("name").notNull(),
  description: text("description"),
  configType: text("config_type").notNull(), // 'color', 'font', 'layout', 'style'
  defaultValue: jsonb("default_value"),
  options: jsonb("options"), // Available configuration options
  isRequired: boolean("is_required").default(false),
  displayOrder: integer("display_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const templateContent = pgTable("template_content", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  templateId: varchar("template_id").notNull(),
  sectionName: text("section_name").notNull(),
  componentType: text("component_type").notNull(), // 'header', 'hero', 'content', 'footer', etc.
  content: jsonb("content").notNull(), // Actual content data (text, images, etc.)
  structure: jsonb("structure").notNull(), // HTML structure/layout
  styles: jsonb("styles"), // CSS styles specific to this section
  displayOrder: integer("display_order").default(0),
  isRequired: boolean("is_required").default(false),
  isCustomizable: boolean("is_customizable").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const userTemplateCustomizations = pgTable("user_template_customizations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  templateId: varchar("template_id").notNull(),
  customizationName: text("customization_name"),
  configurations: jsonb("configurations").notNull(), // User's configuration choices
  contentOverrides: jsonb("content_overrides"), // User's content modifications
  styleOverrides: jsonb("style_overrides"), // User's style modifications
  isTemplate: boolean("is_template").default(false), // If user wants to save as new template
  isFavorite: boolean("is_favorite").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const templateAssets = pgTable("template_assets", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  templateId: varchar("template_id").notNull(),
  assetType: text("asset_type").notNull(), // 'preview', 'thumbnail', 'demo', 'icon'
  fileName: text("file_name").notNull(),
  fileUrl: text("file_url").notNull(),
  fileSize: integer("file_size"),
  mimeType: text("mime_type"),
  altText: text("alt_text"),
  displayOrder: integer("display_order").default(0),
  createdAt: timestamp("created_at").defaultNow(),
});

export const websiteTemplates = pgTable("website_templates", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").references(() => users.id), // Made optional for system templates
  categoryId: varchar("category_id").references(() => templateCategories.id),
  name: text("name").notNull(),
  description: text("description"),
  templateType: text("template_type").notNull(),
  difficulty: text("difficulty").default("beginner"), // 'beginner', 'intermediate', 'advanced'
  industry: text("industry"), // Target industry
  tags: text("tags").array(), // Searchable tags
  demoUrl: text("demo_url"),
  url: text("url"),
  isLive: boolean("is_live").default(false),
  isSystemTemplate: boolean("is_system_template").default(false), // Built-in vs user-created
  isFeatured: boolean("is_featured").default(false),
  isActive: boolean("is_active").default(true),
  downloadCount: integer("download_count").default(0),
  rating: numeric("rating", { precision: 3, scale: 2 }), // Average rating
  ratingCount: integer("rating_count").default(0),
  price: numeric("price", { precision: 10, scale: 2 }).default('0'), // For premium templates
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const notifications = pgTable("notifications", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  message: text("message").notNull(),
  type: text("type").notNull(), // 'info', 'success', 'warning', 'error', 'pro_welcome', 'system'
  isRead: boolean("is_read").default(false),
  createdAt: timestamp("created_at").defaultNow(),
});

export const cancellations = pgTable("cancellations", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  firebaseUid: text("firebase_uid").notNull(),
  email: text("email").notNull(),
  reason: text("reason").notNull(), // 'Too expensive', 'Didn\'t get value', 'Missing feature', 'Technical issues', 'Temporary pause', 'Other'
  note: text("note"), // Optional comments
  createdAt: timestamp("created_at").defaultNow(),
});

export const pauses = pgTable("pauses", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  uid: text("uid").notNull(), // Firebase UID
  email: text("email"),
  stripeSubId: text("stripe_sub_id").notNull(),
  months: integer("months").notNull(),
  reason: text("reason"),
  note: text("note"),
  resumeAt: timestamp("resume_at", { withTimezone: true }).notNull(),
  createdAt: timestamp("created_at", { withTimezone: true }).defaultNow(),
});

export const auditLogs = pgTable("audit_logs", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  adminUserId: varchar("admin_user_id").notNull().references(() => users.id),
  adminRole: text("admin_role").notNull(),
  action: text("action").notNull(),
  targetType: text("target_type"), // 'user', 'brandkit', 'coupon', etc.
  targetId: text("target_id"), // ID of the affected entity
  details: jsonb("details"), // Additional action details
  ipAddress: text("ip_address"),
  userAgent: text("user_agent"),
  createdAt: timestamp("created_at").defaultNow(),
});

export const domainOrders = pgTable("domain_orders", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  domain: text("domain").notNull(),
  years: integer("years").notNull().default(1),
  privacy: boolean("privacy").default(true),
  priceCents: integer("price_cents").notNull(),
  status: text("status").notNull().default("pending"), // 'pending', 'registering', 'paid', 'active', 'failed', 'refunded'
  stripeSessionId: text("stripe_session_id"),
  providerRegId: text("provider_reg_id"), // OpenSRS registration ID
  nameservers: text("nameservers").array(),
  expiresAt: timestamp("expires_at"),
  registrantContact: jsonb("registrant_contact").notNull(),
  errorMessage: text("error_message"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const domainCredits = pgTable("domain_credits", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  status: text("status").notNull().default("available"), // 'available', 'used', 'expired', 'revoked'
  capCents: integer("cap_cents").notNull(), // Maximum price this credit can cover in cents
  eligibleTlds: jsonb("eligible_tlds").notNull(), // Array of TLDs this credit can be used for
  issuedAt: timestamp("issued_at").defaultNow(),
  expiresAt: timestamp("expires_at").notNull(),
  usedDomain: text("used_domain"), // Domain this credit was used for (if status is 'used')
  usedAt: timestamp("used_at"), // When this credit was used
  sourceSubscriptionId: text("source_subscription_id").unique(), // Stripe subscription ID that generated this credit (UNIQUE to prevent duplicates)
});

export const domains = pgTable("domains", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  domain: text("domain").notNull().unique(),
  tld: text("tld").notNull(), // Top-level domain (com, net, org, etc.)
  registeredAt: timestamp("registered_at").defaultNow(),
  expiresAt: timestamp("expires_at").notNull(),
  autorenew: boolean("autorenew").default(true),
  registrar: text("registrar").notNull().default("OpenSRS"),
  premium: boolean("premium").default(false), // Whether this is a premium domain
  subscriptionLinked: boolean("subscription_linked").default(false), // Whether linked to a subscription
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const importedIcons = pgTable("imported_icons", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: text("name").notNull(),
  svg: text("svg").notNull(),
  style: text("style").notNull().default("flat"), // Icon style
  tags: text("tags").array().default([]), // Searchable tags  
  importedBy: varchar("imported_by").notNull().references(() => users.id), // Admin who imported
  createdAt: timestamp("created_at").defaultNow(),
});

export const visitorSessions = pgTable("visitor_sessions", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  sessionId: text("session_id").notNull(), // Unique session identifier (hash of IP + user agent + date)
  ipHash: text("ip_hash").notNull(), // Salted hash of IP address (privacy-compliant)
  country: text("country"), // Country name from geolocation
  countryCode: text("country_code"), // ISO country code  
  city: text("city"), // City name from geolocation
  latitude: numeric("latitude", { precision: 10, scale: 7 }), // Lat/lng for map positioning
  longitude: numeric("longitude", { precision: 10, scale: 7 }),
  userAgent: text("user_agent"), // For analytics (device/browser type)
  pageViews: integer("page_views").default(1), // Number of page views in this session
  firstPageUrl: text("first_page_url"), // First page visited in session
  lastPageUrl: text("last_page_url"), // Last page visited in session
  referrer: text("referrer"), // HTTP referrer
  isPublicPage: boolean("is_public_page").default(true), // Only track public pages
  date: text("date").notNull(), // YYYY-MM-DD for daily unique tracking
  createdAt: timestamp("created_at").defaultNow(), // First visit time
  updatedAt: timestamp("updated_at").defaultNow(), // Last activity time
}, (table) => ({
  // Unique constraint for session ID to prevent duplicates
  sessionIdUnique: unique("visitor_sessions_session_id_unique").on(table.sessionId),
}));

export const dailyVisitorStats = pgTable("daily_visitor_stats", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  date: text("date").notNull(), // YYYY-MM-DD format for easy querying
  uniqueVisitors: integer("unique_visitors").default(0), // Count of unique sessions
  totalPageViews: integer("total_page_views").default(0), // Total page views for the day
  countries: jsonb("countries").default(sql`'{}'`), // Object with country counts: { "US": 10, "CA": 5 }
  topPages: jsonb("top_pages").default(sql`'{}'`), // Object with page view counts
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  // Unique constraint on date to prevent duplicates
  dateUnique: unique().on(table.date),
}));

// Cart Items table for shopping cart functionality
export const cartItems = pgTable("cart_items", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").references(() => users.id),
  sessionId: text("session_id"), // For unauthenticated users
  itemType: text("item_type").notNull(), // 'stock_photo', 'subscription'
  itemId: text("item_id").notNull(), // Stock photo ID or subscription plan ID
  itemName: text("item_name").notNull(),
  itemPrice: integer("item_price_cents").notNull(), // Price in cents
  quantity: integer("quantity").default(1),
  metadata: jsonb("metadata"), // Additional item data (preview URL, etc.)
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Purchases table to track completed purchases
export const purchases = pgTable("purchases", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").references(() => users.id),
  stripeSessionId: text("stripe_session_id").notNull().unique(), // Stripe checkout session ID
  stripePaymentIntentId: text("stripe_payment_intent_id"),
  status: text("status").notNull().default("pending"), // 'pending', 'completed', 'failed', 'refunded'
  totalAmountCents: integer("total_amount_cents").notNull(),
  items: jsonb("items").notNull(), // Array of purchased items
  metadata: jsonb("metadata"), // Additional purchase data
  createdAt: timestamp("created_at").defaultNow(),
  completedAt: timestamp("completed_at"),
});

// User Entitlements table to track what users have access to
export const userEntitlements = pgTable("user_entitlements", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id),
  entitlementType: text("entitlement_type").notNull(), // 'stock_photo', 'subscription', 'icons_no_attribution', 'icon_pack'
  entitlementId: text("entitlement_id").notNull(), // Stock photo ID or subscription plan
  purchaseId: varchar("purchase_id").references(() => purchases.id),
  subscriptionId: text("subscription_id"), // Stripe subscription ID if from subscription
  status: text("status").notNull().default("active"), // 'active', 'expired', 'revoked'
  expiresAt: timestamp("expires_at"), // For time-limited entitlements
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Creator Marketplace tables for creator onboarding, asset management, and earnings tracking

// Creators table to store creator profile and Stripe Connect information
export const creators = pgTable("creators", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().unique().references(() => users.id, { onDelete: 'cascade' }), // Unique constraint ensures one creator per user
  stripeConnectAccountId: text("stripe_connect_account_id").unique(), // Stripe Connect account ID
  onboardingStatus: creatorOnboardingStatusEnum("onboarding_status").notNull().default("pending"), // Database-enforced enum
  onboardingCompletedAt: timestamp("onboarding_completed_at"),
  profileData: jsonb("profile_data"), // Creator bio, portfolio links, etc.
  payoutEnabled: boolean("payout_enabled").default(false),
  totalEarnings: integer("total_earnings_cents").default(0), // Total earnings in cents
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  // Indexes for performance
  userIdIdx: index("creators_user_id_idx").on(table.userId),
  onboardingStatusIdx: index("creators_onboarding_status_idx").on(table.onboardingStatus),
  payoutEnabledIdx: index("creators_payout_enabled_idx").on(table.payoutEnabled),
  isActiveIdx: index("creators_is_active_idx").on(table.isActive),
}));

// Assets table for general marketplace assets
export const assets = pgTable("assets", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  creatorId: varchar("creator_id").notNull().references(() => creators.id, { onDelete: 'cascade' }),
  fileName: text("file_name").notNull(),
  originalFileName: text("original_file_name").notNull(),
  fileUrl: text("file_url").notNull(), // Private original file path
  previewUrl: text("preview_url"), // Public watermarked preview path
  fileSize: integer("file_size").notNull(), // File size in bytes
  mimeType: text("mime_type").notNull(),
  assetType: text("asset_type").notNull(), // 'image', 'video', 'audio', 'document', 'template'
  category: text("category"), // Asset category for organization
  tags: text("tags").array().default([]), // Searchable tags
  metadata: jsonb("metadata"), // Additional asset metadata (dimensions, duration, etc.)
  isPublic: boolean("is_public").default(true),
  downloadCount: integer("download_count").default(0),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  // Indexes for performance - removed redundant ownershipUnique as it's not needed
  creatorIdIdx: index("assets_creator_id_idx").on(table.creatorId),
  assetTypeIdx: index("assets_asset_type_idx").on(table.assetType),
  categoryIdx: index("assets_category_idx").on(table.category),
  isPublicIdx: index("assets_is_public_idx").on(table.isPublic),
  downloadCountIdx: index("assets_download_count_idx").on(table.downloadCount),
}));

// Creator Assets table to track assets with approval status and pricing
export const creatorAssets = pgTable("creator_assets", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  creatorId: varchar("creator_id").notNull().references(() => creators.id, { onDelete: 'cascade' }),
  assetId: varchar("asset_id").notNull().references(() => assets.id, { onDelete: 'cascade' }),
  title: text("title").notNull(),
  description: text("description"),
  price: integer("price_cents").notNull().default(0), // Price in cents
  approvalStatus: creatorApprovalStatusEnum("approval_status").notNull().default("pending"), // Database-enforced enum
  approvedAt: timestamp("approved_at"),
  approvedBy: varchar("approved_by").references(() => users.id, { onDelete: 'set null' }), // Admin who approved
  rejectionReason: text("rejection_reason"), // Reason for rejection if status is 'rejected'
  cancelledAt: timestamp("cancelled_at", { withTimezone: true }),  // When the asset was cancelled by creator
  isExclusive: boolean("is_exclusive").default(false), // Whether this is an exclusive asset
  salesCount: integer("sales_count").default(0),
  totalRevenue: integer("total_revenue_cents").default(0), // Total revenue generated in cents
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  // Prevent duplicate listings from the same creator for the same asset
  creatorAssetUnique: unique("creator_assets_creator_asset_unique").on(table.creatorId, table.assetId),
  
  // Price validation constraints using hardcoded values for now
  priceMinCheck: check("creator_assets_price_min", sql`${table.price} >= 199`),
  priceMaxCheck: check("creator_assets_price_max", sql`${table.price} <= 4999`),
  
  // Indexes for performance
  creatorIdIdx: index("creator_assets_creator_id_idx").on(table.creatorId),
  assetIdIdx: index("creator_assets_asset_id_idx").on(table.assetId),
  approvalStatusIdx: index("creator_assets_approval_status_idx").on(table.approvalStatus),
  priceIdx: index("creator_assets_price_idx").on(table.price),
}));

// Creator Earnings table to track earnings from sales and payout history
export const creatorEarnings = pgTable("creator_earnings", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  creatorId: varchar("creator_id").notNull().references(() => creators.id, { onDelete: 'restrict' }),
  creatorAssetId: varchar("creator_asset_id").references(() => creatorAssets.id, { onDelete: 'restrict' }),
  purchaseId: varchar("purchase_id").references(() => purchases.id, { onDelete: 'restrict' }),
  earningType: creatorEarningTypeEnum("earning_type").notNull(), // Database-enforced enum
  grossAmountCents: integer("gross_amount_cents").notNull(), // Gross earnings in cents (renamed for consistency)
  platformFeeCents: integer("platform_fee_cents").notNull(), // Platform fee in cents
  netAmountCents: integer("net_amount_cents").notNull(), // Net earnings after fees in cents
  payoutStatus: creatorPayoutStatusEnum("payout_status").notNull().default("pending"), // Database-enforced enum
  payoutDate: timestamp("payout_date"),
  payoutReference: text("payout_reference"), // Stripe payout reference
  stripeSessionId: text("stripe_session_id").notNull().unique(), // Enterprise security: enforce idempotency at DB level
  occurredAt: timestamp("occurred_at").notNull().defaultNow(), // When the earning actually occurred
  metadata: jsonb("metadata"), // Additional payout details
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => ({
  // Financial audit trail constraints
  grossAmountNonNegativeCheck: check("creator_earnings_gross_amount_non_negative", sql`${table.grossAmountCents} >= 0`),
  platformFeeNonNegativeCheck: check("creator_earnings_platform_fee_non_negative", sql`${table.platformFeeCents} >= 0`),
  netAmountNonNegativeCheck: check("creator_earnings_net_amount_non_negative", sql`${table.netAmountCents} >= 0`),
  netAmountCalculationCheck: check("creator_earnings_net_calculation", sql`${table.netAmountCents} = ${table.grossAmountCents} - ${table.platformFeeCents}`),
  
  // Indexes for performance and querying
  creatorIdIdx: index("creator_earnings_creator_id_idx").on(table.creatorId),
  creatorAssetIdIdx: index("creator_earnings_creator_asset_id_idx").on(table.creatorAssetId),
  purchaseIdIdx: index("creator_earnings_purchase_id_idx").on(table.purchaseId),
  payoutStatusIdx: index("creator_earnings_payout_status_idx").on(table.payoutStatus),
  occurredAtIdx: index("creator_earnings_occurred_at_idx").on(table.occurredAt),
  stripeSessionIdIdx: index("creator_earnings_stripe_session_id_idx").on(table.stripeSessionId), // Enterprise security: fast lookups
  // Composite index for creator earnings history (most recent first)
  creatorEarningsHistoryIdx: index("creator_earnings_creator_occurred_at_desc_idx").on(table.creatorId, table.occurredAt.desc()),
}));

// Cover Templates table to store template gallery items
export const coverTemplates = pgTable("cover_templates", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: text("title").notNull(),
  category: coverTemplateCategoryEnum("category").notNull(), // Database-enforced enum
  topTier: coverTemplateTopTierEnum("top_tier").notNull().default('General'), // Top-tier industry classification
  subcategories: text("subcategories").array().default(sql`'{}'`), // Array of subcategories
  priceCents: integer("price_cents").notNull().default(1200), // Price in cents, default $12.00
  previewUrl: text("preview_url").notNull(), // Thumbnail/preview image URL (renamed from preview_image_url)
  baseImageUrl: text("base_image_url"), // Optional base image used by template
  // Four preview URLs for lightbox functionality
  coverPreviewUrl: text("cover_preview_url"), // Main cover preview for lightbox
  divider1PreviewUrl: text("divider1_preview_url"), // First divider preview
  divider2PreviewUrl: text("divider2_preview_url"), // Second divider preview  
  divider3PreviewUrl: text("divider3_preview_url"), // Third divider preview
  downloadFile: text("download_file").notNull(), // Path or URL to PPTX/PDF/PNG bundle
  approvalStatus: coverTemplateApprovalStatusEnum("approval_status").notNull().default('pending'), // Approval workflow
  isActive: boolean("is_active").notNull().default(true),
  createdAt: timestamp("created_at", { withTimezone: true }).defaultNow(),
}, (table) => ({
  // Indexes for performance
  categoryIdx: index("cover_templates_category_idx").on(table.category),
  topTierIdx: index("cover_templates_top_tier_idx").on(table.topTier),
  approvalStatusIdx: index("cover_templates_approval_status_idx").on(table.approvalStatus),
  isActiveIdx: index("cover_templates_is_active_idx").on(table.isActive),
  priceCentsIdx: index("cover_templates_price_cents_idx").on(table.priceCents),
}));

// Cover Purchases table to track user purchases and customizations
export const coverPurchases = pgTable("cover_purchases", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: 'cascade' }),
  templateId: varchar("template_id").notNull().references(() => coverTemplates.id, { onDelete: 'cascade' }),
  customImageUrl: text("custom_image_url"), // Optional user-uploaded replacement image
  status: coverPurchaseStatusEnum("status").notNull().default("pending"), // Database-enforced enum
  downloadUrl: text("download_url"), // Final rendered bundle URL
  createdAt: timestamp("created_at", { withTimezone: true }).defaultNow(),
}, (table) => ({
  // Indexes for performance
  userIdIdx: index("cover_purchases_user_id_idx").on(table.userId),
  templateIdIdx: index("cover_purchases_template_id_idx").on(table.templateId),
  statusIdx: index("cover_purchases_status_idx").on(table.status),
}));

// Infographic Templates table for single-slide gallery items  
export const infographicTemplates = pgTable("infographic_templates", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  creatorId: varchar("creator_id").references(() => users.id, { onDelete: 'set null' }), // nullable for admin uploads
  title: text("title").notNull(),
  topTier: coverTemplateTopTierEnum("top_tier").notNull().default('General'), // reuse existing enum
  subcategories: text("subcategories").array().default(sql`'{}'`), // array of subcategories
  descriptors: text("descriptors").array().default(sql`'{}'`), // creator-defined descriptors/tags
  category: text("category").notNull(), // style: business|professional|creative|modern
  priceCents: integer("price_cents").notNull().default(1499), // MSRP but UI uses bundle price
  currency: text("currency").notNull().default("usd"),
  previewImageUrl: text("preview_image_url").notNull(), // thumbnail for gallery
  // Format URLs - at least one should be present
  pptxUrl: text("pptx_url"), // PowerPoint format
  keynoteUrl: text("keynote_url"), // Keynote format  
  gslidesUrl: text("gslides_url"), // Google Slides format
  downloadBundleUrl: text("download_bundle_url"), // ZIP or bundle download
  isActive: boolean("is_active").notNull().default(false), // admin controlled
  approvalStatus: coverTemplateApprovalStatusEnum("approval_status").notNull().default('pending'), // reuse existing enum
  createdAt: timestamp("created_at", { withTimezone: true }).defaultNow(),
}, (table) => ({
  // Indexes for performance
  creatorIdIdx: index("infographic_templates_creator_id_idx").on(table.creatorId),
  topTierIdx: index("infographic_templates_top_tier_idx").on(table.topTier), 
  categoryIdx: index("infographic_templates_category_idx").on(table.category),
  approvalStatusIdx: index("infographic_templates_approval_status_idx").on(table.approvalStatus),
  isActiveIdx: index("infographic_templates_is_active_idx").on(table.isActive),
}));

// Infographic Purchases table for bundle purchases (1-4 items per purchase)
export const infographicPurchases = pgTable("infographic_purchases", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: 'cascade' }),
  selectedIds: text("selected_ids").array().notNull(), // 1-4 template IDs selected at checkout
  status: coverPurchaseStatusEnum("status").notNull().default("pending"), // reuse existing enum
  downloadUrl: text("download_url"), // Final ZIP bundle URL
  stripeSessionId: text("stripe_session_id"), // for webhook correlation
  stripePaymentIntent: text("stripe_payment_intent"), // additional tracking
  amountCents: integer("amount_cents").notNull().default(1499), // bundle price $14.99
  currency: text("currency").notNull().default("usd"),
  createdAt: timestamp("created_at", { withTimezone: true }).defaultNow(),
}, (table) => ({
  // Indexes for performance
  userIdIdx: index("infographic_purchases_user_id_idx").on(table.userId),
  statusIdx: index("infographic_purchases_status_idx").on(table.status),
  stripeSessionIdIdx: index("infographic_purchases_stripe_session_id_idx").on(table.stripeSessionId),
}));

// Presentation Templates table
export const presentationTemplates = pgTable("presentation_templates", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  creatorId: varchar("creator_id").references(() => users.id, { onDelete: 'set null' }), // nullable for admin uploads
  title: text("title").notNull(),
  topTier: coverTemplateTopTierEnum("top_tier").notNull().default('General'), // reuse existing enum
  subcategories: text("subcategories").array().default(sql`'{}'`), // array of subcategories
  category: text("category").notNull(), // style: business|professional|creative|modern
  previewImageUrl: text("preview_image_url").notNull(), // thumbnail for gallery card
  slidePreviews: text("slide_previews").array().notNull().default(sql`'{}'`), // array of 24 slide URLs
  // Format URLs - at least one should be present
  pptxUrl: text("pptx_url"), // PowerPoint format
  keynoteUrl: text("keynote_url"), // Keynote format
  gslidesUrl: text("gslides_url"), // Google Slides format
  downloadBundleUrl: text("download_bundle_url"), // ZIP or bundle download
  isActive: boolean("is_active").notNull().default(false), // admin controlled
  approvalStatus: coverTemplateApprovalStatusEnum("approval_status").notNull().default('pending'), // reuse existing enum
  createdAt: timestamp("created_at", { withTimezone: true }).defaultNow(),
}, (table) => ({
  // Indexes for performance
  creatorIdIdx: index("presentation_templates_creator_id_idx").on(table.creatorId),
  topTierIdx: index("presentation_templates_top_tier_idx").on(table.topTier),
  categoryIdx: index("presentation_templates_category_idx").on(table.category),
  approvalStatusIdx: index("presentation_templates_approval_status_idx").on(table.approvalStatus),
  isActiveIdx: index("presentation_templates_is_active_idx").on(table.isActive),
}));

// Presentation Purchases table  
export const presentationPurchases = pgTable("presentation_purchases", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: 'cascade' }),
  type: text("type").notNull(), // 'regular' | 'premium_plus'
  baseTemplateId: varchar("base_template_id").notNull().references(() => presentationTemplates.id, { onDelete: 'cascade' }),
  selectedCoverId: varchar("selected_cover_id").references(() => coverTemplates.id, { onDelete: 'set null' }), // nullable
  selectedInfographicIds: text("selected_infographic_ids").array().default(sql`'{}'`), // up to 4 infographic IDs
  status: coverPurchaseStatusEnum("status").notNull().default("pending"), // reuse existing enum
  amountCents: integer("amount_cents").notNull(),
  currency: text("currency").notNull().default("usd"),
  stripeSessionId: text("stripe_session_id").unique(),
  stripePaymentIntent: text("stripe_payment_intent"),
  downloadUrl: text("download_url"), // signed URL or file path
  createdAt: timestamp("created_at", { withTimezone: true }).defaultNow(),
}, (table) => ({
  // Indexes for performance
  userIdIdx: index("presentation_purchases_user_id_idx").on(table.userId),
  statusIdx: index("presentation_purchases_status_idx").on(table.status),
  typeIdx: index("presentation_purchases_type_idx").on(table.type),
  baseTemplateIdIdx: index("presentation_purchases_base_template_id_idx").on(table.baseTemplateId),
}));

export const insertUserSchema = createInsertSchema(users).omit({
  id: true,
  createdAt: true,
}).extend({
  role: z.enum(['owner', 'manager', 'staff', 'analyst', 'user', 'pro']).default('user'),
});

export const insertBrandKitSchema = createInsertSchema(brandKits).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertBusinessNameSchema = createInsertSchema(businessNames).omit({
  id: true,
  createdAt: true,
});

export const insertSocialMediaKitSchema = createInsertSchema(socialMediaKits).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertWebsiteTemplateSchema = createInsertSchema(websiteTemplates).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  downloadCount: true,
  rating: true,
  ratingCount: true,
});

export const insertTemplateCategorySchema = createInsertSchema(templateCategories).omit({
  id: true,
  createdAt: true,
});

export const insertTemplateConfigurationSchema = createInsertSchema(templateConfigurations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertTemplateContentSchema = createInsertSchema(templateContent).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertUserTemplateCustomizationSchema = createInsertSchema(userTemplateCustomizations).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertTemplateAssetSchema = createInsertSchema(templateAssets).omit({
  id: true,
  createdAt: true,
});

export const insertNotificationSchema = createInsertSchema(notifications).omit({
  id: true,
  createdAt: true,
});

export const insertCancellationSchema = createInsertSchema(cancellations).omit({
  id: true,
  createdAt: true,
}).extend({
  reason: z.enum(CANCELLATION_REASON_VALUES as [string, ...string[]]),
});

export const insertPauseSchema = createInsertSchema(pauses).omit({
  id: true,
  createdAt: true,
}).extend({
  months: z.number().int().min(1).max(2),
  resumeAt: z.date(),
});

export const insertDomainOrderSchema = createInsertSchema(domainOrders).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  // SECURITY: Remove these server-controlled fields to prevent client tampering
  status: true,
  priceCents: true,
  stripeSessionId: true,
  providerRegId: true,
  expiresAt: true,
  errorMessage: true,
}).extend({
  domain: z.string().min(1).transform(val => val.toLowerCase().trim()),
  years: z.number().int().min(1).max(10),
  registrantContact: z.object({
    firstName: z.string().min(1).trim(),
    lastName: z.string().min(1).trim(),
    email: z.string().email().toLowerCase().trim(),
    phone: z.string().regex(/^\+[1-9]\d{1,14}$/, "Phone must be in E.164 format (e.g., +1234567890)"),
    organization: z.string().trim().optional(),
    address: z.string().min(1).trim(),
    city: z.string().min(1).trim(),
    state: z.string().trim().optional(), // Optional for international addresses
    postalCode: z.string().trim().optional(), // Optional for international addresses  
    country: z.string().min(2).max(2).toUpperCase(), // ISO country code
  }),
});

export const insertDomainCreditSchema = createInsertSchema(domainCredits).omit({
  id: true,
  issuedAt: true,
  usedAt: true,
}).extend({
  status: z.enum(DOMAIN_CREDIT_STATUS_VALUES as [string, ...string[]]).default('available'),
  capCents: z.number().int().min(0),
  eligibleTlds: z.array(z.string().min(1)),
  expiresAt: z.date(),
  usedDomain: z.string().optional(),
  sourceSubscriptionId: z.string().optional(),
});

export const insertDomainSchema = createInsertSchema(domains).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  domain: z.string().min(1).transform(val => val.toLowerCase().trim()),
  tld: z.string().min(1).transform(val => val.toLowerCase().trim()),
  expiresAt: z.date(),
});

export const insertImportedIconSchema = createInsertSchema(importedIcons).omit({
  id: true,
  createdAt: true,
}).extend({
  name: z.string().min(1).trim(),
  svg: z.string().min(1),
  style: z.enum(["modern", "classic", "flat", "outlined", "solid", "handdrawn", "isometric", "material"]).default("flat"),
  tags: z.array(z.string()).default([]),
});

export const insertVisitorSessionSchema = createInsertSchema(visitorSessions).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  sessionId: z.string().min(1),
  ipHash: z.string().min(1), // Salted hash instead of raw IP
  country: z.string().optional(),
  countryCode: z.string().optional(), 
  city: z.string().optional(),
  latitude: z.number().optional(), // Fixed: align with database numeric type
  longitude: z.number().optional(),
  userAgent: z.string().optional(),
  pageViews: z.number().int().min(1).default(1),
  firstPageUrl: z.string().optional(),
  lastPageUrl: z.string().optional(),
  referrer: z.string().optional(),
  isPublicPage: z.boolean().default(true),
  date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/), // YYYY-MM-DD format
});

export const insertDailyVisitorStatsSchema = createInsertSchema(dailyVisitorStats).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  date: z.string().regex(/^\d{4}-\d{2}-\d{2}$/), // YYYY-MM-DD format
  uniqueVisitors: z.number().int().min(0).default(0),
  totalPageViews: z.number().int().min(0).default(0),
  countries: z.record(z.number()).default({}),
  topPages: z.record(z.number()).default({}),
});

// Cart Items insert schema
export const insertCartItemSchema = createInsertSchema(cartItems).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  itemType: z.enum(['stock_photo', 'subscription']),
  itemId: z.string().min(1),
  itemName: z.string().min(1),
  itemPrice: z.number().int().min(0), // Price in cents
  quantity: z.number().int().min(1).default(1),
  metadata: z.record(z.any()).optional(),
});

// Purchases insert schema
export const insertPurchaseSchema = createInsertSchema(purchases).omit({
  id: true,
  createdAt: true,
  completedAt: true,
}).extend({
  stripeSessionId: z.string().min(1),
  stripePaymentIntentId: z.string().optional(),
  status: z.enum(['pending', 'completed', 'failed', 'refunded']).default('pending'),
  totalAmountCents: z.number().int().min(0),
  items: z.array(z.record(z.any())), // Array of purchased items
  metadata: z.record(z.any()).optional(),
});

// User Entitlements insert schema
export const insertUserEntitlementSchema = createInsertSchema(userEntitlements).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
}).extend({
  entitlementType: z.enum(['stock_photo', 'subscription']),
  entitlementId: z.string().min(1),
  status: z.enum(['active', 'expired', 'revoked']).default('active'),
  expiresAt: z.date().optional(),
});

// Creator Marketplace insert schemas
export const insertCreatorSchema = createInsertSchema(creators).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  totalEarnings: true, // Server-calculated field
}).extend({
  onboardingStatus: z.enum(CREATOR_ONBOARDING_STATUS_VALUES).default('pending'),
  stripeConnectAccountId: z.string().optional(),
  profileData: z.record(z.any()).optional(),
  payoutEnabled: z.boolean().default(false),
  isActive: z.boolean().default(true),
});

export const insertAssetSchema = createInsertSchema(assets).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  downloadCount: true, // Server-tracked field
}).extend({
  fileName: z.string().min(1),
  originalFileName: z.string().min(1),
  fileUrl: z.string().url(),
  fileSize: z.number().int().min(0),
  mimeType: z.string().min(1),
  assetType: z.enum(['image', 'video', 'audio', 'document', 'template']),
  category: z.string().optional(),
  tags: z.array(z.string()).default([]),
  metadata: z.record(z.any()).optional(),
  isPublic: z.boolean().default(true),
});

export const insertCreatorAssetSchema = createInsertSchema(creatorAssets).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
  approvedAt: true, // Server-set field
  salesCount: true, // Server-tracked field
  totalRevenue: true, // Server-calculated field
}).extend({
  title: z.string().min(1).trim(),
  description: z.string().optional(),
  price: z.number().int().min(CREATOR_MIN_PRICE_CENTS).max(CREATOR_MAX_PRICE_CENTS), // Price validation with constants
  approvalStatus: z.enum(CREATOR_APPROVAL_STATUS_VALUES).default('pending'),
  rejectionReason: z.string().optional(),
  isExclusive: z.boolean().default(false),
});

export const insertCreatorEarningSchema = createInsertSchema(creatorEarnings).omit({
  id: true,
  createdAt: true,
}).extend({
  earningType: z.enum(CREATOR_EARNING_TYPE_VALUES),
  grossAmount: z.number().int().min(0), // Gross earnings in cents
  platformFeeCents: z.number().int().min(0), // Platform fee in cents (updated field name)
  netAmountCents: z.number().int().min(0), // Net earnings after fees in cents (updated field name)
  payoutStatus: z.enum(CREATOR_PAYOUT_STATUS_VALUES).default('pending'),
  payoutDate: z.date().optional(),
  payoutReference: z.string().optional(),
  occurredAt: z.date().default(() => new Date()), // New field for when earning occurred
  metadata: z.record(z.any()).optional(),
});

// Cover Templates insert schemas
export const insertCoverTemplateSchema = createInsertSchema(coverTemplates).omit({
  id: true,
  createdAt: true,
}).extend({
  title: z.string().min(1).trim(),
  category: z.enum(COVER_TEMPLATE_CATEGORY_VALUES),
  priceCents: z.number().int().min(0).default(1200), // Price in cents, default $12.00
  previewUrl: z.string().url(),
  baseImageUrl: z.string().url().optional(),
  downloadFile: z.string().min(1), // Can be URL or file path
  isActive: z.boolean().default(true),
});

export const insertCoverPurchaseSchema = createInsertSchema(coverPurchases).omit({
  id: true,
  createdAt: true,
}).extend({
  userId: z.string().min(1),
  templateId: z.string().min(1),
  customImageUrl: z.string().url().optional(),
  status: z.enum(COVER_PURCHASE_STATUS_VALUES).default('pending'),
  downloadUrl: z.string().url().optional(),
});

// Infographic Templates insert schemas
export const insertInfographicTemplateSchema = createInsertSchema(infographicTemplates).omit({
  id: true,
  createdAt: true,
}).extend({
  title: z.string().min(1).trim(),
  topTier: z.enum(COVER_TEMPLATE_TOP_TIER_VALUES).default('General'), // reuse existing enum
  subcategories: z.array(z.string()).default([]),
  descriptors: z.array(z.string()).default([]), // creator-defined descriptors/tags
  category: z.string().min(1), // business|professional|creative|modern
  priceCents: z.number().int().min(0).default(1499), // bundle MSRP
  currency: z.string().default("usd"),
  previewImageUrl: z.string().url(),
  // At least one format URL should be provided
  pptxUrl: z.string().url().optional(),
  keynoteUrl: z.string().url().optional(),
  gslidesUrl: z.string().url().optional(),
  downloadBundleUrl: z.string().url().optional(),
  isActive: z.boolean().default(false),
  approvalStatus: z.enum(COVER_TEMPLATE_APPROVAL_STATUS_VALUES).default('pending'),
});

export const insertInfographicPurchaseSchema = createInsertSchema(infographicPurchases).omit({
  id: true,
  createdAt: true,
}).extend({
  userId: z.string().min(1),
  selectedIds: z.array(z.string()).min(1).max(4), // 1-4 template IDs
  status: z.enum(COVER_PURCHASE_STATUS_VALUES).default('pending'),
  downloadUrl: z.string().url().optional(),
  stripeSessionId: z.string().optional(),
  stripePaymentIntent: z.string().optional(),
  amountCents: z.number().int().min(0).default(1499), // $14.99 bundle price
  currency: z.string().default("usd"),
});

// Presentation Templates insert schemas
export const insertPresentationTemplateSchema = createInsertSchema(presentationTemplates).omit({
  id: true,
  createdAt: true,
}).extend({
  title: z.string().min(1).trim(),
  topTier: z.enum(COVER_TEMPLATE_TOP_TIER_VALUES).default('General'), // reuse existing enum
  subcategories: z.array(z.string()).default([]),
  category: z.string().min(1), // business|professional|creative|modern
  previewImageUrl: z.string().url(),
  slidePreviews: z.array(z.string().url()).length(24), // exactly 24 slide URLs required
  // At least one format URL should be provided
  pptxUrl: z.string().url().optional(),
  keynoteUrl: z.string().url().optional(),
  gslidesUrl: z.string().url().optional(),
  downloadBundleUrl: z.string().url().optional(),
  isActive: z.boolean().default(false),
  approvalStatus: z.enum(COVER_TEMPLATE_APPROVAL_STATUS_VALUES).default('pending'),
});

export const insertPresentationPurchaseSchema = createInsertSchema(presentationPurchases).omit({
  id: true,
  createdAt: true,
}).extend({
  userId: z.string().min(1),
  type: z.enum(['regular', 'premium_plus']), 
  baseTemplateId: z.string().min(1),
  selectedCoverId: z.string().optional(),
  selectedInfographicIds: z.array(z.string()).max(4).default([]), // up to 4 infographic IDs
  status: z.enum(COVER_PURCHASE_STATUS_VALUES).default('pending'),
  amountCents: z.number().int().min(0), // price depends on type: 2999 or 4999
  currency: z.string().default("usd"),
  downloadUrl: z.string().url().optional(),
});

export type User = typeof users.$inferSelect;
export type InsertUser = z.infer<typeof insertUserSchema>;
export type BrandKit = typeof brandKits.$inferSelect;
export type InsertBrandKit = z.infer<typeof insertBrandKitSchema>;
export type BusinessName = typeof businessNames.$inferSelect;
export type InsertBusinessName = z.infer<typeof insertBusinessNameSchema>;
export type SocialMediaKit = typeof socialMediaKits.$inferSelect;
export type InsertSocialMediaKit = z.infer<typeof insertSocialMediaKitSchema>;
export type WebsiteTemplate = typeof websiteTemplates.$inferSelect;
export type InsertWebsiteTemplate = z.infer<typeof insertWebsiteTemplateSchema>;

export type TemplateCategory = typeof templateCategories.$inferSelect;
export type InsertTemplateCategory = z.infer<typeof insertTemplateCategorySchema>;
export type TemplateConfiguration = typeof templateConfigurations.$inferSelect;
export type InsertTemplateConfiguration = z.infer<typeof insertTemplateConfigurationSchema>;
export type TemplateContent = typeof templateContent.$inferSelect;
export type InsertTemplateContent = z.infer<typeof insertTemplateContentSchema>;
export type UserTemplateCustomization = typeof userTemplateCustomizations.$inferSelect;
export type InsertUserTemplateCustomization = z.infer<typeof insertUserTemplateCustomizationSchema>;
export type TemplateAsset = typeof templateAssets.$inferSelect;
export type InsertTemplateAsset = z.infer<typeof insertTemplateAssetSchema>;
export type Notification = typeof notifications.$inferSelect;
export type InsertNotification = z.infer<typeof insertNotificationSchema>;
export type Cancellation = typeof cancellations.$inferSelect;
export type InsertCancellation = z.infer<typeof insertCancellationSchema>;
export type Pause = typeof pauses.$inferSelect;
export type InsertPause = z.infer<typeof insertPauseSchema>;
export type DomainOrder = typeof domainOrders.$inferSelect;
export type InsertDomainOrder = z.infer<typeof insertDomainOrderSchema>;
export type DomainCredit = typeof domainCredits.$inferSelect;
export type InsertDomainCredit = z.infer<typeof insertDomainCreditSchema>;
export type Domain = typeof domains.$inferSelect;
export type InsertDomain = z.infer<typeof insertDomainSchema>;
export type ImportedIcon = typeof importedIcons.$inferSelect;
export type InsertImportedIcon = z.infer<typeof insertImportedIconSchema>;
export type VisitorSession = typeof visitorSessions.$inferSelect;
export type InsertVisitorSession = z.infer<typeof insertVisitorSessionSchema>;
export type DailyVisitorStats = typeof dailyVisitorStats.$inferSelect;
export type InsertDailyVisitorStats = z.infer<typeof insertDailyVisitorStatsSchema>;
export type CartItem = typeof cartItems.$inferSelect;
export type InsertCartItem = z.infer<typeof insertCartItemSchema>;
export type Purchase = typeof purchases.$inferSelect;
export type InsertPurchase = z.infer<typeof insertPurchaseSchema>;
export type UserEntitlement = typeof userEntitlements.$inferSelect;
export type InsertUserEntitlement = z.infer<typeof insertUserEntitlementSchema>;

// Creator Marketplace types
export type Creator = typeof creators.$inferSelect;
export type InsertCreator = z.infer<typeof insertCreatorSchema>;
export type Asset = typeof assets.$inferSelect;
export type InsertAsset = z.infer<typeof insertAssetSchema>;
export type CreatorAsset = typeof creatorAssets.$inferSelect;
export type InsertCreatorAsset = z.infer<typeof insertCreatorAssetSchema>;
export type CreatorEarning = typeof creatorEarnings.$inferSelect;
export type InsertCreatorEarning = z.infer<typeof insertCreatorEarningSchema>;

// Cover Templates types
export type CoverTemplate = typeof coverTemplates.$inferSelect;
export type InsertCoverTemplate = z.infer<typeof insertCoverTemplateSchema>;
export type CoverPurchase = typeof coverPurchases.$inferSelect;
export type InsertCoverPurchase = z.infer<typeof insertCoverPurchaseSchema>;

// Infographic Templates types
export type InfographicTemplate = typeof infographicTemplates.$inferSelect;
export type InsertInfographicTemplate = z.infer<typeof insertInfographicTemplateSchema>;
export type InfographicPurchase = typeof infographicPurchases.$inferSelect;
export type InsertInfographicPurchase = z.infer<typeof insertInfographicPurchaseSchema>;

// Presentation Templates types
export type PresentationTemplate = typeof presentationTemplates.$inferSelect;
export type InsertPresentationTemplate = z.infer<typeof insertPresentationTemplateSchema>;
export type PresentationPurchase = typeof presentationPurchases.$inferSelect;
export type InsertPresentationPurchase = z.infer<typeof insertPresentationPurchaseSchema>;

// Template customization structure for frontend
export interface TemplateCustomization {
  colors: {
    primary: string;
    secondary: string;
    accent: string;
    background: string;
    text: string;
  };
  fonts: {
    heading: string;
    body: string;
    sizes: {
      h1: string;
      h2: string;
      h3: string;
      body: string;
    };
  };
  layout: {
    spacing: string;
    maxWidth: string;
    gridColumns: number;
  };
  content: {
    sections: Array<{
      id: string;
      type: string;
      content: any;
      order: number;
    }>;
  };
  metadata?: {
    description?: string;
    [key: string]: any;
  };
}

// Domain registration types
export interface DomainSearchHit {
  domain: string;
  available: boolean;
  priceCents?: number;
  type?: 'standard' | 'premium';
}

export interface DomainSearchResult {
  hits: DomainSearchHit[];
}

export interface RegistrantContact {
  firstName: string;
  lastName: string;
  email: string;
  phone: string;
  organization?: string;
  address: string;
  city: string;
  state?: string; // Optional for international addresses
  postalCode?: string; // Optional for international addresses
  country: string; // ISO country code
}

export interface DomainRegistrationInput {
  domain: string;
  years: number;
  contact: RegistrantContact;
  privacy: boolean;
  nameservers?: string[];
}

export interface DomainRegistrationResult {
  success: boolean;
  providerRegId?: string;
  message?: string;
}
