// SuperNova's comprehensive SVG normalization function
// Strips unsafe content, ensures viewBox, removes fixed dimensions, and sets currentColor
export function sanitizeSvg(input: string): string {
  let s = input.trim();
  
  // Check if input is empty or doesn't contain SVG content
  if (!s || (!s.includes('<svg') && !s.includes('<?xml'))) {
    console.warn('[sanitizeSvg] Input does not contain valid SVG content');
    return s;
  }

  // Remove XML headers and DOCTYPEs
  s = s.replace(/<\?xml[^>]*>/gi, "");
  s = s.replace(/<!DOCTYPE[^>]*>/gi, "");

  // Security: Remove all script elements and dangerous content
  s = s.replace(/<script[\s\S]*?<\/script>/gi, "");
  s = s.replace(/<foreignObject[\s\S]*?<\/foreignObject>/gi, "");
  s = s.replace(/<iframe[\s\S]*?<\/iframe>/gi, "");
  s = s.replace(/<object[\s\S]*?<\/object>/gi, "");
  s = s.replace(/<embed[\s\S]*?<\/embed>/gi, "");
  s = s.replace(/<link[\s\S]*?>/gi, "");

  // Strip ALL event handlers (onload, onclick, etc.)
  s = s.replace(/\son[a-z]+\s*=\s*"[^"]*"/gi, "");
  s = s.replace(/\son[a-z]+\s*=\s*'[^']*'/gi, "");
  s = s.replace(/\son[a-z]+\s*=\s*[^\s>"']+/gi, "");

  // Remove ALL external href references and dangerous schemes
  s = s.replace(/\s(xlink:)?href\s*=\s*"https?:[^"]*"/gi, "");
  s = s.replace(/\s(xlink:)?href\s*=\s*'https?:[^']*'/gi, "");
  s = s.replace(/\s(xlink:)?href\s*=\s*"javascript:[^"]*"/gi, "");
  s = s.replace(/\s(xlink:)?href\s*=\s*'javascript:[^']*'/gi, "");
  s = s.replace(/\s(xlink:)?href\s*=\s*"data:[^"]*"/gi, "");
  s = s.replace(/\s(xlink:)?href\s*=\s*'data:[^']*'/gi, "");
  s = s.replace(/\s(xlink:)?href\s*=\s*"vbscript:[^"]*"/gi, "");
  s = s.replace(/\s(xlink:)?href\s*=\s*'vbscript:[^']*'/gi, "");

  // Remove style attributes and <style> elements (XSS vectors)
  s = s.replace(/\sstyle\s*=\s*"[^"]*"/gi, "");
  s = s.replace(/\sstyle\s*=\s*'[^']*'/gi, "");
  s = s.replace(/<style[\s\S]*?<\/style>/gi, "");

  // Force width/height removal so it scales properly in UI
  s = s.replace(/\swidth\s*=\s*"[^"]*"/gi, "");
  s = s.replace(/\sheight\s*=\s*"[^"]*"/gi, "");
  s = s.replace(/\swidth\s*=\s*'[^']*'/gi, "");
  s = s.replace(/\sheight\s*=\s*'[^']*'/gi, "");

  // Extract SVG content if it's buried in other content
  if (s && !s.startsWith('<svg') && s.includes('<svg')) {
    const svgMatch = s.match(/<svg[\s\S]*?<\/svg>/i);
    if (svgMatch) {
      s = svgMatch[0];
    }
  }

  // Ensure viewBox exists (critical for proper scaling)
  if (s.includes('<svg') && !/viewBox\s*=/i.test(s)) {
    // Try to extract dimensions first if they exist
    const widthMatch = s.match(/width\s*=\s*["']?([\d.]+)/i);
    const heightMatch = s.match(/height\s*=\s*["']?([\d.]+)/i);
    
    let viewBox = "0 0 24 24"; // default
    if (widthMatch && heightMatch) {
      viewBox = `0 0 ${widthMatch[1]} ${heightMatch[1]}`;
    }
    
    s = s.replace(
      /<svg([^>]*)>/i,
      (match, attrs) => {
        // Only add viewBox if not already present
        if (!/viewBox/i.test(attrs)) {
          return `<svg${attrs} viewBox="${viewBox}">`;
        }
        return match;
      }
    );
  }

  // SuperNova's color picker support: replace explicit colors with currentColor
  // This allows the color picker to work properly
  s = s.replace(/fill\s*=\s*"(#[0-9a-fA-F]{3,6}|rgb\([^)]*\)|[a-zA-Z]+)"/gi, 'fill="currentColor"');
  s = s.replace(/stroke\s*=\s*"(#[0-9a-fA-F]{3,6}|rgb\([^)]*\)|[a-zA-Z]+)"/gi, 'stroke="currentColor"');
  
  // Handle single quotes too
  s = s.replace(/fill\s*=\s*'(#[0-9a-fA-F]{3,6}|rgb\([^)]*\)|[a-zA-Z]+)'/gi, 'fill="currentColor"');
  s = s.replace(/stroke\s*=\s*'(#[0-9a-fA-F]{3,6}|rgb\([^)]*\)|[a-zA-Z]+)'/gi, 'stroke="currentColor"');
  
  // But preserve "none" values
  s = s.replace(/fill="currentColor"/gi, (match, offset) => {
    const before = s.substring(Math.max(0, offset - 20), offset);
    if (before.includes('fill="none"') || before.includes("fill='none'")) {
      return 'fill="none"';
    }
    return match;
  });
  
  s = s.replace(/stroke="currentColor"/gi, (match, offset) => {
    const before = s.substring(Math.max(0, offset - 20), offset);
    if (before.includes('stroke="none"') || before.includes("stroke='none'")) {
      return 'stroke="none"';
    }
    return match;
  });

  // Final validation and cleanup
  const finalResult = s.trim();
  
  // Ensure we have a valid SVG structure
  if (!finalResult.startsWith('<svg')) {
    console.warn('[sanitizeSvg] Result does not start with <svg> tag');
    return '';
  }
  
  if (!finalResult.includes('</svg>')) {
    console.warn('[sanitizeSvg] Result does not have closing </svg> tag');
    return '';
  }
  
  return finalResult;
}

// Utility function to determine if SVG content is small enough to inline
export function shouldInlineSvg(svgContent: string): boolean {
  // Inline if less than 2KB
  return svgContent.length < 2048;
}

// Utility function to extract icon ID from various path formats
export function extractIconId(path: string): string {
  const parts = path.split('/');
  const filename = parts[parts.length - 1];
  return filename.replace(/\.(png|svg)$/i, '');
}