import Stripe from "stripe";

if (!process.env.STRIPE_SECRET_KEY) {
  throw new Error('Missing required Stripe secret: STRIPE_SECRET_KEY');
}

const stripe = new Stripe(process.env.STRIPE_SECRET_KEY, { apiVersion: "2024-06-20" });

const PRICE_LOOKUP_KEY = "ibrandbiz_infographic_bundle_price_v1"; // bump if price changes

export async function ensureGlobalInfographicBundlePrice() {
  // First try to find existing price by lookup_key
  const prices = await stripe.prices.list({ lookup_keys: [PRICE_LOOKUP_KEY], active: true, limit: 1 });
  if (prices.data.length > 0) return prices.data[0];

  // Try to find existing product by metadata to avoid duplication
  const products = await stripe.products.list({ active: true, limit: 100 });
  let product = products.data.find(p => p.metadata.code === "ibrandbiz_infographic_bundle");
  
  if (!product) {
    // Create product if not found
    product = await stripe.products.create({
      name: "Infographic Templates Bundle (1-4 Items)",
      description: "Professional infographic templates bundle. Choose 1-4 slides. PowerPoint, Keynote, Google Slides formats included.",
      metadata: { code: "ibrandbiz_infographic_bundle" }
    });
  }

  // Create the canonical price with concurrency safety
  // Fixed bundle price $14.99 regardless of 1-4 items selected
  const unit_amount = parseInt(process.env.INFOGRAPHIC_BUNDLE_PRICE_CENTS || "1499", 10);
  const currency = (process.env.INFOGRAPHIC_BUNDLE_CURRENCY || "usd").toLowerCase();

  try {
    return await stripe.prices.create({
      product: product.id,
      unit_amount,
      currency,
      lookup_key: PRICE_LOOKUP_KEY,
    });
  } catch (error: any) {
    // Handle race condition - check both error.code and error.raw?.code
    if (error.code === 'resource_already_exists' || error.raw?.code === 'resource_already_exists') {
      const retryPrices = await stripe.prices.list({ lookup_keys: [PRICE_LOOKUP_KEY], active: true, limit: 1 });
      if (retryPrices.data.length > 0) return retryPrices.data[0];
    }
    throw error;
  }
}

export async function createInfographicBundleCheckoutSession(args: {
  successUrl: string; cancelUrl: string; metadata?: Record<string, string>; coupon?: string;
}) {
  const price = await ensureGlobalInfographicBundlePrice();
  return stripe.checkout.sessions.create({
    mode: "payment",
    line_items: [{ price: price.id, quantity: 1 }], // Always quantity 1 for bundle
    success_url: args.successUrl,
    cancel_url: args.cancelUrl,
    discounts: args.coupon ? [{ coupon: args.coupon }] : undefined,
    metadata: args.metadata,
  });
}