import { db } from '../storage'; // your drizzle instance
import { eq, and, sql } from 'drizzle-orm';
import { creatorAssets, creators, users } from '../../shared/schema'; // adjust paths/names to your actual schema

export async function getCreatorByUserId(userId: string) {
  const rows = await db.select({
    id: creators.id,
    userId: creators.userId,
    name: creators.name,
    email: creators.email,
    onboardingStatus: creators.onboardingStatus,
    stripeAccountId: creators.stripeAccountId,
    stripeOnboardingCompleted: creators.stripeOnboardingCompleted,
    createdAt: creators.createdAt
  })
    .from(creators)
    .where(eq(creators.userId, userId))
    .limit(1);
  
  return rows[0] || null;
}

export async function getCreatorAssetsByCreatorId(creatorId: string) {
  const rows = await db.select({
      id: creatorAssets.id,
      creatorId: creatorAssets.creatorId,
      assetId: creatorAssets.assetId,
      title: creatorAssets.title,
      description: creatorAssets.description,
      price: creatorAssets.price, // priceCents
      approvalStatus: creatorAssets.approvalStatus,
      approvedAt: creatorAssets.approvedAt,
      cancelledAt: creatorAssets.cancelledAt,
      rejectionReason: creatorAssets.rejectionReason,
      salesCount: creatorAssets.salesCount,
      totalRevenueCents: creatorAssets.totalRevenue, // totalRevenue field in schema
      createdAt: creatorAssets.createdAt,
      // Join to get preview URL from assets table if needed
      previewUrl: sql`null`.as('previewUrl'), // placeholder for now
    })
    .from(creatorAssets)
    .where(eq(creatorAssets.creatorId, creatorId))
    .orderBy(sql`${creatorAssets.createdAt} DESC`);
  
  return rows.map(row => ({
    ...row,
    salesCount: row.salesCount || 0,
    totalRevenueCents: row.totalRevenueCents || 0
  }));
}

export async function getCreatorAssetById(id: string) {
  const rows = await db.select().from(creatorAssets).where(eq(creatorAssets.id, id)).limit(1);
  return rows[0] || null;
}

export async function updateCreatorAsset(id: string, patch: Partial<typeof creatorAssets.$inferInsert>) {
  const rows = await db.update(creatorAssets)
    .set({
      ...patch,
      updatedAt: new Date()
    })
    .where(eq(creatorAssets.id, id))
    .returning();
  return rows[0];
}

export async function getCreatorPayoutSummary(creatorId: string) {
  // Get all approved assets for this creator
  const assets = await getCreatorAssetsByCreatorId(creatorId);
  
  // Calculate totals from assets
  const approvedAssets = assets.filter(a => a.approvalStatus === 'approved');
  const lifetimeEarningsCents = approvedAssets.reduce((total, asset) => {
    // Calculate creator's share (70% after 30% platform fee)
    const creatorShare = Math.floor((asset.totalRevenueCents || 0) * 0.7);
    return total + creatorShare;
  }, 0);
  
  // For now, assume all earnings are pending (would integrate with actual payout system)
  const pendingPayoutCents = lifetimeEarningsCents;
  
  return { 
    lifetimeEarningsCents, 
    pendingPayoutCents, 
    lastPayoutAt: null // Would come from actual payout records
  };
}

export async function getAllCreatorAssetsForAdmin() {
  // Join with creators and users to get display names
  const rows = await db.select({
    id: creatorAssets.id,
    title: creatorAssets.title,
    price: creatorAssets.price,
    approvalStatus: creatorAssets.approvalStatus,
    createdAt: creatorAssets.createdAt,
    previewUrl: sql`null`.as('previewUrl'), // placeholder
    salesCount: creatorAssets.salesCount,
    creatorName: sql`COALESCE(${creators.name}, ${users.displayName}, ${users.email})`.as('creatorName')
  })
    .from(creatorAssets)
    .leftJoin(creators, eq(creators.id, creatorAssets.creatorId))
    .leftJoin(users, eq(users.id, creators.userId))
    .orderBy(sql`${creatorAssets.createdAt} DESC`);
  
  return rows.map(row => ({
    id: row.id,
    title: row.title,
    price: row.price || 0,
    status: row.approvalStatus,
    createdAt: row.createdAt ? row.createdAt.toISOString() : new Date().toISOString(),
    previewUrl: row.previewUrl,
    salesCount: row.salesCount || 0,
    creatorName: row.creatorName || 'Unknown Creator'
  }));
}