import { 
  type User, 
  type InsertUser,
  type BrandKit,
  type InsertBrandKit,
  type BusinessName,
  type InsertBusinessName,
  type SocialMediaKit,
  type InsertSocialMediaKit,
  type WebsiteTemplate,
  type InsertWebsiteTemplate,
  type UserTemplateCustomization,
  type InsertUserTemplateCustomization,
  type Notification,
  type InsertNotification,
  type Cancellation,
  type InsertCancellation,
  type Pause,
  type InsertPause,
  type DomainOrder,
  type InsertDomainOrder,
  type DomainCredit,
  type InsertDomainCredit,
  type Domain,
  type InsertDomain,
  type ImportedIcon,
  type InsertImportedIcon,
  type VisitorSession,
  type InsertVisitorSession,
  type DailyVisitorStats,
  type InsertDailyVisitorStats,
  type CartItem,
  type InsertCartItem,
  type Purchase,
  type InsertPurchase,
  type UserEntitlement,
  type InsertUserEntitlement,
  type Creator,
  type InsertCreator,
  type Asset,
  type InsertAsset,
  type CreatorAsset,
  type InsertCreatorAsset,
  type CreatorEarning,
  type InsertCreatorEarning,
  type CoverTemplate,
  type InsertCoverTemplate,
  type CoverPurchase,
  type InsertCoverPurchase,
  type InfographicTemplate,
  type InsertInfographicTemplate,
  type InfographicPurchase,
  type InsertInfographicPurchase,
  type PresentationTemplate,
  type InsertPresentationTemplate,
  type PresentationPurchase,
  type InsertPresentationPurchase,
  users,
  brandKits,
  businessNames,
  socialMediaKits,
  websiteTemplates,
  userTemplateCustomizations,
  notifications,
  cancellations,
  pauses,
  domainOrders,
  domainCredits,
  domains,
  importedIcons,
  visitorSessions,
  dailyVisitorStats,
  cartItems,
  purchases,
  userEntitlements,
  creators,
  assets,
  creatorAssets,
  creatorEarnings,
  coverTemplates,
  coverPurchases,
  infographicTemplates,
  infographicPurchases,
  presentationTemplates,
  presentationPurchases
} from "@shared/schema";
import { drizzle } from "drizzle-orm/neon-http";
import { eq, and, or, sql, gte, lte, ilike, desc } from "drizzle-orm";
import { neon } from "@neondatabase/serverless";
import { notificationBroadcaster } from "./events/notifications";

export interface IStorage {
  // User operations
  getUser(id: string): Promise<User | undefined>;
  getAllUsers(): Promise<User[]>;
  getUserByEmail(email: string): Promise<User | undefined>;
  getUserByFirebaseUid(firebaseUid: string): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;
  updateUser(id: string, user: Partial<InsertUser>): Promise<User | undefined>;
  updateUserByFirebaseUid(firebaseUid: string, data: { displayName?: string; company?: string; avatarUrl?: string; totpSecret?: string; totpEnabled?: boolean; totpBackupCodes?: string[] }): Promise<User | undefined>;
  deleteUser(id: string): Promise<boolean>;

  // TOTP operations
  getTotpSecret(firebaseUid: string): Promise<string | undefined>;
  setTotpSecret(firebaseUid: string, encryptedSecret: string): Promise<boolean>;
  enableTotp(firebaseUid: string, backupCodes: string[]): Promise<boolean>;
  disableTotp(firebaseUid: string): Promise<boolean>;
  getTotpBackupCodes(firebaseUid: string): Promise<string[] | undefined>;
  updateTotpBackupCodes(firebaseUid: string, backupCodes: string[]): Promise<boolean>;
  useBackupCode(firebaseUid: string, code: string): Promise<boolean>;

  // Quota tracking operations
  updateUserQuota(userId: string, lookupKey: string, quota: number, resetDate: Date): Promise<User | undefined>;
  incrementUserDownloads(userId: string): Promise<{ success: boolean; remainingDownloads: number; quotaExceeded: boolean }>;
  resetUserQuota(userId: string): Promise<User | undefined>;
  getUserQuotaStatus(userId: string): Promise<{ quota: number; used: number; remaining: number; resetDate: Date | null } | undefined>;
  checkQuotaAvailable(userId: string): Promise<boolean>;

  // Brand Kit operations
  getBrandKit(id: string): Promise<BrandKit | undefined>;
  getAllBrandKits(): Promise<BrandKit[]>;
  getBrandKitsByUserId(userId: string): Promise<BrandKit[]>;
  createBrandKit(brandKit: InsertBrandKit): Promise<BrandKit>;
  updateBrandKit(id: string, brandKit: Partial<InsertBrandKit>): Promise<BrandKit | undefined>;
  deleteBrandKit(id: string): Promise<boolean>;

  // Business Name operations
  getBusinessName(id: string): Promise<BusinessName | undefined>;
  getAllBusinessNames(): Promise<BusinessName[]>;
  getBusinessNamesByUserId(userId: string): Promise<BusinessName[]>;
  getSavedBusinessNamesByUserId(userId: string): Promise<BusinessName[]>;
  createBusinessName(businessName: InsertBusinessName): Promise<BusinessName>;
  updateBusinessName(id: string, businessName: Partial<InsertBusinessName>): Promise<BusinessName | undefined>;
  deleteBusinessName(id: string): Promise<boolean>;

  // Social Media Kit operations
  getSocialMediaKit(id: string): Promise<SocialMediaKit | undefined>;
  getSocialMediaKitsByUserId(userId: string): Promise<SocialMediaKit[]>;
  createSocialMediaKit(socialMediaKit: InsertSocialMediaKit): Promise<SocialMediaKit>;
  updateSocialMediaKit(id: string, socialMediaKit: Partial<InsertSocialMediaKit>): Promise<SocialMediaKit | undefined>;
  deleteSocialMediaKit(id: string): Promise<boolean>;

  // Website Template operations
  getWebsiteTemplate(id: string): Promise<WebsiteTemplate | undefined>;
  getWebsiteTemplatesByUserId(userId: string): Promise<WebsiteTemplate[]>;
  createWebsiteTemplate(websiteTemplate: InsertWebsiteTemplate): Promise<WebsiteTemplate>;
  updateWebsiteTemplate(id: string, websiteTemplate: Partial<InsertWebsiteTemplate>): Promise<WebsiteTemplate | undefined>;
  deleteWebsiteTemplate(id: string): Promise<boolean>;

  // User Template Customization operations
  getUserTemplateCustomization(id: string): Promise<UserTemplateCustomization | undefined>;
  getUserTemplateCustomizationsByUserId(userId: string): Promise<UserTemplateCustomization[]>;
  getUserTemplateCustomizationByTemplateId(userId: string, templateId: string): Promise<UserTemplateCustomization | undefined>;
  createUserTemplateCustomization(customization: InsertUserTemplateCustomization): Promise<UserTemplateCustomization>;
  updateUserTemplateCustomization(id: string, customization: Partial<InsertUserTemplateCustomization>): Promise<UserTemplateCustomization | undefined>;
  deleteUserTemplateCustomization(id: string): Promise<boolean>;
  duplicateUserTemplateCustomization(id: string, newName: string): Promise<UserTemplateCustomization | undefined>;

  // Notification operations
  getNotifications(firebaseUid: string): Promise<Notification[]>;
  markNotificationRead(id: string, firebaseUid: string): Promise<boolean>;
  createNotification(userId: string, message: string, type: string): Promise<Notification>;

  // Cancellation operations
  getCancellation(id: string): Promise<Cancellation | undefined>;
  getAllCancellations(): Promise<Cancellation[]>;
  getCancellationsByUserId(userId: string): Promise<Cancellation[]>;
  createCancellation(cancellation: InsertCancellation): Promise<Cancellation>;

  // Pause operations
  getPause(id: string): Promise<Pause | undefined>;
  getAllPauses(): Promise<Pause[]>;
  getPauseByUid(uid: string): Promise<Pause | undefined>;
  getPausesByStripeSubId(stripeSubId: string): Promise<Pause[]>;
  createPause(pause: InsertPause): Promise<Pause>;
  updatePause(id: string, pause: Partial<InsertPause>): Promise<Pause | undefined>;
  deletePause(id: string): Promise<boolean>;
  
  // Pause metrics operations
  getPausedUsersCount(): Promise<number>;
  getRecentPauses(limit?: number): Promise<Array<{
    id: string;
    email: string;
    months: number;
    resumeAt: Date;
    createdAt: Date;
    reason?: string;
    note?: string;
  }>>;

  // Domain Order operations
  getDomainOrder(id: string): Promise<DomainOrder | undefined>;
  getDomainOrdersByUserId(userId: string): Promise<DomainOrder[]>;
  createDomainOrder(domainOrder: InsertDomainOrder): Promise<DomainOrder>;
  updateDomainOrder(id: string, domainOrder: Partial<InsertDomainOrder>): Promise<DomainOrder | undefined>;
  deleteDomainOrder(id: string): Promise<boolean>;

  // Domain Credit operations
  createDomainCredit(credit: InsertDomainCredit): Promise<DomainCredit>;
  getDomainCreditsByUserId(userId: string): Promise<DomainCredit[]>;
  getDomainCreditsBySubscriptionId(subscriptionId: string): Promise<DomainCredit[]>;
  getAvailableDomainCredits(userId: string): Promise<DomainCredit[]>;
  updateDomainCreditStatus(id: string, status: string, usedDomain?: string, usedAt?: Date): Promise<DomainCredit>;
  expireDomainCredits(): Promise<void>;

  // Domain operations
  createDomain(domain: InsertDomain): Promise<Domain>;
  getDomainsByUserId(userId: string): Promise<Domain[]>;
  getDomainByName(domain: string): Promise<Domain | null>;
  updateDomainRenewal(id: string, expiresAt: Date, autorenew?: boolean): Promise<Domain>;

  // Imported Icon operations
  getImportedIcon(id: string): Promise<ImportedIcon | undefined>;
  getAllImportedIcons(): Promise<ImportedIcon[]>;
  getImportedIconsByStyle(style: string): Promise<ImportedIcon[]>;
  searchImportedIcons(query: string): Promise<ImportedIcon[]>;
  createImportedIcon(icon: InsertImportedIcon): Promise<ImportedIcon>;
  createImportedIcons(icons: InsertImportedIcon[]): Promise<ImportedIcon[]>;
  deleteImportedIcon(id: string): Promise<boolean>;

  // Visitor Analytics operations
  getVisitorSessionBySessionId(sessionId: string): Promise<VisitorSession | undefined>;
  createVisitorSession(session: InsertVisitorSession): Promise<VisitorSession>;
  updateVisitorSession(id: string, session: Partial<InsertVisitorSession>): Promise<VisitorSession | undefined>;
  getDailyVisitorStatsByDate(date: string): Promise<DailyVisitorStats | undefined>;
  createDailyVisitorStats(stats: InsertDailyVisitorStats): Promise<DailyVisitorStats>;
  updateDailyVisitorStats(id: string, stats: Partial<InsertDailyVisitorStats>): Promise<DailyVisitorStats | undefined>;
  getRecentVisitorSessions(days?: number, limit?: number): Promise<VisitorSession[]>;
  getDailyVisitorStatsRange(startDate: string, endDate: string): Promise<DailyVisitorStats[]>;
  deleteVisitorSessionsOlderThan(cutoffDate: Date): Promise<void>;

  // Cart operations
  getCartItems(userId?: string, sessionId?: string): Promise<CartItem[]>;
  addCartItem(cartItem: InsertCartItem): Promise<CartItem>;
  updateCartItem(id: string, cartItem: Partial<InsertCartItem>): Promise<CartItem | undefined>;
  removeCartItem(id: string): Promise<boolean>;
  clearCart(userId?: string, sessionId?: string): Promise<void>;

  // Purchase operations
  createPurchase(purchase: InsertPurchase): Promise<Purchase>;
  getPurchase(id: string): Promise<Purchase | undefined>;
  getPurchaseByStripeSessionId(stripeSessionId: string): Promise<Purchase | undefined>;
  updatePurchase(id: string, purchase: Partial<InsertPurchase>): Promise<Purchase | undefined>;
  getUserPurchases(userId: string): Promise<Purchase[]>;

  // User Entitlement operations
  createUserEntitlement(entitlement: InsertUserEntitlement): Promise<UserEntitlement>;
  getUserEntitlements(userId: string): Promise<UserEntitlement[]>;
  getUserEntitlementsByType(userId: string, type: string): Promise<UserEntitlement[]>;
  getUserEntitlementByUserAndItem(userId: string, entitlementType: string, entitlementId: string): Promise<UserEntitlement | undefined>;
  hasEntitlement(userId: string, entitlementType: string, entitlementId: string): Promise<boolean>;
  updateUserEntitlement(id: string, entitlement: Partial<InsertUserEntitlement>): Promise<UserEntitlement | undefined>;

  // Creator Marketplace operations
  
  // Creator operations
  getCreator(id: string): Promise<Creator | undefined>;
  getCreatorByUserId(userId: string): Promise<Creator | undefined>;
  getCreatorByStripeAccountId(stripeAccountId: string): Promise<Creator | undefined>;
  getAllCreators(): Promise<Creator[]>;
  createCreator(creator: InsertCreator): Promise<Creator>;
  updateCreator(id: string, creator: Partial<InsertCreator>): Promise<Creator | undefined>;
  deleteCreator(id: string): Promise<boolean>;
  updateCreatorEarnings(creatorId: string, earningsChange: number): Promise<Creator | undefined>;

  // Asset operations
  getAsset(id: string): Promise<Asset | undefined>;
  getAssetsByCreatorId(creatorId: string): Promise<Asset[]>;
  getAllAssets(): Promise<Asset[]>;
  getPublicAssets(): Promise<Asset[]>;
  createAsset(asset: InsertAsset): Promise<Asset>;
  updateAsset(id: string, asset: Partial<InsertAsset>): Promise<Asset | undefined>;
  deleteAsset(id: string): Promise<boolean>;
  incrementAssetDownloadCount(id: string): Promise<Asset | undefined>;

  // Creator Asset operations
  getCreatorAsset(id: string): Promise<CreatorAsset | undefined>;
  getCreatorAssetsByCreatorId(creatorId: string): Promise<CreatorAsset[]>;
  getCreatorAssetsByAssetId(assetId: string): Promise<CreatorAsset[]>;
  getCreatorAssetsByStatus(status: string): Promise<CreatorAsset[]>;
  getAllCreatorAssets(): Promise<CreatorAsset[]>;
  createCreatorAsset(creatorAsset: InsertCreatorAsset): Promise<CreatorAsset>;
  updateCreatorAsset(id: string, creatorAsset: Partial<InsertCreatorAsset>): Promise<CreatorAsset | undefined>;
  deleteCreatorAsset(id: string): Promise<boolean>;
  approveCreatorAsset(id: string, approvedBy: string): Promise<CreatorAsset | undefined>;
  rejectCreatorAsset(id: string, rejectionReason: string): Promise<CreatorAsset | undefined>;
  incrementCreatorAssetSales(id: string, revenue: number): Promise<CreatorAsset | undefined>;

  // Creator Earning operations
  getCreatorEarning(id: string): Promise<CreatorEarning | undefined>;
  getCreatorEarningsByCreatorId(creatorId: string): Promise<CreatorEarning[]>;
  getCreatorEarningsByStatus(status: string): Promise<CreatorEarning[]>;
  getAllCreatorEarnings(): Promise<CreatorEarning[]>;
  createCreatorEarning(earning: InsertCreatorEarning): Promise<CreatorEarning>;
  updateCreatorEarning(id: string, earning: Partial<InsertCreatorEarning>): Promise<CreatorEarning | undefined>;
  processCreatorPayout(creatorId: string, payoutReference: string): Promise<CreatorEarning[]>;

  // Cover Template operations
  getCoverTemplate(id: string): Promise<CoverTemplate | undefined>;
  getCoverTemplates(opts?: { q?: string; category?: string; min?: number; max?: number; top_tier?: 'FIG'|'TMT'|'Healthcare/Pharma'|'General'; subcat?: string; includeInactive?: boolean }): Promise<CoverTemplate[]>;
  createCoverTemplate(template: InsertCoverTemplate): Promise<CoverTemplate>;
  updateCoverTemplate(id: string, template: Partial<InsertCoverTemplate>): Promise<CoverTemplate | undefined>;
  deleteCoverTemplate(id: string): Promise<boolean>;

  // Cover Purchase operations
  getCoverPurchase(id: string): Promise<CoverPurchase | undefined>;
  getUserCoverPurchases(userId: string): Promise<CoverPurchase[]>;
  createCoverPurchase(purchase: InsertCoverPurchase): Promise<CoverPurchase>;
  updateCoverPurchase(id: string, purchase: Partial<InsertCoverPurchase>): Promise<CoverPurchase | undefined>;
  markPurchasePaidBySession(sessionId: string, paymentIntent: string, downloadUrl: string, actualAmountCents?: number, actualCurrency?: string): Promise<CoverPurchase | undefined>;

  // Infographic Template operations
  getInfographicTemplate(id: string): Promise<InfographicTemplate | undefined>;
  getInfographicTemplates(opts?: { q?: string; category?: string; descriptors?: string[]; min?: number; max?: number; includeInactive?: boolean }): Promise<InfographicTemplate[]>;
  createInfographicTemplate(template: InsertInfographicTemplate): Promise<InfographicTemplate>;
  updateInfographicTemplate(id: string, template: Partial<InsertInfographicTemplate>): Promise<InfographicTemplate | undefined>;
  deleteInfographicTemplate(id: string): Promise<boolean>;

  // Infographic Purchase operations
  getInfographicPurchase(id: string): Promise<InfographicPurchase | undefined>;
  getUserInfographicPurchases(userId: string): Promise<InfographicPurchase[]>;
  createInfographicPurchase(purchase: InsertInfographicPurchase): Promise<InfographicPurchase>;
  updateInfographicPurchase(id: string, purchase: Partial<InsertInfographicPurchase>): Promise<InfographicPurchase | undefined>;
  markInfographicPurchasePaidBySession(sessionId: string, paymentIntent: string, downloadUrl: string, actualAmountCents?: number, actualCurrency?: string): Promise<InfographicPurchase | undefined>;

  // Presentation Template operations
  getPresentationTemplate(id: string): Promise<PresentationTemplate | undefined>;
  getPresentationTemplates(opts?: { q?: string; category?: string; descriptors?: string[]; min?: number; max?: number; includeInactive?: boolean; status?: string }): Promise<PresentationTemplate[]>;
  createPresentationTemplate(template: InsertPresentationTemplate): Promise<PresentationTemplate>;
  updatePresentationTemplate(id: string, template: Partial<InsertPresentationTemplate>): Promise<PresentationTemplate | undefined>;
  deletePresentationTemplate(id: string): Promise<boolean>;

  // Presentation Purchase operations
  getPresentationPurchase(id: string): Promise<PresentationPurchase | undefined>;
  getUserPresentationPurchases(userId: string): Promise<PresentationPurchase[]>;
  createPresentationPurchase(purchase: InsertPresentationPurchase): Promise<PresentationPurchase>;
  updatePresentationPurchase(id: string, purchase: Partial<InsertPresentationPurchase>): Promise<PresentationPurchase | undefined>;
  markPresentationPurchasePaidBySession(sessionId: string, paymentIntent: string, downloadUrl: string, actualAmountCents?: number, actualCurrency?: string): Promise<PresentationPurchase | undefined>;
}

export class PostgreSQLStorage implements IStorage {
  private db: ReturnType<typeof drizzle>;

  constructor() {
    if (!process.env.DATABASE_URL) {
      throw new Error("DATABASE_URL environment variable is required");
    }
    
    // Use HTTP-based connection instead of WebSocket to bypass connection issues in Replit
    const sql = neon(process.env.DATABASE_URL);
    this.db = drizzle(sql);
  }

  // User operations
  async getUser(id: string): Promise<User | undefined> {
    const result = await this.db.select().from(users).where(eq(users.id, id)).limit(1);
    return result[0];
  }

  async getAllUsers(): Promise<User[]> {
    return await this.db.select().from(users);
  }

  async getUserByEmail(email: string): Promise<User | undefined> {
    const result = await this.db.select().from(users).where(eq(users.email, email)).limit(1);
    return result[0];
  }

  async getUserByFirebaseUid(firebaseUid: string): Promise<User | undefined> {
    const result = await this.db.select().from(users).where(eq(users.firebaseUid, firebaseUid)).limit(1);
    return result[0];
  }

  async createUser(user: InsertUser): Promise<User> {
    const result = await this.db.insert(users).values(user).returning();
    return result[0];
  }

  async updateUser(id: string, user: Partial<InsertUser>): Promise<User | undefined> {
    const result = await this.db.update(users).set(user).where(eq(users.id, id)).returning();
    return result[0];
  }

  async updateUserByFirebaseUid(firebaseUid: string, data: { displayName?: string; company?: string; avatarUrl?: string; totpSecret?: string; totpEnabled?: boolean; totpBackupCodes?: string[] }): Promise<User | undefined> {
    const result = await this.db.update(users).set(data).where(eq(users.firebaseUid, firebaseUid)).returning();
    return result[0];
  }

  async getUserPreferences(firebaseUid: string) {
    try {
      // Import sql from drizzle-orm for raw queries
      const { sql } = await import("drizzle-orm");
      
      // First, ensure the preferences table exists
      await this.db.execute(sql`
        CREATE TABLE IF NOT EXISTS user_preferences (
          user_uid TEXT PRIMARY KEY,
          email_news BOOLEAN DEFAULT true,
          marketing_emails BOOLEAN DEFAULT false,
          product_updates BOOLEAN DEFAULT true,
          created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
          updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
      `);

      const result = await this.db.execute(sql`
        SELECT email_news, marketing_emails, product_updates,
               system_notifications, billing_notifications, project_notifications
        FROM user_preferences 
        WHERE user_uid = ${firebaseUid}
      `);

      if (result.rows && result.rows.length > 0) {
        const row = result.rows[0] as any;
        return {
          emailNews: !!row.email_news,
          marketingEmails: !!row.marketing_emails,
          productUpdates: !!row.product_updates,
          systemNotifications: !!row.system_notifications,
          billingNotifications: !!row.billing_notifications,
          projectNotifications: !!row.project_notifications
        };
      }

      // Return defaults if no preferences found
      return {
        emailNews: true,
        marketingEmails: false,
        productUpdates: true,
        systemNotifications: true,
        billingNotifications: true,
        projectNotifications: true
      };
    } catch (error) {
      console.error('Get user preferences error:', error);
      return {
        emailNews: true,
        marketingEmails: false,
        productUpdates: true
      };
    }
  }

  async setUserPreferences(firebaseUid: string, prefs: { emailNews: boolean; marketingEmails: boolean; productUpdates: boolean; systemNotifications?: boolean; billingNotifications?: boolean; projectNotifications?: boolean }) {
    try {
      const { sql } = await import("drizzle-orm");
      
      // First, ensure the preferences table has the new notification columns
      await this.db.execute(sql`
        ALTER TABLE user_preferences 
        ADD COLUMN IF NOT EXISTS system_notifications BOOLEAN DEFAULT true,
        ADD COLUMN IF NOT EXISTS billing_notifications BOOLEAN DEFAULT true,
        ADD COLUMN IF NOT EXISTS project_notifications BOOLEAN DEFAULT true
      `);
      
      await this.db.execute(sql`
        INSERT INTO user_preferences (
          user_uid, email_news, marketing_emails, product_updates, 
          system_notifications, billing_notifications, project_notifications, updated_at
        )
        VALUES (
          ${firebaseUid}, ${prefs.emailNews}, ${prefs.marketingEmails}, ${prefs.productUpdates},
          ${prefs.systemNotifications ?? true}, ${prefs.billingNotifications ?? true}, ${prefs.projectNotifications ?? true},
          CURRENT_TIMESTAMP
        )
        ON CONFLICT(user_uid) DO UPDATE SET
          email_news = EXCLUDED.email_news,
          marketing_emails = EXCLUDED.marketing_emails,
          product_updates = EXCLUDED.product_updates,
          system_notifications = EXCLUDED.system_notifications,
          billing_notifications = EXCLUDED.billing_notifications,
          project_notifications = EXCLUDED.project_notifications,
          updated_at = CURRENT_TIMESTAMP
      `);
      return true;
    } catch (error) {
      console.error('Set user preferences error:', error);
      return false;
    }
  }

  async deleteUser(id: string): Promise<boolean> {
    const result = await this.db.delete(users).where(eq(users.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  // Brand Kit operations
  async getBrandKit(id: string): Promise<BrandKit | undefined> {
    const result = await this.db.select().from(brandKits).where(eq(brandKits.id, id)).limit(1);
    return result[0];
  }

  async getAllBrandKits(): Promise<BrandKit[]> {
    return await this.db.select().from(brandKits);
  }

  async getBrandKitsByUserId(userId: string): Promise<BrandKit[]> {
    return await this.db.select().from(brandKits).where(eq(brandKits.userId, userId));
  }

  async createBrandKit(brandKit: InsertBrandKit): Promise<BrandKit> {
    const result = await this.db.insert(brandKits).values({
      ...brandKit,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateBrandKit(id: string, brandKit: Partial<InsertBrandKit>): Promise<BrandKit | undefined> {
    const result = await this.db.update(brandKits).set({
      ...brandKit,
      updatedAt: new Date()
    }).where(eq(brandKits.id, id)).returning();
    return result[0];
  }

  async deleteBrandKit(id: string): Promise<boolean> {
    const result = await this.db.delete(brandKits).where(eq(brandKits.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  // Business Name operations
  async getBusinessName(id: string): Promise<BusinessName | undefined> {
    const result = await this.db.select().from(businessNames).where(eq(businessNames.id, id)).limit(1);
    return result[0];
  }

  async getAllBusinessNames(): Promise<BusinessName[]> {
    return await this.db.select().from(businessNames);
  }

  async getBusinessNamesByUserId(userId: string): Promise<BusinessName[]> {
    return await this.db.select().from(businessNames).where(eq(businessNames.userId, userId));
  }

  async getSavedBusinessNamesByUserId(userId: string): Promise<BusinessName[]> {
    return await this.db.select().from(businessNames)
      .where(and(eq(businessNames.userId, userId), eq(businessNames.isSaved, true)));
  }

  async createBusinessName(businessName: InsertBusinessName): Promise<BusinessName> {
    const result = await this.db.insert(businessNames).values(businessName).returning();
    return result[0];
  }

  async updateBusinessName(id: string, businessName: Partial<InsertBusinessName>): Promise<BusinessName | undefined> {
    const result = await this.db.update(businessNames).set(businessName).where(eq(businessNames.id, id)).returning();
    return result[0];
  }

  async deleteBusinessName(id: string): Promise<boolean> {
    const result = await this.db.delete(businessNames).where(eq(businessNames.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  // Social Media Kit operations
  async getSocialMediaKit(id: string): Promise<SocialMediaKit | undefined> {
    const result = await this.db.select().from(socialMediaKits).where(eq(socialMediaKits.id, id)).limit(1);
    return result[0];
  }

  async getSocialMediaKitsByUserId(userId: string): Promise<SocialMediaKit[]> {
    return await this.db.select().from(socialMediaKits).where(eq(socialMediaKits.userId, userId));
  }

  async createSocialMediaKit(socialMediaKit: InsertSocialMediaKit): Promise<SocialMediaKit> {
    const result = await this.db.insert(socialMediaKits).values({
      ...socialMediaKit,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateSocialMediaKit(id: string, socialMediaKit: Partial<InsertSocialMediaKit>): Promise<SocialMediaKit | undefined> {
    const result = await this.db.update(socialMediaKits).set({
      ...socialMediaKit,
      updatedAt: new Date()
    }).where(eq(socialMediaKits.id, id)).returning();
    return result[0];
  }

  async deleteSocialMediaKit(id: string): Promise<boolean> {
    const result = await this.db.delete(socialMediaKits).where(eq(socialMediaKits.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  // Website Template operations
  async getWebsiteTemplate(id: string): Promise<WebsiteTemplate | undefined> {
    const result = await this.db.select().from(websiteTemplates).where(eq(websiteTemplates.id, id)).limit(1);
    return result[0];
  }

  async getWebsiteTemplatesByUserId(userId: string): Promise<WebsiteTemplate[]> {
    return await this.db.select().from(websiteTemplates).where(eq(websiteTemplates.userId, userId));
  }

  async createWebsiteTemplate(websiteTemplate: InsertWebsiteTemplate): Promise<WebsiteTemplate> {
    const result = await this.db.insert(websiteTemplates).values({
      ...websiteTemplate,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateWebsiteTemplate(id: string, websiteTemplate: Partial<InsertWebsiteTemplate>): Promise<WebsiteTemplate | undefined> {
    const result = await this.db.update(websiteTemplates).set({
      ...websiteTemplate,
      updatedAt: new Date()
    }).where(eq(websiteTemplates.id, id)).returning();
    return result[0];
  }

  async deleteWebsiteTemplate(id: string): Promise<boolean> {
    const result = await this.db.delete(websiteTemplates).where(eq(websiteTemplates.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  // User Template Customization operations
  async getUserTemplateCustomization(id: string): Promise<UserTemplateCustomization | undefined> {
    const result = await this.db.select().from(userTemplateCustomizations).where(eq(userTemplateCustomizations.id, id)).limit(1);
    return result[0];
  }

  async getUserTemplateCustomizationsByUserId(userId: string): Promise<UserTemplateCustomization[]> {
    return await this.db.select().from(userTemplateCustomizations).where(eq(userTemplateCustomizations.userId, userId));
  }

  async getUserTemplateCustomizationByTemplateId(userId: string, templateId: string): Promise<UserTemplateCustomization | undefined> {
    const result = await this.db.select().from(userTemplateCustomizations)
      .where(and(eq(userTemplateCustomizations.userId, userId), eq(userTemplateCustomizations.templateId, templateId)))
      .limit(1);
    return result[0];
  }

  async createUserTemplateCustomization(customization: InsertUserTemplateCustomization): Promise<UserTemplateCustomization> {
    const result = await this.db.insert(userTemplateCustomizations).values({
      ...customization,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateUserTemplateCustomization(id: string, customization: Partial<InsertUserTemplateCustomization>): Promise<UserTemplateCustomization | undefined> {
    const result = await this.db.update(userTemplateCustomizations).set({
      ...customization,
      updatedAt: new Date()
    }).where(eq(userTemplateCustomizations.id, id)).returning();
    return result[0];
  }

  async deleteUserTemplateCustomization(id: string): Promise<boolean> {
    const result = await this.db.delete(userTemplateCustomizations).where(eq(userTemplateCustomizations.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  async duplicateUserTemplateCustomization(id: string, newName: string): Promise<UserTemplateCustomization | undefined> {
    const original = await this.getUserTemplateCustomization(id);
    if (!original) return undefined;

    const duplicateData: InsertUserTemplateCustomization = {
      userId: original.userId,
      templateId: original.templateId,
      customizationName: newName,
      configurations: original.configurations as any,
      contentOverrides: original.contentOverrides as any,
      styleOverrides: original.styleOverrides as any,
      isTemplate: false,
      isFavorite: false
    };

    return await this.createUserTemplateCustomization(duplicateData);
  }

  // Notification operations
  async getNotifications(firebaseUid: string): Promise<Notification[]> {
    const user = await this.getUserByFirebaseUid(firebaseUid);
    if (!user) return [];
    
    const result = await this.db.select()
      .from(notifications)
      .where(eq(notifications.userId, user.id))
      .orderBy(sql`${notifications.createdAt} DESC`)
      .limit(50);
    return result;
  }

  async markNotificationRead(id: string, firebaseUid: string): Promise<boolean> {
    const user = await this.getUserByFirebaseUid(firebaseUid);
    if (!user) return false;
    
    const result = await this.db.update(notifications)
      .set({ isRead: true })
      .where(and(eq(notifications.id, id), eq(notifications.userId, user.id)))
      .returning();
    return result.length > 0;
  }

  async createNotification(userId: string, message: string, type: string): Promise<Notification> {
    try {
      const result = await this.db.insert(notifications).values({
        userId,
        message,
        type
      }).returning();
      
      const notification = result[0];
      
      // FIXED: Use standalone broadcaster module instead of coupling to routes
      // Get user's Firebase UID for broadcasting
      const user = await this.getUser(userId);
      if (user && user.firebaseUid) {
        notificationBroadcaster.sendNotificationToUser(user.firebaseUid, {
          id: notification.id,
          userId: user.firebaseUid, // Use Firebase UID for frontend matching
          message: notification.message,
          type: notification.type as any,
          isRead: notification.isRead ?? false,
          createdAt: notification.createdAt?.toISOString() ?? new Date().toISOString()
        });
      }
      
      return notification;
    } catch (error) {
      console.error('Failed to create notification:', error);
      throw error;
    }
  }

  // Cancellation operations
  async getCancellation(id: string): Promise<Cancellation | undefined> {
    const result = await this.db.select().from(cancellations).where(eq(cancellations.id, id)).limit(1);
    return result[0];
  }

  async getAllCancellations(): Promise<Cancellation[]> {
    return await this.db.select().from(cancellations);
  }

  async getCancellationsByUserId(userId: string): Promise<Cancellation[]> {
    return await this.db.select().from(cancellations).where(eq(cancellations.userId, userId));
  }

  async createCancellation(cancellation: InsertCancellation): Promise<Cancellation> {
    const result = await this.db.insert(cancellations).values(cancellation).returning();
    return result[0];
  }

  // Pause operations
  async getPause(id: string): Promise<Pause | undefined> {
    const result = await this.db.select().from(pauses).where(eq(pauses.id, id)).limit(1);
    return result[0];
  }

  async getAllPauses(): Promise<Pause[]> {
    return await this.db.select().from(pauses);
  }

  async getPauseByUid(uid: string): Promise<Pause | undefined> {
    const result = await this.db.select().from(pauses).where(eq(pauses.uid, uid)).limit(1);
    return result[0];
  }

  async getPausesByStripeSubId(stripeSubId: string): Promise<Pause[]> {
    return await this.db.select().from(pauses).where(eq(pauses.stripeSubId, stripeSubId));
  }

  async createPause(pause: InsertPause): Promise<Pause> {
    const result = await this.db.insert(pauses).values(pause).returning();
    return result[0];
  }

  async updatePause(id: string, pause: Partial<InsertPause>): Promise<Pause | undefined> {
    const result = await this.db.update(pauses).set(pause).where(eq(pauses.id, id)).returning();
    return result[0];
  }

  async deletePause(id: string): Promise<boolean> {
    const result = await this.db.delete(pauses).where(eq(pauses.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  // Pause metrics operations
  async getPausedUsersCount(): Promise<number> {
    const result = await this.db.select({ count: sql<number>`count(*)` })
      .from(users)
      .where(eq(users.subscriptionStatus, 'paused'));
    return result[0]?.count || 0;
  }

  async getRecentPauses(limit: number = 10): Promise<Array<{
    id: string;
    email: string;
    months: number;
    resumeAt: Date;
    createdAt: Date;
    reason?: string;
    note?: string;
  }>> {
    const result = await this.db.select({
      id: pauses.id,
      email: pauses.email,
      months: pauses.months,
      resumeAt: pauses.resumeAt,
      createdAt: pauses.createdAt,
      reason: pauses.reason,
      note: pauses.note
    })
    .from(pauses)
    .orderBy(sql`${pauses.createdAt} DESC`)
    .limit(limit);
    
    return result.map(p => ({
      id: p.id,
      email: p.email || '',
      months: p.months,
      resumeAt: p.resumeAt,
      createdAt: p.createdAt || new Date(),
      reason: p.reason || undefined,
      note: p.note || undefined
    }));
  }

  // Domain Order operations
  async getDomainOrder(id: string): Promise<DomainOrder | undefined> {
    const result = await this.db.select().from(domainOrders).where(eq(domainOrders.id, id)).limit(1);
    return result[0];
  }

  async getDomainOrdersByUserId(userId: string): Promise<DomainOrder[]> {
    return await this.db.select().from(domainOrders).where(eq(domainOrders.userId, userId));
  }

  async createDomainOrder(domainOrder: InsertDomainOrder): Promise<DomainOrder> {
    const result = await this.db.insert(domainOrders).values({
      ...domainOrder,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateDomainOrder(id: string, domainOrder: Partial<InsertDomainOrder>): Promise<DomainOrder | undefined> {
    const result = await this.db.update(domainOrders).set({
      ...domainOrder,
      updatedAt: new Date()
    }).where(eq(domainOrders.id, id)).returning();
    return result[0];
  }

  async deleteDomainOrder(id: string): Promise<boolean> {
    const result = await this.db.delete(domainOrders).where(eq(domainOrders.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  // TOTP operations
  async getTotpSecret(firebaseUid: string): Promise<string | undefined> {
    const user = await this.getUserByFirebaseUid(firebaseUid);
    return user?.totpSecret || undefined;
  }

  async setTotpSecret(firebaseUid: string, encryptedSecret: string): Promise<boolean> {
    try {
      const result = await this.db.update(users)
        .set({ totpSecret: encryptedSecret })
        .where(eq(users.firebaseUid, firebaseUid))
        .returning();
      return result.length > 0;
    } catch (error) {
      console.error('Set TOTP secret error:', error);
      return false;
    }
  }

  async enableTotp(firebaseUid: string, backupCodes: string[]): Promise<boolean> {
    try {
      const result = await this.db.update(users)
        .set({ 
          totpEnabled: true,
          totpBackupCodes: backupCodes
        })
        .where(eq(users.firebaseUid, firebaseUid))
        .returning();
      return result.length > 0;
    } catch (error) {
      console.error('Enable TOTP error:', error);
      return false;
    }
  }

  async disableTotp(firebaseUid: string): Promise<boolean> {
    try {
      const result = await this.db.update(users)
        .set({ 
          totpEnabled: false,
          totpSecret: null,
          totpBackupCodes: null
        })
        .where(eq(users.firebaseUid, firebaseUid))
        .returning();
      return result.length > 0;
    } catch (error) {
      console.error('Disable TOTP error:', error);
      return false;
    }
  }

  async getTotpBackupCodes(firebaseUid: string): Promise<string[] | undefined> {
    const user = await this.getUserByFirebaseUid(firebaseUid);
    return user?.totpBackupCodes || undefined;
  }

  async updateTotpBackupCodes(firebaseUid: string, backupCodes: string[]): Promise<boolean> {
    try {
      const result = await this.db.update(users)
        .set({ totpBackupCodes: backupCodes })
        .where(eq(users.firebaseUid, firebaseUid))
        .returning();
      return result.length > 0;
    } catch (error) {
      console.error('Update TOTP backup codes error:', error);
      return false;
    }
  }

  async useBackupCode(firebaseUid: string, code: string): Promise<boolean> {
    try {
      const user = await this.getUserByFirebaseUid(firebaseUid);
      if (!user?.totpBackupCodes) return false;

      const backupCodes = user.totpBackupCodes;
      const codeIndex = backupCodes.indexOf(code);
      
      if (codeIndex === -1) return false;

      // Remove the used backup code
      const updatedCodes = backupCodes.filter((_, index) => index !== codeIndex);
      
      await this.updateTotpBackupCodes(firebaseUid, updatedCodes);

      return true;
    } catch (error) {
      console.error('Use backup code error:', error);
      return false;
    }
  }

  // Quota tracking operations
  async updateUserQuota(userId: string, lookupKey: string, quota: number, resetDate: Date): Promise<User | undefined> {
    try {
      const result = await this.db.update(users)
        .set({
          monthlyDownloadQuota: quota,
          currentMonthDownloads: 0, // Reset downloads when quota is updated
          quotaResetDate: resetDate,
          subscriptionLookupKey: lookupKey
        })
        .where(eq(users.id, userId))
        .returning();
      
      console.log(`📊 Updated quota for user ${userId}: ${quota} downloads/month (${lookupKey})`);
      return result[0];
    } catch (error) {
      console.error('Update user quota error:', error);
      return undefined;
    }
  }

  async incrementUserDownloads(userId: string): Promise<{ success: boolean; remainingDownloads: number; quotaExceeded: boolean }> {
    try {
      const user = await this.getUser(userId);
      if (!user) {
        return { success: false, remainingDownloads: 0, quotaExceeded: true };
      }

      const currentDownloads = user.currentMonthDownloads || 0;
      const quota = user.monthlyDownloadQuota || 0;

      // Check if quota would be exceeded
      if (currentDownloads >= quota && quota > 0) {
        return { success: false, remainingDownloads: 0, quotaExceeded: true };
      }

      // Increment download count
      const result = await this.db.update(users)
        .set({ currentMonthDownloads: currentDownloads + 1 })
        .where(eq(users.id, userId))
        .returning();

      const remainingDownloads = Math.max(0, quota - (currentDownloads + 1));
      
      console.log(`📈 User ${userId} download count: ${currentDownloads + 1}/${quota} (${remainingDownloads} remaining)`);
      
      return { 
        success: true, 
        remainingDownloads, 
        quotaExceeded: false 
      };
    } catch (error) {
      console.error('Increment user downloads error:', error);
      return { success: false, remainingDownloads: 0, quotaExceeded: true };
    }
  }

  async resetUserQuota(userId: string): Promise<User | undefined> {
    try {
      // Calculate next reset date (first day of next month)
      const nextResetDate = new Date();
      nextResetDate.setMonth(nextResetDate.getMonth() + 1);
      nextResetDate.setDate(1);
      nextResetDate.setHours(0, 0, 0, 0);

      const result = await this.db.update(users)
        .set({
          currentMonthDownloads: 0,
          quotaResetDate: nextResetDate
        })
        .where(eq(users.id, userId))
        .returning();
      
      console.log(`🔄 Reset quota for user ${userId}, next reset: ${nextResetDate.toISOString()}`);
      return result[0];
    } catch (error) {
      console.error('Reset user quota error:', error);
      return undefined;
    }
  }

  async getUserQuotaStatus(userId: string): Promise<{ quota: number; used: number; remaining: number; resetDate: Date | null } | undefined> {
    try {
      const user = await this.getUser(userId);
      if (!user) return undefined;

      const quota = user.monthlyDownloadQuota || 0;
      const used = user.currentMonthDownloads || 0;
      const remaining = Math.max(0, quota - used);
      const resetDate = user.quotaResetDate;

      return { quota, used, remaining, resetDate };
    } catch (error) {
      console.error('Get user quota status error:', error);
      return undefined;
    }
  }

  async checkQuotaAvailable(userId: string): Promise<boolean> {
    try {
      const status = await this.getUserQuotaStatus(userId);
      if (!status) return false;

      // If quota is 0 (unlimited), allow downloads
      if (status.quota === 0) return true;

      return status.remaining > 0;
    } catch (error) {
      console.error('Check quota available error:', error);
      return false;
    }
  }

  // Domain Credit operations
  async createDomainCredit(credit: InsertDomainCredit): Promise<DomainCredit> {
    try {
      // Try to insert new credit with conflict resolution for idempotency
      const result = await this.db.insert(domainCredits)
        .values(credit)
        .returning();
      return result[0];
    } catch (error: any) {
      // Handle unique constraint violation on sourceSubscriptionId
      if (error.code === '23505' && error.constraint?.includes('source_subscription_id')) {
        console.log(`⚠️ Domain credit already exists for subscription ${credit.sourceSubscriptionId}, returning existing credit`);
        // Return the existing credit for this subscription
        const existing = await this.getDomainCreditsBySubscriptionId(credit.sourceSubscriptionId!);
        if (existing.length > 0) {
          return existing[0];
        }
      }
      // Re-throw other errors
      throw error;
    }
  }

  async getDomainCreditsByUserId(userId: string): Promise<DomainCredit[]> {
    return await this.db.select().from(domainCredits)
      .where(eq(domainCredits.userId, userId))
      .orderBy(sql`${domainCredits.issuedAt} DESC`);
  }

  async getDomainCreditsBySubscriptionId(subscriptionId: string): Promise<DomainCredit[]> {
    return await this.db.select().from(domainCredits)
      .where(eq(domainCredits.sourceSubscriptionId, subscriptionId))
      .orderBy(sql`${domainCredits.issuedAt} DESC`);
  }

  async getAvailableDomainCredits(userId: string): Promise<DomainCredit[]> {
    const now = new Date();
    return await this.db.select().from(domainCredits)
      .where(
        and(
          eq(domainCredits.userId, userId),
          eq(domainCredits.status, 'available'),
          sql`${domainCredits.expiresAt} > ${now}`
        )
      )
      .orderBy(sql`${domainCredits.expiresAt} ASC`);
  }

  async updateDomainCreditStatus(id: string, status: string, usedDomain?: string, usedAt?: Date): Promise<DomainCredit> {
    const updateData: Partial<InsertDomainCredit> = { 
      status: status as any
    };
    
    if (status === 'used' && usedDomain && usedAt) {
      updateData.usedDomain = usedDomain;
      updateData.usedAt = usedAt;
    }

    const result = await this.db.update(domainCredits)
      .set(updateData)
      .where(eq(domainCredits.id, id))
      .returning();
    return result[0];
  }

  async expireDomainCredits(): Promise<void> {
    const now = new Date();
    await this.db.update(domainCredits)
      .set({ status: 'expired' })
      .where(
        and(
          eq(domainCredits.status, 'available'),
          sql`${domainCredits.expiresAt} <= ${now}`
        )
      );
  }

  // Domain operations
  async createDomain(domain: InsertDomain): Promise<Domain> {
    const result = await this.db.insert(domains).values({
      ...domain,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async getDomainsByUserId(userId: string): Promise<Domain[]> {
    return await this.db.select().from(domains)
      .where(eq(domains.userId, userId))
      .orderBy(sql`${domains.createdAt} DESC`);
  }

  async getDomainByName(domain: string): Promise<Domain | null> {
    const result = await this.db.select().from(domains)
      .where(eq(domains.domain, domain.toLowerCase()))
      .limit(1);
    return result[0] || null;
  }

  async updateDomainRenewal(id: string, expiresAt: Date, autorenew?: boolean): Promise<Domain> {
    const updateData: Partial<InsertDomain> = { 
      expiresAt,
      updatedAt: new Date()
    };
    
    if (autorenew !== undefined) {
      updateData.autorenew = autorenew;
    }

    const result = await this.db.update(domains)
      .set(updateData)
      .where(eq(domains.id, id))
      .returning();
    return result[0];
  }

  // Imported Icon operations
  async getImportedIcon(id: string): Promise<ImportedIcon | undefined> {
    const result = await this.db.select().from(importedIcons).where(eq(importedIcons.id, id)).limit(1);
    return result[0];
  }

  async getAllImportedIcons(): Promise<ImportedIcon[]> {
    return await this.db.select().from(importedIcons)
      .orderBy(sql`${importedIcons.createdAt} DESC`);
  }

  async getImportedIconsByStyle(style: string): Promise<ImportedIcon[]> {
    return await this.db.select().from(importedIcons)
      .where(eq(importedIcons.style, style))
      .orderBy(sql`${importedIcons.createdAt} DESC`);
  }

  async searchImportedIcons(query: string): Promise<ImportedIcon[]> {
    const searchTerm = `%${query.toLowerCase()}%`;
    return await this.db.select().from(importedIcons)
      .where(
        sql`LOWER(${importedIcons.name}) LIKE ${searchTerm} OR 
            EXISTS (
              SELECT 1 FROM unnest(${importedIcons.tags}) AS tag 
              WHERE LOWER(tag) LIKE ${searchTerm}
            )`
      )
      .orderBy(sql`${importedIcons.createdAt} DESC`);
  }

  async createImportedIcon(icon: InsertImportedIcon): Promise<ImportedIcon> {
    const result = await this.db.insert(importedIcons).values(icon).returning();
    return result[0];
  }

  async createImportedIcons(icons: InsertImportedIcon[]): Promise<ImportedIcon[]> {
    const result = await this.db.insert(importedIcons).values(icons).returning();
    return result;
  }

  async deleteImportedIcon(id: string): Promise<boolean> {
    try {
      // First, get the icon to check if it exists and get its name for file deletion
      const icon = await this.getImportedIcon(id);
      
      if (!icon) {
        return false; // Icon not found
      }

      // Import ObjectStorageService at the top if not already imported
      const { ObjectStorageService } = await import("./objectStorage");
      const objectStorageService = new ObjectStorageService();

      // Delete all related files from object storage
      // Icons can be stored in multiple formats/locations:
      const iconName = icon.name.toLowerCase().replace(/[^a-z0-9]/g, '_');
      const patterns = [
        `icons/preview/${id}`,      // Preview files by ID
        `icons/svg/${id}`,          // SVG files by ID  
        `icons/png/${id}`,          // PNG files by ID
        `icons/preview/*/${iconName}`, // Preview files by name in UUID folders
        `icons/svg/*/${iconName}`,     // SVG files by name in UUID folders
        `icons/png/*/${iconName}`      // PNG files by name in UUID folders
      ];

      let filesDeleted = false;
      for (const pattern of patterns) {
        try {
          const deleted = await objectStorageService.deleteFilesByPattern(pattern);
          if (deleted) {
            filesDeleted = true;
            console.log(`🗑️ Deleted object storage files matching pattern: ${pattern}`);
          }
        } catch (error) {
          console.warn(`Failed to delete files matching pattern ${pattern}:`, error);
        }
      }

      // Try to delete files using exact icon ID as well
      try {
        const exactPatterns = [
          `icons/preview/${id}.png`,
          `icons/svg/${id}.svg`,
          `icons/png/${id}.png`
        ];
        
        for (const exactPattern of exactPatterns) {
          try {
            const deleted = await objectStorageService.deleteFile(exactPattern);
            if (deleted) {
              filesDeleted = true;
              console.log(`🗑️ Deleted exact file: ${exactPattern}`);
            }
          } catch (error) {
            // Silent fail for exact files - they might not exist
          }
        }
      } catch (error) {
        console.warn(`Error during exact file deletion:`, error);
      }

      // Delete the database record
      const result = await this.db.delete(importedIcons).where(eq(importedIcons.id, id));
      const dbDeleted = (result.rowCount ?? 0) > 0;

      if (dbDeleted) {
        console.log(`✅ Deleted icon "${icon.name}" (ID: ${id}) from database and object storage`);
      }

      return dbDeleted;
    } catch (error) {
      console.error(`Error deleting imported icon ${id}:`, error);
      return false;
    }
  }

  // Visitor Analytics operations
  async getVisitorSessionBySessionId(sessionId: string): Promise<VisitorSession | undefined> {
    const result = await this.db.select().from(visitorSessions)
      .where(eq(visitorSessions.sessionId, sessionId))
      .limit(1);
    return result[0];
  }

  async createVisitorSession(session: InsertVisitorSession): Promise<VisitorSession> {
    const result = await this.db.insert(visitorSessions).values({
      ...session,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateVisitorSession(id: string, session: Partial<InsertVisitorSession>): Promise<VisitorSession | undefined> {
    const result = await this.db.update(visitorSessions).set({
      ...session,
      updatedAt: new Date()
    }).where(eq(visitorSessions.id, id)).returning();
    return result[0];
  }

  async getDailyVisitorStatsByDate(date: string): Promise<DailyVisitorStats | undefined> {
    const result = await this.db.select().from(dailyVisitorStats)
      .where(eq(dailyVisitorStats.date, date))
      .limit(1);
    return result[0];
  }

  async createDailyVisitorStats(stats: InsertDailyVisitorStats): Promise<DailyVisitorStats> {
    const result = await this.db.insert(dailyVisitorStats).values({
      ...stats,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateDailyVisitorStats(id: string, stats: Partial<InsertDailyVisitorStats>): Promise<DailyVisitorStats | undefined> {
    const result = await this.db.update(dailyVisitorStats).set({
      ...stats,
      updatedAt: new Date()
    }).where(eq(dailyVisitorStats.id, id)).returning();
    return result[0];
  }

  async getRecentVisitorSessions(days: number = 7, limit: number = 100): Promise<VisitorSession[]> {
    const cutoffDate = new Date();
    cutoffDate.setDate(cutoffDate.getDate() - days);
    
    return await this.db.select().from(visitorSessions)
      .where(sql`${visitorSessions.createdAt} >= ${cutoffDate}`)
      .orderBy(sql`${visitorSessions.createdAt} DESC`)
      .limit(limit);
  }

  async getDailyVisitorStatsRange(startDate: string, endDate: string): Promise<DailyVisitorStats[]> {
    return await this.db.select().from(dailyVisitorStats)
      .where(
        and(
          sql`${dailyVisitorStats.date} >= ${startDate}`,
          sql`${dailyVisitorStats.date} <= ${endDate}`
        )
      )
      .orderBy(sql`${dailyVisitorStats.date} ASC`);
  }

  async deleteVisitorSessionsOlderThan(cutoffDate: Date): Promise<void> {
    await this.db.delete(visitorSessions)
      .where(sql`${visitorSessions.createdAt} < ${cutoffDate}`);
  }

  // Cart operations
  async getCartItems(userId?: string, sessionId?: string): Promise<CartItem[]> {
    if (!userId && !sessionId) {
      return [];
    }
    
    const whereClause = userId 
      ? eq(cartItems.userId, userId)
      : sessionId 
      ? eq(cartItems.sessionId, sessionId)
      : sql`false`; // No matches if neither provided

    return await this.db.select().from(cartItems).where(whereClause);
  }

  async addCartItem(cartItem: InsertCartItem): Promise<CartItem> {
    const result = await this.db.insert(cartItems).values(cartItem).returning();
    return result[0];
  }

  async updateCartItem(id: string, cartItem: Partial<InsertCartItem>): Promise<CartItem | undefined> {
    const result = await this.db.update(cartItems)
      .set({ ...cartItem, updatedAt: new Date() })
      .where(eq(cartItems.id, id))
      .returning();
    return result[0];
  }

  async removeCartItem(id: string): Promise<boolean> {
    const result = await this.db.delete(cartItems).where(eq(cartItems.id, id)).returning();
    return result.length > 0;
  }

  async clearCart(userId?: string, sessionId?: string): Promise<void> {
    if (!userId && !sessionId) {
      return;
    }
    
    const whereClause = userId 
      ? eq(cartItems.userId, userId)
      : sessionId 
      ? eq(cartItems.sessionId, sessionId)
      : sql`false`;

    await this.db.delete(cartItems).where(whereClause);
  }

  // Purchase operations
  async createPurchase(purchase: InsertPurchase): Promise<Purchase> {
    const result = await this.db.insert(purchases).values(purchase).returning();
    return result[0];
  }

  async getPurchase(id: string): Promise<Purchase | undefined> {
    const result = await this.db.select().from(purchases).where(eq(purchases.id, id)).limit(1);
    return result[0];
  }

  async getPurchaseByStripeSessionId(stripeSessionId: string): Promise<Purchase | undefined> {
    const result = await this.db.select().from(purchases).where(eq(purchases.stripeSessionId, stripeSessionId)).limit(1);
    return result[0];
  }

  async updatePurchase(id: string, purchase: Partial<InsertPurchase>): Promise<Purchase | undefined> {
    const result = await this.db.update(purchases)
      .set(purchase)
      .where(eq(purchases.id, id))
      .returning();
    return result[0];
  }

  async getUserPurchases(userId: string): Promise<Purchase[]> {
    return await this.db.select().from(purchases).where(eq(purchases.userId, userId));
  }

  // User Entitlement operations
  async createUserEntitlement(entitlement: InsertUserEntitlement): Promise<UserEntitlement> {
    const result = await this.db.insert(userEntitlements).values(entitlement).returning();
    return result[0];
  }

  async getUserEntitlements(userId: string): Promise<UserEntitlement[]> {
    return await this.db.select().from(userEntitlements).where(eq(userEntitlements.userId, userId));
  }

  async getUserEntitlementsByType(userId: string, type: string): Promise<UserEntitlement[]> {
    return await this.db.select().from(userEntitlements)
      .where(and(
        eq(userEntitlements.userId, userId),
        eq(userEntitlements.entitlementType, type)
      ));
  }

  async getUserEntitlementByUserAndItem(userId: string, entitlementType: string, entitlementId: string): Promise<UserEntitlement | undefined> {
    const result = await this.db.select().from(userEntitlements)
      .where(and(
        eq(userEntitlements.userId, userId),
        eq(userEntitlements.entitlementType, entitlementType),
        eq(userEntitlements.entitlementId, entitlementId),
        eq(userEntitlements.status, 'active')
      ))
      .limit(1);
    return result[0];
  }

  async hasEntitlement(userId: string, entitlementType: string, entitlementId: string): Promise<boolean> {
    const result = await this.db.select().from(userEntitlements)
      .where(and(
        eq(userEntitlements.userId, userId),
        eq(userEntitlements.entitlementType, entitlementType),
        eq(userEntitlements.entitlementId, entitlementId),
        eq(userEntitlements.status, 'active')
      ))
      .limit(1);
    return result.length > 0;
  }

  async updateUserEntitlement(id: string, entitlement: Partial<InsertUserEntitlement>): Promise<UserEntitlement | undefined> {
    const result = await this.db.update(userEntitlements)
      .set({ ...entitlement, updatedAt: new Date() })
      .where(eq(userEntitlements.id, id))
      .returning();
    return result[0];
  }

  // Creator Marketplace operations

  // Creator operations
  async getCreator(id: string): Promise<Creator | undefined> {
    const result = await this.db.select().from(creators).where(eq(creators.id, id)).limit(1);
    return result[0];
  }

  async getCreatorByUserId(userId: string): Promise<Creator | undefined> {
    const result = await this.db.select().from(creators).where(eq(creators.userId, userId)).limit(1);
    return result[0];
  }

  async getCreatorByStripeAccountId(stripeAccountId: string): Promise<Creator | undefined> {
    const result = await this.db.select().from(creators).where(eq(creators.stripeConnectAccountId, stripeAccountId)).limit(1);
    return result[0];
  }

  async getAllCreators(): Promise<Creator[]> {
    return await this.db.select().from(creators);
  }

  async createCreator(creator: InsertCreator): Promise<Creator> {
    const result = await this.db.insert(creators).values({
      ...creator,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateCreator(id: string, creator: Partial<InsertCreator>): Promise<Creator | undefined> {
    const result = await this.db.update(creators)
      .set({ ...creator, updatedAt: new Date() })
      .where(eq(creators.id, id))
      .returning();
    return result[0];
  }

  async deleteCreator(id: string): Promise<boolean> {
    const result = await this.db.delete(creators).where(eq(creators.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  async updateCreatorEarnings(creatorId: string, earningsChange: number): Promise<Creator | undefined> {
    const result = await this.db.update(creators)
      .set({ 
        totalEarnings: sql`${creators.totalEarnings} + ${earningsChange}`,
        updatedAt: new Date()
      })
      .where(eq(creators.id, creatorId))
      .returning();
    return result[0];
  }

  // Asset operations
  async getAsset(id: string): Promise<Asset | undefined> {
    const result = await this.db.select().from(assets).where(eq(assets.id, id)).limit(1);
    return result[0];
  }

  async getAssetsByCreatorId(creatorId: string): Promise<Asset[]> {
    return await this.db.select().from(assets).where(eq(assets.creatorId, creatorId));
  }

  async getAllAssets(): Promise<Asset[]> {
    return await this.db.select().from(assets);
  }

  async getPublicAssets(): Promise<Asset[]> {
    return await this.db.select().from(assets).where(eq(assets.isPublic, true));
  }

  async createAsset(asset: InsertAsset): Promise<Asset> {
    const result = await this.db.insert(assets).values({
      ...asset,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateAsset(id: string, asset: Partial<InsertAsset>): Promise<Asset | undefined> {
    const result = await this.db.update(assets)
      .set({ ...asset, updatedAt: new Date() })
      .where(eq(assets.id, id))
      .returning();
    return result[0];
  }

  async deleteAsset(id: string): Promise<boolean> {
    const result = await this.db.delete(assets).where(eq(assets.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  async incrementAssetDownloadCount(id: string): Promise<Asset | undefined> {
    const result = await this.db.update(assets)
      .set({ 
        downloadCount: sql`${assets.downloadCount} + 1`,
        updatedAt: new Date()
      })
      .where(eq(assets.id, id))
      .returning();
    return result[0];
  }

  // Creator Asset operations
  async getCreatorAsset(id: string): Promise<CreatorAsset | undefined> {
    const result = await this.db.select().from(creatorAssets).where(eq(creatorAssets.id, id)).limit(1);
    return result[0];
  }

  async getCreatorAssetsByCreatorId(creatorId: string): Promise<CreatorAsset[]> {
    return await this.db.select().from(creatorAssets).where(eq(creatorAssets.creatorId, creatorId));
  }

  async getCreatorAssetsByAssetId(assetId: string): Promise<CreatorAsset[]> {
    return await this.db.select().from(creatorAssets).where(eq(creatorAssets.assetId, assetId));
  }

  async getCreatorAssetsByStatus(status: string): Promise<CreatorAsset[]> {
    return await this.db.select().from(creatorAssets).where(eq(creatorAssets.approvalStatus, status));
  }

  async getAllCreatorAssets(): Promise<CreatorAsset[]> {
    return await this.db.select().from(creatorAssets);
  }

  async createCreatorAsset(creatorAsset: InsertCreatorAsset): Promise<CreatorAsset> {
    const result = await this.db.insert(creatorAssets).values({
      ...creatorAsset,
      updatedAt: new Date()
    }).returning();
    return result[0];
  }

  async updateCreatorAsset(id: string, creatorAsset: Partial<InsertCreatorAsset>): Promise<CreatorAsset | undefined> {
    const result = await this.db.update(creatorAssets)
      .set({ ...creatorAsset, updatedAt: new Date() })
      .where(eq(creatorAssets.id, id))
      .returning();
    return result[0];
  }

  async deleteCreatorAsset(id: string): Promise<boolean> {
    const result = await this.db.delete(creatorAssets).where(eq(creatorAssets.id, id));
    return (result.rowCount ?? 0) > 0;
  }

  async approveCreatorAsset(id: string, approvedBy: string): Promise<CreatorAsset | undefined> {
    // Audit trail: Get current state for logging
    const currentAsset = await this.getCreatorAsset(id);
    if (!currentAsset) {
      throw new Error('Asset not found for approval');
    }
    
    // Security check: Only allow transition from 'pending' to 'approved'
    if (currentAsset.approvalStatus !== 'pending') {
      throw new Error(`Invalid state transition: cannot approve asset with status '${currentAsset.approvalStatus}'`);
    }
    
    const result = await this.db.update(creatorAssets)
      .set({ 
        approvalStatus: 'approved',
        approvedAt: new Date(),
        approvedBy,
        updatedAt: new Date()
      })
      .where(eq(creatorAssets.id, id))
      .returning();
    
    const updatedAsset = result[0];
    if (updatedAsset) {
      // Audit log the approval
      console.log(`📈 AUDIT: Asset '${currentAsset.title}' (${id}) approved by ${approvedBy}. Status: ${currentAsset.approvalStatus} → ${updatedAsset.approvalStatus}`);
    }
    
    return updatedAsset;
  }

  async rejectCreatorAsset(id: string, rejectionReason: string): Promise<CreatorAsset | undefined> {
    // Audit trail: Get current state for logging
    const currentAsset = await this.getCreatorAsset(id);
    if (!currentAsset) {
      throw new Error('Asset not found for rejection');
    }
    
    // Security check: Only allow transition from 'pending' or 'approved' to 'rejected'
    if (!['pending', 'approved'].includes(currentAsset.approvalStatus)) {
      throw new Error(`Invalid state transition: cannot reject asset with status '${currentAsset.approvalStatus}'`);
    }
    
    const result = await this.db.update(creatorAssets)
      .set({ 
        approvalStatus: 'rejected',
        rejectionReason,
        rejectedAt: new Date(),
        approvedBy,
        updatedAt: new Date()
      })
      .where(eq(creatorAssets.id, id))
      .returning();
    return result[0];
  }

  async rejectCreatorAsset(id: string, rejectionReason: string): Promise<CreatorAsset | undefined> {
    const result = await this.db.update(creatorAssets)
      .set({ 
        approvalStatus: 'rejected',
        rejectionReason,
        updatedAt: new Date()
      })
      .where(eq(creatorAssets.id, id))
      .returning();
    return result[0];
  }

  async incrementCreatorAssetSales(id: string, revenue: number): Promise<CreatorAsset | undefined> {
    const result = await this.db.update(creatorAssets)
      .set({ 
        salesCount: sql`${creatorAssets.salesCount} + 1`,
        totalRevenue: sql`${creatorAssets.totalRevenue} + ${revenue}`,
        updatedAt: new Date()
      })
      .where(eq(creatorAssets.id, id))
      .returning();
    return result[0];
  }

  // Creator Earning operations
  async getCreatorEarning(id: string): Promise<CreatorEarning | undefined> {
    const result = await this.db.select().from(creatorEarnings).where(eq(creatorEarnings.id, id)).limit(1);
    return result[0];
  }

  async getCreatorEarningsByCreatorId(creatorId: string): Promise<CreatorEarning[]> {
    return await this.db.select().from(creatorEarnings).where(eq(creatorEarnings.creatorId, creatorId));
  }

  async getCreatorEarningsByStatus(status: string): Promise<CreatorEarning[]> {
    return await this.db.select().from(creatorEarnings).where(eq(creatorEarnings.payoutStatus, status));
  }

  async getAllCreatorEarnings(): Promise<CreatorEarning[]> {
    return await this.db.select().from(creatorEarnings);
  }

  async createCreatorEarning(earning: InsertCreatorEarning): Promise<CreatorEarning> {
    const result = await this.db.insert(creatorEarnings).values(earning).returning();
    return result[0];
  }

  async updateCreatorEarning(id: string, earning: Partial<InsertCreatorEarning>): Promise<CreatorEarning | undefined> {
    const result = await this.db.update(creatorEarnings)
      .set(earning)
      .where(eq(creatorEarnings.id, id))
      .returning();
    return result[0];
  }

  async processCreatorPayout(creatorId: string, payoutReference: string): Promise<CreatorEarning[]> {
    const result = await this.db.update(creatorEarnings)
      .set({
        payoutStatus: 'paid',
        payoutDate: new Date(),
        payoutReference
      })
      .where(and(
        eq(creatorEarnings.creatorId, creatorId),
        eq(creatorEarnings.payoutStatus, 'pending')
      ))
      .returning();
    return result;
  }

  // Cover Template operations
  async getCoverTemplate(id: string): Promise<CoverTemplate | undefined> {
    const result = await this.db.select().from(coverTemplates).where(eq(coverTemplates.id, id)).limit(1);
    return result[0];
  }

  async getCoverTemplates(opts: { q?: string; category?: string; min?: number; max?: number; top_tier?: 'FIG'|'TMT'|'Healthcare/Pharma'|'General'; subcat?: string; includeInactive?: boolean } = {}): Promise<CoverTemplate[]> {
    // Build conditions array
    const conditions = [];
    
    // Only filter by active status unless admin explicitly includes inactive
    if (!opts.includeInactive) {
      conditions.push(eq(coverTemplates.isActive, true));
    }
    
    // Only show approved templates for public access
    if (!opts.includeInactive) {
      conditions.push(eq(coverTemplates.approvalStatus, 'approved'));
    }
    
    if (opts.category) {
      conditions.push(eq(coverTemplates.category, opts.category));
    }
    
    if (opts.top_tier) {
      conditions.push(eq(coverTemplates.topTier, opts.top_tier));
    }
    
    if (opts.subcat) {
      // Filter by subcategory using array contains
      conditions.push(sql`${opts.subcat} = ANY (${coverTemplates.subcategories})`);
    }
    
    if (opts.min !== undefined) {
      conditions.push(gte(coverTemplates.priceCents, opts.min));
    }
    
    if (opts.max !== undefined) {
      conditions.push(lte(coverTemplates.priceCents, opts.max));
    }
    
    if (opts.q) {
      conditions.push(ilike(coverTemplates.title, `%${opts.q}%`));
    }
    
    const whereCondition = conditions.length > 0 ? and(...conditions) : undefined;
    
    return await this.db.select().from(coverTemplates)
      .where(whereCondition)
      .orderBy(desc(coverTemplates.createdAt));
  }

  async createCoverTemplate(template: InsertCoverTemplate): Promise<CoverTemplate> {
    const result = await this.db.insert(coverTemplates).values(template).returning();
    return result[0];
  }

  async updateCoverTemplate(id: string, template: Partial<InsertCoverTemplate>): Promise<CoverTemplate | undefined> {
    const result = await this.db.update(coverTemplates)
      .set(template)
      .where(eq(coverTemplates.id, id))
      .returning();
    return result[0];
  }

  async deleteCoverTemplate(id: string): Promise<boolean> {
    const result = await this.db.delete(coverTemplates).where(eq(coverTemplates.id, id));
    return result.rowCount > 0;
  }

  // Infographic Template operations
  async getInfographicTemplate(id: string): Promise<InfographicTemplate | undefined> {
    const result = await this.db.select().from(infographicTemplates).where(eq(infographicTemplates.id, id)).limit(1);
    return result[0];
  }

  async getInfographicTemplates(opts: { q?: string; category?: string; descriptors?: string[]; min?: number; max?: number; includeInactive?: boolean; status?: string } = {}): Promise<InfographicTemplate[]> {
    // Build conditions array
    const conditions = [];
    
    // Only filter by active status unless admin explicitly includes inactive
    if (!opts.includeInactive) {
      conditions.push(eq(infographicTemplates.isActive, true));
    }
    
    // Filter by approval status
    if (opts.status && opts.status !== 'all') {
      conditions.push(eq(infographicTemplates.approvalStatus, opts.status));
    } else if (!opts.includeInactive) {
      // Only show approved templates for public access
      conditions.push(eq(infographicTemplates.approvalStatus, 'approved'));
    }
    
    if (opts.category) {
      conditions.push(eq(infographicTemplates.category, opts.category));
    }

    // Filter by descriptors - match all provided descriptors (AND logic)
    if (opts.descriptors && opts.descriptors.length > 0) {
      for (const descriptor of opts.descriptors) {
        conditions.push(sql`${descriptor.toLowerCase()} = ANY(${infographicTemplates.descriptors})`);
      }
    }

    if (opts.q) {
      // Search in title and descriptors
      conditions.push(or(
        ilike(infographicTemplates.title, `%${opts.q}%`),
        sql`${infographicTemplates.descriptors}::text ILIKE ${`%${opts.q}%`}`
      ));
    }

    if (typeof opts.min === 'number') {
      conditions.push(gte(infographicTemplates.priceCents, opts.min));
    }

    if (typeof opts.max === 'number') {
      conditions.push(lte(infographicTemplates.priceCents, opts.max));
    }

    // Combine conditions with AND
    const whereCondition = conditions.length > 0 ? and(...conditions) : undefined;

    return await this.db.select().from(infographicTemplates)
      .where(whereCondition)
      .orderBy(desc(infographicTemplates.createdAt));
  }

  async createInfographicTemplate(template: InsertInfographicTemplate): Promise<InfographicTemplate> {
    const result = await this.db.insert(infographicTemplates).values(template).returning();
    return result[0];
  }

  async updateInfographicTemplate(id: string, template: Partial<InsertInfographicTemplate>): Promise<InfographicTemplate | undefined> {
    const result = await this.db.update(infographicTemplates)
      .set(template)
      .where(eq(infographicTemplates.id, id))
      .returning();
    return result[0];
  }

  async deleteInfographicTemplate(id: string): Promise<boolean> {
    const result = await this.db.delete(infographicTemplates).where(eq(infographicTemplates.id, id));
    return result.rowCount > 0;
  }

  // Cover Purchase operations
  async getCoverPurchase(id: string): Promise<CoverPurchase | undefined> {
    const result = await this.db.select().from(coverPurchases).where(eq(coverPurchases.id, id)).limit(1);
    return result[0];
  }

  async getUserCoverPurchases(userId: string): Promise<CoverPurchase[]> {
    return await this.db.select().from(coverPurchases).where(eq(coverPurchases.userId, userId));
  }

  async createCoverPurchase(purchase: InsertCoverPurchase): Promise<CoverPurchase> {
    const result = await this.db.insert(coverPurchases).values(purchase).returning();
    return result[0];
  }

  async updateCoverPurchase(id: string, purchase: Partial<InsertCoverPurchase>): Promise<CoverPurchase | undefined> {
    const result = await this.db.update(coverPurchases)
      .set(purchase)
      .where(eq(coverPurchases.id, id))
      .returning();
    return result[0];
  }

  async markPurchasePaidBySession(sessionId: string, paymentIntent: string, downloadUrl: string, actualAmountCents?: number, actualCurrency?: string): Promise<CoverPurchase | undefined> {
    const updateData: any = {
      status: 'paid',
      stripePaymentIntent: paymentIntent,
      downloadUrl: downloadUrl
    };
    
    // Update with actual paid amount if provided (handles discounts)
    if (actualAmountCents !== undefined) updateData.amountCents = actualAmountCents;
    if (actualCurrency !== undefined) updateData.currency = actualCurrency;
    
    const result = await this.db.update(coverPurchases)
      .set(updateData)
      .where(eq(coverPurchases.stripeSessionId, sessionId))
      .returning();
    return result[0];
  }

  // Infographic Purchase operations
  async getInfographicPurchase(id: string): Promise<InfographicPurchase | undefined> {
    const result = await this.db.select().from(infographicPurchases).where(eq(infographicPurchases.id, id)).limit(1);
    return result[0];
  }

  async getUserInfographicPurchases(userId: string): Promise<InfographicPurchase[]> {
    return await this.db.select().from(infographicPurchases).where(eq(infographicPurchases.userId, userId));
  }

  async createInfographicPurchase(purchase: InsertInfographicPurchase): Promise<InfographicPurchase> {
    const result = await this.db.insert(infographicPurchases).values(purchase).returning();
    return result[0];
  }

  async updateInfographicPurchase(id: string, purchase: Partial<InsertInfographicPurchase>): Promise<InfographicPurchase | undefined> {
    const result = await this.db.update(infographicPurchases)
      .set(purchase)
      .where(eq(infographicPurchases.id, id))
      .returning();
    return result[0];
  }

  async markInfographicPurchasePaidBySession(sessionId: string, paymentIntent: string, downloadUrl: string, actualAmountCents?: number, actualCurrency?: string): Promise<InfographicPurchase | undefined> {
    const updateData: any = {
      status: 'paid',
      stripePaymentIntent: paymentIntent,
      downloadUrl: downloadUrl
    };
    
    // Update with actual paid amount if provided (handles discounts)
    if (actualAmountCents !== undefined) updateData.amountCents = actualAmountCents;
    if (actualCurrency !== undefined) updateData.currency = actualCurrency;
    
    const result = await this.db.update(infographicPurchases)
      .set(updateData)
      .where(eq(infographicPurchases.stripeSessionId, sessionId))
      .returning();
    return result[0];
  }

  // Presentation Template operations
  async getPresentationTemplate(id: string): Promise<PresentationTemplate | undefined> {
    const result = await this.db.select().from(presentationTemplates).where(eq(presentationTemplates.id, id)).limit(1);
    return result[0];
  }

  async getPresentationTemplates(opts: { q?: string; category?: string; descriptors?: string[]; min?: number; max?: number; includeInactive?: boolean; status?: string } = {}): Promise<PresentationTemplate[]> {
    // Build conditions array
    const conditions = [];
    
    // Only filter by active status unless admin explicitly includes inactive
    if (!opts.includeInactive) {
      conditions.push(eq(presentationTemplates.isActive, true));
    }
    
    // Filter by approval status
    if (opts.status && opts.status !== 'all') {
      conditions.push(eq(presentationTemplates.approvalStatus, opts.status));
    } else if (!opts.includeInactive) {
      // Only show approved templates for public access
      conditions.push(eq(presentationTemplates.approvalStatus, 'approved'));
    }
    
    if (opts.category) {
      conditions.push(eq(presentationTemplates.category, opts.category));
    }

    // Filter by descriptors - match all provided descriptors (AND logic)
    if (opts.descriptors && opts.descriptors.length > 0) {
      for (const descriptor of opts.descriptors) {
        conditions.push(sql`${descriptor.toLowerCase()} = ANY(${presentationTemplates.subcategories})`);
      }
    }

    if (opts.q) {
      // Search in title and subcategories
      conditions.push(or(
        ilike(presentationTemplates.title, `%${opts.q}%`),
        sql`${presentationTemplates.subcategories}::text ILIKE ${`%${opts.q}%`}`
      ));
    }

    // Combine conditions with AND
    const whereCondition = conditions.length > 0 ? and(...conditions) : undefined;

    return await this.db.select().from(presentationTemplates)
      .where(whereCondition)
      .orderBy(desc(presentationTemplates.createdAt));
  }

  async createPresentationTemplate(template: InsertPresentationTemplate): Promise<PresentationTemplate> {
    const result = await this.db.insert(presentationTemplates).values(template).returning();
    return result[0];
  }

  async updatePresentationTemplate(id: string, template: Partial<InsertPresentationTemplate>): Promise<PresentationTemplate | undefined> {
    const result = await this.db.update(presentationTemplates)
      .set(template)
      .where(eq(presentationTemplates.id, id))
      .returning();
    return result[0];
  }

  async deletePresentationTemplate(id: string): Promise<boolean> {
    const result = await this.db.delete(presentationTemplates).where(eq(presentationTemplates.id, id));
    return result.rowCount > 0;
  }

  // Presentation Purchase operations
  async getPresentationPurchase(id: string): Promise<PresentationPurchase | undefined> {
    const result = await this.db.select().from(presentationPurchases).where(eq(presentationPurchases.id, id)).limit(1);
    return result[0];
  }

  async getUserPresentationPurchases(userId: string): Promise<PresentationPurchase[]> {
    return await this.db.select().from(presentationPurchases).where(eq(presentationPurchases.userId, userId));
  }

  async createPresentationPurchase(purchase: InsertPresentationPurchase): Promise<PresentationPurchase> {
    const result = await this.db.insert(presentationPurchases).values(purchase).returning();
    return result[0];
  }

  async updatePresentationPurchase(id: string, purchase: Partial<InsertPresentationPurchase>): Promise<PresentationPurchase | undefined> {
    const result = await this.db.update(presentationPurchases)
      .set(purchase)
      .where(eq(presentationPurchases.id, id))
      .returning();
    return result[0];
  }

  async markPresentationPurchasePaidBySession(sessionId: string, paymentIntent: string, downloadUrl: string, actualAmountCents?: number, actualCurrency?: string): Promise<PresentationPurchase | undefined> {
    const updateData: any = {
      status: 'paid',
      stripePaymentIntent: paymentIntent,
      downloadUrl: downloadUrl
    };
    
    // Update with actual paid amount if provided (handles discounts)
    if (actualAmountCents !== undefined) updateData.amountCents = actualAmountCents;
    if (actualCurrency !== undefined) updateData.currency = actualCurrency;
    
    const result = await this.db.update(presentationPurchases)
      .set(updateData)
      .where(eq(presentationPurchases.stripeSessionId, sessionId))
      .returning();
    return result[0];
  }
}

export const storage = new PostgreSQLStorage();
