import crypto from 'crypto';

// Interface for stream token data
interface StreamToken {
  token: string;
  userId: string;
  createdAt: Date;
  expiresAt: Date;
  used: boolean;
}

class StreamTokenService {
  private tokens = new Map<string, StreamToken>();
  private cleanupInterval: NodeJS.Timeout | null = null;
  private readonly TOKEN_TTL = 5 * 60 * 1000; // 5 minutes
  private readonly CLEANUP_INTERVAL = 60 * 1000; // 1 minute

  constructor() {
    this.startCleanup();
  }

  // Generate a secure opaque token
  generateToken(userId: string): string {
    // Clean up expired tokens first
    this.cleanupExpiredTokens();

    // Generate cryptographically secure random token
    const token = crypto.randomBytes(32).toString('hex');
    const now = new Date();
    const expiresAt = new Date(now.getTime() + this.TOKEN_TTL);

    // Store token
    this.tokens.set(token, {
      token,
      userId,
      createdAt: now,
      expiresAt,
      used: false
    });

    console.log(`Generated stream token for user ${userId} (expires in 5 minutes)`);
    return token;
  }

  // Validate and consume a token (single-use)
  validateAndConsumeToken(token: string): { valid: boolean; userId?: string; reason?: string } {
    const tokenData = this.tokens.get(token);

    if (!tokenData) {
      return { valid: false, reason: 'Token not found' };
    }

    if (tokenData.used) {
      return { valid: false, reason: 'Token already used' };
    }

    if (new Date() > tokenData.expiresAt) {
      this.tokens.delete(token);
      return { valid: false, reason: 'Token expired' };
    }

    // Mark token as used and return success
    tokenData.used = true;
    const userId = tokenData.userId;

    // Remove token immediately after use (single-use)
    this.tokens.delete(token);

    console.log(`Stream token validated and consumed for user ${userId}`);
    return { valid: true, userId };
  }

  // Revoke a specific token
  revokeToken(token: string): boolean {
    return this.tokens.delete(token);
  }

  // Revoke all tokens for a user
  revokeUserTokens(userId: string): number {
    let revokedCount = 0;
    for (const [token, tokenData] of this.tokens) {
      if (tokenData.userId === userId) {
        this.tokens.delete(token);
        revokedCount++;
      }
    }
    console.log(`Revoked ${revokedCount} tokens for user ${userId}`);
    return revokedCount;
  }

  // Clean up expired tokens
  private cleanupExpiredTokens(): void {
    const now = new Date();
    let cleanedCount = 0;

    for (const [token, tokenData] of this.tokens) {
      if (now > tokenData.expiresAt) {
        this.tokens.delete(token);
        cleanedCount++;
      }
    }

    if (cleanedCount > 0) {
      console.log(`Cleaned up ${cleanedCount} expired stream tokens`);
    }
  }

  // Start periodic cleanup
  private startCleanup(): void {
    if (this.cleanupInterval) {
      clearInterval(this.cleanupInterval);
    }

    this.cleanupInterval = setInterval(() => {
      this.cleanupExpiredTokens();
    }, this.CLEANUP_INTERVAL);
  }

  // Get service stats
  getStats() {
    const now = new Date();
    const validTokens = Array.from(this.tokens.values()).filter(t => !t.used && now <= t.expiresAt);
    
    return {
      totalTokens: this.tokens.size,
      validTokens: validTokens.length,
      expiredTokens: this.tokens.size - validTokens.length,
      tokensByUser: validTokens.reduce((acc, token) => {
        acc[token.userId] = (acc[token.userId] || 0) + 1;
        return acc;
      }, {} as Record<string, number>)
    };
  }

  // Cleanup all tokens (for shutdown)
  cleanup(): void {
    if (this.cleanupInterval) {
      clearInterval(this.cleanupInterval);
      this.cleanupInterval = null;
    }
    this.tokens.clear();
    console.log('Stream token service cleanup completed');
  }
}

// Export singleton instance
export const streamTokenService = new StreamTokenService();

// Graceful shutdown handling
process.on('SIGTERM', () => {
  console.log('SIGTERM received, cleaning up stream token service...');
  streamTokenService.cleanup();
});

process.on('SIGINT', () => {
  console.log('SIGINT received, cleaning up stream token service...');
  streamTokenService.cleanup();
});