import { bucket, db } from '../firebaseAdmin';

export interface LogoTemplateMetadata {
  id: string;
  name: string;
  description?: string;
  tags?: string[];
  version?: number;
  status?: "active" | "draft" | "archived";
  storagePaths?: {
    svg?: string;
    preview?: string;
    raster?: string;
  };
  downloadURLs?: {
    svg?: string;
    preview?: string;
    raster?: string;
  };
  defaults?: {
    brandName?: string;
    tagline?: string;
    estYear?: string;
    Brand_Name?: string;
    Tagline?: string;
    Est_Year?: string;
  };
  createdBy?: string;
  createdAt?: any;
  updatedAt?: any;
}

/**
 * List all logo templates from both 'templates' and 'logo_templates' collections
 */
export async function listLogoTemplates(): Promise<LogoTemplateMetadata[]> {
  const allTemplates: LogoTemplateMetadata[] = [];
  const seenIds = new Set<string>();
  
  // Query both collections
  const collections = ["templates", "logo_templates"];
  
  for (const collectionName of collections) {
    try {
      const snapshot = await db.collection(collectionName).get();
      
      for (const doc of snapshot.docs) {
        const data = doc.data();
        const templateId = data.templateId || doc.id;
        
        // Skip duplicates
        if (seenIds.has(templateId)) continue;
        seenIds.add(templateId);
        
        // Map to consistent structure
        const template: LogoTemplateMetadata = {
          id: templateId,
          name: data.name || templateId,
          description: data.description || "",
          tags: data.tags || [],
          version: data.version || 1,
          status: data.status || "active",
          storagePaths: {
            svg: data.storagePaths?.svg || "",
            preview: data.storagePaths?.preview || data.storagePaths?.raster || "",
            raster: data.storagePaths?.raster || ""
          },
          downloadURLs: {
            svg: data.downloadURLs?.svg || "",
            preview: data.downloadURLs?.preview || data.downloadURLs?.raster || "",
            raster: data.downloadURLs?.raster || ""
          },
          defaults: {
            brandName: data.defaults?.brandName || data.brandName || "Your Brand",
            tagline: data.defaults?.tagline || data.tagline || "Your Tagline",
            estYear: data.defaults?.estYear || data.estYear || "2025"
          },
          createdBy: data.ownerId || data.createdBy || "",
          createdAt: data.createdAt,
          updatedAt: data.updatedAt
        };
        
        allTemplates.push(template);
      }
    } catch (error) {
      console.warn(`Failed to read ${collectionName} collection:`, error);
    }
  }
  
  return allTemplates;
}

/**
 * Get a single logo template by ID from either collection
 */
export async function getLogoTemplateById(id: string): Promise<LogoTemplateMetadata | null> {
  const collections = ["templates", "logo_templates"];
  const searchIds = [id];
  
  // Also try without prefix
  if (id.startsWith("logo-wordmark-")) {
    searchIds.push(id.replace("logo-wordmark-", ""));
  }
  
  for (const collectionName of collections) {
    try {
      const snapshot = await db.collection(collectionName).get();
      
      for (const doc of snapshot.docs) {
        const data = doc.data();
        const templateId = data.templateId || doc.id;
        
        if (!searchIds.includes(templateId)) continue;
        
        return {
          id: templateId,
          name: data.name || templateId,
          description: data.description || "",
          tags: data.tags || [],
          version: data.version || 1,
          status: data.status || "active",
          storagePaths: {
            svg: data.storagePaths?.svg || "",
            preview: data.storagePaths?.preview || data.storagePaths?.raster || "",
            raster: data.storagePaths?.raster || ""
          },
          downloadURLs: {
            svg: data.downloadURLs?.svg || "",
            preview: data.downloadURLs?.preview || data.downloadURLs?.raster || "",
            raster: data.downloadURLs?.raster || ""
          },
          defaults: {
            brandName: data.defaults?.brandName || data.brandName || "Your Brand",
            tagline: data.defaults?.tagline || data.tagline || "Your Tagline",
            estYear: data.defaults?.estYear || data.estYear || "2025"
          },
          createdBy: data.ownerId || data.createdBy || "",
          createdAt: data.createdAt,
          updatedAt: data.updatedAt
        };
      }
    } catch (error) {
      console.warn(`Failed to read ${collectionName} collection:`, error);
    }
  }
  
  return null;
}

/**
 * Generate signed download URL for a storage path
 */
export async function getSignedUrl(storagePath: string): Promise<string> {
  if (!storagePath) return "";
  
  try {
    const file = bucket.file(storagePath);
    const [url] = await file.getSignedUrl({
      action: 'read',
      expires: Date.now() + 15 * 60 * 1000, // 15 minutes
    });
    return url;
  } catch (error) {
    console.error(`Failed to generate signed URL for ${storagePath}:`, error);
    return "";
  }
}
