import archiver from "archiver";
import fetch from "node-fetch";
import fs from "fs/promises";
import path from "path";
import { ObjectStorageService } from "../objectStorage";
import { storage as db } from "../storage";

const objectStorage = new ObjectStorageService();

// SECURITY: Allowed domains for template file URLs to prevent SSRF
const ALLOWED_DOMAINS = [
  'example.com',
  'your-cdn.com', 
  'firebasestorage.googleapis.com',
  'storage.googleapis.com',
  'drive.google.com',
  'docs.google.com'
];

const MAX_FILE_SIZE = 50 * 1024 * 1024; // 50MB per file
const REQUEST_TIMEOUT = 30000; // 30 seconds per file

function isUrlAllowed(url: string): boolean {
  try {
    const urlObj = new URL(url);
    return ALLOWED_DOMAINS.some(domain => 
      urlObj.hostname === domain || urlObj.hostname.endsWith('.' + domain)
    );
  } catch {
    return false;
  }
}

// FIXED: Add filename sanitization to prevent path traversal and invalid characters
function sanitizeFilename(filename: string): string {
  return filename
    .replace(/[\/\\:*?"<>|]/g, '_') // Replace invalid characters with underscore
    .replace(/^\./g, '_') // Replace leading dots
    .replace(/\s+/g, '_') // Replace spaces with underscores
    .substring(0, 100) // Limit length to 100 characters
    .replace(/_+$/, '') || 'untitled'; // Remove trailing underscores, fallback if empty
}

export interface InfographicFile {
  id: string;
  title: string;
  pptxUrl?: string | null;
  keynoteUrl?: string | null;
  gslidesUrl?: string | null;
  downloadBundleUrl?: string | null;
}

/**
 * Create a ZIP bundle containing selected infographic templates
 * Returns the object storage URL where the ZIP was uploaded
 */
export async function createInfographicBundle(
  purchaseId: string,
  selectedIds: string[]
): Promise<string> {
  // Get the selected templates
  const templates = await Promise.all(
    selectedIds.map(id => db.getInfographicTemplate(id))
  );

  // Filter out any null/undefined templates
  const validTemplates = templates.filter(t => t !== null) as InfographicFile[];

  if (validTemplates.length === 0) {
    throw new Error("No valid templates found for bundle creation");
  }

  // Create temporary directory for assembly
  const tmpDir = `/tmp/infographic-bundle-${purchaseId}`;
  await fs.mkdir(tmpDir, { recursive: true });

  try {
    // Create ZIP archive
    const zipPath = path.join(tmpDir, `infographic-bundle-${purchaseId}.zip`);
    const output = await fs.open(zipPath, 'w');
    const stream = output.createWriteStream();
    
    const archive = archiver('zip', {
      zlib: { level: 9 } // Maximum compression
    });

    archive.pipe(stream);

    // Add a README file
    const readmeContent = `Infographic Templates Bundle

Purchase ID: ${purchaseId}
Templates included: ${validTemplates.length}
Date: ${new Date().toISOString().split('T')[0]}

Templates:
${validTemplates.map((t, i) => `${i + 1}. ${t.title}`).join('\n')}

Files included:
- PowerPoint (.pptx) files when available
- Keynote (.key) files when available  
- Google Slides links in text files when available
- Download bundles (.zip) when available

Compatible with:
- Microsoft PowerPoint
- Apple Keynote
- Google Slides

For support, contact support@ibrandbiz.com
`;

    archive.append(readmeContent, { name: 'README.txt' });

    // Process each template
    for (let i = 0; i < validTemplates.length; i++) {
      const template = validTemplates[i];
      const folderName = `${i + 1}. ${template.title.replace(/[^a-zA-Z0-9\-_\s]/g, '')}`;

      // Add template info file
      const templateInfo = `Template: ${template.title}
ID: ${template.id}
Formats available: ${[
        template.pptxUrl ? 'PowerPoint (.pptx)' : null,
        template.keynoteUrl ? 'Keynote (.key)' : null, 
        template.gslidesUrl ? 'Google Slides (link)' : null,
        template.downloadBundleUrl ? 'Bundle (.zip)' : null
      ].filter(Boolean).join(', ')}
`;

      archive.append(templateInfo, { name: `${folderName}/info.txt` });

      // SECURITY: Safely download and include available format files
      if (template.pptxUrl) {
        try {
          if (!isUrlAllowed(template.pptxUrl)) {
            console.error(`PPTX URL not allowed for template ${template.id}: ${template.pptxUrl}`);
          } else {
            const response = await fetch(template.pptxUrl, {
              timeout: REQUEST_TIMEOUT,
              size: MAX_FILE_SIZE,
              headers: { 'User-Agent': 'IBrandBiz-ZipAssembly/1.0' }
            });
            
            if (response.ok && response.headers.get('content-length')) {
              const contentLength = parseInt(response.headers.get('content-length') || '0');
              if (contentLength > MAX_FILE_SIZE) {
                console.error(`PPTX file too large for template ${template.id}: ${contentLength} bytes`);
              } else {
                const buffer = Buffer.from(await response.arrayBuffer());
                archive.append(buffer, { name: `${folderName}/${sanitizeFilename(template.title)}.pptx` });
              }
            }
          }
        } catch (error) {
          console.error(`Failed to download PPTX for template ${template.id}:`, error);
        }
      }

      if (template.keynoteUrl) {
        try {
          if (!isUrlAllowed(template.keynoteUrl)) {
            console.error(`Keynote URL not allowed for template ${template.id}: ${template.keynoteUrl}`);
          } else {
            const response = await fetch(template.keynoteUrl, {
              timeout: REQUEST_TIMEOUT,
              size: MAX_FILE_SIZE,
              headers: { 'User-Agent': 'IBrandBiz-ZipAssembly/1.0' }
            });
            
            if (response.ok && response.headers.get('content-length')) {
              const contentLength = parseInt(response.headers.get('content-length') || '0');
              if (contentLength > MAX_FILE_SIZE) {
                console.error(`Keynote file too large for template ${template.id}: ${contentLength} bytes`);
              } else {
                const buffer = Buffer.from(await response.arrayBuffer());
                archive.append(buffer, { name: `${folderName}/${sanitizeFilename(template.title)}.key` });
              }
            }
          }
        } catch (error) {
          console.error(`Failed to download Keynote for template ${template.id}:`, error);
        }
      }

      if (template.downloadBundleUrl) {
        try {
          if (!isUrlAllowed(template.downloadBundleUrl)) {
            console.error(`Bundle URL not allowed for template ${template.id}: ${template.downloadBundleUrl}`);
          } else {
            const response = await fetch(template.downloadBundleUrl, {
              timeout: REQUEST_TIMEOUT,
              size: MAX_FILE_SIZE,
              headers: { 'User-Agent': 'IBrandBiz-ZipAssembly/1.0' }
            });
            
            if (response.ok && response.headers.get('content-length')) {
              const contentLength = parseInt(response.headers.get('content-length') || '0');
              if (contentLength > MAX_FILE_SIZE) {
                console.error(`Bundle file too large for template ${template.id}: ${contentLength} bytes`);
              } else {
                const buffer = Buffer.from(await response.arrayBuffer());
                archive.append(buffer, { name: `${folderName}/${sanitizeFilename(template.title)}-bundle.zip` });
              }
            }
          }
        } catch (error) {
          console.error(`Failed to download bundle for template ${template.id}:`, error);
        }
      }

      if (template.gslidesUrl) {
        // Create a text file with the Google Slides link
        const linkContent = `Google Slides Template: ${template.title}

Link: ${template.gslidesUrl}

Instructions:
1. Click the link above or copy/paste it into your browser
2. Sign in to your Google account
3. Make a copy of the template to your Google Drive
4. Edit and customize as needed

Note: You'll need a Google account to access Google Slides templates.
`;
        archive.append(linkContent, { name: `${folderName}/${sanitizeFilename(template.title)}-Google-Slides-Link.txt` });
      }
    }

    // FIXED: Properly finalize the archive and wait for stream completion
    await new Promise<void>((resolve, reject) => {
      stream.on('close', resolve);
      stream.on('error', reject);
      archive.on('error', reject);
      archive.finalize();
    });

    // Upload ZIP to object storage
    const zipBuffer = await fs.readFile(zipPath);
    const storageKey = `infographic-bundles/${purchaseId}/bundle-${Date.now()}.zip`;
    
    const uploadedUrl = await objectStorage.uploadFile(
      zipBuffer,
      storageKey,
      'application/zip'
    );

    console.log(`[infographic-zip] Created bundle for purchase ${purchaseId}: ${validTemplates.length} templates, ${zipBuffer.length} bytes`);

    return uploadedUrl;

  } finally {
    // Clean up temporary directory
    try {
      await fs.rm(tmpDir, { recursive: true, force: true });
    } catch (error) {
      console.error(`Failed to clean up temp directory ${tmpDir}:`, error);
    }
  }
}