/**
 * SuperNova's Icon Entitlements Service
 * 
 * Implements exact functions as specified by SuperNova for icon licensing system:
 * - grantIconsNoAttribution(userId)
 * - grantIconPack(userId, packId, count) 
 * - userHasIconsNoAttribution(userId)
 * - userOwnsIcon(userId, iconId)
 * 
 * Uses PostgreSQL database via storage interface instead of JSON files
 */

import { storage } from "../storage";
import type { InsertUserEntitlement } from "../../shared/schema";

/**
 * Grant global "no attribution" license for all icons
 * This is the $4.99 license that removes attribution requirements for all icons
 */
export async function grantIconsNoAttribution(userId: string): Promise<void> {
  if (!userId) {
    throw new Error("User ID is required");
  }

  try {
    // Check if user already has global no-attribution license
    const existing = await storage.hasEntitlement(userId, 'icons_no_attribution', 'global');
    
    if (existing) {
      console.log(`✅ User ${userId} already has global icons no-attribution license`);
      return;
    }

    // Grant the global no-attribution entitlement
    const entitlement: InsertUserEntitlement = {
      userId,
      entitlementType: 'icons_no_attribution',
      entitlementId: 'global', // Global license applies to all icons
      status: 'active',
      // No expiration for purchased licenses
      expiresAt: null
    };

    await storage.createUserEntitlement(entitlement);
    console.log(`✅ Granted global icons no-attribution license to user: ${userId}`);
  } catch (error) {
    console.error(`❌ Error granting icons no-attribution to user ${userId}:`, error);
    throw error;
  }
}

/**
 * Grant license for a specific icon pack
 * @param userId - User ID to grant license to
 * @param packId - Icon pack identifier
 * @param count - Number of icons in the pack (for tracking purposes)
 */
export async function grantIconPack(userId: string, packId: string, count: number): Promise<void> {
  if (!userId || !packId) {
    throw new Error("User ID and pack ID are required");
  }

  try {
    // Check if user already has this pack license
    const existing = await storage.hasEntitlement(userId, 'icon_pack', packId);
    
    if (existing) {
      console.log(`✅ User ${userId} already has license for icon pack: ${packId}`);
      return;
    }

    // Grant the icon pack entitlement
    const entitlement: InsertUserEntitlement = {
      userId,
      entitlementType: 'icon_pack',
      entitlementId: packId,
      status: 'active',
      // No expiration for purchased licenses
      expiresAt: null
    };

    await storage.createUserEntitlement(entitlement);
    console.log(`✅ Granted icon pack license to user ${userId}: ${packId} (${count} icons)`);
  } catch (error) {
    console.error(`❌ Error granting icon pack ${packId} to user ${userId}:`, error);
    throw error;
  }
}

/**
 * Check if user has global "no attribution" license for all icons
 * @param userId - User ID to check
 * @returns true if user has global no-attribution license
 */
export async function userHasIconsNoAttribution(userId: string): Promise<boolean> {
  if (!userId) {
    return false;
  }

  try {
    return await storage.hasEntitlement(userId, 'icons_no_attribution', 'global');
  } catch (error) {
    console.error(`❌ Error checking icons no-attribution for user ${userId}:`, error);
    return false;
  }
}

/**
 * Check if user owns/has license for a specific icon
 * This checks both global no-attribution license and specific icon pack licenses
 * @param userId - User ID to check
 * @param iconId - Specific icon ID to check
 * @returns true if user has license for this icon
 */
export async function userOwnsIcon(userId: string, iconId: string): Promise<boolean> {
  if (!userId || !iconId) {
    return false;
  }

  try {
    // First check if user has global no-attribution license (covers all icons)
    const hasGlobalLicense = await userHasIconsNoAttribution(userId);
    if (hasGlobalLicense) {
      return true;
    }

    // DISABLED: Icon pack ownership logic until proper icon→pack mapping is implemented
    // The current system has a fundamental issue:
    // - grantIconPack() stores packId as entitlementId
    // - userOwnsIcon() tries to find iconId as entitlementId
    // This will never match since iconId ≠ packId
    
    // TODO: Implement proper icon→pack mapping before re-enabling:
    // 1. Create icon_pack_mappings table linking iconId to packId
    // 2. Query packs user owns, then check if iconId is in any owned pack
    // 3. Or restructure to use iconId as entitlementId for individual purchases
    
    // For now, only global licenses work (users must buy $4.99 global license)
    return false;
  } catch (error) {
    console.error(`❌ Error checking icon ownership for user ${userId}, icon ${iconId}:`, error);
    return false;
  }
}

/**
 * Get user's icon entitlement summary
 * Helper function for debugging and admin purposes
 */
export async function getUserIconEntitlements(userId: string): Promise<{
  hasGlobalLicense: boolean;
  iconPacks: string[];
  totalEntitlements: number;
}> {
  if (!userId) {
    return {
      hasGlobalLicense: false,
      iconPacks: [],
      totalEntitlements: 0
    };
  }

  try {
    const hasGlobalLicense = await userHasIconsNoAttribution(userId);
    
    // Get all icon-related entitlements
    const iconEntitlements = await storage.getUserEntitlementsByType(userId, 'icon_pack');
    const iconPacks = iconEntitlements.map(e => e.entitlementId);
    
    const globalEntitlements = await storage.getUserEntitlementsByType(userId, 'icons_no_attribution');
    
    return {
      hasGlobalLicense,
      iconPacks,
      totalEntitlements: iconEntitlements.length + globalEntitlements.length
    };
  } catch (error) {
    console.error(`❌ Error getting icon entitlements for user ${userId}:`, error);
    return {
      hasGlobalLicense: false,
      iconPacks: [],
      totalEntitlements: 0
    };
  }
}

/**
 * Revoke icon entitlements (for admin/refund purposes)
 */
export async function revokeIconEntitlement(userId: string, entitlementType: 'icons_no_attribution' | 'icon_pack', entitlementId: string): Promise<boolean> {
  if (!userId || !entitlementType || !entitlementId) {
    throw new Error("User ID, entitlement type, and entitlement ID are required");
  }

  try {
    const entitlements = await storage.getUserEntitlementsByType(userId, entitlementType);
    const targetEntitlement = entitlements.find(e => e.entitlementId === entitlementId && e.status === 'active');
    
    if (!targetEntitlement) {
      console.log(`⚠️ No active entitlement found to revoke: ${entitlementType}:${entitlementId} for user ${userId}`);
      return false;
    }

    await storage.updateUserEntitlement(targetEntitlement.id, { 
      status: 'revoked',
      updatedAt: new Date()
    });

    console.log(`✅ Revoked entitlement: ${entitlementType}:${entitlementId} for user ${userId}`);
    return true;
  } catch (error) {
    console.error(`❌ Error revoking entitlement for user ${userId}:`, error);
    throw error;
  }
}