import { bucket } from "../firebaseAdmin";

export async function uploadOriginalToFirebase(
  buffer: Buffer,
  filename: string,
  photoId: string,
  mimeType: string = 'image/png'
): Promise<string> {
  try {
    // Store in private path: stock/originals/{photoId}/{filename}
    const file = bucket.file(`stock/originals/${photoId}/${filename}`);
    
    await file.save(buffer, {
      metadata: {
        contentType: mimeType,
        cacheControl: 'public, max-age=31536000', // 1 year cache
      },
    });

    // Get the download URL
    const [url] = await file.getSignedUrl({
      action: 'read',
      expires: Date.now() + 1000 * 60 * 60 * 24 * 365, // 1 year from now
    });

    return url;
  } catch (error) {
    console.error("Failed to upload to Firebase Storage:", error);
    throw error;
  }
}

export async function getOriginalFromFirebase(photoId: string): Promise<Buffer | null> {
  try {
    // List files in the photo directory to find the original
    const [files] = await bucket.getFiles({
      prefix: `stock/originals/${photoId}/`
    });

    if (files.length === 0) {
      return null;
    }

    // Download the first (and should be only) file
    const [buffer] = await files[0].download();
    return buffer;
  } catch (error) {
    console.error("Failed to get original from Firebase Storage:", error);
    return null;
  }
}