import { bucket as adminBucket } from "../firebaseAdmin";
import https from "https";

const TEST_TTL_MS = 10 * 60 * 1000; // 10 min stickiness
let lastMode: "public" | "signed" | null = null;
let lastChecked = 0;

function publicUrl(storagePath: string): string {
  const bkt = adminBucket.name;
  return `https://firebasestorage.googleapis.com/v0/b/${bkt}/o/${encodeURIComponent(storagePath)}?alt=media`;
}

async function head(url: string): Promise<number> {
  return new Promise((resolve) => {
    const req = https.request(url, { method: "HEAD" }, (res) => {
      resolve(res.statusCode || 0);
    });
    req.on("error", () => resolve(0));
    req.end();
  });
}

async function signedUrl(storagePath: string, ttlSeconds = 900): Promise<string> {
  const file = adminBucket.file(storagePath);
  const [url] = await file.getSignedUrl({
    version: "v4",
    action: "read",
    expires: Date.now() + ttlSeconds * 1000,
    responseDisposition: `attachment; filename="${storagePath.split("/").pop()}"`,
  });
  return url;
}

async function signedUrlInline(storagePath: string, ttlSeconds = 900): Promise<string> {
  const file = adminBucket.file(storagePath);
  const [url] = await file.getSignedUrl({
    version: "v4",
    action: "read",
    expires: Date.now() + ttlSeconds * 1000,
    responseDisposition: "inline",
  });
  return url;
}

export async function getDownloadUrlAuto(storagePath: string): Promise<string> {
  const now = Date.now();

  // If we recently decided, re-use that mode
  if (lastMode && (now - lastChecked) < TEST_TTL_MS) {
    return lastMode === "public" ? publicUrl(storagePath) : signedUrl(storagePath);
  }

  // Probe public URL once
  const url = publicUrl(storagePath);
  const status = await head(url);

  if (status === 200) {
    lastMode = "public";
    lastChecked = now;
    return url;
  }

  // Fallback: sign
  lastMode = "signed";
  lastChecked = now;
  return signedUrl(storagePath);
}

export async function getViewerUrlAuto(storagePath: string): Promise<string> {
  // Always use signed URL with inline disposition for PDF viewing
  // This ensures reliable preview rendering with proper content-disposition header
  return signedUrlInline(storagePath);
}
