import { FieldValue } from 'firebase-admin/firestore';
import { bucket, db } from '../firebaseAdmin';

/**
 * SN's Firestore Schema Specification
 * Path: templates/business-plan/types/<slug>
 */
export interface BusinessPlanTemplateMetadata {
  title: string;
  slug: string;
  category: string;
  isActive: boolean;
  isMaster: boolean;  // Only one per category should be true
  currentVersion: string;  // e.g., "v1"
  storagePaths: {
    docx: string;
    pdf: string;
    preview: string;
    thumb: string;
  };
  manifestRef?: string;  // Path to manifest JSON in storage
  tags: string[];
  sections: any[];  // Structured section manifest (see SN's sample)
  createdAt: any;  // Firestore serverTimestamp
  updatedAt: any;  // Firestore serverTimestamp
}

/**
 * Version subdocument
 * Path: templates/business-plan/types/<slug>/versions/<version>
 */
export interface BusinessPlanTemplateVersion {
  version: string;  // "v1", "v2", etc.
  isMaster: boolean;
  notes?: string;
  storagePaths: {
    docx: string;
    pdf: string;
    preview: string;
    thumb: string;
  };
  sections: any[];  // Snapshot of sections for this version
  createdAt: any;  // Firestore serverTimestamp
}

/**
 * Global settings document
 * Path: settings/templates
 */
export interface TemplateSettings {
  'business-plan': {
    masterSlug: string;
    masterVersion: string;
    lastRotatedAt: any;  // Firestore serverTimestamp
  };
}

export class BusinessPlanTemplateService {
  private bucket: any;
  private db: any;
  
  constructor() {
    // Use singleton instances from firebaseAdmin.ts
    this.bucket = bucket;
    this.db = db;
    console.log("[BusinessPlanTemplates] Using bucket:", this.bucket.name);
  }

  /**
   * Upload DOCX file to Firebase Storage
   * SN's path: templates/business-plan/docs/<slug>/<slug>-<version>.docx
   */
  async uploadDocx(slug: string, version: string, buffer: Buffer): Promise<string> {
    const filePath = `templates/business-plan/docs/${slug}/${slug}-${version}.docx`;
    const file = this.bucket.file(filePath);
    
    await file.save(buffer, {
      metadata: {
        contentType: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        cacheControl: 'public, max-age=3600',
      },
      public: true,
    });

    await file.makePublic();

    return filePath;
  }

  /**
   * Upload PDF file to Firebase Storage
   * SN's path: templates/business-plan/pdfs/<slug>/<slug>-<version>.pdf
   */
  async uploadPdf(slug: string, version: string, buffer: Buffer): Promise<string> {
    const filePath = `templates/business-plan/pdfs/${slug}/${slug}-${version}.pdf`;
    const file = this.bucket.file(filePath);
    
    await file.save(buffer, {
      metadata: {
        contentType: 'application/pdf',
        cacheControl: 'public, max-age=3600',
      },
      public: true,
    });

    await file.makePublic();

    return filePath;
  }

  /**
   * Upload preview image (WebP) to Firebase Storage
   * SN's path: templates/business-plan/previews/<slug>/<slug>-<version>-preview.webp
   */
  async uploadPreviewImage(slug: string, version: string, buffer: Buffer): Promise<string> {
    const filePath = `templates/business-plan/previews/${slug}/${slug}-${version}-preview.webp`;
    const file = this.bucket.file(filePath);
    
    await file.save(buffer, {
      metadata: {
        contentType: 'image/webp',
        cacheControl: 'public, max-age=31536000', // 1 year for images
      },
      public: true,
    });

    await file.makePublic();

    return filePath;
  }

  /**
   * Upload thumbnail image (WebP) to Firebase Storage
   * SN's canonical path: templates/business-plan/thumbs/<slug>/<slug>-<version>-thumb.webp
   */
  async uploadThumbImage(slug: string, version: string, buffer: Buffer): Promise<string> {
    const filePath = `templates/business-plan/thumbs/${slug}/${slug}-${version}-thumb.webp`;
    const file = this.bucket.file(filePath);
    
    await file.save(buffer, {
      metadata: {
        contentType: 'image/webp',
        cacheControl: 'public, max-age=31536000', // 1 year for images
      },
      public: true,
    });

    await file.makePublic();

    return filePath;
  }

  /**
   * Create or update template metadata in Firestore
   * SN's path: templates/business-plan/types/<slug>
   */
  async saveMetadata(metadata: BusinessPlanTemplateMetadata): Promise<void> {
    const docRef = this.db.collection('templates').doc('business-plan').collection('types').doc(metadata.slug);
    await docRef.set(metadata, { merge: true });
  }

  /**
   * Get template metadata from Firestore
   * SN's path: templates/business-plan/types/<slug>
   */
  async getMetadata(slug: string): Promise<BusinessPlanTemplateMetadata | null> {
    const docRef = this.db.collection('templates').doc('business-plan').collection('types').doc(slug);
    const doc = await docRef.get();
    
    if (!doc.exists) {
      return null;
    }
    
    return doc.data() as BusinessPlanTemplateMetadata;
  }

  /**
   * Update template metadata (title, category, tags/industries)
   */
  async updateMetadata(slug: string, updates: { title?: string; category?: string; tags?: string[] }): Promise<void> {
    const docRef = this.db.collection('templates').doc('business-plan').collection('types').doc(slug);
    
    await docRef.update({
      ...updates,
      updatedAt: FieldValue.serverTimestamp()
    });
  }

  /**
   * List all active templates from Firestore
   */
  async listTemplates(): Promise<BusinessPlanTemplateMetadata[]> {
    const snapshot = await this.db.collection('templates').doc('business-plan').collection('types')
      .where('isActive', '==', true)
      .get();
    
    return snapshot.docs.map((doc: any) => doc.data() as BusinessPlanTemplateMetadata);
  }

  /**
   * Save version subdocument
   * SN's path: templates/business-plan/types/<slug>/versions/<version>
   */
  async saveVersion(slug: string, versionData: BusinessPlanTemplateVersion): Promise<void> {
    const versionRef = this.db.collection('templates').doc('business-plan').collection('types').doc(slug)
      .collection('versions').doc(versionData.version);
    await versionRef.set(versionData);
  }

  /**
   * Atomic master template flag management
   * Implements SN's transaction requirement:
   * 1. Set isMaster: false on all other templates AND their versions
   * 2. Set isMaster: true on the target template and target version
   * 3. Update global settings document
   */
  async setMasterTemplate(slug: string, version: string): Promise<void> {
    await this.db.runTransaction(async (transaction) => {
      const typesRef = this.db.collection('templates').doc('business-plan').collection('types');
      const settingsRef = this.db.collection('settings').doc('templates');
      const targetRef = typesRef.doc(slug);
      const targetVersionsRef = targetRef.collection('versions');
      
      // IMPORTANT: ALL READS MUST COME BEFORE ANY WRITES IN FIRESTORE TRANSACTIONS
      // 1. Get all templates
      const snapshot = await transaction.get(typesRef);
      
      // 2. Get all versions of target template
      const targetVersionsSnapshot = await transaction.get(targetVersionsRef);
      
      // NOW DO ALL WRITES
      // 3. Set isMaster: false on all other templates
      snapshot.docs.forEach((doc) => {
        if (doc.id !== slug && doc.data().isMaster) {
          transaction.update(doc.ref, { isMaster: false });
        }
      });
      
      // 4. Set isMaster: true on target template
      transaction.update(targetRef, { 
        isMaster: true, 
        currentVersion: version,
        updatedAt: FieldValue.serverTimestamp()
      });
      
      // 5. Demote all versions of the target template to isMaster: false
      targetVersionsSnapshot.docs.forEach((versionDoc) => {
        transaction.update(versionDoc.ref, { isMaster: false });
      });
      
      // 6. Promote the specific target version to isMaster: true
      const targetVersionRef = targetVersionsRef.doc(version);
      transaction.update(targetVersionRef, { isMaster: true });
      
      // 7. Update global settings
      transaction.set(settingsRef, {
        'business-plan': {
          masterSlug: slug,
          masterVersion: version,
          lastRotatedAt: FieldValue.serverTimestamp()
        }
      }, { merge: true });
    });
  }

  /**
   * Generate Google Docs viewer URL
   */
  generateGDocsUrl(docxUrl: string): string {
    return `https://docs.google.com/gview?embedded=1&url=${encodeURIComponent(docxUrl)}`;
  }

  /**
   * Generate signed download URL for public access
   * Expires in 1 hour
   */
  async getSignedDownloadUrl(storagePath: string, filename: string): Promise<string> {
    const file = this.storage.file(storagePath);
    
    const [url] = await file.getSignedUrl({
      action: 'read',
      expires: Date.now() + 60 * 60 * 1000, // 1 hour
      responseDisposition: `attachment; filename="${filename}.docx"`
    });
    
    return url;
  }

  /**
   * Delete template completely (Firestore + Storage)
   */
  async deleteTemplate(slug: string): Promise<void> {
    // 1. Get template metadata to find storage paths
    const templateRef = this.db.collection('templates').doc('business-plan').collection('types').doc(slug);
    const templateDoc = await templateRef.get();
    
    if (!templateDoc.exists) {
      throw new Error(`Template ${slug} not found`);
    }
    
    const data = templateDoc.data();
    
    // 2. Delete all files from Firebase Storage
    const pathsToDelete: string[] = [];
    
    // Add main storage paths
    if (data?.storagePaths?.docx) pathsToDelete.push(data.storagePaths.docx);
    if (data?.storagePaths?.pdf) pathsToDelete.push(data.storagePaths.pdf);
    if (data?.storagePaths?.preview) pathsToDelete.push(data.storagePaths.preview);
    if (data?.storagePaths?.thumb) pathsToDelete.push(data.storagePaths.thumb);
    
    // Delete all files from storage directories
    const prefixes = [
      `templates/business-plan/docs/${slug}/`,
      `templates/business-plan/pdfs/${slug}/`,
      `templates/business-plan/previews/${slug}/`,
      `templates/business-plan/thumbs/${slug}/`
    ];
    
    for (const prefix of prefixes) {
      try {
        const [files] = await this.bucket.getFiles({ prefix });
        for (const file of files) {
          await file.delete();
          console.log(`🗑️ Deleted: ${file.name}`);
        }
      } catch (error) {
        console.warn(`Warning: Could not delete files with prefix ${prefix}:`, error);
      }
    }
    
    // 3. Delete version subcollection
    const versionsRef = templateRef.collection('versions');
    const versionsSnapshot = await versionsRef.get();
    for (const versionDoc of versionsSnapshot.docs) {
      await versionDoc.ref.delete();
    }
    
    // 4. Delete main template document
    await templateRef.delete();
    
    // 5. Update global settings if this was the master template
    if (data?.isMaster) {
      const settingsRef = this.db.collection('settings').doc('templates');
      await settingsRef.set({
        'business-plan': {
          masterSlug: null,
          masterVersion: null,
          lastRotatedAt: FieldValue.serverTimestamp()
        }
      }, { merge: true });
    }
    
    console.log(`✅ Template ${slug} deleted completely`);
  }
}

export const businessPlanTemplateService = new BusinessPlanTemplateService();

// PUBLIC API HELPERS (for free template downloads)
export type TemplateDoc = {
  title: string;
  slug: string;
  category: string;
  industries?: string[];
  tags?: string[];
  isActive?: boolean;
  isMaster?: boolean;
  currentVersion?: string;
  storagePaths?: {
    docx?: string;
    pdf?: string;
    preview?: string;
    thumb?: string;
  };
  sections?: any[];
  createdAt?: any;
  updatedAt?: any;
};

function docToTemplate(data: any, id: string): TemplateDoc {
  return {
    title: data.title,
    slug: data.slug ?? id,
    category: data.category ?? "General",
    industries: data.industries || data.tags || [],
    tags: data.tags || [],
    isActive: data.isActive ?? true,
    isMaster: data.isMaster ?? false,
    currentVersion: data.currentVersion ?? null,
    storagePaths: data.storagePaths || {},
    sections: data.sections || [],
    createdAt: data.createdAt || null,
    updatedAt: data.updatedAt || null
  };
}

export async function listBusinessPlanTemplates(): Promise<TemplateDoc[]> {
  const snap = await db.collection('templates').doc('business-plan').collection('types')
    .where('isActive', '==', true)
    .get();
  return snap.docs.map(d => docToTemplate(d.data(), d.id));
}

export async function getMasterBusinessPlanTemplate(): Promise<TemplateDoc | null> {
  // First try pointer doc for speed
  const pointer = await db.collection('settings').doc('templates').get();
  if (pointer.exists) {
    const data = pointer.data();
    const slug = data?.['business-plan']?.masterSlug;
    if (slug) {
      const doc = await db.collection('templates').doc('business-plan').collection('types').doc(slug).get();
      if (doc.exists) return docToTemplate(doc.data()!, doc.id);
    }
  }
  // Fallback scan if pointer missing
  const q = await db.collection('templates').doc('business-plan').collection('types')
    .where('isMaster', '==', true)
    .limit(1)
    .get();
  if (q.empty) return null;
  const d = q.docs[0];
  return docToTemplate(d.data(), d.id);
}

export async function getBusinessPlanTemplateBySlug(slug: string): Promise<TemplateDoc | null> {
  const doc = await db.collection('templates').doc('business-plan').collection('types').doc(slug).get();
  return doc.exists ? docToTemplate(doc.data()!, doc.id) : null;
}

/**
 * Public direct URL builder (no signature needed)
 * Works because Storage rules allow: allow read: if true;
 */
export function getPublicDocxUrl(storagePath: string): string {
  const bkt = bucket.name;
  const encoded = encodeURIComponent(storagePath);
  return `https://firebasestorage.googleapis.com/v0/b/${bkt}/o/${encoded}?alt=media`;
}

/**
 * Signed URL (use if you later lock Storage reads)
 */
export async function getSignedDocxUrl(storagePath: string, ttlSeconds = 900): Promise<string> {
  const file = bucket.file(storagePath);
  const expires = Date.now() + ttlSeconds * 1000;
  const [url] = await file.getSignedUrl({
    version: 'v4',
    action: 'read',
    expires
  });
  return url;
}
