import sharp from 'sharp';

/**
 * Business Plan Preview Generator
 * 
 * FUTURE IMPLEMENTATION: Master DOCX with Placeholder Map
 * =====================================================
 * Per SN's specification, the production system will use a single styled master DOCX
 * (with the professional blue gradient design) containing bookmarks or curly placeholders:
 * 
 * Placeholders:
 *   {COVER_TITLE}, {COVER_SUBTITLE}
 *   {EXEC_SUMMARY}, {COMPANY_OVERVIEW}, {PRODUCTS_SERVICES}
 *   {MARKET_ANALYSIS}, {MARKETING_SALES}, {OPERATIONS}
 *   {ORG_MANAGEMENT}, {FINANCIAL_PLAN}, {MILESTONES_ROADMAP}
 *   {FUTURE_EXPANSION}, {PARTNERSHIPS_NOTE}, {EXIT_STRATEGY}, {APPENDIX}
 * 
 * Pipeline:
 *   1. Load master DOCX template
 *   2. Replace placeholders with actual content
 *   3. Save filled DOCX
 *   4. Convert to PDF via LibreOffice (convertDocxToPdf)
 *   5. Generate previews via Poppler (pdfToJpegPages)
 *   6. Upload all files to Firebase Storage
 * 
 * Preview specs:
 *   - Long edge: 1200–1600px
 *   - Quality: 80–85
 *   - Format: JPEG
 */
export class BusinessPlanPreviewGenerator {
  /**
   * TODO: Convert DOCX to PDF using LibreOffice headless
   * Command: soffice --headless --convert-to pdf --outdir <dir> <docx>
   * When implemented, this will enable real document conversion instead of placeholder previews
   */
  async convertDocxToPdf(docxBuffer: Buffer): Promise<Buffer> {
    throw new Error('TODO: Implement LibreOffice DOCX → PDF conversion');
  }

  /**
   * TODO: Convert PDF pages to JPEG images using Poppler or Ghostscript
   * Poppler command: pdftoppm -jpeg -r 150 <pdf> <output-prefix>
   * Ghostscript command: gs -dBATCH -dNOPAUSE -sDEVICE=jpeg -r150 -sOutputFile=page-%d.jpg <pdf>
   * When implemented, this will extract real preview images from PDFs
   */
  async pdfToJpegPages(pdfBuffer: Buffer): Promise<Buffer[]> {
    throw new Error('TODO: Implement Poppler/Ghostscript PDF → JPEG conversion');
  }

  /**
   * Generate preview images from DOCX content
   * Current: Generates styled placeholder images using Sharp
   * SN's spec: Output WebP format
   * - Preview: 1024px wide (full preview)
   * - Thumb: 360px wide (card thumbnail)
   */
  async generatePreviewPages(
    title: string,
    category: string,
    sections: string[]
  ): Promise<{ preview: Buffer; thumb: Buffer }> {
    const previewWidth = 1024;
    const thumbWidth = 360;
    const aspectRatio = 1.294;  // 8.5 x 11 aspect ratio
    
    const previewHeight = Math.round(previewWidth * aspectRatio);
    const thumbHeight = Math.round(thumbWidth * aspectRatio);

    // Generate cover page SVG
    const coverSvg = this.generateCoverPage(title, category, previewWidth, previewHeight);

    // Convert to WebP for both sizes
    const previewBuffer = await sharp(Buffer.from(coverSvg))
      .webp({ quality: 85 })
      .toBuffer();

    const thumbBuffer = await sharp(Buffer.from(coverSvg))
      .resize(thumbWidth, thumbHeight)
      .webp({ quality: 80 })
      .toBuffer();

    return { preview: previewBuffer, thumb: thumbBuffer };
  }

  private generateCoverPage(title: string, category: string, width: number, height: number): string {
    const escapedTitle = this.escapeXml(title);
    const escapedCategory = this.escapeXml(category);
    
    return `
      <svg width="${width}" height="${height}" xmlns="http://www.w3.org/2000/svg">
        <defs>
          <linearGradient id="grad" x1="0%" y1="0%" x2="0%" y2="100%">
            <stop offset="0%" style="stop-color:#1e3a5f;stop-opacity:1" />
            <stop offset="100%" style="stop-color:#2d5a8a;stop-opacity:1" />
          </linearGradient>
        </defs>
        
        <rect width="${width}" height="${height}" fill="url(#grad)"/>
        <rect x="0" y="${height * 0.15}" width="${width}" height="8" fill="#ff8800"/>
        
        <text x="${width / 2}" y="${height * 0.35}" 
              font-family="Arial, sans-serif" font-size="48" font-weight="bold" 
              fill="#ffffff" text-anchor="middle">BUSINESS PLAN</text>
        
        <text x="${width / 2}" y="${height * 0.45}" 
              font-family="Arial, sans-serif" font-size="36" 
              fill="#ffffff" text-anchor="middle">${escapedTitle}</text>
        
        <line x1="${width * 0.3}" y1="${height * 0.50}" x2="${width * 0.7}" y2="${height * 0.50}" 
              stroke="#ffffff" stroke-width="2"/>
        
        <text x="${width / 2}" y="${height * 0.85}" 
              font-family="Arial, sans-serif" font-size="18" 
              fill="#cccccc" text-anchor="middle">${escapedCategory}</text>
        
        <text x="${width / 2}" y="${height * 0.90}" 
              font-family="Arial, sans-serif" font-size="16" 
              fill="#cccccc" text-anchor="middle">Professional Business Plan Template</text>
      </svg>
    `;
  }

  private generateContentPage(sectionTitle: string, planTitle: string, width: number, height: number): string {
    const escapedSection = this.escapeXml(sectionTitle.toUpperCase());
    const escapedPlanTitle = this.escapeXml(planTitle);
    
    const contentItems = [
      'Overview and introduction',
      'Key strategies and approaches',
      'Detailed analysis',
      'Future considerations'
    ];
    
    const contentBlocks = contentItems.map((item, i) => {
      const y = 280 + (i * 120);
      return `
        <rect x="80" y="${y}" width="640" height="80" fill="#f5f5f5" rx="8"/>
        <text x="100" y="${y + 45}" font-family="Arial, sans-serif" font-size="20" fill="#333333">${this.escapeXml(item)}</text>
        <line x1="100" y1="${y + 60}" x2="720" y2="${y + 60}" stroke="#e0e0e0" stroke-width="1"/>
      `;
    }).join('');
    
    return `
      <svg width="${width}" height="${height}" xmlns="http://www.w3.org/2000/svg">
        <rect width="${width}" height="${height}" fill="#ffffff"/>
        
        <!-- Header -->
        <rect x="0" y="0" width="${width}" height="120" fill="#1e3a5f"/>
        <rect x="0" y="120" width="${width}" height="6" fill="#ff8800"/>
        
        <text x="${width / 2}" y="75" 
              font-family="Arial, sans-serif" font-size="32" font-weight="bold" 
              fill="#ffffff" text-anchor="middle">${escapedSection}</text>
        
        <!-- Content sections -->
        ${contentBlocks}
        
        <!-- Footer -->
        <rect x="0" y="${height - 60}" width="${width}" height="60" fill="#f8f8f8"/>
        <text x="${width / 2}" y="${height - 25}" 
              font-family="Arial, sans-serif" font-size="14" 
              fill="#666666" text-anchor="middle">${escapedPlanTitle} Business Plan</text>
      </svg>
    `;
  }

  private escapeXml(text: string): string {
    return text
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&apos;');
  }
}

export const previewGenerator = new BusinessPlanPreviewGenerator();
