import { createHash } from "crypto";
import { getFirestore } from "firebase-admin/firestore";

export type DownloadEvent = {
  slug: string;
  version: string;
  mode: "public" | "signed";
  ts: number;
  ua?: string;
  ref?: string;
  ipHash?: string;
};

/**
 * Logs a template download event to Firestore analytics collection.
 * Stores SHA-256 hash of IP instead of raw IP for privacy compliance.
 */
export async function logDownloadEvent(event: {
  slug: string;
  version: string;
  mode: "public" | "signed";
  userAgent?: string;
  referrer?: string;
  ip?: string;
}): Promise<void> {
  try {
    const db = getFirestore();
    const eventData: DownloadEvent = {
      slug: event.slug,
      version: event.version,
      mode: event.mode,
      ts: Date.now(),
    };

    if (event.userAgent) {
      eventData.ua = event.userAgent;
    }

    if (event.referrer) {
      eventData.ref = event.referrer;
    }

    // Hash IP address for privacy (SHA-256)
    if (event.ip) {
      eventData.ipHash = createHash("sha256").update(event.ip).digest("hex");
    }

    await db.collection("analytics").doc("downloads").collection("events").add(eventData);
  } catch (err) {
    console.error("[Analytics] Failed to log download event:", err);
    // Don't throw - analytics failure shouldn't break downloads
  }
}
