import express from "express";
import Stripe from "stripe";
import { storage as db } from "../storage";
import { createInfographicBundle } from "../services/infographicZipAssembly";

const router = express.Router();

router.post("/", express.raw({ type: "application/json" }), async (req, res) => {
  const sig = req.headers["stripe-signature"];
  const whSecret = process.env.STRIPE_WEBHOOK_SECRET!;
  let event: Stripe.Event;

  try {
    const stripe = new Stripe(process.env.STRIPE_SECRET_KEY!, { apiVersion: "2024-06-20" });
    event = stripe.webhooks.constructEvent(req.body, sig as string, whSecret);
  } catch (err: any) {
    console.error("[stripe/webhook] Signature verification failed:", err.message);
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }

  try {
    if (event.type === "checkout.session.completed") {
      const session = event.data.object as Stripe.Checkout.Session;
      const purchaseType = session.metadata?.type;

      if (purchaseType === 'infographic_bundle') {
        // Handle infographic bundle purchase
        const purchaseId = session.metadata?.purchaseId!;
        const selectedIds = session.metadata?.selectedIds?.split(',') || [];
        
        // Get actual paid amount and currency from session
        const actualAmountCents = session.amount_total || 0;
        const actualCurrency = session.currency || 'usd';
        
        try {
          // Create and upload ZIP bundle containing selected templates
          const downloadUrl = await createInfographicBundle(purchaseId, selectedIds);
          
          // Mark purchase as paid with real download URL
          const updatedPurchase = await db.updateInfographicPurchase(purchaseId, {
            status: 'paid',
            stripePaymentIntent: session.payment_intent as string,
            downloadUrl: downloadUrl,
            actualAmountCents: actualAmountCents,
            actualCurrency: actualCurrency
          });

          console.log(`[stripe/webhook] Infographic bundle purchase ${purchaseId} marked as paid - ${selectedIds.length} items - Amount: ${actualAmountCents} ${actualCurrency} - ZIP created`);
          
        } catch (error) {
          console.error(`[stripe/webhook] Failed to create ZIP bundle for purchase ${purchaseId}:`, error);
          
          // Mark purchase as paid but without download URL - user can retry later
          const updatedPurchase = await db.updateInfographicPurchase(purchaseId, {
            status: 'paid',
            stripePaymentIntent: session.payment_intent as string,
            downloadUrl: null,
            actualAmountCents: actualAmountCents,
            actualCurrency: actualCurrency
          });
        }
        
      } else {
        // Handle cover template purchase (existing logic)
        const tplId = session.metadata?.templateId!;
        const customImageUrl = session.metadata?.customImageUrl || null;

        // Get template to determine download URL
        const tpl = await db.getCoverTemplate(tplId);
        const download = tpl?.downloadFile || tpl?.pptxUrl || tpl?.keynoteUrl || tpl?.gslidesUrl || "";
        
        // Get actual paid amount and currency from session (handles discounts)
        const actualAmountCents = session.amount_total || 0;  // Amount actually paid (after discounts)
        const actualCurrency = session.currency || 'usd';
        
        // Mark purchase as paid with actual payment details
        const updatedPurchase = await db.markPurchasePaidBySession(
          session.id,
          session.payment_intent as string,
          download,
          actualAmountCents,
          actualCurrency
        );

        console.log(`[stripe/webhook] Cover template purchase ${session.metadata?.purchaseId || 'unknown'} marked as paid for template ${tplId} - Amount: ${actualAmountCents} ${actualCurrency}`);
      }
    }
  } catch (e) {
    console.error("[stripe/webhook] Error processing webhook:", e);
  }
  
  res.json({ received: true });
});

export default router;