// routes/stockLibraryRoutes.js
// Upload + serve stock photos with server-side watermarking for previews/unlicensed
import path from "path";
import fs from "fs/promises";
import express from "express";
import multer from "multer";
import {
  applyWatermark,
  toTenDigitCode,
} from "../services/watermark.js";
import { ObjectStorageService, objectStorageClient } from "../objectStorage";

// NEW: entitlements
import {
  hasLicense,
  grantLicenses,
  getTotalRemaining,
  decrementOne,
} from "../services/entitlements.js";
import { requireAdmin } from "../middleware/adminGuard.js";

const router = express.Router();

// Helper: load photos database
async function getAllPhotos() {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "photos-db.json");
    const data = await fs.readFile(dbPath, "utf8");
    return JSON.parse(data);
  } catch (error) {
    return [];
  }
}

// Helper: save photo to database
async function savePhotoToDB(photo) {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "photos-db.json");
    const photos = await getAllPhotos();
    photos.push(photo);
    await fs.writeFile(dbPath, JSON.stringify(photos, null, 2));
    return photo;
  } catch (error) {
    console.error("Error saving photo to DB:", error);
    throw error;
  }
}

// Helper: load mockups database
async function getAllMockups() {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "mockups-db.json");
    const data = await fs.readFile(dbPath, "utf8");
    return JSON.parse(data);
  } catch (error) {
    return [];
  }
}

// Helper: save mockup to database
async function saveMockupToDB(mockup) {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "mockups-db.json");
    const mockups = await getAllMockups();
    mockups.push(mockup);
    await fs.writeFile(dbPath, JSON.stringify(mockups, null, 2));
    return mockup;
  } catch (error) {
    console.error("Error saving mockup to DB:", error);
    throw error;
  }
}

// Helper: delete photo from database
async function deletePhotoFromDB(id) {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "photos-db.json");
    const photos = await getAllPhotos();
    const filteredPhotos = photos.filter(photo => photo.id !== id);
    await fs.writeFile(dbPath, JSON.stringify(filteredPhotos, null, 2));
    console.log(`✅ Deleted photo ${id} from database`);
    return true;
  } catch (error) {
    console.error("Error deleting photo from DB:", error);
    throw error;
  }
}

// Helper: delete mockup from database
async function deleteMockupFromDB(id) {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "mockups-db.json");
    const mockups = await getAllMockups();
    const filteredMockups = mockups.filter(mockup => mockup.id !== id);
    await fs.writeFile(dbPath, JSON.stringify(filteredMockups, null, 2));
    console.log(`✅ Deleted mockup ${id} from database`);
    return true;
  } catch (error) {
    console.error("Error deleting mockup from DB:", error);
    throw error;
  }
}

// --- Multer storage (temp to local before pushing to object storage) ---
const uploadDir = path.join(process.cwd(), "uploads", "photos");
const storage = multer.diskStorage({
  destination: async (_req, _file, cb) => {
    await fs.mkdir(uploadDir, { recursive: true });
    cb(null, uploadDir);
  },
  filename: (_req, file, cb) => {
    const stamp = Date.now();
    const safe = file.originalname.replace(/\s+/g, "_");
    cb(null, `${stamp}_${safe}`);
  },
});
const upload = multer({
  storage,
  limits: { fileSize: 25 * 1024 * 1024 }, // 25MB
});

// Helper: object storage keys
function objKeyOriginal(dbId, filename) {
  return `stock/original/${dbId}/${filename}`;
}
function objKeyPreview(dbId, filename) {
  return `stock/preview/${dbId}/${filename}`;
}

// Mock DB insert
async function insertStockDBRecord({ originalName, mimeType, uploaderId }) {
  const id =
    Math.floor(Date.now() / 1000).toString(36) +
    Math.floor(Math.random() * 1e6).toString(36);
  return { id, originalName, mimeType, uploaderId };
}

// POST /api/stock/upload (admin)
router.post("/upload", upload.single("file"), async (req, res) => {
  try {
    if (!req.file) return res.status(400).json({ error: "No file uploaded" });

    // 1) Create DB record
    const dbRow = await insertStockDBRecord({
      originalName: file.originalname,
      mimeType: file.mimetype,
      uploaderId: req.user?.id || "admin",
    });

    const tenDigit = toTenDigitCode(dbRow.id);

    // 2) Push ORIGINAL (private) to object storage
    const originalBuf = await fs.readFile(file.path);
    const objectStorage = new ObjectStorageService();
    const publicPaths = objectStorage.getPublicObjectSearchPaths();
    const originalPath = `${publicPaths[0]}/stock/original/${dbRow.id}/${file.filename}`;
    
    try {
      await objectStorage.uploadFile(originalPath, originalBuf);
      console.log(`✅ Uploaded original to object storage: ${originalPath}`);
    } catch (uploadError) {
      console.warn(`⚠️ Object storage upload failed, saving locally:`, uploadError);
    }

    // 3) WATERMARKED PREVIEW (public)
    const wmBuf = await applyWatermark(originalBuf, { stockSeed: dbRow.id, opacity: 0.15 });
    const previewPath = `${publicPaths[0]}/stock/preview/${dbRow.id}/${file.filename}`;
    
    try {
      await objectStorage.uploadFile(previewPath, wmBuf);
      console.log(`✅ Uploaded preview to object storage: ${previewPath}`);
    } catch (uploadError) {
      console.warn(`⚠️ Preview upload failed, saving locally:`, uploadError);
      // Save locally as fallback
      const localPreviewDir = path.join(process.cwd(), "uploads", "previews", dbRow.id);
      await fs.mkdir(localPreviewDir, { recursive: true });
      const localPreviewPath = path.join(localPreviewDir, file.filename);
      await fs.writeFile(localPreviewPath, wmBuf);
    }

    // 4) Clean local temp
    await fs.unlink(file.path).catch(() => {});

    // 5) Save to JSON database
    const photoRecord = {
      id: dbRow.id,
      name: file.originalname,
      filename: file.filename,
      mimeType: file.mimetype,
      uploadedAt: new Date().toISOString(),
      uploaderId: req.user?.id || "admin"
    };
    await savePhotoToDB(photoRecord);

    // 6) Response metadata
    res.json({
      id: dbRow.id,
      code: tenDigit,
      name: file.originalname,
      mimeType: file.mimetype,
      previewUrl: `/api/stock/${dbRow.id}/preview`,
      originalKey: originalPath, // private
    });
  } catch (err) {
    console.error("Upload error:", err);
    res.status(500).json({ error: "Upload failed" });
  }
});

// POST /api/stock/photos (admin) - photo upload (what StockUploader actually calls)
router.post("/photos", upload.array("files"), async (req, res) => {
  try {
    if (!req.files || req.files.length === 0) return res.status(400).json({ error: "No files uploaded" });

    // Handle multiple files - process first file for now
    const file = req.files[0];

    // 1) Create DB record
    const dbRow = await insertStockDBRecord({
      originalName: file.originalname,
      mimeType: file.mimetype,
      uploaderId: req.user?.id || "admin",
    });

    const tenDigit = toTenDigitCode(dbRow.id);

    // 2) Push ORIGINAL (private) to object storage
    const originalBuf = await fs.readFile(file.path);
    const objectStorage = new ObjectStorageService();
    const publicPaths = objectStorage.getPublicObjectSearchPaths();
    const originalPath = `${publicPaths[0]}/stock/original/${dbRow.id}/${file.filename}`;
    
    try {
      await objectStorage.uploadFile(originalPath, originalBuf);
      console.log(`✅ Uploaded original to object storage: ${originalPath}`);
    } catch (uploadError) {
      console.warn(`⚠️ Object storage upload failed, saving locally:`, uploadError);
    }

    // 3) WATERMARKED PREVIEW (public)
    const wmBuf = await applyWatermark(originalBuf, { stockSeed: dbRow.id, opacity: 0.15 });
    const previewPath = `${publicPaths[0]}/stock/preview/${dbRow.id}/${file.filename}`;
    
    try {
      await objectStorage.uploadFile(previewPath, wmBuf);
      console.log(`✅ Uploaded preview to object storage: ${previewPath}`);
    } catch (uploadError) {
      console.warn(`⚠️ Preview upload failed, saving locally:`, uploadError);
      // Save locally as fallback
      const localPreviewDir = path.join(process.cwd(), "uploads", "previews", dbRow.id);
      await fs.mkdir(localPreviewDir, { recursive: true });
      const localPreviewPath = path.join(localPreviewDir, file.filename);
      await fs.writeFile(localPreviewPath, wmBuf);
    }

    // 4) Clean local temp
    await fs.unlink(file.path).catch(() => {});

    // 5) Save to JSON database
    const photoRecord = {
      id: dbRow.id,
      name: file.originalname,
      filename: file.filename,
      mimeType: file.mimetype,
      uploadedAt: new Date().toISOString(),
      uploaderId: req.user?.id || "admin"
    };
    await savePhotoToDB(photoRecord);

    // 6) Response metadata
    res.json({
      id: dbRow.id,
      code: tenDigit,
      name: file.originalname,
      mimeType: file.mimetype,
      previewUrl: `/api/stock/${dbRow.id}/preview`,
      originalKey: originalPath, // private
    });
  } catch (err) {
    console.error("Upload error:", err);
    res.status(500).json({ error: "Upload failed" });
  }
});

// POST /api/stock/mockups (admin) - mockup upload
router.post("/mockups", upload.array("files", 10), async (req, res) => {
  try {
    if (!req.files || req.files.length === 0) return res.status(400).json({ error: "No files uploaded" });

    const results = [];
    const objectStorage = new ObjectStorageService();
    const publicPaths = objectStorage.getPublicObjectSearchPaths();

    // Process multiple files in a loop
    for (const file of req.files) {
      // 1) Create DB record
      const dbRow = await insertStockDBRecord({
        originalName: file.originalname,
        mimeType: file.mimetype,
        uploaderId: req.user?.id || "admin",
      });

      const tenDigit = toTenDigitCode(dbRow.id);

      // 2) Push ORIGINAL (private) to object storage
      const originalBuf = await fs.readFile(file.path);
      const originalPath = `${publicPaths[0]}/mockups/original/${dbRow.id}/${file.filename}`;
      
      try {
        await objectStorage.uploadFile(originalPath, originalBuf, file.mimetype);
        console.log(`✅ Uploaded original mockup to object storage: ${originalPath}`);
      } catch (uploadError) {
        console.warn(`⚠️ Object storage upload failed, saving locally:`, uploadError);
      }

      // 3) WATERMARKED PREVIEW (public)
      const wmBuf = await applyWatermark(originalBuf, { stockSeed: dbRow.id, opacity: 0.15 });
      const previewPath = `${publicPaths[0]}/mockups/preview/${dbRow.id}/${file.filename}`;
      
      try {
        await objectStorage.uploadFile(previewPath, wmBuf, file.mimetype);
        console.log(`✅ Uploaded mockup preview to object storage: ${previewPath}`);
      } catch (uploadError) {
        console.warn(`⚠️ Mockup preview upload failed, saving locally:`, uploadError);
        // Save locally as fallback
        const localPreviewDir = path.join(process.cwd(), "uploads", "mockup-previews", dbRow.id);
        await fs.mkdir(localPreviewDir, { recursive: true });
        const localPreviewPath = path.join(localPreviewDir, file.filename);
        await fs.writeFile(localPreviewPath, wmBuf);
      }

      // 4) Clean local temp
      await fs.unlink(file.path).catch(() => {});

      // 5) Save to JSON database
      const mockupRecord = {
        id: dbRow.id,
        name: file.originalname,
        filename: file.filename,
        mimeType: file.mimetype,
        uploadedAt: new Date().toISOString(),
        uploaderId: req.user?.id || "admin"
      };
      await saveMockupToDB(mockupRecord);

      // 6) Add to results
      results.push({
        id: dbRow.id,
        code: tenDigit,
        name: file.originalname,
        mimeType: file.mimetype,
        previewUrl: `/api/mockups/${dbRow.id}/preview`,
        originalKey: originalPath, // private
      });
    }

    res.json({ uploaded: results });
  } catch (err) {
    console.error("Mockup upload error:", err);
    res.status(500).json({ error: "Upload failed" });
  }
});

// DELETE /api/stock/photos/:id - Delete a photo (Admin only)
router.delete("/photos/:id", requireAdmin, async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`🗑️ Deleting photo ${id}...`);

    const objectStorage = new ObjectStorageService();
    const searchPaths = objectStorage.getPublicObjectSearchPaths();

    // 1) Delete from object storage first - both original and preview
    for (const searchPath of searchPaths) {
      const parts = searchPath.split('/');
      const bucketName = parts[1];
      const basePrefix = parts.slice(2).join('/');
      const bucket = objectStorageClient.bucket(bucketName);

      // Delete original files
      try {
        const originalPrefix = `${basePrefix}/stock/original/${id}/`;
        const [originalFiles] = await bucket.getFiles({ prefix: originalPrefix });
        for (const file of originalFiles) {
          await file.delete();
          console.log(`✅ Deleted original from object storage: ${file.name}`);
        }
      } catch (originalError) {
        console.warn(`⚠️ Failed to delete original from object storage:`, originalError);
      }

      // Delete preview files
      try {
        const previewPrefix = `${basePrefix}/stock/preview/${id}/`;
        const [previewFiles] = await bucket.getFiles({ prefix: previewPrefix });
        for (const file of previewFiles) {
          await file.delete();
          console.log(`✅ Deleted preview from object storage: ${file.name}`);
        }
      } catch (previewError) {
        console.warn(`⚠️ Failed to delete preview from object storage:`, previewError);
      }
    }

    // 3) Clean up local files
    try {
      // Delete local original files in uploads/photos/
      const localPhotosDir = path.join(process.cwd(), "uploads", "photos");
      try {
        const localFiles = await fs.readdir(localPhotosDir);
        for (const filename of localFiles) {
          if (filename.includes(id)) {
            const localFilePath = path.join(localPhotosDir, filename);
            await fs.unlink(localFilePath);
            console.log(`✅ Deleted local photo file: ${filename}`);
          }
        }
      } catch (localPhotosError) {
        console.warn(`⚠️ No local photos to clean up or error:`, localPhotosError);
      }

      // Delete local preview files
      const localPreviewDir = path.join(process.cwd(), "uploads", "previews", id);
      try {
        await fs.rmdir(localPreviewDir, { recursive: true });
        console.log(`✅ Deleted local preview directory: ${localPreviewDir}`);
      } catch (localPreviewError) {
        console.warn(`⚠️ No local preview to clean up or error:`, localPreviewError);
      }
    } catch (cleanupError) {
      console.warn(`⚠️ Local file cleanup error:`, cleanupError);
    }

    // 4) Delete from database last (after storage cleanup)
    await deletePhotoFromDB(id);

    console.log(`✅ Successfully deleted photo ${id}`);
    res.json({ success: true, message: `Photo ${id} deleted successfully` });
  } catch (err) {
    console.error("Delete photo error:", err);
    res.status(500).json({ error: "Failed to delete photo" });
  }
});

// DELETE /api/stock/mockups/:id - Delete a mockup (Admin only)
router.delete("/mockups/:id", requireAdmin, async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`🗑️ Deleting mockup ${id}...`);

    const objectStorage = new ObjectStorageService();
    const searchPaths = objectStorage.getPublicObjectSearchPaths();

    // 1) Delete from object storage first - both original and preview
    for (const searchPath of searchPaths) {
      const parts = searchPath.split('/');
      const bucketName = parts[1];
      const basePrefix = parts.slice(2).join('/');
      const bucket = objectStorageClient.bucket(bucketName);

      // Delete original files
      try {
        const originalPrefix = `${basePrefix}/mockups/original/${id}/`;
        const [originalFiles] = await bucket.getFiles({ prefix: originalPrefix });
        for (const file of originalFiles) {
          await file.delete();
          console.log(`✅ Deleted original mockup from object storage: ${file.name}`);
        }
      } catch (originalError) {
        console.warn(`⚠️ Failed to delete original mockup from object storage:`, originalError);
      }

      // Delete preview files
      try {
        const previewPrefix = `${basePrefix}/mockups/preview/${id}/`;
        const [previewFiles] = await bucket.getFiles({ prefix: previewPrefix });
        for (const file of previewFiles) {
          await file.delete();
          console.log(`✅ Deleted mockup preview from object storage: ${file.name}`);
        }
      } catch (previewError) {
        console.warn(`⚠️ Failed to delete mockup preview from object storage:`, previewError);
      }
    }

    // 3) Clean up local files
    try {
      // Delete local original files in uploads/mockups/
      const localMockupsDir = path.join(process.cwd(), "uploads", "mockups");
      try {
        const localFiles = await fs.readdir(localMockupsDir);
        for (const filename of localFiles) {
          if (filename.includes(id)) {
            const localFilePath = path.join(localMockupsDir, filename);
            await fs.unlink(localFilePath);
            console.log(`✅ Deleted local mockup file: ${filename}`);
          }
        }
      } catch (localMockupsError) {
        console.warn(`⚠️ No local mockups to clean up or error:`, localMockupsError);
      }

      // Delete local preview files
      const localPreviewDir = path.join(process.cwd(), "uploads", "mockup-previews", id);
      try {
        await fs.rmdir(localPreviewDir, { recursive: true });
        console.log(`✅ Deleted local mockup preview directory: ${localPreviewDir}`);
      } catch (localPreviewError) {
        console.warn(`⚠️ No local mockup preview to clean up or error:`, localPreviewError);
      }
    } catch (cleanupError) {
      console.warn(`⚠️ Local file cleanup error:`, cleanupError);
    }

    // 4) Delete from database last (after storage cleanup)
    await deleteMockupFromDB(id);

    console.log(`✅ Successfully deleted mockup ${id}`);
    res.json({ success: true, message: `Mockup ${id} deleted successfully` });
  } catch (err) {
    console.error("Delete mockup error:", err);
    res.status(500).json({ error: "Failed to delete mockup" });
  }
});

// GET /api/stock/photos - List all uploaded photos
router.get("/photos", async (req, res) => {
  try {
    const photos = await getAllPhotos();
    res.json(photos);
  } catch (err) {
    console.error("Error listing photos:", err);
    res.status(500).json({ error: "Failed to list photos" });
  }
});

// GET /api/stock/mockups - List all uploaded mockups
router.get("/mockups", async (req, res) => {
  try {
    const mockups = await getAllMockups();
    // Format for frontend compatibility
    const formattedMockups = mockups.map(mockup => ({
      id: mockup.id,
      name: mockup.name,
      mimeType: mockup.mimeType,
      code: toTenDigitCode(mockup.id),
      previewUrl: `/api/stock/${mockup.id}/mockup-preview`,
      createdAt: new Date(mockup.uploadedAt).getTime(),
      tags: mockup.tags || [],
      category: mockup.category || "mockup",
      url: `/api/stock/${mockup.id}/mockup-preview` // For compatibility
    }));
    res.json({ mockups: formattedMockups });
  } catch (err) {
    console.error("Error listing mockups:", err);
    res.status(500).json({ error: "Failed to list mockups" });
  }
});

// GET /api/stock/:id/preview (always public, watermarked)
router.get("/:id/preview", async (req, res) => {
  try {
    const { id } = req.params;
    const objectStorage = new ObjectStorageService();
    const searchPaths = objectStorage.getPublicObjectSearchPaths();
    
    // Try cloud storage first
    let file = null;
    for (const searchPath of searchPaths) {
      const parts = searchPath.split('/');
      const bucketName = parts[1];
      const basePrefix = parts.slice(2).join('/');
      const prefix = `${basePrefix}/stock/preview/${id}/`;
      const bucket = objectStorageClient.bucket(bucketName);
      
      try {
        const [files] = await bucket.getFiles({ prefix });
        if (files.length > 0) {
          file = files[0];
          break;
        }
      } catch (searchError) {
        continue;
      }
    }
    
    if (file) {
      console.log(`✅ Found cloud preview for stock ID ${id}: ${file.name}`);
      return await objectStorage.downloadObject(file, res);
    }
    
    // Fallback to local storage
    try {
      const localPreviewDir = path.join(process.cwd(), "uploads", "previews", id);
      const files = await fs.readdir(localPreviewDir);
      
      if (files.length > 0) {
        const filename = files[0];
        const localFilePath = path.join(localPreviewDir, filename);
        const ext = path.extname(filename).toLowerCase();
        const mimeType = ext === '.png' ? 'image/png' : 'image/jpeg';
        
        res.setHeader('Content-Type', mimeType);
        res.setHeader('Cache-Control', 'public, max-age=31536000');
        
        const fileStream = fs.createReadStream(localFilePath);
        return fileStream.pipe(res);
      }
    } catch (localError) {
      console.warn(`Local fallback failed for ${id}:`, localError);
    }
    
    res.status(404).send("Preview not found");
  } catch (err) {
    console.error("Preview error:", err);
    res.status(500).send("Preview error");
  }
});

// GET /api/stock/:id/download
// If user has license -> serve original.
// Else if user has quota -> grant license, decrement quota, serve original.
// Else -> serve watermarked preview (blocks original).
router.get("/:id/download", async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id || null;

    // Helper: stream original by first key under /stock/original/{id}/
    async function streamOriginal() {
      const objectStorage = new ObjectStorageService();
      const searchPaths = objectStorage.getPublicObjectSearchPaths();
      
      // Try to find original in object storage
      let file = null;
      for (const searchPath of searchPaths) {
        const parts = searchPath.split('/');
        const bucketName = parts[1];
        const basePrefix = parts.slice(2).join('/');
        const prefix = `${basePrefix}/stock/original/${id}/`;
        const bucket = objectStorageClient.bucket(bucketName);
        
        try {
          const [files] = await bucket.getFiles({ prefix });
          if (files.length > 0) {
            file = files[0];
            break;
          }
        } catch (searchError) {
          continue;
        }
      }
      
      if (!file) {
        return res.status(404).send("Original not found");
      }
      
      res.setHeader("Content-Disposition", `attachment; filename="IBrandBiz_${id}.jpg"`);
      res.setHeader("Content-Type", file.metadata?.contentType || "image/jpeg");
      return await objectStorage.downloadObject(file, res);
    }

    // Helper: stream preview (watermarked)
    async function streamPreviewWithHeaders(extraHeaders = {}) {
      const objectStorage = new ObjectStorageService();
      const searchPaths = objectStorage.getPublicObjectSearchPaths();
      
      // Try cloud storage first
      let file = null;
      for (const searchPath of searchPaths) {
        const parts = searchPath.split('/');
        const bucketName = parts[1];
        const basePrefix = parts.slice(2).join('/');
        const prefix = `${basePrefix}/stock/preview/${id}/`;
        const bucket = objectStorageClient.bucket(bucketName);
        
        try {
          const [files] = await bucket.getFiles({ prefix });
          if (files.length > 0) {
            file = files[0];
            break;
          }
        } catch (searchError) {
          continue;
        }
      }
      
      if (file) {
        for (const [k, v] of Object.entries(extraHeaders)) res.setHeader(k, v);
        res.setHeader("Content-Type", file.metadata?.contentType || "image/jpeg");
        return await objectStorage.downloadObject(file, res);
      }
      
      // Fallback to local preview
      try {
        const localPreviewDir = path.join(process.cwd(), "uploads", "previews", id);
        const files = await fs.readdir(localPreviewDir);
        
        if (files.length > 0) {
          const filename = files[0];
          const localFilePath = path.join(localPreviewDir, filename);
          const ext = path.extname(filename).toLowerCase();
          const mimeType = ext === '.png' ? 'image/png' : 'image/jpeg';
          
          for (const [k, v] of Object.entries(extraHeaders)) res.setHeader(k, v);
          res.setHeader('Content-Type', mimeType);
          
          const fileStream = fs.createReadStream(localFilePath);
          return fileStream.pipe(res);
        }
      } catch (localError) {
        console.warn(`Local preview fallback failed for ${id}:`, localError);
      }
      
      return res.status(404).send("Preview not found");
    }

    // Anonymous users never get original
    if (!userId) {
      return streamPreviewWithHeaders({ "X-Quota-Remaining": "0", "X-License": "none" });
    }

    // 1) If already licensed → original
    if (hasLicense(userId, id)) {
      res.setHeader("X-License", "owned");
      return streamOriginal();
    }

    // 2) Not licensed → try to burn a quota
    const remaining = getTotalRemaining(userId);
    if (remaining > 0) {
      const ok = decrementOne(userId);
      if (ok) {
        // Grant a permanent license for this asset so repeat downloads don't burn quota again.
        grantLicenses(userId, [id]);
        res.setHeader("X-Quota-Remaining", String(remaining - 1));
        res.setHeader("X-License", "newly-granted");
        return streamOriginal();
      }
    }

    // 3) No license + no quota → serve preview (block original)
    return streamPreviewWithHeaders({ "X-Quota-Remaining": "0", "X-License": "none" });
  } catch (err) {
    console.error(err);
    res.status(500).send("Download error");
  }
});

// NEW: entitlement status for a single asset (needs auth to get user context)
router.get("/:id/entitlement", async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id || null;

    const licensed = userId ? hasLicense(userId, id) : false;
    const quotaRemaining = userId ? getTotalRemaining(userId) : 0;

    res.json({
      userId: userId || null,
      licensed,
      quotaRemaining,
      canDownloadOriginal: licensed || quotaRemaining > 0,
    });
  } catch (e) {
    console.error("entitlement error", e);
    res.status(500).json({ error: "entitlement_failed" });
  }
});

export default router;