import { Router } from "express";
import fetch from "node-fetch";

const router = Router();

// Type definitions for stock photo responses
interface StockPhoto {
  id: string;
  thumb: string;
  full: string;
  credit?: string;
}

interface UnsplashPhoto {
  id: string;
  urls: {
    thumb: string;
    regular: string;
    full: string;
  };
  user: {
    name: string;
    username: string;
  };
}

interface PexelsPhoto {
  id: number;
  src: {
    medium: string;
    large: string;
    original: string;
  };
  photographer: string;
}

/**
 * GET /api/stock/photos
 * Search stock photos from Unsplash and/or Pexels
 * Query params: query (search term), limit (max 30, default 12)
 */
router.get("/photos", async (req, res) => {
  try {
    const query = req.query.query as string;
    const limit = Math.min(parseInt(req.query.limit as string) || 12, 30);

    if (!query || query.trim().length === 0) {
      return res.status(400).json({ 
        error: "Query parameter is required",
        results: []
      });
    }

    const results: StockPhoto[] = [];

    // Try Unsplash first if API key is available
    const unsplashKey = process.env.UNSPLASH_ACCESS_KEY;
    if (unsplashKey) {
      try {
        const unsplashResults = await searchUnsplash(query, limit, unsplashKey);
        results.push(...unsplashResults);
      } catch (error) {
        console.warn('Unsplash API error:', error);
        // Continue to try Pexels if Unsplash fails
      }
    }

    // Try Pexels if we don't have enough results or Unsplash failed
    const pexelsKey = process.env.PEXELS_API_KEY;
    if (pexelsKey && results.length < limit) {
      try {
        const remaining = limit - results.length;
        const pexelsResults = await searchPexels(query, remaining, pexelsKey);
        results.push(...pexelsResults);
      } catch (error) {
        console.warn('Pexels API error:', error);
      }
    }

    // If no API keys available, return a helpful message
    if (!unsplashKey && !pexelsKey) {
      return res.status(503).json({
        error: "Stock photo search unavailable - no API keys configured",
        message: "Configure UNSPLASH_ACCESS_KEY or PEXELS_API_KEY environment variables to enable stock photo search",
        results: []
      });
    }

    // Return results (may be empty if all APIs failed)
    res.json({
      query,
      total: results.length,
      results: results.slice(0, limit)
    });

  } catch (error) {
    console.error('Stock photo search error:', error);
    res.status(500).json({ 
      error: "Failed to search stock photos",
      results: []
    });
  }
});

/**
 * Search Unsplash for photos
 */
async function searchUnsplash(query: string, limit: number, apiKey: string): Promise<StockPhoto[]> {
  const url = `https://api.unsplash.com/search/photos?query=${encodeURIComponent(query)}&per_page=${limit}&orientation=landscape`;
  
  const response = await fetch(url, {
    headers: {
      'Authorization': `Client-ID ${apiKey}`,
      'Accept-Version': 'v1'
    }
  });

  if (!response.ok) {
    throw new Error(`Unsplash API error: ${response.status} ${response.statusText}`);
  }

  const data = await response.json() as { results: UnsplashPhoto[] };
  
  return data.results.map((photo: UnsplashPhoto): StockPhoto => ({
    id: `unsplash-${photo.id}`,
    thumb: photo.urls.thumb,
    full: photo.urls.regular, // Use regular instead of full for better performance
    credit: `Photo by ${photo.user.name} on Unsplash`
  }));
}

/**
 * Search Pexels for photos
 */
async function searchPexels(query: string, limit: number, apiKey: string): Promise<StockPhoto[]> {
  const url = `https://api.pexels.com/v1/search?query=${encodeURIComponent(query)}&per_page=${limit}&orientation=landscape`;
  
  const response = await fetch(url, {
    headers: {
      'Authorization': apiKey
    }
  });

  if (!response.ok) {
    throw new Error(`Pexels API error: ${response.status} ${response.statusText}`);
  }

  const data = await response.json() as { photos: PexelsPhoto[] };
  
  return data.photos.map((photo: PexelsPhoto): StockPhoto => ({
    id: `pexels-${photo.id}`,
    thumb: photo.src.medium,
    full: photo.src.large,
    credit: `Photo by ${photo.photographer} on Pexels`
  }));
}

export default router;