// routes/stockLibraryRoutes.ts
// Upload + serve stock photos with server-side watermarking for previews/unlicensed

import path from "path";
import fs from "fs/promises";
import express, { type Request, type Response } from "express";
import multer from "multer";
import { applyWatermark, toTenDigitCode } from "../services/watermark";
import { ObjectStorageService, objectStorageClient } from "../objectStorage";
import { requireAdmin } from "../middleware/adminGuard";
import { uploadOriginalToFirebase, getOriginalFromFirebase } from "../services/firebaseStorage";
import { storage } from "../storage";

// Import authenticateToken middleware from routes.ts
// We'll define it locally since it's not exported from routes.ts
import { getFirebaseAdmin } from "../admin/firebaseAdmin";
import type { NextFunction } from "express";

// Define AuthenticatedRequest interface
interface AuthenticatedRequest extends Request {
  user?: {
    uid: string;
    email?: string;
    displayName?: string;
  };
}

// Authentication middleware - matches the one in routes.ts
async function authenticateToken(req: AuthenticatedRequest, res: Response, next: NextFunction) {
  try {
    // Explicit development bypass - ONLY if DEV_BYPASS=true AND NODE_ENV=development
    const isDevelopment = process.env.NODE_ENV === 'development';
    const devBypassEnabled = process.env.DEV_BYPASS === 'true';
    const devBypassHeader = req.headers['x-dev-bypass'];
    
    if (isDevelopment && devBypassEnabled && devBypassHeader === 'development') {
      console.warn('🚨 DEV AUTH BYPASS ACTIVE - Only use in development!');
      req.user = {
        uid: 'dev-user-123',
        email: 'dev@example.com',
        displayName: 'Development User',
      };
      return next();
    }

    const admin = await getFirebaseAdmin();
    
    if (!admin) {
      // In production, fail hard if Firebase is not available
      if (process.env.NODE_ENV === 'production') {
        console.error('🚨 CRITICAL: Firebase Admin not available in production');
        return res.status(500).json({ error: 'Authentication service unavailable' });
      }
      
      // In development, only bypass if explicitly enabled
      if (isDevelopment && devBypassEnabled) {
        console.warn('🚨 Firebase unavailable in development with DEV_BYPASS enabled, using mock user');
        req.user = {
          uid: 'dev-user-123',
          email: 'dev@example.com',
          displayName: 'Development User',
        };
        return next();
      }
      
      return res.status(500).json({ error: 'Authentication service unavailable' });
    }

    const authHeader = req.headers.authorization;
    const token = authHeader && authHeader.split(' ')[1]; // Bearer TOKEN

    if (!token) {
      return res.status(401).json({ error: 'Access token required' });
    }

    const decodedToken = await admin.auth().verifyIdToken(token);
    req.user = {
      uid: decodedToken.uid,
      email: decodedToken.email,
      displayName: decodedToken.name,
    };
    next();
  } catch (error) {
    console.error('Authentication error:', error);
    return res.status(403).json({ error: 'Invalid or expired token' });
  }
}

const router = express.Router();
const objectStorage = new ObjectStorageService();

// --- Multer storage (temp to local before pushing to object storage) ---
const uploadDir = path.join(process.cwd(), "uploads", "photos");
const storage = multer.diskStorage({
  destination: async (_req, _file, cb) => {
    await fs.mkdir(uploadDir, { recursive: true });
    cb(null, uploadDir);
  },
  filename: (_req, file, cb) => {
    const stamp = Date.now();
    const safe = file.originalname.replace(/\s+/g, "_");
    cb(null, `${stamp}_${safe}`);
  },
});
const upload = multer({
  storage,
  limits: { fileSize: 25 * 1024 * 1024 }, // 25MB
});

// Helper: construct object paths for stock images
function objPathOriginal(dbId: string, filename: string): string {
  // Store in private directory: .private/stock/original/{id}/{filename}
  const privateDir = process.env.PRIVATE_OBJECT_DIR || '';
  return `${privateDir}/stock/original/${dbId}/${filename}`;
}

function objPathPreview(dbId: string, filename: string): string {
  // Store in public directory with full bucket path
  const publicPath = objectStorage.getPublicObjectSearchPaths()[0]; // "/bucket/public"
  return `${publicPath}/stock/preview/${dbId}/${filename}`;
}

// Helper: get search path for previews (relative to search paths)
function previewSearchPath(dbId: string, filename: string): string {
  return `stock/preview/${dbId}/${filename}`;
}

// Helper: Save file locally as fallback
async function saveLocalFile(filePath: string, buffer: Buffer): Promise<void> {
  const fullPath = path.join(process.cwd(), "uploads", filePath);
  await fs.mkdir(path.dirname(fullPath), { recursive: true });
  await fs.writeFile(fullPath, buffer);
}

// Helper: Try to upload to cloud storage, fallback to local on failure
async function uploadWithFallback(cloudPath: string, localPath: string, buffer: Buffer, mimeType: string): Promise<void> {
  try {
    // First save locally (guaranteed to work)
    await saveLocalFile(localPath, buffer);
    console.log(`✅ Saved locally: ${localPath}`);
    
    // Then try cloud upload (best effort)
    if (process.env.FIREBASE_SERVICE_ACCOUNT_KEY || process.env.GOOGLE_APPLICATION_CREDENTIALS) {
      await uploadToObjectStorage(cloudPath, buffer, mimeType);
      console.log(`☁️ Cloud upload successful: ${cloudPath}`);
    } else {
      console.log(`⚠️ No cloud credentials configured, using local storage only`);
    }
  } catch (error) {
    console.log(`⚠️ Cloud upload failed, using local fallback: ${error}`);
    // Local save already succeeded, so continue
  }
}

// Helper: Upload buffer to object storage at specific path
async function uploadToObjectStorage(objectPath: string, buffer: Buffer, mimeType: string): Promise<void> {
  try {
    // Parse the object path to get bucket and object name
    const pathParts = objectPath.startsWith("/") ? objectPath.substring(1).split("/") : objectPath.split("/");
    if (pathParts.length < 2) {
      throw new Error(`Invalid object path: ${objectPath}`);
    }
    
    const bucketName = pathParts[0];
    const objectName = pathParts.slice(1).join("/");
    
    // Get the bucket and file reference
    const bucket = objectStorageClient.bucket(bucketName);
    const file = bucket.file(objectName);
    
    // Create a write stream and upload the buffer
    const stream = file.createWriteStream({
      metadata: {
        contentType: mimeType,
      },
      // Remove public flag to avoid bucket policy conflicts - we'll serve files through our API
    });
    
    return new Promise<void>((resolve, reject) => {
      stream.on('error', (error) => {
        console.error('Upload stream error:', error);
        reject(error);
      });
      
      stream.on('finish', () => {
        console.log(`✅ Uploaded to ${objectPath}`);
        resolve();
      });
      
      // Write the buffer and end the stream
      stream.end(buffer);
    });
  } catch (error) {
    console.error(`❌ Upload failed for ${objectPath}:`, error);
    throw error;
  }
}

// Helper: Get file from object storage by path
async function getFileFromObjectStorage(objectPath: string): Promise<import("@google-cloud/storage").File | null> {
  try {
    // Parse the object path to get bucket and object name
    const pathParts = objectPath.startsWith("/") ? objectPath.substring(1).split("/") : objectPath.split("/");
    if (pathParts.length < 2) return null;
    
    const bucketName = pathParts[0];
    const objectName = pathParts.slice(1).join("/");
    
    const bucket = objectStorageClient.bucket(bucketName);
    const file = bucket.file(objectName);
    
    const [exists] = await file.exists();
    return exists ? file : null;
  } catch (error) {
    console.error("Error getting file from object storage:", error);
    return null;
  }
}

// Simple JSON file storage for uploaded photos
const photosDbPath = path.join(process.cwd(), "uploads", "photos-db.json");

// Ensure photos database file exists
async function ensurePhotosDB() {
  try {
    await fs.access(photosDbPath);
  } catch {
    await fs.writeFile(photosDbPath, JSON.stringify([]));
  }
}

// Read all photos from database
async function getAllPhotos() {
  await ensurePhotosDB();
  const data = await fs.readFile(photosDbPath, 'utf-8');
  return JSON.parse(data);
}

// Add photo to database
async function addPhotoToDB(photo: any) {
  const photos = await getAllPhotos();
  photos.push(photo);
  await fs.writeFile(photosDbPath, JSON.stringify(photos, null, 2));
  return photo;
}

// NOTE: You likely already insert a DB row for each stock asset.
// Here we mock a DB insert and return a stable id.
async function insertStockDBRecord({ originalName, mimeType, uploaderId, tags = '', category = '' }: { originalName: string; mimeType: string; uploaderId: string; tags?: string; category?: string }) {
  // Generate stable id
  const id = Math.floor(Date.now() / 1000).toString(36) + Math.floor(Math.random() * 1e6).toString(36);
  
  // Create photo record
  const photo = {
    id,
    name: originalName,
    url: `/api/stock/${id}/preview`,
    tags: tags.split(',').map((t: string) => t.trim()).filter(Boolean),
    category: category.trim(),
    createdAt: Date.now(),
    originalName,
    mimeType,
    uploaderId
  };
  
  // Save to JSON database
  await addPhotoToDB(photo);
  
  return { id, originalName, mimeType, uploaderId };
}

// Helper: Handle stock upload for both photos and mockups
async function handleStockUpload(req: Request, res: Response, type: 'photos' | 'mockups') {
  try {
    const files = req.files as Express.Multer.File[];
    if (!files || files.length === 0) {
      return res.status(400).json({ error: "No files uploaded" });
    }

    const { tags = '', category = '' } = req.body;
    const results = [];

    for (const file of files) {
      // 1) Create DB record to get stable id
      const dbRow = await insertStockDBRecord({
        originalName: file.originalname,
        mimeType: file.mimetype,
        uploaderId: "admin", // Admin-only route
        tags,
        category,
      });

      const tenDigit = toTenDigitCode(dbRow.id);

      // 2) Read the uploaded file
      const originalBuf = await fs.readFile(file.path);

      // 3) Upload ORIGINAL to Firebase Storage (secure, non-watermarked)
      console.log(`🔥 Uploading original to Firebase Storage for photo ID: ${dbRow.id}`);
      let firebaseUrl = null;
      try {
        firebaseUrl = await uploadOriginalToFirebase(originalBuf, file.filename, dbRow.id, file.mimetype);
        console.log(`✅ Firebase Storage upload successful`);
      } catch (error) {
        console.log(`⚠️ Firebase Storage upload failed, skipping (bucket may not exist):`, (error as Error).message);
        // Continue with watermark generation even if Firebase fails
      }

      // 4) Generate WATERMARKED PREVIEW (Adobe-style) with local fallback
      const wmBuf = await applyWatermark(originalBuf, { stockSeed: dbRow.id, opacity: 0.15 });
      const previewPath = objPathPreview(dbRow.id, file.filename);
      const localPreviewPath = `previews/${dbRow.id}/${file.filename}`;
      await uploadWithFallback(previewPath, localPreviewPath, wmBuf, file.mimetype);
      
      // Also save original locally as fallback
      const localOriginalPath = `originals/${dbRow.id}/${file.filename}`;
      await saveLocalFile(localOriginalPath, originalBuf);

      // 5) Clean local temp
      await fs.unlink(file.path).catch(() => {});

      // 6) Build result
      results.push({
        id: dbRow.id,
        name: file.originalname,
        url: `/api/stock/${dbRow.id}/preview`,
        tags: tags.split(',').map((t: string) => t.trim()).filter(Boolean),
        category: category.trim()
      });
    }

    // 7) Return metadata that frontend expects
    res.json({
      success: true,
      message: `Successfully uploaded ${results.length} ${type}`,
      items: results
    });
  } catch (err) {
    console.error(`Upload error for ${type}:`, err);
    res.status(500).json({ error: "Upload failed" });
  }
}

// POST /api/stock/photos (admin only) - Expected by frontend StockUploader
router.post("/photos", requireAdmin, upload.array("files", 10), async (req: Request, res: Response) => {
  await handleStockUpload(req, res, 'photos');
});

// POST /api/stock/mockups (admin only) - Expected by frontend StockUploader  
router.post("/mockups", requireAdmin, upload.array("files", 10), async (req: Request, res: Response) => {
  await handleStockUpload(req, res, 'mockups');
});

// POST /api/stock/upload (admin only) - Keep for backward compatibility
router.post("/upload", requireAdmin, upload.single("file"), async (req: Request, res: Response) => {
  try {
    if (!req.file) return res.status(400).json({ error: "No file uploaded" });

    // 1) Create DB record to get stable id (or use your real DB).
    const dbRow = await insertStockDBRecord({
      originalName: req.file.originalname,
      mimeType: req.file.mimetype,
      uploaderId: "admin", // Admin-only route
      tags: req.body.tags || '',
      category: req.body.category || '',
    });

    const tenDigit = toTenDigitCode(dbRow.id);

    // 2) Read the uploaded file
    const originalBuf = await fs.readFile(req.file.path);

    // 3) Upload ORIGINAL to Firebase Storage (secure, non-watermarked)
    console.log(`🔥 Uploading original to Firebase Storage for photo ID: ${dbRow.id}`);
    let firebaseUrl = null;
    try {
      firebaseUrl = await uploadOriginalToFirebase(originalBuf, req.file.filename, dbRow.id, req.file.mimetype);
      console.log(`✅ Firebase Storage upload successful`);
    } catch (error) {
      console.log(`⚠️ Firebase Storage upload failed, skipping (bucket may not exist):`, (error as Error).message);
      // Continue with watermark generation even if Firebase fails
    }

    // 4) Generate WATERMARKED PREVIEW (Adobe-style) with local fallback
    const wmBuf = await applyWatermark(originalBuf, { stockSeed: dbRow.id, opacity: 0.15 });
    const previewPath = objPathPreview(dbRow.id, req.file.filename);
    const localPreviewPath = `previews/${dbRow.id}/${req.file.filename}`;
    await uploadWithFallback(previewPath, localPreviewPath, wmBuf, req.file.mimetype);
    
    // Also save original locally as fallback
    const localOriginalPath = `originals/${dbRow.id}/${req.file.filename}`;
    await saveLocalFile(localOriginalPath, originalBuf);

    // 5) Clean local temp
    await fs.unlink(req.file.path).catch(() => {});

    // 6) Get public URL for preview
    const previewUrl = `/api/stock/${dbRow.id}/preview`;

    // 7) Return metadata your frontend expects
    res.json({
      id: dbRow.id,
      code: tenDigit,
      name: req.file.originalname,
      mimeType: req.file.mimetype,
      previewUrl,
      // originalPath removed as it's not defined in this scope
    });
  } catch (err) {
    console.error("Upload error:", err);
    res.status(500).json({ error: "Upload failed" });
  }
});

// Helper: Migrate existing uploaded files to database
async function migrateExistingPhotos() {
  try {
    const photosDir = path.join(process.cwd(), "uploads", "photos");
    const files = await fs.readdir(photosDir);
    const existingPhotos = await getAllPhotos();
    
    // Get filenames already processed (to avoid re-processing)
    const processedFilenames = new Set(existingPhotos.map((p: any) => p.originalName));
    
    for (const filename of files) {
      if (!filename.endsWith('.png') && !filename.endsWith('.jpg') && !filename.endsWith('.jpeg')) continue;
      
      // Extract original name (remove timestamp prefix)
      const originalName = filename.substring(filename.indexOf('_') + 1);
      
      // Skip if already processed
      if (processedFilenames.has(originalName)) continue;
      
      console.log(`🔄 Processing existing photo: ${originalName}`);
      
      // Generate stable ID
      const timestamp = filename.split('_')[0];
      const id = Math.floor(parseInt(timestamp) / 1000).toString(36) + Math.floor(Math.random() * 1e6).toString(36);
      
      // Read the original file
      const filePath = path.join(photosDir, filename);
      const originalBuffer = await fs.readFile(filePath);
      
      // Upload original to Firebase Storage
      try {
        console.log(`🔥 Uploading original to Firebase Storage for migrated photo: ${id}`);
        await uploadOriginalToFirebase(originalBuffer, filename, id, 'image/png');
      } catch (fbError) {
        console.warn(`⚠️ Firebase upload failed for ${id}, continuing with migration:`, fbError);
      }
      
      // Generate watermarked preview with local fallback
      const watermarkedBuffer = await applyWatermark(originalBuffer, { stockSeed: id, opacity: 0.15 });
      const previewPath = objPathPreview(id, filename);
      const localPreviewPath = `previews/${id}/${filename}`;
      await uploadWithFallback(previewPath, localPreviewPath, watermarkedBuffer, 'image/png');
      
      // Also save original locally for migration
      const localOriginalPath = `originals/${id}/${filename}`;
      await saveLocalFile(localOriginalPath, originalBuffer);
      
      // Create photo record
      const photo = {
        id,
        name: originalName,
        url: `/api/stock/${id}/preview`,
        tags: [],
        category: '',
        createdAt: parseInt(timestamp),
        originalName,
        mimeType: 'image/png',
        uploaderId: 'migrated'
      };
      
      await addPhotoToDB(photo);
      console.log(`✅ Migrated and processed: ${originalName} -> ${id}`);
    }
  } catch (error) {
    console.warn("Migration failed:", error);
  }
}

// Helper: Regenerate watermarked previews for migrated photos
async function regenerateWatermarkedPreviews() {
  try {
    const photos = await getAllPhotos();
    const photosDir = path.join(process.cwd(), "uploads", "photos");
    
    for (const photo of photos) {
      if (photo.uploaderId !== 'migrated') continue;
      
      // Check if preview already exists
      const searchPaths = objectStorage.getPublicObjectSearchPaths();
      let previewExists = false;
      
      for (const searchPath of searchPaths) {
        const bucket = objectStorageClient.bucket(searchPath.split('/')[1]);
        const prefix = `stock/preview/${photo.id}/`;
        
        try {
          const [files] = await bucket.getFiles({ prefix });
          if (files.length > 0) {
            previewExists = true;
            break;
          }
        } catch (error) {
          continue;
        }
      }
      
      if (previewExists) {
        console.log(`✅ Preview already exists for ${photo.id}`);
        continue;
      }
      
      console.log(`🔄 Regenerating watermarked preview for ${photo.id}`);
      
      // Find the original file in uploads directory
      const files = await fs.readdir(photosDir);
      // File names are formatted as: {timestamp}_{originalname}
      const safeName = photo.originalName.replace(/\s+/g, "_");
      const originalFile = files.find(f => f.endsWith(safeName));
      
      if (!originalFile) {
        console.warn(`⚠️ Original file not found for ${photo.id}: ${photo.originalName}`);
        console.warn(`   Available files: ${files.join(', ')}`);
        continue;
      }
      
      console.log(`📁 Found file ${originalFile} for photo ${photo.id}`);
      
      try {
        // Read the original file
        const filePath = path.join(photosDir, originalFile);
        const originalBuffer = await fs.readFile(filePath);
        
        // Generate watermarked preview
        const watermarkedBuffer = await applyWatermark(originalBuffer, { stockSeed: photo.id, opacity: 0.15 });
        
        // Upload to object storage using proper search path structure
        const publicPaths = objectStorage.getPublicObjectSearchPaths();
        const publicPath = publicPaths[0]; // e.g. "/bucket-name/public"
        const fullPath = `${publicPath}/stock/preview/${photo.id}/${originalFile}`;
        
        // Use the existing object storage infrastructure
        const { bucketName, objectName } = parseObjectPath(fullPath);
        const bucket = objectStorageClient.bucket(bucketName);
        const file = bucket.file(objectName);
        
        const stream = file.createWriteStream({
          metadata: {
            contentType: photo.mimeType,
          },
        });
        
        await new Promise<void>((resolve, reject) => {
          stream.on('error', reject);
          stream.on('finish', resolve);
          stream.end(watermarkedBuffer);
        });
        
        console.log(`✅ Regenerated watermarked preview for ${photo.id}`);
      } catch (error) {
        console.error(`❌ Failed to regenerate preview for ${photo.id}:`, error);
      }
    }
  } catch (error) {
    console.error("❌ Preview regeneration failed:", error);
  }
}

// Import parseObjectPath function  
function parseObjectPath(path: string): { bucketName: string; objectName: string } {
  if (!path.startsWith("/")) {
    path = `/${path}`;
  }
  const pathParts = path.split("/");
  if (pathParts.length < 3) {
    throw new Error("Invalid path: must contain at least a bucket name");
  }
  
  const bucketName = pathParts[1];
  const objectName = pathParts.slice(2).join("/");
  
  return {
    bucketName,
    objectName,
  };
}

// POST /api/stock/regenerate-previews (admin only) - Manual trigger for preview regeneration
router.post("/regenerate-previews", requireAdmin, async (req: Request, res: Response) => {
  try {
    console.log("🔧 Manual preview regeneration triggered");
    await regenerateWatermarkedPreviews();
    res.json({ success: true, message: "Preview regeneration completed" });
  } catch (err) {
    console.error("Error regenerating previews:", err);
    res.status(500).json({ error: "Failed to regenerate previews" });
  }
});

// GET /api/stock/photos - List all uploaded photos
router.get("/photos", async (req: Request, res: Response) => {
  try {
    // Migrate existing photos on first access
    await migrateExistingPhotos();
    
    // Regenerate missing watermarked previews
    await regenerateWatermarkedPreviews();
    
    // Get all uploaded photos from JSON database
    const photos = await getAllPhotos();
    res.json(photos);
  } catch (err) {
    console.error("Error listing photos:", err);
    res.status(500).json({ error: "Failed to list photos" });
  }
});

// GET /api/stock/:id/preview
// Serves the already-watermarked preview (public). Frontend can use this in the modal zoom.
router.get("/:id/preview", async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    
    // Search for preview file using the search function
    // This will look in all public search paths for stock/preview/{id}/*
    const searchPattern = `stock/preview/${id}`;
    
    // Search in public paths
    let file = null;
    const searchPaths = objectStorage.getPublicObjectSearchPaths();
    
    // Try to find any file in the preview directory for this ID
    for (const searchPath of searchPaths) {
      const parts = searchPath.split('/');         // ["", "bucket-name", "public"]
      const bucketName = parts[1];
      const basePrefix = parts.slice(2).join('/'); // "public"
      const prefix = `${basePrefix}/stock/preview/${id}/`; // "public/stock/preview/{id}/"
      const bucket = objectStorageClient.bucket(bucketName);
      
      try {
        const [files] = await bucket.getFiles({ prefix });
        if (files.length > 0) {
          file = files[0]; // Use the first file found
          break;
        }
      } catch (searchError) {
        console.warn(`Search failed in ${searchPath}:`, searchError);
        continue;
      }
    }
    
    if (!file) {
      // Fallback to local file storage
      console.log(`☁️ Cloud preview not found for ${id}, trying local fallback...`);
      
      try {
        const localPreviewDir = path.join(process.cwd(), "uploads", "previews", id);
        const files = await fs.readdir(localPreviewDir);
        
        if (files.length > 0) {
          const filename = files[0]; // Use first file found
          const localFilePath = path.join(localPreviewDir, filename);
          
          console.log(`✅ Found local preview for stock ID ${id}: ${filename}`);
          
          // Set appropriate headers
          const ext = path.extname(filename).toLowerCase();
          const mimeType = ext === '.png' ? 'image/png' : ext === '.jpg' || ext === '.jpeg' ? 'image/jpeg' : 'image/png';
          res.setHeader('Content-Type', mimeType);
          res.setHeader('Cache-Control', 'public, max-age=31536000');
          
          // Stream the local file using Node.js built-in fs
          const fileStream = require('fs').createReadStream(localFilePath);
          fileStream.pipe(res);
          return;
        }
      } catch (localError) {
        console.warn(`Local fallback also failed for ${id}:`, localError);
      }
      
      console.log(`❌ Preview not found for stock ID: ${id} (tried cloud and local)`);
      return res.status(404).send("Preview not found");
    }
    
    console.log(`✅ Found cloud preview for stock ID ${id}: ${file.name}`);
    
    // Stream the file to response
    await objectStorage.downloadObject(file, res);
  } catch (err) {
    console.error("Preview error:", err);
    res.status(500).send("Preview error");
  }
});

// GET /api/stock/:id/download
// Check user quota and serve original file if available, otherwise serve watermarked preview
router.get("/:id/download", authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
  try {
    const { id } = req.params;
    // Get user database ID from Firebase UID
    let userId: string | null = null;
    let dbUser = null;
    
    if (req.user?.uid) {
      try {
        dbUser = await storage.getUserByFirebaseUid(req.user.uid);
        userId = dbUser?.id || null;
      } catch (error) {
        console.error(`❌ Error looking up user by Firebase UID ${req.user.uid}:`, error);
      }
    }

    console.log(`📥 Download request for stock photo ${id} by user ${userId || 'anonymous'} (Firebase UID: ${req.user?.uid || 'none'})`);

    // Check if user is authenticated and has quota available
    let canDownloadOriginal = false;
    let quotaMessage = '';

    if (userId && dbUser) {
      try {
        const hasQuota = await storage.checkQuotaAvailable(userId);
        if (hasQuota) {
          // User has quota available, increment their usage
          await storage.incrementUserDownloads(userId);
          canDownloadOriginal = true;
          console.log(`✅ User ${userId} has quota available, serving original file`);
        } else {
          // Check user's quota status for better error message
          const quotaStatus = await storage.getUserQuotaStatus(userId);
          if (!quotaStatus || quotaStatus.quota === 0) {
            quotaMessage = 'You need an active subscription to download original stock photos. Please upgrade your plan to access unlimited downloads.';
          } else {
            quotaMessage = `You've reached your monthly download limit of ${quotaStatus.quota} downloads. Your quota will reset on ${quotaStatus.resetDate?.toLocaleDateString() || 'your next billing cycle'}.`;
          }
          console.log(`⛔ User ${userId} quota exceeded: ${quotaMessage}`);
        }
      } catch (quotaError) {
        console.error(`❌ Error checking quota for user ${userId}:`, quotaError);
        quotaMessage = 'Unable to verify your download quota. Please try again later.';
      }
    } else {
      if (req.user?.uid && !dbUser) {
        quotaMessage = 'User account not found. Please contact support.';
        console.log(`⚠️ Firebase user ${req.user.uid} not found in database`);
      } else {
        quotaMessage = 'Please sign in to download original stock photos.';
        console.log(`ℹ️ Unauthenticated user requesting download, serving preview only`);
      }
    }

    if (!canDownloadOriginal) {
      // Serve watermarked preview - same logic as preview endpoint
      let file = null;
      const searchPaths = objectStorage.getPublicObjectSearchPaths();
      
      // Try to find any file in the preview directory for this ID
      for (const searchPath of searchPaths) {
        const parts = searchPath.split('/');         // ["", "bucket-name", "public"]
        const bucketName = parts[1];
        const basePrefix = parts.slice(2).join('/'); // "public"
        const prefix = `${basePrefix}/stock/preview/${id}/`; // "public/stock/preview/{id}/"
        const bucket = objectStorageClient.bucket(bucketName);
        
        try {
          const [files] = await bucket.getFiles({ prefix });
          if (files.length > 0) {
            file = files[0]; // Use the first file found
            break;
          }
        } catch (searchError) {
          console.warn(`Search failed in ${searchPath}:`, searchError);
          continue;
        }
      }
      
      if (!file) {
        return res.status(404).json({ 
          error: "Preview not found",
          message: "The requested stock photo preview is not available."
        });
      }
      
      // If no quota but preview available, add headers to inform about limitation
      if (quotaMessage) {
        res.setHeader("X-Quota-Message", quotaMessage);
        res.setHeader("X-Download-Type", "preview");
      }
      
      res.setHeader("Content-Disposition", `inline; filename="preview_${id}.jpg"`);
      return await objectStorage.downloadObject(file, res);
    }

    // User has quota: serve original from private storage
    console.log(`🎨 Serving original file for stock photo ${id} to user ${userId}`);
    
    // Search in private directory for the original file
    const privateDir = objectStorage.getPrivateObjectDir();
    const bucket = objectStorageClient.bucket(privateDir.split('/')[1]);
    const prefix = `stock/original/${id}/`;
    
    let originalFile = null;
    try {
      const [files] = await bucket.getFiles({ prefix });
      if (files.length > 0) {
        originalFile = files[0]; // Use the first file found
      }
    } catch (searchError) {
      console.error(`❌ Original file search failed for ${id}:`, searchError);
    }
    
    if (!originalFile) {
      console.error(`❌ Original file not found for stock photo ${id}`);
      return res.status(404).json({
        error: "Original file not found",
        message: "The requested stock photo original file is not available."
      });
    }
    
    // Get original filename from the file path
    const filename = path.basename(originalFile.name);
    const fileExt = path.extname(filename);
    
    // Set headers to indicate this is an original download
    res.setHeader("Content-Disposition", `attachment; filename="IBrandBiz_${id}${fileExt}"`);
    res.setHeader("X-Download-Type", "original");
    res.setHeader("X-Quota-Used", "true");
    
    console.log(`✅ Successfully serving original file: ${filename} for stock photo ${id}`);
    await objectStorage.downloadObject(originalFile, res);
  } catch (err) {
    console.error("Download error:", err);
    res.status(500).send("Download error");
  }
});

export default router;