import { Router } from "express";
import multer from "multer";
import bcrypt from "bcrypt";
import { storage } from "../storage.js";
import { withAuth } from "../auth/middleware.js";

const router = Router();

// Configure multer for file uploads (temporarily to local disk - should move to S3 later)
const upload = multer({ 
  dest: "uploads/",
  limits: {
    fileSize: 5 * 1024 * 1024, // 5MB limit
  },
  fileFilter: (req, file, cb) => {
    // Only allow image files
    if (file.mimetype.startsWith('image/')) {
      cb(null, true);
    } else {
      cb(new Error('Only image files are allowed'));
    }
  }
});

// Update profile info (name, company)
router.post("/api/profile/update", withAuth, async (req: any, res) => {
  try {
    const { name, company } = req.body || {};
    
    // Update user profile
    await storage.updateUserByFirebaseUid(req.user.uid, { 
      displayName: name, 
      company 
    });
    
    res.json({ name, company });
  } catch (error) {
    console.error('Profile update error:', error);
    res.status(500).json({ error: "Failed to update profile" });
  }
});

// Upload avatar
router.post("/api/profile/avatar", withAuth, upload.single("file"), async (req: any, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: "No file uploaded" });
    }

    // In a real app, you'd upload to S3/CloudStorage and get a permanent URL
    // For now, just store the local path
    const url = `/uploads/${req.file.filename}`;
    
    // Update user avatar URL
    await storage.updateUserByFirebaseUid(req.user.uid, { avatarUrl: url });
    
    res.json({ url });
  } catch (error) {
    console.error('Avatar upload error:', error);
    res.status(500).json({ error: "Upload failed" });
  }
});

// Get user preferences
router.get("/api/profile/preferences", withAuth, async (req: any, res) => {
  try {
    const preferences = await storage.getUserPreferences(req.user.uid);
    res.json(preferences);
  } catch (error) {
    console.error('Get preferences error:', error);
    res.json({ 
      emailNews: true, 
      marketingEmails: false, 
      productUpdates: true 
    }); // Default preferences
  }
});

// Update user preferences
router.post("/api/profile/preferences", withAuth, async (req: any, res) => {
  try {
    const { emailNews, marketingEmails, productUpdates } = req.body || {};
    
    await storage.setUserPreferences(req.user.uid, {
      emailNews: !!emailNews,
      marketingEmails: !!marketingEmails,
      productUpdates: !!productUpdates
    });
    
    res.json({ ok: true });
  } catch (error) {
    console.error('Save preferences error:', error);
    res.status(500).json({ error: "Failed to save preferences" });
  }
});

// Change password (Note: This is a placeholder - Firebase handles auth)
router.post("/api/profile/change-password", withAuth, async (req: any, res) => {
  try {
    const { current, next } = req.body || {};
    
    if (!current || !next) {
      return res.status(400).json({ error: "Current and new passwords are required" });
    }

    if (next.length < 6) {
      return res.status(400).json({ error: "New password must be at least 6 characters long" });
    }

    // Note: For Firebase Auth, password changes should be handled client-side
    // This is a placeholder that would typically integrate with Firebase Admin SDK
    // or your custom authentication system
    
    // For now, we'll simulate success since Firebase handles this client-side
    res.json({ ok: true });
  } catch (error) {
    console.error('Password change error:', error);
    res.status(500).json({ error: "Failed to update password" });
  }
});

export default router;