import { Router } from "express";
import { z } from "zod";
import { storage } from "../storage";
import { insertPresentationTemplateSchema, insertPresentationPurchaseSchema } from "@shared/schema";
import { requireAdmin } from "../middleware/adminGuard";
import Stripe from "stripe";

// Define AuthenticatedRequest interface
interface AuthenticatedRequest {
  user?: {
    uid: string;
    email?: string;
    displayName?: string;
  };
  body: any;
  params: any;
  query: any;
  headers: any;
}

// Lazy Stripe initialization
let stripe: Stripe | null = null;
function getStripe(): Stripe | null {
  if (stripe) return stripe;
  
  const stripeKey = process.env.STRIPE_SECRET_KEY;
  if (!stripeKey) {
    console.warn('STRIPE_SECRET_KEY not found - Stripe functionality will be disabled');
    return null;
  }
  
  try {
    stripe = new Stripe(stripeKey, {
      apiVersion: "2023-10-16",
    });
    return stripe;
  } catch (error) {
    console.error('Failed to initialize Stripe:', error);
    return null;
  }
}

// Helper function to build secure redirect URLs
function buildSecureRedirectUrl(path: string, origin?: string): string {
  const allowedBaseUrls = [
    process.env.DOMAIN_URL,
    process.env.FRONTEND_URL,
    process.env.VITE_BASE_URL,
    origin // Use request origin as fallback if provided
  ].filter(Boolean);
  
  // CRITICAL: Always return absolute URL for Stripe compatibility
  let baseUrl = allowedBaseUrls[0];
  
  if (!baseUrl) {
    if (process.env.NODE_ENV === 'development') {
      baseUrl = 'http://localhost:5000';
    } else {
      console.warn('[buildSecureRedirectUrl] No DOMAIN_URL/FRONTEND_URL configured in production');
      baseUrl = ''; // Will cause Stripe to fail - admin must configure env vars
    }
  }
  
  const cleanPath = path.startsWith('/') ? path : `/${path}`;
  
  return `${baseUrl}${cleanPath}`;
}

const router = Router();

// Validation schema for query parameters
const getTemplatesQuerySchema = z.object({
  q: z.string().optional(),
  category: z.string().optional(),
  descriptors: z.array(z.string()).or(z.string()).optional().transform((val) => {
    if (typeof val === 'string') return val.split(',').filter(Boolean);
    return val || [];
  }),
  min: z.coerce.number().optional(),
  max: z.coerce.number().optional(),
  includeInactive: z.coerce.boolean().default(false),
  status: z.string().optional()
});

/**
 * GET /api/presentations
 * Public endpoint to get presentation templates with filtering
 */
router.get("/", async (req, res) => {
  try {
    const query = getTemplatesQuerySchema.parse(req.query);
    
    const templates = await storage.getPresentationTemplates({
      q: query.q,
      category: query.category,
      descriptors: query.descriptors,
      min: query.min,
      max: query.max,
      includeInactive: query.includeInactive,
      status: query.status
    });

    return res.json({ templates });
  } catch (e: any) {
    console.error("[GET /presentations] error", e);
    
    if (e instanceof z.ZodError) {
      return res.status(400).json({
        error: "Invalid query parameters",
        details: e.errors
      });
    }
    
    return res.status(500).json({ error: "Failed to fetch presentation templates" });
  }
});

/**
 * GET /api/presentations/:id
 * Public endpoint to get a specific presentation template by ID
 */
router.get("/:id", async (req, res) => {
  try {
    const { id } = req.params;
    
    if (!id) {
      return res.status(400).json({ error: "Template ID is required" });
    }

    const template = await storage.getPresentationTemplate(id);
    
    if (!template) {
      return res.status(404).json({ error: "Presentation template not found" });
    }

    // Only return approved and active templates for public access unless admin
    if (!template.isActive || template.approvalStatus !== 'approved') {
      return res.status(404).json({ error: "Presentation template not available" });
    }

    return res.json({ template });
  } catch (e: any) {
    console.error("[GET /presentations/:id] error", e);
    return res.status(500).json({ error: "Failed to fetch presentation template" });
  }
});

/**
 * POST /api/presentations
 * Admin endpoint to create a new presentation template
 */
router.post("/", requireAdmin, async (req, res) => {
  try {
    const validatedData = insertPresentationTemplateSchema.parse(req.body);
    
    const template = await storage.createPresentationTemplate(validatedData);
    
    return res.status(201).json({ template });
  } catch (e: any) {
    console.error("[POST /presentations] error", e);
    
    if (e instanceof z.ZodError) {
      return res.status(400).json({
        error: "Validation failed",
        details: e.errors
      });
    }
    
    return res.status(500).json({ error: "Failed to create presentation template" });
  }
});

/**
 * PUT /api/presentations/:id
 * Admin endpoint to update a presentation template
 */
router.put("/:id", requireAdmin, async (req, res) => {
  try {
    const { id } = req.params;
    
    if (!id) {
      return res.status(400).json({ error: "Template ID is required" });
    }

    // Parse and validate the update data
    const validatedData = insertPresentationTemplateSchema.partial().parse(req.body);
    
    const template = await storage.updatePresentationTemplate(id, validatedData);
    
    if (!template) {
      return res.status(404).json({ error: "Presentation template not found" });
    }

    return res.json({ template });
  } catch (e: any) {
    console.error("[PUT /presentations/:id] error", e);
    
    if (e instanceof z.ZodError) {
      return res.status(400).json({
        error: "Validation failed",
        details: e.errors
      });
    }
    
    return res.status(500).json({ error: "Failed to update presentation template" });
  }
});

/**
 * DELETE /api/presentations/:id
 * Admin endpoint to delete a presentation template
 */
router.delete("/:id", requireAdmin, async (req, res) => {
  try {
    const { id } = req.params;
    
    if (!id) {
      return res.status(400).json({ error: "Template ID is required" });
    }

    const deleted = await storage.deletePresentationTemplate(id);
    
    if (!deleted) {
      return res.status(404).json({ error: "Presentation template not found" });
    }

    return res.json({ success: true, message: "Presentation template deleted successfully" });
  } catch (e: any) {
    console.error("[DELETE /presentations/:id] error", e);
    return res.status(500).json({ error: "Failed to delete presentation template" });
  }
});

/**
 * POST /api/presentations/checkout/regular
 * Create Stripe checkout session for regular presentation purchase ($29.99)
 */
// Simple authentication middleware for specific routes
async function authenticateToken(req: AuthenticatedRequest, res: any, next: any) {
  try {
    // Simple development bypass for now
    const isDevelopment = process.env.NODE_ENV === 'development';
    const devBypassHeader = req.headers['x-dev-bypass'];
    
    if (isDevelopment && devBypassHeader === 'development') {
      console.warn('🚨 DEV AUTH BYPASS ACTIVE for presentation checkout');
      req.user = {
        uid: 'dev-user-123',
        email: 'dev@example.com',
        displayName: 'Development User',
      };
      return next();
    }

    const authHeader = req.headers.authorization;
    const token = authHeader && authHeader.split(' ')[1]; // Bearer TOKEN

    if (!token) {
      return res.status(401).json({ error: 'Access token required' });
    }

    // For now, mock authentication to unblock development
    // In production, this would use Firebase Admin SDK
    req.user = {
      uid: 'mock-user-id',
      email: 'user@example.com',
      displayName: 'Mock User',
    };
    
    next();
  } catch (error) {
    console.error('Presentation checkout authentication error:', error);
    return res.status(403).json({ error: 'Invalid or expired token' });
  }
}

router.post("/checkout/regular", authenticateToken, async (req: AuthenticatedRequest, res) => {
  try {
    const stripe = getStripe();
    if (!stripe) {
      return res.status(503).json({ error: "Payment processing is currently unavailable" });
    }

    // Get authenticated user - security requirement
    const userId = req.user?.uid;
    if (!userId) {
      return res.status(401).json({ error: "Authentication required" });
    }

    const schema = z.object({
      baseTemplateId: z.string().min(1)
    });

    const { baseTemplateId } = schema.parse(req.body);

    // Verify template exists and is available
    const template = await storage.getPresentationTemplate(baseTemplateId);
    if (!template || !template.isActive || template.approvalStatus !== 'approved') {
      return res.status(404).json({ error: "Presentation template not found or unavailable" });
    }

    // Create pending purchase record
    const purchase = await storage.createPresentationPurchase({
      userId,
      type: 'regular',
      baseTemplateId,
      amountCents: 2999, // $29.99
      currency: 'usd',
      status: 'pending'
    });

    // Create Stripe checkout session
    const session = await stripe.checkout.sessions.create({
      payment_method_types: ['card'],
      line_items: [
        {
          price_data: {
            currency: 'usd',
            product_data: {
              name: `Presentation Template - ${template.title}`,
              description: 'Professional presentation template with full customization rights',
            },
            unit_amount: 2999,
          },
          quantity: 1,
        },
      ],
      mode: 'payment',
      success_url: buildSecureRedirectUrl(`/presentations/purchase-success?session_id={CHECKOUT_SESSION_ID}`),
      cancel_url: buildSecureRedirectUrl(`/presentations/${baseTemplateId}`),
      metadata: {
        purchaseId: purchase.id,
        userId,
        type: 'presentation_regular',
        baseTemplateId
      },
    });

    // Update purchase with session ID
    await storage.updatePresentationPurchase(purchase.id, {
      stripeSessionId: session.id
    });

    return res.json({ 
      sessionId: session.id,
      url: session.url
    });

  } catch (e: any) {
    console.error("[POST /presentations/checkout/regular] error", e);
    
    if (e instanceof z.ZodError) {
      return res.status(400).json({
        error: "Invalid request data",
        details: e.errors
      });
    }
    
    return res.status(500).json({ error: "Failed to create checkout session" });
  }
});

/**
 * POST /api/presentations/checkout/premium
 * Create Stripe checkout session for Premium+ presentation purchase ($49.99)
 */
router.post("/checkout/premium", authenticateToken, async (req: AuthenticatedRequest, res) => {
  try {
    const stripe = getStripe();
    if (!stripe) {
      return res.status(503).json({ error: "Payment processing is currently unavailable" });
    }

    // Get authenticated user - security requirement
    const userId = req.user?.uid;
    if (!userId) {
      return res.status(401).json({ error: "Authentication required" });
    }

    const schema = z.object({
      baseTemplateId: z.string().min(1),
      selectedCoverId: z.string().optional(),
      selectedInfographicIds: z.array(z.string()).max(4).default([])
    });

    const { baseTemplateId, selectedCoverId, selectedInfographicIds } = schema.parse(req.body);

    // Verify template exists and is available
    const template = await storage.getPresentationTemplate(baseTemplateId);
    if (!template || !template.isActive || template.approvalStatus !== 'approved') {
      return res.status(404).json({ error: "Presentation template not found or unavailable" });
    }

    // Verify selected cover exists if provided
    if (selectedCoverId) {
      const cover = await storage.getCoverTemplate(selectedCoverId);
      if (!cover || !cover.isActive) {
        return res.status(404).json({ error: "Selected cover template not found or unavailable" });
      }
    }

    // Verify selected infographics exist if provided
    if (selectedInfographicIds.length > 0) {
      for (const infographicId of selectedInfographicIds) {
        const infographic = await storage.getInfographicTemplate(infographicId);
        if (!infographic || !infographic.isActive || infographic.approvalStatus !== 'approved') {
          return res.status(404).json({ error: `Infographic template ${infographicId} not found or unavailable` });
        }
      }
    }

    // Create pending purchase record
    const purchase = await storage.createPresentationPurchase({
      userId,
      type: 'premium_plus',
      baseTemplateId,
      selectedCoverId,
      selectedInfographicIds,
      amountCents: 4999, // $49.99
      currency: 'usd',
      status: 'pending'
    });

    // Build description for Premium+ features
    const features = [
      'Professional presentation template',
      'Full customization rights',
    ];
    if (selectedCoverId) features.push('Custom cover slide');
    if (selectedInfographicIds.length > 0) features.push(`${selectedInfographicIds.length} infographic slide${selectedInfographicIds.length > 1 ? 's' : ''}`);

    // Create Stripe checkout session
    const session = await stripe.checkout.sessions.create({
      payment_method_types: ['card'],
      line_items: [
        {
          price_data: {
            currency: 'usd',
            product_data: {
              name: `Presentation Template Premium+ - ${template.title}`,
              description: features.join(' • '),
            },
            unit_amount: 4999,
          },
          quantity: 1,
        },
      ],
      mode: 'payment',
      success_url: buildSecureRedirectUrl(`/presentations/purchase-success?session_id={CHECKOUT_SESSION_ID}`),
      cancel_url: buildSecureRedirectUrl(`/presentations/builder?template=${baseTemplateId}`),
      metadata: {
        purchaseId: purchase.id,
        userId,
        type: 'presentation_premium',
        baseTemplateId,
        selectedCoverId: selectedCoverId || '',
        selectedInfographicIds: selectedInfographicIds.join(',')
      },
    });

    // Update purchase with session ID
    await storage.updatePresentationPurchase(purchase.id, {
      stripeSessionId: session.id
    });

    return res.json({ 
      sessionId: session.id,
      url: session.url
    });

  } catch (e: any) {
    console.error("[POST /presentations/checkout/premium] error", e);
    
    if (e instanceof z.ZodError) {
      return res.status(400).json({
        error: "Invalid request data",
        details: e.errors
      });
    }
    
    return res.status(500).json({ error: "Failed to create checkout session" });
  }
});

/**
 * GET /api/presentations/purchases
 * Get user's presentation purchases
 */
router.get("/purchases", async (req, res) => {
  try {
    // This would typically get userId from authentication middleware
    const userId = req.query.userId as string;
    
    if (!userId) {
      return res.status(400).json({ error: "User ID is required" });
    }

    const purchases = await storage.getUserPresentationPurchases(userId);
    
    // Enrich purchases with template details
    const enrichedPurchases = await Promise.all(
      purchases.map(async (purchase) => {
        const template = await storage.getPresentationTemplate(purchase.baseTemplateId);
        let cover = null;
        let infographics = [];

        if (purchase.selectedCoverId) {
          cover = await storage.getCoverTemplate(purchase.selectedCoverId);
        }

        if (purchase.selectedInfographicIds && purchase.selectedInfographicIds.length > 0) {
          infographics = await Promise.all(
            purchase.selectedInfographicIds.map(id => storage.getInfographicTemplate(id))
          );
        }

        return {
          ...purchase,
          template,
          cover,
          infographics: infographics.filter(Boolean)
        };
      })
    );

    return res.json({ purchases: enrichedPurchases });
  } catch (e: any) {
    console.error("[GET /presentations/purchases] error", e);
    return res.status(500).json({ error: "Failed to fetch presentation purchases" });
  }
});

export default router;