import { Router } from "express";
import PPTXGenJS from "pptxgenjs";

const router = Router();

type LogoPlacement = "none"|"top-left"|"top-right"|"bottom-left"|"bottom-right";

type CoverPayload = {
  size: "16x9"|"4x3";
  style: "centered"|"band"|"left"|"wash";
  brand: { name?: string; fonts: { heading: string; body: string }; accents: string[] };
  content: { title: string; subtitle?: string };
  assets: { logoDataUrl: string|null; logoPlacement: LogoPlacement; logoScale: number };
};

type BuildBody = {
  size: "16x9" | "4x3";
  brand: { name?: string; fonts: { heading: string; body: string }; accents: string[] };
  assets: { logoDataUrl: string|null; logoPlacement: LogoPlacement; logoScale: number };
  selections: { layouts: string[]; infographics: string[] };
  infographicsDetailed?: any[];
  coverDividers?: CoverPayload | null;
};

router.post("/build-template", async (req, res) => {
  try {
    const b = req.body as BuildBody;

    const pptx = new PPTXGenJS();
    pptx.layout = b.size === "4x3" ? "LAYOUT_4x3" : "LAYOUT_16x9";
    const slideW = pptx.width, slideH = pptx.height;

    // --- Masters: use handoff if present ---
    if (b.coverDividers) {
      const cd = normalizeCD(b.coverDividers, b.size);
      defineCoverMasterFromCD(pptx, cd, slideW, slideH);
      defineDividerMasterFromCD(pptx, cd, slideW, slideH);
    } else {
      // minimal defaults
      pptx.defineSlideMaster({
        title: "CoverMaster",
        background: { color: rgb(b.brand.accents[0], "0EA5E9") },
        objects: [
          { text: { text: b.brand.name || "Your Company", options: { x: 1, y: 2, fontSize: 36, color: "FFFFFF", fontFace: b.brand.fonts.heading, bold: true } } },
          ...(b.assets.logoDataUrl && b.assets.logoPlacement !== "none" ? [logoObject(b.assets, slideW, slideH)] : []),
        ],
      });
      pptx.defineSlideMaster({
        title: "DividerMaster",
        background: { color: rgb(b.brand.accents[1], "F99F1B") },
        objects: [
          { text: { text: "Section Title", options: { x: 1, y: 2, fontSize: 28, color: "FFFFFF", fontFace: b.brand.fonts.heading, bold: true } } },
          ...(b.assets.logoDataUrl && b.assets.logoPlacement !== "none" ? [logoObject(b.assets, slideW, slideH)] : []),
        ],
      });
    }

    // Title slide
    pptx.addSlide({ masterName: "CoverMaster" }).addText("", { x: 0, y: 0, w: 0, h: 0 }); // noop to ensure slide created
    // Section divider
    pptx.addSlide({ masterName: "DividerMaster" });

    // Body layouts
    b.selections.layouts.forEach((layoutId) => {
      const s = pptx.addSlide();
      s.addText(layoutId, { x: 1, y: 0.6, fontSize: 20, fontFace: b.brand.fonts.heading });
      s.addText("Content area", { x: 1, y: 1.4, fontSize: 14, fontFace: b.brand.fonts.body, color: rgb(b.brand.accents[2], "333333") });
      addLogoToSlide(s, b.assets, slideW, slideH);
    });

    // Infographics
    b.selections.infographics.forEach((inf) => {
      const s = pptx.addSlide();
      s.addText(`Infographic: ${inf}`, { x: 1, y: 0.6, fontSize: 20, fontFace: b.brand.fonts.heading });
      const sections = parseInt(inf.match(/\d+/)?.[0] || "2", 10);
      const totalW = slideW - 1.0;
      const boxW = (totalW - (sections - 1) * 0.1) / sections;
      for (let i = 0; i < sections; i++) {
        s.addShape(pptx.ShapeType.rect, {
          x: 0.5 + i * (boxW + 0.1),
          y: 1.6,
          w: boxW,
          h: 2.2,
          fill: { color: rgb(b.brand.accents[i % b.brand.accents.length], "6B7280") },
        });
      }
      addLogoToSlide(s, b.assets, slideW, slideH);
    });

    // Detailed Infographics from Infographics page
    if (b.infographicsDetailed && b.infographicsDetailed.length > 0) {
      b.infographicsDetailed.forEach((item) => {
        const s = pptx.addSlide();
        addInfographicDetailed(pptx, s, item, b.brand.fonts, b.brand.accents, slideW, slideH);
        addLogoToSlide(s, b.assets, slideW, slideH);
      });
    }

    const buf = await pptx.write("nodebuffer");
    res.setHeader("Content-Disposition", 'attachment; filename="ibrandbiz-template.pptx"');
    res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.presentationml.presentation");
    res.send(buf);
  } catch (e) {
    console.error("PPT build error", e);
    res.status(500).json({ error: "Failed to build PPTX" });
  }
});

export default router;

// ---------- helpers ----------
function addInfographicDetailed(
  pptx: PPTXGenJS,
  s: PPTXGenJS.Slide,
  item: any,
  fonts: { heading: string; body: string },
  colors: string[],
  slideW: number,
  slideH: number
) {
  const col = (i: number) => rgb(colors[i % colors.length], "0EA5E9");
  const center = { x: slideW/2, y: slideH/2 };
  const count = Math.max(2, Math.min(item.count || 6, 12));

  if (item.layout === "hubSpoke") {
    const R = Math.min(slideW, slideH)*0.22;
    const node = Math.min(slideW, slideH)*0.11;
    s.addShape(pptx.ShapeType.ellipse, { x: center.x - node*0.7/2, y: center.y - node*0.7/2, w: node*0.7, h: node*0.7, fill: { color: "0F172A" }, line: { color: "FFFFFF", width: 1 } });
    s.addText(item.centerTitle || "Topic", { x: center.x - 2.5, y: center.y - 0.4, w: 5, h: 0.8, align: "center", fontFace: fonts.heading, fontSize: 22, bold: true, color: "FFFFFF" });

    for (let i=0;i<count;i++) {
      const a = (Math.PI*2 * i)/count - Math.PI/2;
      const nx = center.x + R*Math.cos(a), ny = center.y + R*Math.sin(a);

      s.addShape(pptx.ShapeType.line, { x: center.x, y: center.y, w: nx-center.x, h: ny-center.y, line: { color: "CBD5E1", width: 1.5 } });
      s.addShape(pptx.ShapeType.ellipse, { x: nx - node/2, y: ny - node/2, w: node, h: node, fill: { color: col(i) }, line: { color: "FFFFFF", width: 1 } });

      const lbl = (item.labels?.[i] ?? `Point ${i+1}`);
      const note = (item.notes?.[i] ?? "");
      s.addText(
        [
          { text: `${lbl}\n`, options: { bold: true, fontFace: fonts.heading, fontSize: 14, color: "1F2937" } },
          ...(note ? [{ text: note, options: { fontFace: fonts.body, fontSize: 12, color: "475569" } }] : [])
        ],
        { x: nx - 1.6, y: ny + node/2 + 0.1, w: 3.2, h: 1.0, align: "center" }
      );
    }
  }

  if (item.layout === "stepsHorizontal") {
    const marginX = 0.7, gap = 0.25;
    const availableW = slideW - marginX*2 - gap*(count-1);
    const boxW = availableW / count;
    const boxH = 1.4;
    const y = slideH*0.42;

    for (let i=0;i<count;i++) {
      const x = marginX + i*(boxW + gap);
      s.addShape(pptx.ShapeType.roundRect, { x, y, w: boxW, h: boxH, fill: { color: col(i) }, line: { color: "FFFFFF", width: 1 }, rectRadius: 0.15 });
      s.addText(String(i+1).padStart(2,"0"), { x: x+0.15, y: y+0.18, w: 0.6, h: 0.4, fontFace: fonts.heading, fontSize: 14, bold: true, color: "FFFFFF" });
      
      const lbl = (item.labels?.[i] ?? `Step ${i+1}`);
      const note = (item.notes?.[i] ?? "");
      s.addText(
        [
          { text: `${lbl}\n`, options: { bold: true, fontFace: fonts.heading, fontSize: 14, color: "FFFFFF" } },
          ...(note ? [{ text: note, options: { fontFace: fonts.body, fontSize: 12, color: "F8FAFC" } }] : [])
        ],
        { x: x+0.85, y: y+0.2, w: boxW-1.05, h: boxH-0.4, align: "left" }
      );
    }
    s.addShape(pptx.ShapeType.line, { x: marginX, y: y+boxH+0.15, w: slideW - marginX*2, h: 0, line: { color: "CBD5E1", width: 1 } });
  }

  if (item.layout === "mindmap") {
    const node = 1.3;
    s.addShape(pptx.ShapeType.ellipse, { x: center.x - node, y: center.y - node*0.7, w: node*2, h: node*1.4, fill: { color: "0F172A" }, line: { color: "FFFFFF", width: 1 } });
    s.addText(item.centerTitle || "Topic", { x: center.x - 2.5, y: center.y - 0.2, w: 5, h: 0.8, align: "center", fontFace: fonts.heading, fontSize: 22, bold: true, color: "FFFFFF" });

    const perSide = Math.ceil(count/2);
    const leftX = 1.0, rightX = slideW - 4.0;
    const topY = slideH*0.20, stepY = (slideH*0.60)/(perSide-1 || 1);

    let idx = 0;
    for (let j=0;j<perSide && idx<count;j++, idx++) {
      const y = topY + j*stepY;
      s.addShape(pptx.ShapeType.line, { x: center.x - 0.2, y: center.y, w: (leftX+2.2)-(center.x - 0.2), h: y - center.y, line: { color: "94A3B8", width: 1.25 } });
      s.addShape(pptx.ShapeType.roundRect, { x: leftX, y, w: 3.2, h: 1.0, rectRadius: 0.12, fill: { color: col(idx) }, line: { color: "FFFFFF", width: 1 } });
      
      const lbl = item.labels?.[idx] ?? `Point ${idx+1}`;
      const note = item.notes?.[idx] ?? "";
      s.addText(
        [
          { text: lbl + "\n", options: { bold: true, fontFace: fonts.heading, fontSize: 14, color: "FFFFFF" } },
          ...(note ? [{ text: note, options: { fontFace: fonts.body, fontSize: 12, color: "F8FAFC" } }] : [])
        ],
        { x: leftX+0.25, y: y+0.12, w: 2.7, h: 0.8, align: "left" }
      );
    }
    for (let j=0;j<perSide && idx<count;j++, idx++) {
      const y = topY + j*stepY;
      s.addShape(pptx.ShapeType.line, { x: center.x + 0.2, y: center.y, w: (rightX)-(center.x + 0.2), h: y - center.y, line: { color: "94A3B8", width: 1.25 } });
      s.addShape(pptx.ShapeType.roundRect, { x: rightX, y, w: 3.2, h: 1.0, rectRadius: 0.12, fill: { color: col(idx) }, line: { color: "FFFFFF", width: 1 } });
      
      const lbl = item.labels?.[idx] ?? `Point ${idx+1}`;
      const note = item.notes?.[idx] ?? "";
      s.addText(
        [
          { text: lbl + "\n", options: { bold: true, fontFace: fonts.heading, fontSize: 14, color: "FFFFFF" } },
          ...(note ? [{ text: note, options: { fontFace: fonts.body, fontSize: 12, color: "F8FAFC" } }] : [])
        ],
        { x: rightX+0.25, y: y+0.12, w: 2.7, h: 0.8, align: "left" }
      );
    }
  }

  if (item.layout === "circularList") {
    const R = Math.min(slideW, slideH) * 0.25; // Circle radius
    const dotSize = Math.min(slideW, slideH) * 0.08; // Dot size

    for (let i = 0; i < count; i++) {
      const angle = (Math.PI * 2 * i) / count - Math.PI / 2; // Start from top (-90°)
      const dotX = center.x + R * Math.cos(angle);
      const dotY = center.y + R * Math.sin(angle);

      // Draw numbered dot
      s.addShape(pptx.ShapeType.ellipse, {
        x: dotX - dotSize / 2,
        y: dotY - dotSize / 2,
        w: dotSize,
        h: dotSize,
        fill: { color: col(i) },
        line: { color: "FFFFFF", width: 2 }
      });

      // Add number inside dot
      s.addText(String(i + 1), {
        x: dotX - dotSize / 2,
        y: dotY - dotSize / 2,
        w: dotSize,
        h: dotSize,
        align: "center",
        valign: "middle",
        fontFace: fonts.heading,
        fontSize: Math.max(10, Math.floor(dotSize * 18)),
        bold: true,
        color: "FFFFFF"
      });

      // Calculate label position (outside the circle)
      const labelRadius = R + dotSize / 2 + 0.5;
      const labelX = center.x + labelRadius * Math.cos(angle);
      const labelY = center.y + labelRadius * Math.sin(angle);

      // Draw connecting line from dot to label area
      s.addShape(pptx.ShapeType.line, {
        x: dotX,
        y: dotY,
        w: labelX - dotX,
        h: labelY - dotY,
        line: { color: "CBD5E1", width: 1.5 }
      });

      // Position text box around the label point
      const textBoxW = 2.5;
      const textBoxH = 1.0;
      let textX = labelX - textBoxW / 2;
      let textY = labelY - textBoxH / 2;
      let textAlign: "left" | "center" | "right" = "center";

      // Adjust text positioning based on angle for better readability
      if (Math.cos(angle) < -0.3) {
        // Left side
        textX = labelX - textBoxW;
        textAlign = "right";
      } else if (Math.cos(angle) > 0.3) {
        // Right side
        textX = labelX;
        textAlign = "left";
      }

      const lbl = item.labels?.[i] ?? `Point ${i + 1}`;
      const note = item.notes?.[i] ?? "";

      s.addText(
        [
          { text: `${lbl}\n`, options: { bold: true, fontFace: fonts.heading, fontSize: 12, color: "1F2937" } },
          ...(note ? [{ text: note, options: { fontFace: fonts.body, fontSize: 10, color: "475569" } }] : [])
        ],
        {
          x: textX,
          y: textY,
          w: textBoxW,
          h: textBoxH,
          align: textAlign
        }
      );
    }
  }
}

function rgb(maybeHex?: string, fallbackHex?: string) {
  const h = (maybeHex || fallbackHex || "000000").replace("#", "");
  return h.length === 6 ? h.toUpperCase() : "000000";
}

function addLogoToSlide(
  slide: PPTXGenJS.Slide,
  assets: { logoDataUrl: string|null; logoPlacement: LogoPlacement; logoScale: number },
  slideW: number,
  slideH: number
) {
  if (!assets.logoDataUrl || assets.logoPlacement === "none") return;
  const base = 1.2;
  const w = base * (assets.logoScale || 1);
  const h = w;
  const m = 0.25;
  let x = m, y = m;
  switch (assets.logoPlacement) {
    case "top-left": x = m; y = m; break;
    case "top-right": x = slideW - w - m; y = m; break;
    case "bottom-left": x = m; y = slideH - h - m; break;
    case "bottom-right": x = slideW - w - m; y = slideH - h - m; break;
  }
  slide.addImage({ data: assets.logoDataUrl, x, y, w, h });
}

function logoObject(
  assets: { logoDataUrl: string|null; logoPlacement: LogoPlacement; logoScale: number },
  slideW: number,
  slideH: number
): PPTXGenJS.SlideMasterObject {
  const base = 1.2;
  const w = base * (assets.logoScale || 1);
  const h = w;
  const m = 0.25;
  let x = -999, y = -999;
  switch (assets.logoPlacement) {
    case "top-left": x = m; y = m; break;
    case "top-right": x = slideW - w - m; y = m; break;
    case "bottom-left": x = m; y = slideH - h - m; break;
    case "bottom-right": x = slideW - w - m; y = slideH - h - m; break;
    case "none": default: break;
  }
  return { image: { data: assets.logoDataUrl!, x, y, w, h } };
}

// Normalize incoming cover/dividers to current size (in case user saved at a different size)
function normalizeCD(cd: CoverPayload, currentSize: "16x9"|"4x3"): CoverPayload {
  if (cd.size === currentSize) return cd;
  return { ...cd, size: currentSize }; // we only use relative placements
}

function defineCoverMasterFromCD(
  pptx: PPTXGenJS,
  cd: CoverPayload,
  slideW: number,
  slideH: number
) {
  const objects: PPTXGenJS.SlideMasterObject[] = [];

  if (cd.style === "band") {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[2], "111827") } } });
    objects.push({ rect: { x: 0, y: slideH*0.32, w: "100%", h: slideH*0.36, fill: { color: rgb(cd.brand.accents[0], "0EA5E9") } } });
  } else if (cd.style === "left") {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[2], "111827") } } });
    objects.push({ rect: { x: 0.6, y: slideH*0.26, w: 0.12, h: slideH*0.48, fill: { color: rgb(cd.brand.accents[1], "F99F1B") } } });
  } else if (cd.style === "wash") {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[2], "111827") } } });
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[0], "0EA5E9"), transparency: 60 } } });
  } else {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[0], "0EA5E9") } } });
  }

  objects.push({
    text: { text: cd.content.title || "Title", options: { x: 0.8, y: slideH*0.32, w: slideW-1.6, h: 1.6, align: "center", fontFace: cd.brand.fonts.heading, fontSize: 42, bold: true, color: "FFFFFF" } }
  });
  if (cd.content.subtitle?.trim()) {
    objects.push({
      text: { text: cd.content.subtitle!, options: { x: 1.2, y: slideH*0.48, w: slideW-2.4, h: 1.1, align: "center", fontFace: cd.brand.fonts.body, fontSize: 20, color: "FFFFFF" } }
    });
  }
  if (cd.assets.logoDataUrl && cd.assets.logoPlacement !== "none") {
    objects.push(logoObject(cd.assets, slideW, slideH));
  }

  pptx.defineSlideMaster({ title: "CoverMaster", objects });
}

function defineDividerMasterFromCD(
  pptx: PPTXGenJS,
  cd: CoverPayload,
  slideW: number,
  slideH: number
) {
  const objects: PPTXGenJS.SlideMasterObject[] = [];

  if (cd.style === "band") {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[2], "111827") } } });
    objects.push({ rect: { x: 0, y: slideH*0.35, w: "100%", h: slideH*0.30, fill: { color: rgb(cd.brand.accents[1], "F99F1B") } } });
  } else if (cd.style === "left") {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[0], "0EA5E9") } } });
    objects.push({ rect: { x: 0.6, y: slideH*0.3, w: 0.12, h: slideH*0.40, fill: { color: "FFFFFF" } } });
  } else if (cd.style === "wash") {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[2], "111827") } } });
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[1], "F99F1B"), transparency: 65 } } });
  } else {
    objects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: rgb(cd.brand.accents[1], "F99F1B") } } });
  }

  objects.push({
    text: { text: "Section Title", options: { x: 0.8, y: slideH*0.40, w: slideW-1.6, h: 1.2, align: "center", fontFace: cd.brand.fonts.heading, fontSize: 32, bold: true, color: "FFFFFF" } }
  });
  if (cd.assets.logoDataUrl && cd.assets.logoPlacement !== "none") {
    objects.push(logoObject(cd.assets, slideW, slideH));
  }

  pptx.defineSlideMaster({ title: "DividerMaster", objects });
}