// routes/mockupLibraryRoutes.js
// Mockups upload/serve pipeline (mirrors stock photos)

import path from "path";
import fs from "fs";
import fsPromises from "fs/promises";
import express from "express";
import multer from "multer";
import {
  applyWatermark,
  toTenDigitCode,
} from "../services/watermark.js";
import { ObjectStorageService, objectStorageClient } from "../objectStorage";
import {
  hasLicense,
  grantLicenses,
  getTotalRemaining,
  decrementOne,
} from "../services/entitlements.js";

const router = express.Router();

// Helper: load mockups database
async function getAllMockups() {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "mockups-db.json");
    const data = await fsPromises.readFile(dbPath, "utf8");
    return JSON.parse(data);
  } catch (error) {
    return [];
  }
}

// Helper: save mockup to database
async function saveMockupToDB(mockup) {
  try {
    const dbPath = path.join(process.cwd(), "uploads", "mockups-db.json");
    const mockups = await getAllMockups();
    mockups.push(mockup);
    await fsPromises.writeFile(dbPath, JSON.stringify(mockups, null, 2));
    return mockup;
  } catch (error) {
    console.error("Error saving mockup to DB:", error);
    throw error;
  }
}

// --- Multer storage (temp to local before pushing to object storage) ---
const uploadDir = path.join(process.cwd(), "uploads", "mockups");
const storage = multer.diskStorage({
  destination: async (_req, _file, cb) => {
    await fsPromises.mkdir(uploadDir, { recursive: true });
    cb(null, uploadDir);
  },
  filename: (_req, file, cb) => {
    const stamp = Date.now();
    const safe = file.originalname.replace(/\s+/g, "_");
    cb(null, `${stamp}_${safe}`);
  },
});
const upload = multer({
  storage,
  limits: { fileSize: 25 * 1024 * 1024 }, // 25MB
});

// Helper: object storage keys
function objKeyOriginal(dbId, filename) {
  return `mockups/original/${dbId}/${filename}`;
}
function objKeyPreview(dbId, filename) {
  return `mockups/preview/${dbId}/${filename}`;
}

// Mock DB insert
async function insertMockupDBRecord({ originalName, mimeType, uploaderId }) {
  const id =
    Math.floor(Date.now() / 1000).toString(36) +
    Math.floor(Math.random() * 1e6).toString(36);
  return { id, originalName, mimeType, uploaderId };
}

// POST /api/mockups/upload (admin)
router.post("/upload", upload.single("file"), async (req, res) => {
  try {
    if (!req.file) return res.status(400).json({ error: "No file uploaded" });

    // 1) Create DB record
    const dbRow = await insertMockupDBRecord({
      originalName: req.file.originalname,
      mimeType: req.file.mimetype,
      uploaderId: req.user?.id || "admin",
    });

    const tenDigit = toTenDigitCode(dbRow.id);

    // 2) Push ORIGINAL (private) to object storage
    const originalBuf = await fsPromises.readFile(req.file.path);
    const objectStorage = new ObjectStorageService();
    const publicPaths = objectStorage.getPublicObjectSearchPaths();
    const originalPath = `${publicPaths[0]}/mockups/original/${dbRow.id}/${req.file.filename}`;
    
    try {
      await objectStorage.uploadObject(originalPath, originalBuf);
      console.log(`✅ Uploaded original mockup to object storage: ${originalPath}`);
    } catch (uploadError) {
      console.warn(`⚠️ Object storage upload failed, saving locally:`, uploadError);
    }

    // 3) WATERMARKED PREVIEW (public)
    const wmBuf = await applyWatermark(originalBuf, { stockSeed: dbRow.id, opacity: 0.15 });
    const previewPath = `${publicPaths[0]}/mockups/preview/${dbRow.id}/${req.file.filename}`;
    
    try {
      await objectStorage.uploadObject(previewPath, wmBuf);
      console.log(`✅ Uploaded mockup preview to object storage: ${previewPath}`);
    } catch (uploadError) {
      console.warn(`⚠️ Mockup preview upload failed, saving locally:`, uploadError);
      // Save locally as fallback
      const localPreviewDir = path.join(process.cwd(), "uploads", "mockup-previews", dbRow.id);
      await fsPromises.mkdir(localPreviewDir, { recursive: true });
      const localPreviewPath = path.join(localPreviewDir, req.file.filename);
      await fsPromises.writeFile(localPreviewPath, wmBuf);
    }

    // 4) Clean local temp
    await fsPromises.unlink(req.file.path).catch(() => {});

    // 5) Save to JSON database
    const mockupRecord = {
      id: dbRow.id,
      name: req.file.originalname,
      filename: req.file.filename,
      mimeType: req.file.mimetype,
      uploadedAt: new Date().toISOString(),
      uploaderId: req.user?.id || "admin"
    };
    await saveMockupToDB(mockupRecord);

    // 6) Response metadata
    res.json({
      id: dbRow.id,
      code: tenDigit,
      name: req.file.originalname,
      mimeType: req.file.mimetype,
      previewUrl: `/api/mockups/${dbRow.id}/preview`,
      originalKey: originalPath, // private
    });
  } catch (err) {
    console.error("Mockup upload error:", err);
    res.status(500).json({ error: "Upload failed" });
  }
});

// GET /api/mockups/list - List all uploaded mockups
router.get("/list", async (req, res) => {
  try {
    const mockups = await getAllMockups();
    // Format for frontend compatibility
    const formattedMockups = mockups.map(mockup => ({
      id: mockup.id,
      name: mockup.name,
      mimeType: mockup.mimeType,
      code: toTenDigitCode(mockup.id),
      previewUrl: `/api/mockups/${mockup.id}/preview`,
      createdAt: new Date(mockup.uploadedAt).getTime(),
      tags: [],
      category: "mockup",
      url: `/api/mockups/${mockup.id}/preview` // For compatibility
    }));
    res.json({ mockups: formattedMockups });
  } catch (err) {
    console.error("Error listing mockups:", err);
    res.status(500).json({ error: "Failed to list mockups" });
  }
});

// GET /api/mockups/:id/preview (always public, watermarked)
router.get("/:id/preview", async (req, res) => {
  try {
    const { id } = req.params;
    const objectStorage = new ObjectStorageService();
    const searchPaths = objectStorage.getPublicObjectSearchPaths();
    
    // Try cloud storage first
    let file = null;
    for (const searchPath of searchPaths) {
      const parts = searchPath.split('/');
      const bucketName = parts[1];
      const basePrefix = parts.slice(2).join('/');
      const prefix = `${basePrefix}/mockups/preview/${id}/`;
      const bucket = objectStorageClient.bucket(bucketName);
      
      try {
        const [files] = await bucket.getFiles({ prefix });
        if (files.length > 0) {
          file = files[0];
          break;
        }
      } catch (searchError) {
        continue;
      }
    }
    
    if (file) {
      console.log(`✅ Found cloud preview for mockup ID ${id}: ${file.name}`);
      return await objectStorage.downloadObject(file, res);
    }
    
    // Fallback to local storage
    try {
      const localPreviewDir = path.join(process.cwd(), "uploads", "mockup-previews", id);
      const files = await fsPromises.readdir(localPreviewDir);
      
      if (files.length > 0) {
        const filename = files[0];
        const localFilePath = path.join(localPreviewDir, filename);
        const ext = path.extname(filename).toLowerCase();
        const mimeType = ext === '.png' ? 'image/png' : 'image/jpeg';
        
        res.setHeader('Content-Type', mimeType);
        res.setHeader('Cache-Control', 'public, max-age=31536000');
        
        const fileStream = fs.createReadStream(localFilePath);
        return fileStream.pipe(res);
      }
    } catch (localError) {
      console.warn(`Local fallback failed for mockup ${id}:`, localError);
    }
    
    res.status(404).send("Preview not found");
  } catch (err) {
    console.error("Mockup preview error:", err);
    res.status(500).send("Preview error");
  }
});

// GET /api/mockups/:id/download
// If user has license → serve original.
// Else if user has quota → grant license, decrement quota, serve original.
// Else → serve watermarked preview (blocks original).
router.get("/:id/download", async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id || null;

    // Helper: stream original by first key under /mockups/original/{id}/
    async function streamOriginal() {
      const objectStorage = new ObjectStorageService();
      const searchPaths = objectStorage.getPublicObjectSearchPaths();
      
      // Try to find original in object storage
      let file = null;
      for (const searchPath of searchPaths) {
        const parts = searchPath.split('/');
        const bucketName = parts[1];
        const basePrefix = parts.slice(2).join('/');
        const prefix = `${basePrefix}/mockups/original/${id}/`;
        const bucket = objectStorageClient.bucket(bucketName);
        
        try {
          const [files] = await bucket.getFiles({ prefix });
          if (files.length > 0) {
            file = files[0];
            break;
          }
        } catch (searchError) {
          continue;
        }
      }
      
      if (!file) {
        return res.status(404).send("Original not found");
      }
      
      res.setHeader("Content-Disposition", `attachment; filename="IBrandBiz_Mockup_${id}.png"`);
      res.setHeader("Content-Type", file.metadata?.contentType || "image/png");
      return await objectStorage.downloadObject(file, res);
    }

    // Helper: stream preview (watermarked)
    async function streamPreviewWithHeaders(extraHeaders = {}) {
      const objectStorage = new ObjectStorageService();
      const searchPaths = objectStorage.getPublicObjectSearchPaths();
      
      // Try cloud storage first
      let file = null;
      for (const searchPath of searchPaths) {
        const parts = searchPath.split('/');
        const bucketName = parts[1];
        const basePrefix = parts.slice(2).join('/');
        const prefix = `${basePrefix}/mockups/preview/${id}/`;
        const bucket = objectStorageClient.bucket(bucketName);
        
        try {
          const [files] = await bucket.getFiles({ prefix });
          if (files.length > 0) {
            file = files[0];
            break;
          }
        } catch (searchError) {
          continue;
        }
      }
      
      if (file) {
        for (const [k, v] of Object.entries(extraHeaders)) res.setHeader(k, v);
        res.setHeader("Content-Type", file.metadata?.contentType || "image/png");
        return await objectStorage.downloadObject(file, res);
      }
      
      // Fallback to local preview
      try {
        const localPreviewDir = path.join(process.cwd(), "uploads", "mockup-previews", id);
        const files = await fsPromises.readdir(localPreviewDir);
        
        if (files.length > 0) {
          const filename = files[0];
          const localFilePath = path.join(localPreviewDir, filename);
          const ext = path.extname(filename).toLowerCase();
          const mimeType = ext === '.png' ? 'image/png' : 'image/jpeg';
          
          for (const [k, v] of Object.entries(extraHeaders)) res.setHeader(k, v);
          res.setHeader('Content-Type', mimeType);
          
          const fileStream = fs.createReadStream(localFilePath);
          return fileStream.pipe(res);
        }
      } catch (localError) {
        console.warn(`Local mockup preview fallback failed for ${id}:`, localError);
      }
      
      return res.status(404).send("Preview not found");
    }

    // Anonymous users never get original
    if (!userId) {
      return streamPreviewWithHeaders({ "X-Quota-Remaining": "0", "X-License": "none" });
    }

    // 1) If already licensed → original
    if (hasLicense(userId, id)) {
      res.setHeader("X-License", "owned");
      return streamOriginal();
    }

    // 2) Not licensed → try to burn a quota
    const remaining = getTotalRemaining(userId);
    if (remaining > 0) {
      const ok = decrementOne(userId);
      if (ok) {
        // Grant a permanent license for this asset so repeat downloads don't burn quota again.
        grantLicenses(userId, [id]);
        res.setHeader("X-Quota-Remaining", String(remaining - 1));
        res.setHeader("X-License", "newly-granted");
        return streamOriginal();
      }
    }

    // 3) No license + no quota → serve preview (block original)
    return streamPreviewWithHeaders({ "X-Quota-Remaining": "0", "X-License": "none" });
  } catch (err) {
    console.error(err);
    res.status(500).send("Download error");
  }
});

// NEW: entitlement status for a single asset (needs auth to get user context)
router.get("/:id/entitlement", async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.user?.id || null;

    const licensed = userId ? hasLicense(userId, id) : false;
    const quotaRemaining = userId ? getTotalRemaining(userId) : 0;

    res.json({
      userId: userId || null,
      licensed,
      quotaRemaining,
      canDownloadOriginal: licensed || quotaRemaining > 0,
    });
  } catch (e) {
    console.error("mockup entitlement error", e);
    res.status(500).json({ error: "entitlement_failed" });
  }
});

export default router;