import express, { Request, Response } from "express";
import { listLogoTemplates, getLogoTemplateById, getSignedUrl } from "../services/logoTemplates";

const router = express.Router();

// GET /api/logo-templates-firebase/list
router.get("/list", async (_req: Request, res: Response) => {
  try {
    const items = await listLogoTemplates();
    
    // Convert storage paths to signed URLs
    const dto = await Promise.all(items.map(async (t) => {
      let svgDownloadUrl = t.downloadURLs?.svg || "";
      let previewDownloadUrl = t.downloadURLs?.preview || t.downloadURLs?.raster || "";
      
      // If no download URLs, generate from storage paths
      if (!svgDownloadUrl && t.storagePaths?.svg) {
        svgDownloadUrl = await getSignedUrl(t.storagePaths.svg);
      }
      if (!previewDownloadUrl && (t.storagePaths?.preview || t.storagePaths?.raster)) {
        previewDownloadUrl = await getSignedUrl(t.storagePaths?.preview || t.storagePaths?.raster || "");
      }
      
      return {
        id: t.id,
        name: t.name,
        description: t.description,
        tags: t.tags || [],
        version: t.version,
        status: t.status,
        assets: {
          svgPath: t.storagePaths?.svg || "",
          previewPngPath: t.storagePaths?.preview || t.storagePaths?.raster || "",
          svgDownloadUrl,
          previewDownloadUrl
        },
        defaults: {
          brandName: t.defaults?.brandName || t.defaults?.Brand_Name || "Your Brand",
          tagline: t.defaults?.tagline || t.defaults?.Tagline || "Your Tagline",
          estYear: t.defaults?.estYear || t.defaults?.Est_Year || "2025"
        },
        createdBy: t.createdBy,
        createdAt: t.createdAt,
        updatedAt: t.updatedAt
      };
    }));
    
    // Cache list briefly
    res.set("Cache-Control", "public, max-age=60");
    res.json({ items: dto });
  } catch (err: any) {
    console.error("[logo-templates:list] error", err);
    res.status(500).json({ error: "Failed to list logo templates" });
  }
});

// GET /api/logo-templates-firebase/:id
router.get("/:id", async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const template = await getLogoTemplateById(id);
    
    if (!template) {
      return res.status(404).json({ error: "Template not found" });
    }
    
    // Convert storage paths to signed URLs
    let svgDownloadUrl = template.downloadURLs?.svg || "";
    let previewDownloadUrl = template.downloadURLs?.preview || template.downloadURLs?.raster || "";
    
    if (!svgDownloadUrl && template.storagePaths?.svg) {
      svgDownloadUrl = await getSignedUrl(template.storagePaths.svg);
    }
    if (!previewDownloadUrl && (template.storagePaths?.preview || template.storagePaths?.raster)) {
      previewDownloadUrl = await getSignedUrl(template.storagePaths?.preview || template.storagePaths?.raster || "");
    }
    
    const dto = {
      id: template.id,
      name: template.name,
      description: template.description,
      tags: template.tags || [],
      version: template.version,
      status: template.status,
      assets: {
        svgPath: template.storagePaths?.svg || "",
        previewPngPath: template.storagePaths?.preview || template.storagePaths?.raster || "",
        svgDownloadUrl,
        previewDownloadUrl
      },
      defaults: {
        brandName: template.defaults?.brandName || template.defaults?.Brand_Name || "Your Brand",
        tagline: template.defaults?.tagline || template.defaults?.Tagline || "Your Tagline",
        estYear: template.defaults?.estYear || template.defaults?.Est_Year || "2025"
      },
      createdBy: template.createdBy,
      createdAt: template.createdAt,
      updatedAt: template.updatedAt
    };
    
    res.set("Cache-Control", "public, max-age=300");
    res.json(dto);
  } catch (err: any) {
    console.error("[logo-templates:get] error", err);
    res.status(500).json({ error: "Failed to get logo template" });
  }
});

export default router;
