import { Router } from "express";
import formidable, { File } from "formidable";
import fs from "fs";
import path from "path";

const router = Router();

type ManifestItem = {
  id: string;
  name: string;
  tags: string[];
  svgPath: string;
  previewPath: string;
  createdAt: string;
};

const ensureDir = (p: string) => {
  if (!fs.existsSync(p)) fs.mkdirSync(p, { recursive: true });
};

const sanitizeId = (id: string) =>
  (id || "")
    .toLowerCase()
    .trim()
    .replace(/[^a-z0-9\-]/g, "-")
    .replace(/\-+/g, "-");

router.post("/upload", (req, res) => {
  const form = formidable({ keepExtensions: true, maxFileSize: 20 * 1024 * 1024 });
  form.parse(req, (err, fields, files) => {
    if (err) return res.status(400).json({ error: err.message });

    const id = sanitizeId(String(fields.id || ""));
    const name = String(fields.name || "");
    const tags = String(fields.tags || "")
      .split(",")
      .map((t) => t.trim())
      .filter(Boolean);

    if (!id || !name) return res.status(400).json({ error: "id and name are required" });

    const svg = files.svg as File | undefined;
    const preview = files.preview as File | undefined;
    if (!svg || !preview) return res.status(400).json({ error: "svg and preview are required" });

    const svgOk =
      (svg.mimetype && svg.mimetype === "image/svg+xml") ||
      (svg.originalFilename && svg.originalFilename.toLowerCase().endsWith(".svg"));
    if (!svgOk) return res.status(400).json({ error: "SVG must be image/svg+xml" });

    const previewOk =
      (preview.mimetype && /image\/(png|jpeg)/i.test(preview.mimetype)) ||
      (preview.originalFilename && /\.(png|jpe?g)$/i.test(preview.originalFilename || ""));
    if (!previewOk) return res.status(400).json({ error: "Preview must be PNG or JPG" });

    try {
      const baseDir = path.join(process.cwd(), "public", "logo-templates", id);
      ensureDir(baseDir);

      const svgPathAbs = path.join(baseDir, "template.svg");
      const previewExt = (preview.originalFilename || "").split(".").pop() || "png";
      const previewPathAbs = path.join(baseDir, `preview.${previewExt}`);

      fs.copyFileSync(svg.filepath, svgPathAbs);
      fs.copyFileSync(preview.filepath, previewPathAbs);

      const manifestDir = path.join(process.cwd(), "public", "logo-templates");
      ensureDir(manifestDir);
      const manifestPath = path.join(manifestDir, "manifest.json");

      let manifest: ManifestItem[] = [];
      if (fs.existsSync(manifestPath)) {
        try {
          manifest = JSON.parse(fs.readFileSync(manifestPath, "utf8"));
          if (!Array.isArray(manifest)) manifest = [];
        } catch {
          manifest = [];
        }
      }

      const svgPath = `/logo-templates/${id}/template.svg`;
      const previewPath = `/logo-templates/${id}/` + path.basename(previewPathAbs);

      const item: ManifestItem = {
        id,
        name,
        tags,
        svgPath,
        previewPath,
        createdAt: new Date().toISOString(),
      };

      const idx = manifest.findIndex((m) => m.id === id);
      if (idx >= 0) manifest[idx] = item;
      else manifest.push(item);

      fs.writeFileSync(manifestPath, JSON.stringify(manifest, null, 2), "utf8");
      res.json({ ok: true, item });
    } catch (error: any) {
      console.error("Logo template upload error:", error);
      res.status(500).json({ error: error?.message || "Upload failed" });
    }
  });
});

// Get all logo templates from manifest
router.get("/", (req, res) => {
  try {
    const manifestPath = path.join(process.cwd(), "public", "logo-templates", "manifest.json");
    
    if (!fs.existsSync(manifestPath)) {
      return res.json([]);
    }

    const manifest = JSON.parse(fs.readFileSync(manifestPath, "utf8"));
    res.json(Array.isArray(manifest) ? manifest : []);
  } catch (error) {
    console.error("Error reading logo templates manifest:", error);
    res.status(500).json({ error: "Failed to load logo templates" });
  }
});

export default router;