import express from "express";
import { storage as db } from "../storage";
import { insertInfographicPurchaseSchema } from "@shared/schema";
import { createInfographicBundleCheckoutSession } from "../stripe/infographicCheckout";
import { eq, desc } from "drizzle-orm";
import { z } from "zod";

const router = express.Router();

// FIXED: Dedicated validation schema for checkout requests (decoupled from insert schema)
const checkoutRequestSchema = z.object({
  selectedIds: z.array(z.string().min(1, "Template ID cannot be empty"))
    .min(1, "Please select at least 1 infographic")
    .max(4, "Maximum 4 infographics allowed per bundle")
    .refine(arr => new Set(arr).size === arr.length, "Duplicate selections not allowed")
});

// Authentication middleware - using existing auth system
const authenticate = (req: any, res: any, next: any) => {
  // For development, mock authentication
  if (process.env.NODE_ENV === 'development') {
    req.user = {
      id: 'dev-user-123',
      uid: 'dev-user-123',
      email: 'dev@example.com'
    };
    return next();
  }
  
  // In production, add proper Firebase token verification here
  if (!req.user) {
    return res.status(401).json({ error: "Authentication required" });
  }
  next();
};

const authenticateAdmin = (req: any, res: any, next: any) => {
  // Add admin check logic here - for now allow in development
  if (process.env.NODE_ENV === 'development') {
    return next();
  }
  // In production, add proper admin role verification
  if (!req.user?.isAdmin) {
    return res.status(403).json({ error: "Admin access required" });
  }
  next();
};

// Gallery with filters using descriptors
router.get("/api/infographics", async (req, res) => {
  try {
    // Parse descriptors from comma-separated string
    const descriptorsParam = req.query.descriptors as string | undefined;
    const descriptors = descriptorsParam ? descriptorsParam.split(',').map(d => d.trim()) : undefined;
    
    const rows = await db.getInfographicTemplates({
      q: req.query.q as string | undefined,
      category: req.query.category as string | undefined,
      descriptors: descriptors,
      min: req.query.min ? parseInt(req.query.min as string, 10) : undefined,
      max: req.query.max ? parseInt(req.query.max as string, 10) : undefined,
    });
    res.json(rows);
  } catch (error) {
    console.error("[infographics] Gallery error:", error);
    res.status(500).json({ error: "Failed to fetch templates" });
  }
});

// Single template detail for lightbox preview
router.get("/api/infographics/:id", async (req, res) => {
  try {
    const tpl = await db.getInfographicTemplate(req.params.id);
    if (!tpl || !tpl.isActive || tpl.approvalStatus !== 'approved') {
      return res.status(404).json({ error: "Not found" });
    }
    res.json(tpl);
  } catch (error) {
    console.error("[infographics] Single template error:", error);
    res.status(500).json({ error: "Failed to fetch template" });
  }
});

// Contributor upload (SN's specification) - Fixed to handle descriptors
router.post("/api/creator/infographics", authenticate, async (req: any, res) => {
  try {
    const { 
      title, category, priceCents, previewImageUrl, descriptors,
      topTier, subcategories, 
      pptxUrl, keynoteUrl, gslidesUrl, downloadBundleUrl 
    } = req.body || {};
    
    if (!title || !category || !previewImageUrl) {
      return res.status(400).json({ error: "Missing required fields: title, category, previewImageUrl" });
    }

    if (!descriptors || !Array.isArray(descriptors) || descriptors.length === 0) {
      return res.status(400).json({ error: "At least one descriptor is required" });
    }

    // At least one format URL should be provided
    if (!pptxUrl && !keynoteUrl && !gslidesUrl && !downloadBundleUrl) {
      return res.status(400).json({ error: "At least one format URL must be provided" });
    }

    const rec = await db.createInfographicTemplate({
      creatorId: req.user.id, // can be null for admin uploads
      title,
      topTier: topTier || 'General',
      subcategories: subcategories || [],
      descriptors: descriptors, // FIXED: Now properly accepting and storing descriptors
      category,
      priceCents: priceCents ?? 1499, // Default to bundle price
      currency: "usd",
      previewImageUrl,
      pptxUrl: pptxUrl || null,
      keynoteUrl: keynoteUrl || null,
      gslidesUrl: gslidesUrl || null,
      downloadBundleUrl: downloadBundleUrl || null,
      isActive: false, // Awaiting approval
      approvalStatus: 'pending'
    });
    
    res.json({ success: true, template: rec });
  } catch (error) {
    console.error("[infographics] Creator upload error:", error);
    res.status(500).json({ error: "Failed to create template" });
  }
});

// Temporary seed endpoint for development
router.post("/api/admin/seed-infographics", async (req, res) => {
  if (process.env.NODE_ENV !== 'development') {
    return res.status(403).json({ error: "Only available in development" });
  }

  const sampleInfographics = [
    {
      title: "Financial Performance Dashboard",
      topTier: "FIG",
      subcategories: ["Finance & Insurance"],
      category: "business",
      priceCents: 1499,
      currency: "usd",
      previewImageUrl: "https://images.unsplash.com/photo-1551288049-bebda4e38f71?w=800&h=600",
      pptxUrl: "https://example.com/financial-dashboard.pptx",
      keynoteUrl: "https://example.com/financial-dashboard.key",
      gslidesUrl: "https://example.com/financial-dashboard",
      downloadBundleUrl: "https://example.com/financial-dashboard.zip",
      isActive: true,
      approvalStatus: "approved"
    },
    {
      title: "Technology Market Analysis",
      topTier: "TMT",
      subcategories: ["Information Technology (IT) / Information"],
      category: "professional",
      priceCents: 1499,
      currency: "usd",
      previewImageUrl: "https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=800&h=600",
      pptxUrl: "https://example.com/tech-analysis.pptx",
      keynoteUrl: "https://example.com/tech-analysis.key",
      gslidesUrl: "https://example.com/tech-analysis",
      downloadBundleUrl: "https://example.com/tech-analysis.zip",
      isActive: true,
      approvalStatus: "approved"
    },
    {
      title: "Healthcare Industry Overview", 
      topTier: "Healthcare/Pharma",
      subcategories: ["Healthcare"],
      category: "professional",
      priceCents: 1499,
      currency: "usd",
      previewImageUrl: "https://images.unsplash.com/photo-1559757148-5c350d0d3c56?w=800&h=600",
      pptxUrl: "https://example.com/healthcare-overview.pptx",
      keynoteUrl: "https://example.com/healthcare-overview.key",
      gslidesUrl: "https://example.com/healthcare-overview",
      downloadBundleUrl: "https://example.com/healthcare-overview.zip",
      isActive: true,
      approvalStatus: "approved"
    },
    {
      title: "Modern Business Process Flow",
      topTier: "General",
      subcategories: ["Professional, Scientific, & Technical Services"],
      category: "modern",
      priceCents: 1499,
      currency: "usd",
      previewImageUrl: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=800&h=600",
      pptxUrl: "https://example.com/business-process.pptx",
      keynoteUrl: "https://example.com/business-process.key",
      gslidesUrl: "https://example.com/business-process",
      downloadBundleUrl: "https://example.com/business-process.zip",
      isActive: true,
      approvalStatus: "approved"
    },
    {
      title: "Creative Marketing Funnel",
      topTier: "General",
      subcategories: ["Professional, Scientific, & Technical Services"],
      category: "creative",
      priceCents: 1499,
      currency: "usd",
      previewImageUrl: "https://images.unsplash.com/photo-1432888622747-4eb9a8efeb07?w=800&h=600",
      pptxUrl: "https://example.com/marketing-funnel.pptx",
      keynoteUrl: "https://example.com/marketing-funnel.key",
      gslidesUrl: "https://example.com/marketing-funnel",
      downloadBundleUrl: "https://example.com/marketing-funnel.zip",
      isActive: true,
      approvalStatus: "approved"
    }
  ];

  try {
    const created = [];
    for (const infographic of sampleInfographics) {
      const result = await db.createInfographicTemplate(infographic);
      created.push(result);
    }
    
    res.json({ success: true, created: created.length, templates: created });
  } catch (error) {
    console.error("Seed error:", error);
    res.status(500).json({ error: "Failed to seed infographics" });
  }
});

// Admin list all templates (for admin management interface)
router.get("/api/admin/infographics", authenticateAdmin, async (req, res) => {
  try {
    const rows = await db.getInfographicTemplates({
      q: req.query.q as string | undefined,
      category: req.query.category as string | undefined,
      descriptors: req.query.descriptors ? (req.query.descriptors as string).split(',') : undefined,
      min: req.query.min ? parseInt(req.query.min as string, 10) : undefined,
      max: req.query.max ? parseInt(req.query.max as string, 10) : undefined,
      includeInactive: req.query.includeInactive === 'true',
      status: req.query.status as string | undefined
    });
    res.json(rows);
  } catch (error) {
    console.error("[admin-infographics] List templates error:", error);
    res.status(500).json({ error: "Failed to fetch templates" });
  }
});

// Admin approve template
router.patch("/api/admin/infographics/:id/activate", authenticateAdmin, async (req, res) => {
  try {
    const { active } = req.body;
    
    const updateData: any = {
      isActive: Boolean(active),
      approvalStatus: active ? 'approved' : 'pending'
    };

    const updated = await db.updateInfographicTemplate(req.params.id, updateData);
    
    if (!updated) {
      return res.status(404).json({ error: "Template not found" });
    }

    res.json({ success: true, template: updated });
  } catch (error) {
    console.error("[infographics] Admin activate error:", error);
    res.status(500).json({ error: "Failed to update template" });
  }
});

// Bundle Checkout: Buy 1-4 selected infographics (SN's specification)
router.post("/api/infographics/checkout", authenticate, async (req: any, res) => {
  try {
    // FIXED: Use dedicated request validation schema (decoupled from insert schema)
    const validationResult = checkoutRequestSchema.safeParse(req.body);

    if (!validationResult.success) {
      const errors = validationResult.error.errors.map(err => ({
        field: err.path.join('.'),
        message: err.message
      }));
      return res.status(400).json({ 
        error: "Validation failed", 
        details: errors 
      });
    }

    const { selectedIds } = validationResult.data;

    // Validate all selected templates exist and are available
    const templates = await Promise.all(
      selectedIds.map(id => db.getInfographicTemplate(id))
    );

    const unavailableTemplates = templates.filter((tpl, index) => 
      !tpl || !tpl.isActive || tpl.approvalStatus !== 'approved'
    );

    if (unavailableTemplates.length > 0) {
      return res.status(400).json({ 
        error: "One or more items are unavailable. Please reselect." 
      });
    }

    // Create the bundle purchase with server-controlled pricing
    const purchase = await db.createInfographicPurchase({
      userId: req.user.id, // Server-controlled, not from client
      selectedIds: selectedIds,
      status: 'pending',
      amountCents: 1499, // Server-controlled bundle price
      currency: 'usd'
    });

    // Build secure redirect URLs using environment variables
    const baseUrl = process.env.DOMAIN_URL || process.env.FRONTEND_URL || process.env.APP_BASE_URL || req.headers.origin || '';
    
    const session = await createInfographicBundleCheckoutSession({
      successUrl: `${baseUrl}/business-assets/templates/infographics?success=true&purchase=${purchase.id}`,
      cancelUrl: `${baseUrl}/business-assets/templates/infographics?canceled=true`,
      metadata: {
        type: 'infographic_bundle',
        purchaseId: purchase.id,
        selectedIds: selectedIds.join(','),
        userId: req.user.id
      }
    });

    // Update purchase with Stripe session ID
    await db.updateInfographicPurchase(purchase.id, {
      stripeSessionId: session.id
    });

    res.json({
      sessionId: session.id,
      url: session.url,
      purchaseId: purchase.id
    });
  } catch (error) {
    console.error("[infographics] Checkout error:", error);
    res.status(500).json({ error: "Failed to process checkout" });
  }
});

// Get purchase status (for polling after checkout)
router.get("/api/infographics/purchases/:id", authenticate, async (req, res) => {
  try {
    const purchase = await db.getInfographicPurchase(req.params.id);
    
    if (!purchase || purchase.userId !== req.user.id) {
      return res.status(404).json({ error: "Purchase not found" });
    }

    res.json(purchase);
  } catch (error) {
    console.error("[infographics] Purchase status error:", error);
    res.status(500).json({ error: "Failed to fetch purchase" });
  }
});

// Download purchased bundle
router.get("/api/infographics/purchases/:id/download", authenticate, async (req, res) => {
  try {
    const purchase = await db.getInfographicPurchase(req.params.id);
    
    if (!purchase || purchase.userId !== req.user.id) {
      return res.status(404).json({ error: "Purchase not found" });
    }

    if (purchase.status !== 'paid') {
      return res.status(400).json({ error: "Purchase not completed" });
    }

    if (!purchase.downloadUrl) {
      return res.status(404).json({ error: "Download not ready" });
    }

    // Redirect to the signed ZIP URL or stream the file
    res.redirect(purchase.downloadUrl);
  } catch (error) {
    console.error("[infographics] Download error:", error);
    res.status(500).json({ error: "Failed to process download" });
  }
});

// Get user's infographic purchases (for My Purchases page)
router.get("/api/me/infographic-purchases", authenticate, async (req, res) => {
  try {
    const purchases = await db.getUserInfographicPurchases(req.user.id);
    res.json(purchases);
  } catch (error) {
    console.error("[infographics] User purchases error:", error);
    res.status(500).json({ error: "Failed to fetch purchases" });
  }
});

export default router;