import { Router } from "express";
import multer from "multer";
import sharp from "sharp";
import { Potrace } from "potrace";
import { z } from "zod";
import pLimit from "p-limit";
import { sanitizeSvg } from "../../util/sanitize";

// Enhanced file validation with magic number checking
function validateFileFormat(buffer: Buffer, mimetype: string, filename: string): boolean {
  if (buffer.length < 8) return false;
  
  const signature = buffer.subarray(0, 8);
  
  switch (mimetype) {
    case 'image/png':
      // PNG: 89 50 4E 47 0D 0A 1A 0A
      return signature[0] === 0x89 && signature[1] === 0x50 && signature[2] === 0x4E && signature[3] === 0x47;
      
    case 'image/jpeg':
    case 'image/jpg':
      // JPEG: FF D8 FF
      return signature[0] === 0xFF && signature[1] === 0xD8 && signature[2] === 0xFF;
      
    case 'image/svg+xml':
      // SVG: Check for XML or SVG tag at start
      const content = buffer.toString('utf-8', 0, Math.min(100, buffer.length));
      return content.includes('<svg') || content.includes('<?xml');
      
    default:
      return false;
  }
}

const upload = multer({ 
  storage: multer.memoryStorage(), 
  limits: { 
    fileSize: 10 * 1024 * 1024, // 10MB per file
    files: 20 // Max 20 files
  },
  fileFilter: (req, file, cb) => {
    // Strict MIME type validation
    const allowedTypes = ['image/png', 'image/jpg', 'image/jpeg', 'image/svg+xml'];
    if (!allowedTypes.includes(file.mimetype)) {
      return cb(new Error(`File ${file.originalname}: Only PNG, JPG, JPEG, and SVG files are allowed`));
    }
    
    // File extension validation
    const allowedExtensions = ['.png', '.jpg', '.jpeg', '.svg'];
    const fileExtension = file.originalname.toLowerCase().match(/\.[^.]+$/)?.[0];
    if (!fileExtension || !allowedExtensions.includes(fileExtension)) {
      return cb(new Error(`File ${file.originalname}: Invalid file extension`));
    }
    
    cb(null, true);
  }
});

// Validation schema for import parameters
const importIndividualFilesSchema = z.object({
  threshold: z.coerce.number().int().min(0).max(255).default(180),
  invert: z.coerce.boolean().default(false)
});

const router = Router();

/**
 * POST /api/icons/import-individual-files
 * Body (multipart/form-data):
 * - files: array of individual icon files (PNG, JPG, SVG)
 * - threshold: 0–255 binarize threshold for PNG/JPG conversion (optional, default 180)
 * - invert: "true"/"false" – invert colors for PNG/JPG conversion (optional, default false)
 */
router.post("/", upload.array("files", 20), async (req, res) => {
  try {
    const files = req.files as Express.Multer.File[];
    if (!files || files.length === 0) {
      return res.status(400).json({ error: "No files uploaded" });
    }
    
    // Additional server-side validation
    if (files.length > 20) {
      return res.status(400).json({ error: "Maximum 20 files allowed" });
    }
    
    // Validate each file with magic number checking
    for (const file of files) {
      if (!validateFileFormat(file.buffer, file.mimetype, file.originalname)) {
        return res.status(400).json({ 
          error: `Invalid file format for ${file.originalname}: file signature doesn't match MIME type` 
        });
      }
    }

    // Validate parameters with Zod
    const validatedParams = importIndividualFilesSchema.parse(req.body);
    const { threshold, invert } = validatedParams;

    // Concurrency limit for processing files (max 4 concurrent conversions)
    const limit = pLimit(4);
    
    async function convertImageToSvg(buffer: Buffer, filename: string): Promise<string> {
      return new Promise((resolve, reject) => {
        const startTime = Date.now();
        
        // Set up timeout for individual file processing
        const timeout = setTimeout(() => {
          reject(new Error(`Timeout processing ${filename}: operation took too long`));
        }, 45000); // 45 second timeout per file
        
        // First, get image metadata to calculate optimal resize dimensions
        const sharpImage = sharp(buffer);
        
        sharpImage.metadata((metaErr, metadata) => {
          if (metaErr) {
            clearTimeout(timeout);
            return reject(new Error(`Failed to read image metadata for ${filename}: ${metaErr.message}`));
          }

          const originalWidth = metadata.width || 256;
          const originalHeight = metadata.height || 256;
          
          // Calculate resize dimensions (max 256px on longest side)
          const resizeWidth = Math.min(originalWidth, 256);
          const resizeHeight = Math.min(originalHeight, 256);
          
          console.log(`[icons/import-individual-files] Processing ${filename}: ${originalWidth}x${originalHeight} -> ${resizeWidth}x${resizeHeight}`);
          
          // Process the image with Sharp including resizing optimization
          let imageProcessor = sharp(buffer)
            .grayscale()
            .threshold(threshold);

          // Apply inversion if requested
          if (invert) {
            imageProcessor = imageProcessor.negate();
          }

          // Add resizing optimization before toBuffer
          imageProcessor = imageProcessor.resize({ 
            width: resizeWidth, 
            height: resizeHeight, 
            fit: "inside" 
          });

          const sharpStartTime = Date.now();
          imageProcessor.toBuffer((sharpErr, processedBuffer) => {
            const sharpProcessingTime = Date.now() - sharpStartTime;
            
            if (sharpErr) {
              clearTimeout(timeout);
              return reject(new Error(`Sharp processing failed for ${filename}: ${sharpErr.message}`));
            }

            console.log(`[icons/import-individual-files] Sharp processing for ${filename} completed in ${sharpProcessingTime}ms`);

            // Then trace with Potrace
            const potrace = new Potrace({
              threshold,
              turdSize: 2,
              turnPolicy: Potrace.TURNPOLICY_MINORITY,
              color: "#000000",
              background: "#FFFFFF",
            });

            const potraceStartTime = Date.now();
            potrace.loadImage(processedBuffer, (loadErr: any) => {
              if (loadErr) {
                clearTimeout(timeout);
                return reject(new Error(`Potrace loading failed for ${filename}: ${loadErr.message}`));
              }

              potrace.getSVG((svgErr: any, svg: string) => {
                clearTimeout(timeout);
                const potraceProcessingTime = Date.now() - potraceStartTime;
                const totalProcessingTime = Date.now() - startTime;
                
                if (svgErr) {
                  return reject(new Error(`SVG generation failed for ${filename}: ${svgErr.message}`));
                }
                
                console.log(`[icons/import-individual-files] Potrace processing for ${filename} completed in ${potraceProcessingTime}ms`);
                console.log(`[icons/import-individual-files] Total processing time for ${filename}: ${totalProcessingTime}ms`);
                resolve(svg);
              });
            });
          });
        });
      });
    }

    const processedIcons: { id: string; svg: string; filename: string; error?: string }[] = [];

    // Create processing tasks for concurrency control
    const processingTasks = files.map((file, i) => {
      const id = `file-${i}`;
      
      return limit(async () => {
        try {
          let svg: string;

          if (file.mimetype === 'image/svg+xml') {
            // For SVG files, just sanitize and pass through
            const svgContent = file.buffer.toString('utf-8');
            console.log(`[icons/import-individual-files] Original SVG content for ${file.originalname} (first 200 chars):`, svgContent.substring(0, 200));
            
            svg = sanitizeSvg(svgContent);
            console.log(`[icons/import-individual-files] Sanitized SVG content for ${file.originalname} (first 200 chars):`, svg.substring(0, 200));
            console.log(`[icons/import-individual-files] SVG starts with '<svg':`, svg.startsWith('<svg'));
            
            if (!svg.trim().startsWith('<svg')) {
              console.error(`[icons/import-individual-files] Invalid SVG content for ${file.originalname}. Content after sanitization:`, svg.substring(0, 500));
              throw new Error('Invalid SVG content after sanitization');
            }
          } else {
            // For PNG/JPG files, convert to SVG using Potrace
            svg = await convertImageToSvg(file.buffer, file.originalname);
            svg = sanitizeSvg(svg); // Sanitize the generated SVG as well
            
            if (!svg.startsWith('<svg')) {
              throw new Error('Invalid SVG generated after conversion and sanitization');
            }
          }

          return {
            id,
            svg,
            filename: file.originalname,
          };

        } catch (fileError: any) {
          console.error(`[icons/import-individual-files] Error processing file ${file.originalname}:`, fileError);
          return {
            id,
            svg: "",
            filename: file.originalname,
            error: `Failed to process ${file.originalname}: ${fileError.message}`
          };
        }
      });
    });
    
    // Process all files with concurrency control and timeout
    const processingPromise = Promise.all(processingTasks);
    const timeoutPromise = new Promise((_, reject) => {
      setTimeout(() => reject(new Error("Processing timeout: batch operation took too long")), 180000); // 3 minute overall timeout
    });
    
    const results = await Promise.race([processingPromise, timeoutPromise]) as any[];
    processedIcons.push(...results);

    // Filter out failed conversions for the response
    const successfulIcons = processedIcons.filter(icon => !icon.error);
    const failedIcons = processedIcons.filter(icon => icon.error);

    res.json({ 
      icons: successfulIcons,
      successful: successfulIcons.length,
      failed: failedIcons.length,
      errors: failedIcons.length > 0 ? failedIcons.map(icon => ({ filename: icon.filename, error: icon.error })) : undefined
    });

  } catch (e: any) {
    console.error("[icons/import-individual-files]", e);
    
    // Handle Zod validation errors
    if (e instanceof z.ZodError) {
      return res.status(400).json({ 
        error: "Invalid parameters", 
        details: e.errors 
      });
    }
    
    res.status(500).json({ error: "Failed to process individual files" });
  }
});

export default router;