import { Router } from "express";
import { sanitizeSvg } from "../util/sanitize";
import { generateSvgFromPrompt } from "../util/svg-gen";
import { z } from "zod";
import importIndividualFilesRoute from "./icons/import-individual-files";
import { storage } from "../storage";
import { requireAdmin } from "../middleware/adminGuard";
import { insertImportedIconSchema, type InsertImportedIcon } from "../../shared/schema";
import { ObjectStorageService } from "../objectStorage";
import { randomUUID } from "crypto";
import sharp from "sharp";

const router = Router();
const objectStorageService = new ObjectStorageService();

// Helper function to generate PNG preview from SVG
async function generatePreviewFromSvg(svgContent: string, size: number = 256): Promise<Buffer> {
  try {
    // Create PNG buffer from SVG using Sharp
    const pngBuffer = await sharp(Buffer.from(svgContent))
      .resize(size, size, { 
        fit: 'inside',
        background: { r: 255, g: 255, b: 255, alpha: 0 } // transparent background
      })
      .png()
      .toBuffer();
    
    return pngBuffer;
  } catch (error) {
    console.error("Error generating preview from SVG:", error);
    throw new Error("Failed to generate preview image");
  }
}

// Validation schema for icon generation request
const generateIconSchema = z.object({
  prompt: z.string()
    .min(2, "Prompt must be at least 2 characters")
    .max(120, "Prompt must not exceed 120 characters")
    .transform(s => s.trim()),
  style: z.enum([
    "modern",
    "classic", 
    "flat",
    "outlined",
    "solid",
    "handdrawn",
    "isometric",
    "material"
  ]).optional().default("outlined")
});

/**
 * POST /api/icons/generate
 * body: { prompt: string, style?: "modern"|"classic"|"flat"|"outlined"|"solid"|"handdrawn"|"isometric"|"material" }
 * returns: { svg: string }
 */
router.post("/generate", async (req, res) => {
  try {
    // Validate input using Zod schema
    const validatedInput = generateIconSchema.parse(req.body);
    const { prompt, style } = validatedInput;

    // Local generator (fast, offline) — returns valid <svg>
    const rawSvg = generateSvgFromPrompt(prompt, style as any);

    // Safety pass
    const svg = sanitizeSvg(rawSvg);

    // Hard stop if sanitization fails
    if (!svg.startsWith("<svg") || svg.length > 32_000) {
      return res.status(422).json({ error: "Invalid SVG generated." });
    }

    return res.json({ svg });
  } catch (e: any) {
    console.error("[/icons/generate] error", e);
    
    // Handle Zod validation errors specifically
    if (e instanceof z.ZodError) {
      return res.status(400).json({ 
        error: "Validation failed", 
        details: e.errors 
      });
    }
    
    return res.status(500).json({ error: "Failed to generate icon." });
  }
});

// Validation schema for saving imported icons
const saveImportedIconsSchema = z.object({
  icons: z.array(z.object({
    name: z.string().min(1).trim(),
    svg: z.string().min(1),
    style: z.enum(["modern", "classic", "flat", "outlined", "solid", "handdrawn", "isometric", "material"]).default("flat"),
    tags: z.array(z.string()).default([])
  })).min(1, "At least one icon is required")
});

/**
 * POST /api/icons/save-imported
 * Admin-only endpoint to save imported icons to object storage (main Icons Library)
 * Requires Management or Owner role
 */
router.post("/save-imported", requireAdmin, async (req: any, res) => {
  try {
    // Validate input
    const validatedInput = saveImportedIconsSchema.parse(req.body);
    const { icons } = validatedInput;

    // Get admin user ID from auth middleware
    const adminUserId = req.user?.uid;
    if (!adminUserId) {
      return res.status(401).json({ error: "Admin authentication required" });
    }

    const savedIconsInfo = [];
    const errors = [];

    // Process each icon
    for (const icon of icons) {
      try {
        // Generate unique ID for this icon
        const iconId = randomUUID();
        
        // Sanitize SVG for security
        const cleanSvg = sanitizeSvg(icon.svg);
        if (!cleanSvg || !cleanSvg.startsWith("<svg")) {
          errors.push({ name: icon.name, error: "Invalid SVG content" });
          continue;
        }

        // Generate filename from icon name (sanitized)
        const sanitizedName = icon.name.replace(/[^a-zA-Z0-9-_]/g, '_');
        const svgFileName = `${sanitizedName}.svg`;
        const pngFileName = `${sanitizedName}.png`;

        // Get the public search path for storage
        const publicPaths = objectStorageService.getPublicObjectSearchPaths();
        const basePath = publicPaths[0]; // Use first public path

        // Define storage paths
        const svgPath = `${basePath}/icons/svg/${iconId}/${svgFileName}`;
        const previewPath = `${basePath}/icons/preview/${iconId}/${pngFileName}`;

        // Save SVG file to object storage
        await objectStorageService.uploadFile(
          svgPath,
          Buffer.from(cleanSvg, 'utf-8'),
          'image/svg+xml'
        );

        // Generate and save PNG preview
        const previewBuffer = await generatePreviewFromSvg(cleanSvg, 256);
        await objectStorageService.uploadFile(
          previewPath,
          previewBuffer,
          'image/png'
        );

        savedIconsInfo.push({
          id: iconId,
          name: icon.name,
          svgPath: `icons/svg/${iconId}/${svgFileName}`,
          previewPath: `icons/preview/${iconId}/${pngFileName}`,
          style: icon.style,
          tags: icon.tags
        });

      } catch (iconError: any) {
        console.error(`Error saving icon "${icon.name}":`, iconError);
        errors.push({ 
          name: icon.name, 
          error: iconError.message || "Failed to save icon" 
        });
      }
    }

    // Return success response with details
    const response: any = {
      success: true,
      count: savedIconsInfo.length,
      saved: savedIconsInfo.length,
      icons: savedIconsInfo
    };

    if (errors.length > 0) {
      response.errors = errors;
      response.failed = errors.length;
    }

    return res.json(response);

  } catch (e: any) {
    console.error("[/icons/save-imported] error", e);
    
    if (e instanceof z.ZodError) {
      return res.status(400).json({ 
        error: "Validation failed", 
        details: e.errors 
      });
    }
    
    return res.status(500).json({ error: "Failed to save icons to object storage" });
  }
});

/**
 * GET /api/icons/imported
 * Public endpoint to fetch all imported icons for display
 */
router.get("/imported", async (req, res) => {
  try {
    const { style, search } = req.query;

    let icons;
    if (search && typeof search === 'string') {
      icons = await storage.searchImportedIcons(search);
    } else if (style && typeof style === 'string') {
      icons = await storage.getImportedIconsByStyle(style);
    } else {
      icons = await storage.getAllImportedIcons();
    }

    return res.json({ icons });

  } catch (e: any) {
    console.error("[/icons/imported] error", e);
    return res.status(500).json({ error: "Failed to fetch imported icons" });
  }
});

/**
 * DELETE /api/icons/imported/:id
 * Admin-only endpoint to delete imported icons
 * Requires Management or Owner role
 */
router.delete("/imported/:id", requireAdmin, async (req, res) => {
  try {
    const { id } = req.params;
    
    if (!id) {
      return res.status(400).json({ error: "Icon ID is required" });
    }

    const deleted = await storage.deleteImportedIcon(id);
    
    if (!deleted) {
      return res.status(404).json({ error: "Icon not found or already deleted" });
    }

    return res.json({ success: true, message: "Icon deleted successfully" });

  } catch (e: any) {
    console.error("[/icons/imported/:id] error", e);
    return res.status(500).json({ error: "Failed to delete imported icon" });
  }
});


// Mount import-individual-files route with admin protection
router.use("/import-individual-files", requireAdmin, importIndividualFilesRoute);

export default router;