import express from "express";
import archiver from "archiver";
import fs from "fs/promises";
import path from "path";
import { ObjectStorageService } from "../objectStorage";
import { hasLicense } from "../services/entitlements.js";
import { userOwnsIcon, userHasIconsNoAttribution } from "../services/iconEntitlements";
import { buildAttribution } from "../services/iconsAttribution.js";
import { sanitizeSvg, shouldInlineSvg, extractIconId } from "../util/sanitize.js";
import { requireAdmin } from "../middleware/adminGuard.js";

const router = express.Router();
const objectStorageService = new ObjectStorageService();

// SuperNova's helper functions for cloud + local fallback
async function fetchFromStorage(cloudKey, localPath) {
  try {
    // Try cloud storage first
    const stream = await objectStorageService.getStream(cloudKey);
    const chunks = [];
    for await (const chunk of stream) {
      chunks.push(chunk);
    }
    return Buffer.concat(chunks);
  } catch (cloudError) {
    console.warn(`Cloud storage failed for ${cloudKey}, trying local:`, cloudError.message);
    // Fallback to local file
    return await readLocal(localPath);
  }
}

async function readLocal(localPath) {
  try {
    const fullPath = path.resolve(process.cwd(), localPath);
    return await fs.readFile(fullPath);
  } catch (localError) {
    throw new Error(`Local file not found: ${localPath}`);
  }
}

function guessType(filename) {
  const ext = path.extname(filename).toLowerCase();
  const mimeTypes = {
    '.png': 'image/png',
    '.jpg': 'image/jpeg',
    '.jpeg': 'image/jpeg',
    '.gif': 'image/gif',
    '.svg': 'image/svg+xml',
    '.webp': 'image/webp'
  };
  return mimeTypes[ext] || 'application/octet-stream';
}

// SuperNova's improved /api/icons/list endpoint with proper data structure
router.get("/list", async (req, res) => {
  try {
    // List all preview files to discover available icons
    const files = await objectStorageService.listPrefix("icons/preview/");
    const processedIcons = new Map(); // Use Map to deduplicate by baseName
    
    await Promise.all(files.map(async f => {
      try {
        // Extract baseName from filename, not folder path
        const baseName = extractIconId(f.key);
        const id = baseName;
        const name = baseName.replace(/[_-]/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
        
        // Skip if we already processed this icon (deduplicate)
        if (processedIcons.has(baseName)) {
          return;
        }
        
        // Try to get the SVG content with improved error handling
        let svgContent = null;
        let svgUrl = null;
        
        try {
          // NEW LOGIC: Use the preview file structure to find the corresponding SVG
          // From preview URL like: /public-objects/public/icons/preview/62c89e35-e776-4f19-ac86-d1d94eac5460/minus-circle_filled.png
          // Extract the UUID from the current file path to find the SVG
          const pathParts = f.key.split('/');
          let svgKey = null;
          let rawSvgContent = null;
          
          if (pathParts.length >= 3) {
            // Extract UUID from preview path: icons/preview/{UUID}/{filename}
            const uuid = pathParts[pathParts.length - 2]; // Get UUID from path
            const iconFileName = pathParts[pathParts.length - 1].replace('.png', '.svg');
            
            // Try UUID-based layout: icons/svg/{UUID}/{iconName}.svg
            svgKey = `icons/svg/${uuid}/${iconFileName}`;
            
            try {
              const svgStream = await objectStorageService.getStream(svgKey);
              const chunks = [];
              for await (const chunk of svgStream) {
                chunks.push(chunk);
              }
              rawSvgContent = Buffer.concat(chunks).toString('utf8');
            } catch (uuidError) {
              // Fallback: Try to find SVG by listing all files in icons/svg/ that match the icon name
              try {
                const allSvgFiles = await objectStorageService.listPrefix(`icons/svg/`);
                const matchingFile = allSvgFiles.find(file => 
                  file.key.includes(baseName) && file.key.endsWith('.svg')
                );
                
                if (matchingFile) {
                  svgKey = matchingFile.key;
                  const svgStream = await objectStorageService.getStream(svgKey);
                  const chunks = [];
                  for await (const chunk of svgStream) {
                    chunks.push(chunk);
                  }
                  rawSvgContent = Buffer.concat(chunks).toString('utf8');
                }
              } catch (fallbackError) {
                console.warn(`No SVG found for icon ${baseName} using any method`);
              }
            }
          }
          
          if (rawSvgContent) {
            const sanitizedSvg = sanitizeSvg(rawSvgContent);
            if (sanitizedSvg) {
              // SuperNova's logic: inline small SVGs, provide URL for large ones
              if (shouldInlineSvg(sanitizedSvg)) {
                svgContent = sanitizedSvg;
              } else {
                svgUrl = `/api/icons/${id}/svg`;
              }
            }
          }
        } catch (svgError) {
          console.warn(`Could not fetch SVG content for icon ${baseName}:`, svgError.message);
        }
        
        const iconData = {
          id,
          name,
          style: "outlined",
          tags: [],
          previewUrl: objectStorageService.publicUrlFromKey(f.key),
          formats: ["svg", "png"],
          createdAt: Date.now(),
          // SuperNova's data structure: svg field for small files, svgUrl for large ones
          ...(svgContent ? { svg: svgContent } : {}),
          ...(svgUrl ? { svgUrl } : {})
        };
        
        processedIcons.set(baseName, iconData);
      } catch (itemError) {
        console.warn(`Error processing icon file ${f.key}:`, itemError.message);
      }
    }));
    
    const items = Array.from(processedIcons.values());
    
    // Backend Soft-Delete System: Filter out hidden/disabled/deleted icons
    const visibleIcons = items.filter(x => !hiddenIcons.has(x.id));
    
    // PERFORMANCE FIX: Embed entitlement data to prevent N+1 queries
    const userId = req.user?.id || null;
    if (userId) {
      // Check user's global license once
      const hasGlobalLicense = await userHasIconsNoAttribution(userId);
      
      // Add entitlement info to each icon to prevent individual API calls
      visibleIcons.forEach(icon => {
        const licensed = hasGlobalLicense; // Pack logic disabled due to mapping issues
        icon.entitlement = {
          licensed,
          requiresAttribution: !licensed,
          canDownload: true
        };
      });
      
      console.log(`[Icons] Listed ${visibleIcons.length} icons with embedded entitlements for user ${userId}`);
    } else {
      // Anonymous user - all require attribution
      visibleIcons.forEach(icon => {
        icon.entitlement = {
          licensed: false,
          requiresAttribution: true,
          canDownload: true
        };
      });
      
      console.log(`[Icons] Listed ${visibleIcons.length} icons for anonymous user`);
    }
    
    console.log(`[Icons] ${visibleIcons.filter(i => i.svg).length} with inline SVG, ${visibleIcons.filter(i => i.svgUrl).length} with SVG URLs`);
    res.json({ icons: visibleIcons });
  } catch (error) {
    console.error("Error listing icons:", error);
    res.json({ icons: [] }); // Return empty array on error
  }
});

// SuperNova's enhanced entitlement endpoint
router.get("/:id/entitlement", async (req, res) => {
  try {
    const userId = req.user?.id || null;
    const iconId = req.params.id;
    
    if (!userId) {
      // Anonymous users require attribution
      return res.json({ 
        licensed: false, 
        requiresAttribution: true, 
        canDownload: true 
      });
    }

    // Check both global license and specific icon ownership using SuperNova's functions
    const hasGlobalLicense = await userHasIconsNoAttribution(userId);
    const hasIconLicense = await userOwnsIcon(userId, iconId);
    
    // User is licensed if they have global license OR specific icon license
    const licensed = hasGlobalLicense || hasIconLicense;
    
    // Legacy fallback: check old licensing system for backward compatibility
    const legacyLicense = hasLicense(userId, iconId);
    const finalLicensed = licensed || legacyLicense;
    
    res.json({ 
      licensed: finalLicensed, 
      requiresAttribution: !finalLicensed, 
      canDownload: true
    });
  } catch (error) {
    console.error("Error checking entitlement:", error);
    res.status(500).json({ error: "Failed to check entitlement" });
  }
});

router.get("/:id/download", async (req, res) => {
  try {
    const id = req.params.id;
    const fmt = (req.query.format || "svg").toString().toLowerCase(); // svg|png

    // Resolve storage key
    const base = fmt === "png" ? "icons/png" : "icons/svg";
    const files = await objectStorageService.listPrefix(`${base}/${id}/`);
    const file = files[0];
    
    if (!file) {
      return res.status(404).json({ error: "Icon not found" });
    }

    const userId = req.user?.id || null;
    const licensed = userId ? (hasLicense(userId, id) || userHasPro(userId)) : false;

    if (licensed) {
      const stream = await objectStorageService.getStream(file.key);
      res.setHeader("Content-Type", fmt === "png" ? "image/png" : "image/svg+xml");
      res.setHeader("Content-Disposition", `attachment; filename="IBrandBiz_${id}.${fmt}"`);
      return stream.pipe(res);
    }

    // Free with attribution → build a zip {icon + CREDIT.txt}
    const archive = archiver("zip", { zlib: { level: 9 } });
    res.setHeader("Content-Type", "application/zip");
    res.setHeader("Content-Disposition", `attachment; filename="IBrandBiz_${id}_${fmt}_FREE.zip"`);

    const credit = [
      "Thank you for using IBrandBiz Icons!",
      "",
      "License: Free with Attribution",
      'You MUST include visible credit: "Icons by IBrandBiz Icons — Free with Attribution (https://ibrandbiz.com/icons)"',
      "Full license: https://ibrandbiz.com/license/icons/free",
      "Upgrade to IBrandBiz Pro to remove attribution: https://ibrandbiz.com/pricing",
      ""
    ].join("\n");

    archive.append(credit, { name: "CREDIT.txt" });
    
    const fileStream = await objectStorageService.getStream(file.key);
    archive.append(fileStream, { name: `icon.${fmt}` });
    
    archive.finalize();
    archive.pipe(res);
  } catch (error) {
    console.error("Error downloading icon:", error);
    res.status(500).json({ error: "Failed to download icon" });
  }
});

// SuperNova's new /api/icons/:id/svg endpoint
router.get("/:id/svg", async (req, res) => {
  try {
    const id = req.params.id;
    
    // Try to find the SVG file in object storage
    let svgKey = null;
    let rawSvgContent = null;
    
    try {
      // NEW LOGIC: Find SVG using the actual storage structure
      // First try to find any SVG file that matches the icon name
      const allSvgFiles = await objectStorageService.listPrefix(`icons/svg/`);
      const matchingFile = allSvgFiles.find(file => 
        file.key.includes(id) && file.key.endsWith('.svg')
      );
      
      if (matchingFile) {
        svgKey = matchingFile.key;
        const svgStream = await objectStorageService.getStream(svgKey);
        const chunks = [];
        for await (const chunk of svgStream) {
          chunks.push(chunk);
        }
        rawSvgContent = Buffer.concat(chunks).toString('utf8');
      } else {
        return res.status(404).json({ error: "SVG file not found" });
      }
    } catch (error) {
      console.error(`Error finding SVG for icon ${id}:`, error);
      return res.status(404).json({ error: "SVG file not found" });
    }
    
    if (!rawSvgContent) {
      return res.status(404).json({ error: "SVG content not found" });
    }
    
    // Sanitize and normalize the SVG content
    const sanitizedSvg = sanitizeSvg(rawSvgContent);
    
    if (!sanitizedSvg) {
      return res.status(500).json({ error: "Failed to process SVG content" });
    }
    
    // Serve with proper headers for SVG
    res.setHeader("Content-Type", "image/svg+xml");
    res.setHeader("Cache-Control", "public, max-age=31536000"); // 1 year cache
    res.setHeader("Access-Control-Allow-Origin", "*");
    
    res.send(sanitizedSvg);
  } catch (error) {
    console.error(`Error serving SVG for icon ${req.params.id}:`, error);
    res.status(500).json({ error: "Failed to serve SVG" });
  }
});

// SuperNova's new /api/icons/:id/preview endpoint
router.get("/:id/preview", async (req, res) => {
  try {
    const id = req.params.id;
    
    // NEW LOGIC: Find preview using the actual storage structure
    let previewBuffer = null;
    let contentType = 'image/png';
    
    try {
      // Find the preview file by searching for any file matching the icon name
      const allPreviewFiles = await objectStorageService.listPrefix(`icons/preview/`);
      const matchingFile = allPreviewFiles.find(file => 
        file.key.includes(id) && (file.key.endsWith('.png') || file.key.endsWith('.jpg'))
      );
      
      if (matchingFile) {
        const previewStream = await objectStorageService.getStream(matchingFile.key);
        const chunks = [];
        for await (const chunk of previewStream) {
          chunks.push(chunk);
        }
        previewBuffer = Buffer.concat(chunks);
        contentType = guessType(matchingFile.key);
      }
    } catch (error) {
      console.warn(`Error finding preview for icon ${id}:`, error);
    }
    
    if (!previewBuffer) {
      return res.status(404).json({ error: "Preview image not found" });
    }
    
    // Serve with proper headers for preview images
    res.setHeader("Content-Type", contentType);
    res.setHeader("Cache-Control", "public, max-age=31536000"); // 1 year cache
    res.setHeader("Access-Control-Allow-Origin", "*");
    
    res.send(previewBuffer);
  } catch (error) {
    console.error(`Error serving preview for icon ${req.params.id}:`, error);
    res.status(500).json({ error: "Failed to serve preview image" });
  }
});

// DELETE /api/icons/:id - Delete icon from object storage (admin only)
router.delete("/:id", requireAdmin, async (req, res) => {
  try {
    const id = req.params.id;
    
    if (!id) {
      return res.status(400).json({ error: "Icon ID is required" });
    }
    
    console.log(`🗑️ Admin attempting to delete icon: ${id}`);
    
    // Delete both SVG and preview files from object storage
    let deletedAny = false;
    
    // Delete SVG files (try multiple possible paths)
    try {
      const svgDeleted = await objectStorageService.deleteFilesByPattern(`icons/svg/*/${id}.svg`);
      if (svgDeleted) {
        deletedAny = true;
        console.log(`✅ Deleted SVG files for icon: ${id}`);
      }
    } catch (svgError) {
      console.warn(`Error deleting SVG files for ${id}:`, svgError);
    }
    
    // Delete preview files (try multiple possible paths)
    try {
      const previewDeleted = await objectStorageService.deleteFilesByPattern(`icons/preview/*/${id}.*`);
      if (previewDeleted) {
        deletedAny = true;
        console.log(`✅ Deleted preview files for icon: ${id}`);
      }
    } catch (previewError) {
      console.warn(`Error deleting preview files for ${id}:`, previewError);
    }
    
    // Also try deleting with exact filename matches from the list
    try {
      const allFiles = await objectStorageService.listPrefix(`icons/`);
      const matchingFiles = allFiles.filter(file => 
        file.key.includes(id) && (
          file.key.includes('svg/') || 
          file.key.includes('preview/') || 
          file.key.includes('png/')
        )
      );
      
      for (const file of matchingFiles) {
        try {
          await objectStorageService.deleteFile(file.key);
          deletedAny = true;
          console.log(`✅ Deleted specific file: ${file.key}`);
        } catch (fileError) {
          console.warn(`Failed to delete file ${file.key}:`, fileError);
        }
      }
    } catch (listError) {
      console.warn(`Error listing files for cleanup:`, listError);
    }
    
    if (!deletedAny) {
      console.log(`⚠️ No files found to delete for icon: ${id}`);
      return res.status(404).json({ error: "Icon not found or already deleted" });
    }
    
    console.log(`✅ Successfully deleted icon: ${id}`);
    return res.json({ success: true, message: "Icon deleted successfully from object storage" });
    
  } catch (error) {
    console.error(`❌ Error deleting icon ${req.params.id}:`, error);
    return res.status(500).json({ error: "Failed to delete icon from object storage" });
  }
});

// Simple hidden icons tracking (JSON file-based)
let hiddenIcons = new Set();
const HIDDEN_ICONS_FILE = path.join(process.cwd(), 'server', 'data', 'hidden-icons.json');

// Load hidden icons on startup
async function loadHiddenIcons() {
  try {
    const data = await fs.readFile(HIDDEN_ICONS_FILE, 'utf8');
    const hiddenArray = JSON.parse(data);
    hiddenIcons = new Set(hiddenArray);
    console.log(`[Icons] Loaded ${hiddenIcons.size} hidden icons from file`);
  } catch (error) {
    // File doesn't exist or invalid JSON - start with empty set
    hiddenIcons = new Set();
    console.log(`[Icons] No hidden icons file found, starting fresh`);
  }
}

// Save hidden icons to file
async function saveHiddenIcons() {
  try {
    await fs.mkdir(path.dirname(HIDDEN_ICONS_FILE), { recursive: true });
    await fs.writeFile(HIDDEN_ICONS_FILE, JSON.stringify([...hiddenIcons], null, 2));
    console.log(`[Icons] Saved ${hiddenIcons.size} hidden icons to file`);
  } catch (error) {
    console.error(`[Icons] Failed to save hidden icons:`, error);
  }
}

// Load hidden icons on startup
loadHiddenIcons();

// Admin route: POST /admin/icons/:id/disable - Mark icon as hidden
router.post("/admin/:id/disable", requireAdmin, async (req, res) => {
  try {
    const iconId = req.params.id;
    
    if (!iconId) {
      return res.status(400).json({ error: "Icon ID is required" });
    }
    
    // Add to hidden icons set
    hiddenIcons.add(iconId);
    await saveHiddenIcons();
    
    console.log(`[Icons] Admin disabled icon: ${iconId}`);
    
    res.json({ 
      success: true, 
      message: `Icon ${iconId} has been marked as hidden`,
      hiddenIconsCount: hiddenIcons.size
    });
  } catch (error) {
    console.error(`Error disabling icon ${req.params.id}:`, error);
    res.status(500).json({ error: "Failed to disable icon" });
  }
});

// Admin route: POST /admin/icons/:id/purge-files - Clean up files for hidden icon
router.post("/admin/:id/purge-files", requireAdmin, async (req, res) => {
  try {
    const iconId = req.params.id;
    
    if (!iconId) {
      return res.status(400).json({ error: "Icon ID is required" });
    }
    
    // First check if icon is hidden
    if (!hiddenIcons.has(iconId)) {
      return res.status(400).json({ error: "Icon must be disabled before purging files" });
    }
    
    console.log(`🗑️ Admin purging files for hidden icon: ${iconId}`);
    
    // Use the same deletion logic as the DELETE route
    let deletedAny = false;
    
    // Delete SVG files (try multiple possible paths)
    try {
      const svgDeleted = await objectStorageService.deleteFilesByPattern(`icons/svg/*/${iconId}.svg`);
      if (svgDeleted) {
        deletedAny = true;
        console.log(`✅ Purged SVG files for icon: ${iconId}`);
      }
    } catch (svgError) {
      console.warn(`Error purging SVG files for ${iconId}:`, svgError);
    }
    
    // Delete preview files (try multiple possible paths)
    try {
      const previewDeleted = await objectStorageService.deleteFilesByPattern(`icons/preview/*/${iconId}.*`);
      if (previewDeleted) {
        deletedAny = true;
        console.log(`✅ Purged preview files for icon: ${iconId}`);
      }
    } catch (previewError) {
      console.warn(`Error purging preview files for ${iconId}:`, previewError);
    }
    
    // Also try deleting with exact filename matches from the list
    try {
      const allFiles = await objectStorageService.listPrefix(`icons/`);
      const matchingFiles = allFiles.filter(file => 
        file.key.includes(iconId) && (
          file.key.includes('svg/') || 
          file.key.includes('preview/') || 
          file.key.includes('png/')
        )
      );
      
      for (const file of matchingFiles) {
        try {
          await objectStorageService.deleteFile(file.key);
          deletedAny = true;
          console.log(`✅ Purged specific file: ${file.key}`);
        } catch (fileError) {
          console.warn(`Failed to purge file ${file.key}:`, fileError);
        }
      }
    } catch (listError) {
      console.warn(`Error listing files for purge:`, listError);
    }
    
    res.json({ 
      success: true, 
      message: `Files purged for hidden icon ${iconId}`,
      filesDeleted: deletedAny
    });
    
  } catch (error) {
    console.error(`❌ Error purging files for icon ${req.params.id}:`, error);
    return res.status(500).json({ error: "Failed to purge icon files" });
  }
});

export default router;