import express from 'express';
import { requireAuth } from '../middleware/authz';
import { requireAdmin } from '../middleware/adminGuard';
import {
  getCreatorByUserId,
  getCreatorAssetsByCreatorId,
  getCreatorAssetById,
  updateCreatorAsset,
  getCreatorPayoutSummary,
  getAllCreatorAssetsForAdmin,
} from '../storage/creatorMarketplace';

const router = express.Router();

// Helper function for error handling
const handleError = (res: express.Response, error: any, message: string) => {
  console.error(`${message}:`, error);
  res.status(500).json({ error: message });
};

// CREATOR – overview (grouped by status)
router.get('/api/creator/me/marketplace', requireAuth, async (req, res) => {
  try {
    const userId = req.user!.uid; // Firebase UID
    const creator = await getCreatorByUserId(userId);
    if (!creator) {
      return res.status(403).json({ error: 'Not a creator' });
    }

    const rows = await getCreatorAssetsByCreatorId(creator.id);
    const group = (s: string) => rows.filter(r => r.approvalStatus === s);

    res.json({
      assets: {
        approved: group('approved'),
        pending: group('pending'),
        rejected: group('rejected'),
        cancelled: group('cancelled'),
      },
      stats: {
        totalSales: rows.reduce((n, r) => n + (r.salesCount || 0), 0),
        totalRevenueCents: rows.reduce((n, r) => n + (r.totalRevenueCents || 0), 0),
      }
    });
  } catch (e) {
    handleError(res, e, 'Failed to load creator marketplace');
  }
});

// CREATOR – cancel listing
router.patch('/api/creator/assets/:id/cancel', requireAuth, async (req, res) => {
  try {
    const userId = req.user!.uid; // Firebase UID
    const creator = await getCreatorByUserId(userId);
    if (!creator) {
      return res.status(403).json({ error: 'Not a creator' });
    }

    const rec = await getCreatorAssetById(req.params.id);
    if (!rec || rec.creatorId !== creator.id) {
      return res.status(404).json({ error: 'Asset not found' });
    }

    if (rec.approvalStatus === 'cancelled') {
      return res.json({ success: true, asset: rec });
    }

    const updated = await updateCreatorAsset(rec.id, { 
      approvalStatus: 'cancelled', 
      cancelledAt: new Date() 
    });
    res.json({ success: true, asset: updated });
  } catch (e) {
    handleError(res, e, 'Failed to cancel listing');
  }
});

// CREATOR – payouts summary
router.get('/api/creator/me/payouts/summary', requireAuth, async (req, res) => {
  try {
    const userId = req.user!.uid; // Firebase UID
    const creator = await getCreatorByUserId(userId);
    if (!creator) {
      return res.status(403).json({ error: 'Not a creator' });
    }

    const summary = await getCreatorPayoutSummary(creator.id);
    res.json(summary || { lifetimeEarningsCents: 0, pendingPayoutCents: 0, lastPayoutAt: null });
  } catch (e) {
    handleError(res, e, 'Failed to load payouts summary');
  }
});

/* ------------------- Admin endpoints ------------------- */

// Admin overview
router.get('/api/admin/marketplace/overview', requireAdmin, async (req, res) => {
  try {
    const q = (req.query.search as string || '').toLowerCase();
    const rows = await getAllCreatorAssetsForAdmin();
    const filtered = q
      ? rows.filter(r =>
          (r.title || '').toLowerCase().includes(q) ||
          (r.creatorName || '').toLowerCase().includes(q)
        )
      : rows;

    res.json({
      pending: filtered.filter(r => r.status === 'pending'),
      approved: filtered.filter(r => r.status === 'approved'),
      rejected: filtered.filter(r => r.status === 'rejected'),
    });
  } catch (e) {
    handleError(res, e, 'Failed to load admin marketplace overview');
  }
});

// Admin approve asset
router.patch('/api/admin/marketplace/assets/:id/approve', requireAdmin, async (req, res) => {
  try {
    const updated = await updateCreatorAsset(req.params.id, { 
      approvalStatus: 'approved', 
      approvedAt: new Date() 
    });
    res.json({ success: true, asset: updated });
  } catch (e) {
    handleError(res, e, 'Failed to approve asset');
  }
});

// Admin reject asset
router.patch('/api/admin/marketplace/assets/:id/reject', requireAdmin, async (req, res) => {
  try {
    const updated = await updateCreatorAsset(req.params.id, { 
      approvalStatus: 'rejected', 
      rejectionReason: req.body?.reason || null 
    });
    res.json({ success: true, asset: updated });
  } catch (e) {
    handleError(res, e, 'Failed to reject asset');
  }
});

export default router;