import { Router } from "express";
import PPTXGenJS from "pptxgenjs";
import { CoverTemplate, CoverPurchase, insertCoverPurchaseSchema } from "../../shared/schema";
import { storage as db } from "../storage";
import { ensurePriceForTemplate, createCheckoutSession } from "../stripe/coverCheckout";

const router = Router();

// Import authenticateToken from main routes.ts
interface AuthenticatedRequest extends Request {
  user?: {
    uid: string;
    id: string;
    email?: string;
    [key: string]: any;
  };
}

// For now, we'll create a simple authentication middleware placeholder
// In production, this would properly verify Firebase tokens
const requireAuth = (req: any, res: any, next: any) => {
  // Mock authentication for development
  if (process.env.NODE_ENV === 'development') {
    req.user = {
      uid: 'dev-user-123',
      id: 'dev-user-123',
      email: 'dev@example.com'
    };
    return next();
  }
  
  // In production, add proper Firebase token verification here
  const authHeader = req.headers.authorization;
  if (!authHeader) {
    return res.status(401).json({ error: 'Authentication required' });
  }
  
  // TODO: Implement proper token verification
  req.user = {
    uid: 'user-123',
    id: 'user-123',
    email: 'user@example.com'
  };
  next();
};

type ReqBody = {
  size: "16x9" | "4x3";
  prompt: string;
  brand: { name?: string; fonts: { heading: string; body: string }; accents: string[] };
  content: { title: string; subtitle?: string };
  assets: { logoDataUrl: string | null; logoPlacement: "none"|"top-left"|"top-right"|"bottom-left"|"bottom-right"; logoScale: number };
};

router.post("/generate", async (req, res) => {
  try {
    const b = req.body as ReqBody;

    const pptx = new PPTXGenJS();
    pptx.layout = b.size === "4x3" ? "LAYOUT_4x3" : "LAYOUT_16x9";
    const slideW = pptx.width;
    const slideH = pptx.height;

    // Choose a style based on prompt keywords
    const style = pickStyle(b.prompt);

    // Masters: Cover
    const coverMasterObjects: PPTXGenJS.SlideMasterObject[] = [];
    // Backgrounds by style
    if (style === "band") {
      coverMasterObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[2] || "#111827") } } });
      // middle band
      coverMasterObjects.push({ rect: { x: 0, y: slideH*0.32, w: "100%", h: slideH*0.36, fill: { color: hex(b.brand.accents[0] || "#0ea5e9") } } });
    } else if (style === "left") {
      coverMasterObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[2] || "#111827") } } });
      // left bar
      coverMasterObjects.push({ rect: { x: 0.6, y: slideH*0.26, w: 0.12, h: slideH*0.48, fill: { color: hex(b.brand.accents[1] || "#ff8800") } } });
    } else if (style === "wash") {
      coverMasterObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[2] || "#111827") } } });
      // translucent wash
      coverMasterObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[0] || "#0ea5e9"), transparency: 60 } } });
    } else {
      // centered default
      coverMasterObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[0] || "#0ea5e9") } } });
    }

    // Title/Sub placeholders in master
    coverMasterObjects.push({
      text: { text: b.content.title || "Title", options: { x: 0.8, y: slideH*0.32, w: slideW-1.6, h: 1.6, align: "center", fontFace: b.brand.fonts.heading, fontSize: 42, bold: true, color: "FFFFFF" } }
    });
    if (b.content.subtitle?.trim()) {
      coverMasterObjects.push({
        text: { text: b.content.subtitle!, options: { x: 1.2, y: slideH*0.48, w: slideW-2.4, h: 1.1, align: "center", fontFace: b.brand.fonts.body, fontSize: 20, color: "FFFFFF" } }
      });
    }

    // Add logo (if any) to master objects
    if (b.assets.logoDataUrl && b.assets.logoPlacement !== "none") {
      coverMasterObjects.push(logoObject(b.assets, slideW, slideH));
    }

    pptx.defineSlideMaster({ title: "CoverMaster", objects: coverMasterObjects });

    // Masters: Divider
    const dividerObjects: PPTXGenJS.SlideMasterObject[] = [];
    if (style === "band") {
      dividerObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[2] || "#111827") } } });
      dividerObjects.push({ rect: { x: 0, y: slideH*0.35, w: "100%", h: slideH*0.30, fill: { color: hex(b.brand.accents[1] || "#ff8800") } } });
    } else if (style === "left") {
      dividerObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[0] || "#0ea5e9") } } });
      dividerObjects.push({ rect: { x: 0.6, y: slideH*0.3, w: 0.12, h: slideH*0.40, fill: { color: "FFFFFF" } } });
    } else if (style === "wash") {
      dividerObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[2] || "#111827") } } });
      dividerObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[1] || "#ff8800"), transparency: 65 } } });
    } else {
      dividerObjects.push({ rect: { x: 0, y: 0, w: "100%", h: "100%", fill: { color: hex(b.brand.accents[1] || "#ff8800") } } });
    }
    dividerObjects.push({
      text: { text: "Section Title", options: { x: 0.8, y: slideH*0.40, w: slideW-1.6, h: 1.2, align: "center", fontFace: b.brand.fonts.heading, fontSize: 32, bold: true, color: "FFFFFF" } }
    });
    if (b.assets.logoDataUrl && b.assets.logoPlacement !== "none") {
      dividerObjects.push(logoObject(b.assets, slideW, slideH));
    }

    pptx.defineSlideMaster({ title: "DividerMaster", objects: dividerObjects });

    // Slides using masters
    pptx.addSlide({ masterName: "CoverMaster" });
    pptx.addSlide({ masterName: "DividerMaster" });

    // Return file
    const buf = await pptx.write("nodebuffer");
    res.setHeader("Content-Disposition", 'attachment; filename="cover+dividers.pptx"');
    res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.presentationml.presentation");
    res.send(buf);
  } catch (e: any) {
    console.error("[covers/generate] error", e);
    res.status(500).json({ error: "Failed to generate Cover & Dividers" });
  }
});

// Get all active cover templates for the gallery
router.get("/templates", async (req, res) => {
  try {
    const { q, category, min, max } = req.query;
    
    const options = {
      q: q as string | undefined,
      category: category as string | undefined,
      min: min ? parseInt(min as string) : undefined,
      max: max ? parseInt(max as string) : undefined
    };
    
    const templates = await db.getCoverTemplates(options);
    res.json(templates);
  } catch (error) {
    console.error("[covers/templates] error", error);
    res.status(500).json({ error: "Failed to fetch cover templates" });
  }
});

// Purchase a cover template
router.post("/purchase", requireAuth, async (req, res) => {
  try {
    const { templateId, customImageUrl } = req.body;
    const userId = req.user!.id;

    // Check if template exists and is active
    const template = await db.getCoverTemplate(templateId);
    if (!template || !template.isActive) {
      return res.status(404).json({ error: "Template not found or inactive" });
    }

    // Create Stripe price for this template
    const price = await ensurePriceForTemplate(
      template.id,
      template.title,
      template.priceCents,
      template.currency
    );

    // Create purchase record with pending status
    const purchase = await db.createCoverPurchase({
      userId,
      templateId,
      customImageUrl,
      amountCents: template.priceCents,
      currency: template.currency,
      status: 'pending'
    });

    // Build secure redirect URLs using environment variables
    const baseUrl = process.env.DOMAIN_URL || process.env.FRONTEND_URL || req.headers.origin || '';
    
    // Create Stripe checkout session
    const session = await createCheckoutSession({
      priceId: price.id,
      successUrl: `${baseUrl}/business-assets/templates/cover-dividers?success=true&session_id={CHECKOUT_SESSION_ID}`,
      cancelUrl: `${baseUrl}/business-assets/templates/cover-dividers?canceled=true`,
      metadata: {
        purchaseId: purchase.id,
        templateId: template.id,
        userId: userId
      }
    });

    // Update purchase with Stripe session ID
    await db.updateCoverPurchase(purchase.id, {
      stripeSessionId: session.id
    });

    res.json({
      sessionId: session.id,
      url: session.url,
      purchaseId: purchase.id
    });
  } catch (error) {
    console.error("[covers/purchase] error", error);
    res.status(500).json({ error: "Failed to process purchase" });
  }
});

// Get user's cover template purchases
router.get("/purchases", requireAuth, async (req, res) => {
  try {
    const userId = req.user!.id;
    const purchases = await db.getUserCoverPurchases(userId);
    res.json(purchases);
  } catch (error) {
    console.error("[covers/purchases] error", error);
    res.status(500).json({ error: "Failed to fetch purchases" });
  }
});

// Download purchased template
router.get("/download/:purchaseId", requireAuth, async (req, res) => {
  try {
    const { purchaseId } = req.params;
    const userId = req.user!.id;

    // Verify purchase ownership and status
    const purchase = await db.getCoverPurchase(purchaseId);
    if (!purchase || purchase.userId !== userId || purchase.status !== 'paid') {
      return res.status(404).json({ error: "Purchase not found or not accessible" });
    }

    // Get template details
    const template = await db.getCoverTemplate(purchase.templateId);
    if (!template) {
      return res.status(404).json({ error: "Template not found" });
    }

    // For now, return mock download URL
    // In production, this would generate the actual customized template file
    res.json({
      downloadUrl: template.downloadFile,
      fileName: `${template.title.replace(/[^a-zA-Z0-9]/g, '_')}_Cover_Template.zip`,
      customImageUrl: purchase.customImageUrl
    });
  } catch (error) {
    console.error("[covers/download] error", error);
    res.status(500).json({ error: "Failed to process download" });
  }
});


export default router;

// ----- helpers -----
function hex(x?: string) {
  const h = (x || "").replace("#","");
  return (h.length===6 ? h : "000000").toUpperCase();
}

function logoObject(
  assets: ReqBody["assets"],
  slideW: number,
  slideH: number
): PPTXGenJS.SlideMasterObject {
  const base = 1.2;
  const w = base * (assets.logoScale || 1);
  const h = w;
  const m = 0.25;
  let x = m, y = m;
  switch (assets.logoPlacement) {
    case "top-left": x = m; y = m; break;
    case "top-right": x = slideW - w - m; y = m; break;
    case "bottom-left": x = m; y = slideH - h - m; break;
    case "bottom-right": x = slideW - w - m; y = slideH - h - m; break;
    default: x = -999; y = -999; // hidden
  }
  return { image: { data: assets.logoDataUrl!, x, y, w, h } };
}

// crude style inference from prompt
function pickStyle(prompt: string): "centered"|"band"|"left"|"wash" {
  const p = (prompt || "").toLowerCase();
  if (p.includes("band") || p.includes("stripe") || p.includes("bar")) return "band";
  if (p.includes("left") || p.includes("editorial") || p.includes("sidebar")) return "left";
  if (p.includes("wash") || p.includes("overlay") || p.includes("photo")) return "wash";
  if (p.includes("center") || p.includes("hero")) return "centered";
  return "centered";
}