import express from "express";
import { storage as db } from "../storage";
import { createCheckoutSession } from "../stripe/coverCheckout";
import { coverPurchases } from "@shared/schema";
import { and, eq, desc } from "drizzle-orm";

const router = express.Router();

// Authentication middleware - using existing auth system
const authenticate = (req: any, res: any, next: any) => {
  // For development, mock authentication
  if (process.env.NODE_ENV === 'development') {
    req.user = {
      id: 'dev-user-123',
      uid: 'dev-user-123',
      email: 'dev@example.com'
    };
    return next();
  }
  
  // In production, add proper Firebase token verification here
  if (!req.user) {
    return res.status(401).json({ error: "Authentication required" });
  }
  next();
};

const authenticateAdmin = (req: any, res: any, next: any) => {
  // Add admin check logic here - for now allow in development
  if (process.env.NODE_ENV === 'development') {
    return next();
  }
  // In production, add proper admin role verification
  if (!req.user?.isAdmin) {
    return res.status(403).json({ error: "Admin access required" });
  }
  next();
};

// Gallery with filters (SN's specification)
router.get("/api/cover-templates", async (req, res) => {
  try {
    const rows = await db.getCoverTemplates({
      q: req.query.q as string | undefined,
      category: req.query.category as string | undefined,
      top_tier: req.query.toptier as any,
      subcat: req.query.subcat as string | undefined,
      min: req.query.min ? parseInt(req.query.min as string, 10) : undefined,
      max: req.query.max ? parseInt(req.query.max as string, 10) : undefined,
    });
    res.json(rows);
  } catch (error) {
    console.error("[cover-templates] Gallery error:", error);
    res.status(500).json({ error: "Failed to fetch templates" });
  }
});

router.get("/api/cover-templates/:id", async (req, res) => {
  try {
    const tpl = await db.getCoverTemplate(req.params.id);
    if (!tpl || !tpl.isActive || tpl.approvalStatus !== 'approved') {
      return res.status(404).json({ error: "Not found" });
    }
    res.json(tpl); // includes cover_preview_url + divider1..3
  } catch (error) {
    console.error("[cover-templates] Single template error:", error);
    res.status(500).json({ error: "Failed to fetch template" });
  }
});

// Contributor upload: extend payload to include tier/subcats + 4 previews (SN's specification)
router.post("/api/creator/cover-templates", authenticate, async (req: any, res) => {
  try {
    const { 
      title, category, priceCents, previewUrl, downloadFile,
      top_tier, subcategories, cover_preview_url, divider1_preview_url, divider2_preview_url, divider3_preview_url 
    } = req.body || {};
    
    if (!title || !category || !previewUrl) {
      return res.status(400).json({ error: "Missing required fields" });
    }

    const rec = await db.createCoverTemplate({
      title,
      category,
      topTier: top_tier || 'General',
      subcategories: subcategories || [],
      priceCents: priceCents ?? 1499, // Default to global price
      previewUrl, // Card thumbnail
      coverPreviewUrl: cover_preview_url,
      divider1PreviewUrl: divider1_preview_url,
      divider2PreviewUrl: divider2_preview_url,
      divider3PreviewUrl: divider3_preview_url,
      downloadFile,
      isActive: false, // Awaiting approval
      approvalStatus: 'pending'
    });
    
    res.json({ success: true, template: rec });
  } catch (error) {
    console.error("[cover-templates] Creator upload error:", error);
    res.status(500).json({ error: "Failed to create template" });
  }
});

// Checkout: Buy As-Is (SN's updated version with purchaseId in success URL)
router.post("/api/cover-templates/:id/checkout", authenticate, async (req: any, res) => {
  try {
    const tpl = await db.getCoverTemplate(req.params.id);
    if (!tpl || !tpl.isActive || tpl.approvalStatus !== 'approved') {
      return res.status(404).json({ error: "Template not found" });
    }

    const { customImageUrl, applyProDiscount } = req.body || {};
    
    // 1) Create the DB purchase now (pending)
    const purchase = await db.createCoverPurchase({
      userId: req.user.id,
      templateId: tpl.id,
      customImageUrl: customImageUrl || null,
      status: 'pending'
    });

    // 2) Create Checkout; embed purchaseId so success page can poll
    const baseUrl = process.env.DOMAIN_URL || process.env.FRONTEND_URL || process.env.APP_BASE_URL || req.headers.origin || '';
    const successUrl = `${baseUrl}/business-assets/templates/cover-dividers?success=true&purchase=${purchase.id}`;
    const cancelUrl = `${baseUrl}/business-assets/templates/cover-dividers?cancel=true`;

    // Validate Pro discount if requested
    let coupon;
    if (applyProDiscount) {
      if (!process.env.STRIPE_COUPON_PRO_20) {
        return res.status(400).json({ error: "Pro discount not available" });
      }
      
      const user = await db.getUser(req.user.id);
      if (!user || !user.isPaid) {
        return res.status(403).json({ error: "Pro discount requires active Pro subscription" });
      }
      
      coupon = process.env.STRIPE_COUPON_PRO_20;
    }

    const session = await createCheckoutSession({
      successUrl,
      cancelUrl,
      coupon,
      metadata: { 
        templateId: tpl.id, 
        customImageUrl: customImageUrl || "",
        userId: req.user.id,
        purchaseId: purchase.id
      }
    });

    // 3) Backfill session id on the purchase
    // Note: stripeSessionId not stored in cover_purchases table - webhook handles purchase completion

    res.json({ checkoutUrl: session.url, purchaseId: purchase.id });
  } catch (error) {
    console.error("[cover-templates] Checkout error:", error);
    res.status(500).json({ error: "Failed to create checkout session" });
  }
});

// Get user's purchases for backward compatibility
router.get("/api/covers/purchases", authenticate, async (req: any, res) => {
  try {
    const purchases = await db.getUserCoverPurchases(req.user.id);
    res.json(purchases);
  } catch (error) {
    console.error("[cover-templates] Purchases error:", error);
    res.status(500).json({ error: "Failed to fetch purchases" });
  }
});

// Legacy template endpoint for backward compatibility
router.get("/api/covers/templates", async (req, res) => {
  try {
    const rows = await db.getCoverTemplates({
      q: req.query.q as string | undefined,
      category: req.query.category as string | undefined,
      top_tier: req.query.toptier as any,
      subcat: req.query.subcat as string | undefined,
      min: req.query.min ? parseInt(req.query.min as string, 10) : undefined,
      max: req.query.max ? parseInt(req.query.max as string, 10) : undefined,
    });
    res.json(rows);
  } catch (error) {
    console.error("[cover-templates] Legacy templates error:", error);
    res.status(500).json({ error: "Failed to fetch templates" });
  }
});

// Admin endpoints
router.get("/api/admin/cover-templates", authenticateAdmin, async (req, res) => {
  try {
    const q = (req.query.q as string) || undefined;
    const category = (req.query.category as string) || undefined;
    const min = req.query.min ? parseInt(req.query.min as string, 10) : undefined;
    const max = req.query.max ? parseInt(req.query.max as string, 10) : undefined;
    const includeInactive = req.query.includeInactive === 'true';

    // Admin can see all templates including inactive ones
    const rows = await db.getCoverTemplates({ 
      q, category, min, max, 
      top_tier: req.query.toptier as any,
      subcat: req.query.subcat as string | undefined,
      includeInactive 
    });
    res.json(rows);
  } catch (error) {
    console.error("[admin-cover-templates] Gallery error:", error);
    res.status(500).json({ error: "Failed to fetch templates" });
  }
});

router.patch("/api/admin/cover-templates/:id/activate", authenticateAdmin, async (req, res) => {
  try {
    const { active } = req.body || {};
    const updated = await db.updateCoverTemplate(req.params.id, { isActive: !!active });
    
    if (!updated) {
      return res.status(404).json({ error: "Template not found" });
    }
    
    res.json({ success: true, isActive: updated.isActive });
  } catch (error) {
    console.error("[admin-cover-templates] Activation error:", error);
    res.status(500).json({ error: "Failed to update template" });
  }
});

// Has this user purchased this template? (SN's exact implementation)
router.get("/api/cover-templates/:id/my-status", authenticate, async (req: any, res) => {
  try {
    const purchases = await db.db.select().from(coverPurchases)
      .where(and(
        eq(coverPurchases.userId, req.user.id),
        eq(coverPurchases.templateId, req.params.id),
        eq(coverPurchases.status, "paid")
      ))
      .orderBy(desc(coverPurchases.createdAt))
      .limit(1);

    if (!purchases.length) {
      return res.json({ owned: false });
    }

    return res.json({ 
      owned: true, 
      download_url: purchases[0].downloadUrl, 
      purchase_id: purchases[0].id 
    });
  } catch (error) {
    console.error("[cover-templates] Ownership check error:", error);
    res.status(500).json({ error: "Failed to check ownership" });
  }
});

// Secure download: only for owners (SN's exact implementation)
router.get("/api/cover-templates/:id/download", authenticate, async (req: any, res) => {
  try {
    const purchase = await db.db.select().from(coverPurchases)
      .where(and(
        eq(coverPurchases.userId, req.user.id),
        eq(coverPurchases.templateId, req.params.id),
        eq(coverPurchases.status, "paid")
      ))
      .orderBy(desc(coverPurchases.createdAt))
      .limit(1);

    if (!purchase.length || !purchase[0].downloadUrl) {
      return res.status(403).json({ error: "Not purchased" });
    }

    // Option A: redirect to signed URL (SN's preferred approach)
    return res.redirect(purchase[0].downloadUrl);
  } catch (error) {
    console.error("[cover-templates] Download error:", error);
    res.status(500).json({ error: "Failed to process download" });
  }
});

// Lookup a specific purchase (used by success page polling) - SN's exact implementation
router.get("/api/purchases/:id", authenticate, async (req: any, res) => {
  try {
    const purchase = await db.db.select().from(coverPurchases)
      .where(and(
        eq(coverPurchases.id, req.params.id),
        eq(coverPurchases.userId, req.user.id)
      ))
      .limit(1);
    
    if (!purchase.length) {
      return res.status(404).json({ error: "Not found" });
    }
    
    // Return with snake_case for client compatibility
    res.json({
      ...purchase[0],
      download_url: purchase[0].downloadUrl
    });
  } catch (error) {
    console.error("[cover-templates] Purchase lookup error:", error);
    res.status(500).json({ error: "Failed to fetch purchase" });
  }
});

// List my purchases (SN's My Purchases page endpoint)
router.get("/api/me/cover-purchases", authenticate, async (req: any, res) => {
  try {
    const purchases = await db.db.select().from(coverPurchases)
      .where(eq(coverPurchases.userId, req.user.id))
      .orderBy(desc(coverPurchases.createdAt));
    res.json(purchases);
  } catch (error) {
    console.error("[cover-templates] My purchases error:", error);
    res.status(500).json({ error: "Failed to fetch purchases" });
  }
});

export default router;