import { Router } from 'express';
import formidable from 'formidable';
import { businessPlanTemplateService } from '../services/businessPlanTemplates';
import { previewGenerator } from '../services/businessPlanPreviewGenerator';
import { nanoid } from 'nanoid';

const router = Router();

/**
 * Upload business plan template to Firebase
 * POST /api/bp-templates-firebase/upload
 */
router.post('/upload', async (req, res) => {
  const form = formidable({ keepExtensions: true, maxFileSize: 50 * 1024 * 1024 });

  form.parse(req, async (err, fields, files) => {
    if (err) {
      return res.status(400).json({ error: err.message });
    }

    try {
      // Extract form data
      const title = String(fields.title || '');
      const category = String(fields.category || '');
      const tags = String(fields.tags || '')
        .split(',')
        .map((t) => t.trim())
        .filter(Boolean);
      const sections = String(fields.sections || '')
        .split('\n')
        .map((s) => s.trim())
        .filter(Boolean);

      console.log('📋 Upload form data:', { title, category, tags, sections: sections.length });

      if (!title || !category) {
        return res.status(400).json({ error: 'Title and category are required' });
      }

      // Get DOCX file (formidable v3 returns files as arrays or single file)
      const docxFileRaw = files.docx;
      const docxFile = Array.isArray(docxFileRaw) ? docxFileRaw[0] : docxFileRaw;
      
      // Get PDF file
      const pdfFileRaw = files.pdf;
      const pdfFile = Array.isArray(pdfFileRaw) ? pdfFileRaw[0] : pdfFileRaw;
      
      console.log('📁 Upload debug:', {
        hasDocxRaw: !!docxFileRaw,
        hasPdfRaw: !!pdfFileRaw,
        isArray: Array.isArray(docxFileRaw),
        docxFile: docxFile ? {
          originalFilename: docxFile.originalFilename,
          newFilename: docxFile.newFilename,
          mimetype: docxFile.mimetype,
          size: docxFile.size
        } : null,
        pdfFile: pdfFile ? {
          originalFilename: pdfFile.originalFilename,
          newFilename: pdfFile.newFilename,
          mimetype: pdfFile.mimetype,
          size: pdfFile.size
        } : null,
        allFileKeys: Object.keys(files)
      });
      
      if (!docxFile) {
        return res.status(400).json({ error: 'DOCX file is required' });
      }

      if (!pdfFile) {
        return res.status(400).json({ error: 'PDF file is required' });
      }

      // Validate DOCX file (check file extension)
      const docxFilename = docxFile.originalFilename || docxFile.newFilename || '';
      const isDocx = docxFilename.toLowerCase().endsWith('.docx');
      
      if (!isDocx) {
        return res.status(400).json({ 
          error: 'File must be a valid DOCX document',
          debug: { filename: docxFilename, isDocx }
        });
      }

      // Validate PDF file
      const pdfFilename = pdfFile.originalFilename || pdfFile.newFilename || '';
      const isPdf = pdfFilename.toLowerCase().endsWith('.pdf');
      
      if (!isPdf) {
        return res.status(400).json({ 
          error: 'File must be a valid PDF document',
          debug: { filename: pdfFilename, isPdf }
        });
      }

      // Read file buffers
      const fs = await import('fs');
      const docxBuffer = fs.readFileSync(docxFile.filepath);
      const pdfBuffer = fs.readFileSync(pdfFile.filepath);

      // Generate slug and version (SN's spec)
      const slug = title.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-|-$/g, '');
      const version = 'v1';  // SN's spec: string version like "v1", "v2"

      // Create initial metadata (SN's exact schema)
      const metadata = {
        slug,
        title,
        category,
        isActive: true,
        isMaster: false,  // Will be set via transaction
        currentVersion: version,
        storagePaths: {
          docx: '',
          pdf: '',
          preview: '',
          thumb: ''
        },
        tags,
        sections: sections.length > 0 ? sections : ['Executive Summary', 'Market Analysis', 'Financial Projections'],
        createdAt: null,  // Will be set to serverTimestamp
        updatedAt: null
      };

      // Save initial metadata
      await businessPlanTemplateService.saveMetadata(metadata);

      // Process asynchronously
      processTemplate(slug, version, title, category, tags, sections, docxBuffer, pdfBuffer).catch(console.error);

      res.json({
        ok: true,
        slug,
        message: 'Template upload started. Processing previews...'
      });
    } catch (error: any) {
      console.error('Business plan template upload error:', error);
      res.status(500).json({ error: error?.message || 'Upload failed' });
    }
  });
});

/**
 * Get all templates from Firestore
 * GET /api/bp-templates-firebase/list
 */
router.get('/list', async (req, res) => {
  try {
    const templates = await businessPlanTemplateService.listTemplates();
    res.json({ items: templates });
  } catch (error: any) {
    console.error('Error fetching templates:', error);
    res.status(500).json({ error: error?.message || 'Failed to fetch templates' });
  }
});

/**
 * Get specific template metadata
 * GET /api/bp-templates-firebase/:slug
 */
router.get('/:slug', async (req, res) => {
  try {
    const { slug } = req.params;
    const metadata = await businessPlanTemplateService.getMetadata(slug);
    
    if (!metadata) {
      return res.status(404).json({ error: 'Template not found' });
    }
    
    res.json(metadata);
  } catch (error: any) {
    console.error('Error fetching template:', error);
    res.status(500).json({ error: error?.message || 'Failed to fetch template' });
  }
});

/**
 * Process template: upload files, generate previews, update metadata
 * Implements SN's exact specification with hierarchical Firestore structure
 */
async function processTemplate(
  slug: string,
  version: string,
  title: string,
  category: string,
  tags: string[],
  sections: string[],
  docxBuffer: Buffer,
  pdfBuffer: Buffer
): Promise<void> {
  try {
    // 1. Upload DOCX to Storage (SN's path structure)
    const docxPath = await businessPlanTemplateService.uploadDocx(slug, version, docxBuffer);

    // 2. Upload PDF to Storage
    const pdfPath = await businessPlanTemplateService.uploadPdf(slug, version, pdfBuffer);

    // 3. Generate preview images (WebP format per SN's spec)
    const { preview, thumb } = await previewGenerator.generatePreviewPages(
      title,
      category,
      sections
    );

    // 4. Upload preview and thumbnail images
    const previewPath = await businessPlanTemplateService.uploadPreviewImage(slug, version, preview);
    const thumbPath = await businessPlanTemplateService.uploadThumbImage(slug, version, thumb);

    // 5. Update main template document with storagePaths
    const { FieldValue } = await import('firebase-admin/firestore');
    const updatedMetadata = {
      slug,
      title,
      category,
      isActive: true,
      isMaster: false,  // Set via transaction below
      currentVersion: version,
      storagePaths: {
        docx: docxPath,
        pdf: pdfPath,
        preview: previewPath,
        thumb: thumbPath
      },
      tags,
      sections,
      createdAt: FieldValue.serverTimestamp(),
      updatedAt: FieldValue.serverTimestamp()
    };

    await businessPlanTemplateService.saveMetadata(updatedMetadata);

    // 6. Save version subdocument (isMaster: false by default, transaction will set true)
    const versionData = {
      version,
      isMaster: false,  // Will be set to true by setMasterTemplate transaction
      notes: `Initial release of ${title}`,
      storagePaths: {
        docx: docxPath,
        pdf: pdfPath,
        preview: previewPath,
        thumb: thumbPath
      },
      sections,
      createdAt: FieldValue.serverTimestamp()
    };

    await businessPlanTemplateService.saveVersion(slug, versionData);

    // 7. Run atomic transaction to set as master template
    await businessPlanTemplateService.setMasterTemplate(slug, version);

    console.log(`✅ Template ${slug} ${version} processed and set as master successfully`);
  } catch (error: any) {
    console.error(`❌ Error processing template ${slug}:`, error);
    // Note: Error handling simplified - in production, update a status field
  }
}

/**
 * Update business plan template metadata
 * PATCH /api/bp-templates-firebase/admin/:slug
 */
router.patch('/:slug', async (req, res) => {
  try {
    const { slug } = req.params;
    const { title, category, industries } = req.body;
    
    if (!slug) {
      return res.status(400).json({ error: 'Slug is required' });
    }
    
    const updates: any = {};
    if (title) updates.title = title;
    if (category) updates.category = category;
    if (industries) {
      // industries can be comma-separated string or array
      updates.tags = typeof industries === 'string' 
        ? industries.split(',').map(i => i.trim()).filter(Boolean)
        : industries;
    }
    
    await businessPlanTemplateService.updateMetadata(slug, updates);
    
    res.json({
      ok: true,
      message: `Template ${slug} updated successfully`
    });
  } catch (error: any) {
    console.error('Error updating template:', error);
    res.status(500).json({ error: error?.message || 'Failed to update template' });
  }
});

/**
 * Delete business plan template
 * DELETE /api/bp-templates-firebase/admin/:slug
 */
router.delete('/:slug', async (req, res) => {
  try {
    const { slug } = req.params;
    
    if (!slug) {
      return res.status(400).json({ error: 'Slug is required' });
    }
    
    await businessPlanTemplateService.deleteTemplate(slug);
    
    res.json({
      ok: true,
      message: `Template ${slug} deleted successfully`
    });
  } catch (error: any) {
    console.error('Error deleting template:', error);
    res.status(500).json({ error: error?.message || 'Failed to delete template' });
  }
});

export default router;
