import express, { Request, Response } from "express";
import {
  listBusinessPlanTemplates,
  getBusinessPlanTemplateBySlug,
  getMasterBusinessPlanTemplate
} from "../services/businessPlanTemplates";
import { getDownloadUrlAuto, getViewerUrlAuto } from "../services/downloadUrl";
import { logDownloadEvent } from "../services/analytics";

const router = express.Router();

// GET /api/bp-templates-firebase/list
router.get("/list", async (_req: Request, res: Response) => {
  try {
    const items = await listBusinessPlanTemplates();
    
    // Convert storage paths to full URLs
    const dto = await Promise.all(items.map(async (t) => {
      const previewUrl = t.storagePaths?.preview 
        ? await getDownloadUrlAuto(t.storagePaths.preview)
        : null;
      const thumbUrl = t.storagePaths?.thumb
        ? await getDownloadUrlAuto(t.storagePaths.thumb)
        : null;
      const docxUrl = t.storagePaths?.docx
        ? await getDownloadUrlAuto(t.storagePaths.docx)
        : null;
      const pdfUrl = t.storagePaths?.pdf
        ? await getDownloadUrlAuto(t.storagePaths.pdf)
        : null;
      const viewerUrl = t.storagePaths?.pdf
        ? await getViewerUrlAuto(t.storagePaths.pdf)
        : (t.storagePaths?.docx ? await getViewerUrlAuto(t.storagePaths.docx) : null);
      
      return {
        title: t.title,
        slug: t.slug,
        category: t.category,
        tags: t.tags || [],
        industries: t.industries || [],
        sections: t.sections || [],
        isMaster: !!t.isMaster,
        currentVersion: t.currentVersion,
        storagePaths: {
          preview: previewUrl,
          thumb: thumbUrl,
          docx: docxUrl,
          pdf: pdfUrl
        },
        viewerUrl,
        updatedAt: t.updatedAt || null
      };
    }));
    
    // cache list briefly
    res.set("Cache-Control", "public, max-age=60");
    res.json(dto);
  } catch (err: any) {
    console.error("[public:list] error", err);
    res.status(500).json({ error: "Failed to list templates" });
  }
});

// GET /api/bp-templates-firebase/master
router.get("/master", async (_req: Request, res: Response) => {
  try {
    const t = await getMasterBusinessPlanTemplate();
    if (!t) return res.status(404).json({ error: "No master template found" });
    
    // Save original paths for viewerUrl generation
    const originalPdfPath = t.storagePaths?.pdf;
    const originalDocxPath = t.storagePaths?.docx;
    
    // Convert storage paths to full URLs
    if (t.storagePaths?.preview) {
      t.storagePaths.preview = await getDownloadUrlAuto(t.storagePaths.preview);
    }
    if (t.storagePaths?.thumb) {
      t.storagePaths.thumb = await getDownloadUrlAuto(t.storagePaths.thumb);
    }
    if (t.storagePaths?.docx) {
      t.storagePaths.docx = await getDownloadUrlAuto(t.storagePaths.docx);
    }
    if (t.storagePaths?.pdf) {
      t.storagePaths.pdf = await getDownloadUrlAuto(t.storagePaths.pdf);
    }
    
    // Add viewer URL for inline preview (prefer PDF, fallback to DOCX)
    const viewerUrl = originalPdfPath
      ? await getViewerUrlAuto(originalPdfPath)
      : (originalDocxPath ? await getViewerUrlAuto(originalDocxPath) : null);
    
    res.set("Cache-Control", "public, max-age=60");
    res.json({ ...t, viewerUrl });
  } catch (err: any) {
    console.error("[public:master] error", err);
    res.status(500).json({ error: "Failed to resolve master template" });
  }
});

// GET /api/bp-templates-firebase/:slug
router.get("/:slug", async (req: Request, res: Response) => {
  try {
    const t = await getBusinessPlanTemplateBySlug(req.params.slug);
    if (!t) return res.status(404).json({ error: "Template not found" });
    
    // Save original paths for viewerUrl generation
    const originalPdfPath = t.storagePaths?.pdf;
    const originalDocxPath = t.storagePaths?.docx;
    
    // Convert storage paths to full URLs
    if (t.storagePaths?.preview) {
      t.storagePaths.preview = await getDownloadUrlAuto(t.storagePaths.preview);
    }
    if (t.storagePaths?.thumb) {
      t.storagePaths.thumb = await getDownloadUrlAuto(t.storagePaths.thumb);
    }
    if (t.storagePaths?.docx) {
      t.storagePaths.docx = await getDownloadUrlAuto(t.storagePaths.docx);
    }
    if (t.storagePaths?.pdf) {
      t.storagePaths.pdf = await getDownloadUrlAuto(t.storagePaths.pdf);
    }
    
    // Add viewer URL for inline preview (prefer PDF, fallback to DOCX)
    const viewerUrl = originalPdfPath
      ? await getViewerUrlAuto(originalPdfPath)
      : (originalDocxPath ? await getViewerUrlAuto(originalDocxPath) : null);
    
    res.set("Cache-Control", "public, max-age=60");
    res.json({ ...t, viewerUrl });
  } catch (err: any) {
    console.error("[public:getBySlug] error", err);
    res.status(500).json({ error: "Failed to fetch template" });
  }
});

// GET /api/bp-templates-firebase/:slug/download
// Auto-detects if public URLs work, falls back to signed URLs
router.get("/:slug/download", async (req: Request, res: Response) => {
  try {
    const t = await getBusinessPlanTemplateBySlug(req.params.slug);
    if (!t) return res.status(404).json({ error: "Template not found" });

    const docxPath = t.storagePaths?.docx;
    if (!docxPath) return res.status(400).json({ error: "Template has no DOCX path" });

    // Smart auto-detection: tries public first, falls back to signed if needed
    const url = await getDownloadUrlAuto(docxPath);
    
    // Determine URL mode for analytics
    const mode = url.includes("token=") ? "signed" : "public";

    // Log download event (non-blocking)
    logDownloadEvent({
      slug: t.slug,
      version: t.currentVersion || "1.0",
      mode,
      userAgent: req.headers["user-agent"],
      referrer: req.headers.referer || req.headers.referrer,
      ip: req.ip || req.socket.remoteAddress
    }).catch(err => console.error("[Analytics] Log failed:", err));

    // short cache (URL is versioned by path)
    res.set("Cache-Control", "public, max-age=60");
    return res.redirect(302, url);
  } catch (err: any) {
    console.error("[public:download] error", err);
    res.status(500).json({ error: "Failed to generate download URL" });
  }
});

// DEBUG: List ALL files for a template
router.get("/:slug/debug-all", async (req: Request, res: Response) => {
  try {
    const { bucket } = await import("../firebaseAdmin");
    const slug = req.params.slug;
    const prefix = `templates/business-plan/`;
    const [files] = await bucket.getFiles({ prefix });
    
    const related = files.filter(f => f.name.includes(slug)).map(f => ({
      name: f.name,
      publicUrl: `https://storage.googleapis.com/${bucket.name}/${f.name}`
    }));
    
    res.json({ count: related.length, files: related });
  } catch (err: any) {
    res.status(500).json({ error: err.message });
  }
});

export default router;
