import { Router } from "express";
import formidable, { File } from "formidable";
import fs from "fs";
import path from "path";

const router = Router();

// Helper to create directory if it doesn't exist
function ensureDir(dir: string) {
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
}

// Helper to sanitize ID (kebab-case)
function sanitizeId(raw: string): string {
  return raw
    .toLowerCase()
    .replace(/[^a-z0-9\-]/g, "-")
    .replace(/-+/g, "-")
    .replace(/^-|-$/g, "");
}

interface BPManifestItem {
  id: string;
  name: string;
  category?: string;
  tags?: string[];
  previewUrl: string;
  docxUrl: string;
  updatedAt: string;
  sections?: string[];
}

interface BPManifest {
  collection: "business-plan";
  version: number;
  items: BPManifestItem[];
}

// POST /api/bp-templates/upload
router.post("/upload", (req, res) => {
  const form = formidable({ keepExtensions: true, maxFileSize: 50 * 1024 * 1024 }); // 50MB
  
  form.parse(req, (err, fields, files) => {
    if (err) return res.status(400).json({ error: err.message });

    const id = sanitizeId(String(fields.id || ""));
    const name = String(fields.name || "");
    const category = String(fields.category || "");
    const tags = String(fields.tags || "")
      .split(",")
      .map((t) => t.trim())
      .filter(Boolean);
    const sections = String(fields.sections || "")
      .split("\n")
      .map((s) => s.trim())
      .filter(Boolean);

    if (!id || !name) {
      return res.status(400).json({ error: "id and name are required" });
    }

    const docx = files.docx as File | undefined;
    const preview = files.preview as File | undefined;
    
    if (!docx || !preview) {
      return res.status(400).json({ error: "docx and preview files are required" });
    }

    // Validate DOCX file
    const docxOk =
      (docx.mimetype && docx.mimetype === "application/vnd.openxmlformats-officedocument.wordprocessingml.document") ||
      (docx.originalFilename && docx.originalFilename.toLowerCase().endsWith(".docx"));
    if (!docxOk) {
      return res.status(400).json({ error: "DOCX file must be a valid Word document" });
    }

    // Validate preview file
    const previewOk =
      (preview.mimetype && /image\/(png|jpeg)/i.test(preview.mimetype)) ||
      (preview.originalFilename && /\.(png|jpe?g)$/i.test(preview.originalFilename || ""));
    if (!previewOk) {
      return res.status(400).json({ error: "Preview must be PNG or JPG" });
    }

    try {
      // Create directories
      const templatesDir = path.join(process.cwd(), "public", "templates", "business-plan");
      const docsDir = path.join(templatesDir, "docs");
      const previewsDir = path.join(templatesDir, "previews");
      const dataDir = path.join(process.cwd(), "public", "site", "data");
      
      ensureDir(docsDir);
      ensureDir(previewsDir);
      ensureDir(dataDir);

      // Copy files with proper naming
      const docxExt = (docx.originalFilename || "").split(".").pop() || "docx";
      const previewExt = (preview.originalFilename || "").split(".").pop() || "jpg";
      
      const docxFileName = `${id}.${docxExt}`;
      const previewFileName = `${id}.${previewExt}`;
      
      const docxPathAbs = path.join(docsDir, docxFileName);
      const previewPathAbs = path.join(previewsDir, previewFileName);

      fs.copyFileSync(docx.filepath, docxPathAbs);
      fs.copyFileSync(preview.filepath, previewPathAbs);

      // Update manifest
      const manifestPath = path.join(dataDir, "manifest.bp.json");
      
      let manifest: BPManifest = {
        collection: "business-plan",
        version: 1,
        items: []
      };

      if (fs.existsSync(manifestPath)) {
        try {
          const data = JSON.parse(fs.readFileSync(manifestPath, "utf8"));
          if (data && data.collection === "business-plan" && Array.isArray(data.items)) {
            manifest = data;
          }
        } catch {
          // Keep default manifest
        }
      }

      const docxUrl = `/templates/business-plan/docs/${docxFileName}`;
      const previewUrl = `/templates/business-plan/previews/${previewFileName}`;

      const item: BPManifestItem = {
        id,
        name,
        category: category || undefined,
        tags: tags.length > 0 ? tags : undefined,
        previewUrl,
        docxUrl,
        updatedAt: new Date().toISOString().split("T")[0], // YYYY-MM-DD
        sections: sections.length > 0 ? sections : undefined,
      };

      // Update or add item
      const idx = manifest.items.findIndex((m) => m.id === id);
      if (idx >= 0) {
        manifest.items[idx] = item;
      } else {
        manifest.items.push(item);
      }

      // Sort items by name
      manifest.items.sort((a, b) => a.name.localeCompare(b.name));

      fs.writeFileSync(manifestPath, JSON.stringify(manifest, null, 2), "utf8");
      
      res.json({ ok: true, item });
    } catch (error: any) {
      console.error("Business plan template upload error:", error);
      res.status(500).json({ error: error?.message || "Upload failed" });
    }
  });
});

export default router;