import { Router } from "express";
import multer from "multer";
import { parse } from "csv-parse/sync";
import { loadManifest, saveManifest, upsert, kebab, BPItem } from "../../client/src/lib/bp-io";

const upload = multer();
const router = Router();

router.post("/import-csv", upload.single("csv"), (req, res) => {
  try {
    if (!req.file) return res.status(400).json({ error: "CSV file required" });
    const csv = req.file.buffer.toString("utf8");
    const rows: any[] = parse(csv, { columns: true, skip_empty_lines: true, trim: true });

    const manifest = loadManifest();
    const today = new Date().toISOString().slice(0,10);

    let count = 0;
    for (const r of rows) {
      const id = kebab(r.id || r.name);
      if (!id) continue;
      const item: BPItem = {
        id,
        name: r.name || `Business Plan — ${id}`,
        category: r.category || "General",
        tags: (r.tags || "").split("|").map((t: string)=>t.trim()).filter(Boolean),
        previewUrl: r.previewUrl,
        docxUrl: r.docxUrl,
        updatedAt: r.updatedAt || today,
        sections: (r.sections || "").split("|").map((t: string)=>t.trim()).filter(Boolean),
      };
      if (!item.previewUrl || !item.docxUrl) continue;
      upsert(manifest.items, item);
      count++;
    }

    saveManifest(manifest);
    res.json({ ok: true, imported: count, total: manifest.items.length });
  } catch (e:any) {
    console.error(e);
    res.status(500).json({ error: e?.message || "Server error" });
  }
});

export default router;