import type { Request, Response } from "express";
import express from "express";
import Stripe from "stripe";

// Set these in your Replit / env vars
// VITE_STRIPE_PUBLISHABLE_KEY (client)
// STRIPE_SECRET_KEY (server)
const stripe = new Stripe(process.env.STRIPE_SECRET_KEY as string, {
  apiVersion: "2024-06-20",
});

const router = express.Router();

/**
 * POST /api/billing/create-subscription
 * Body: { priceId: string, email?: string }
 * Returns: { clientSecret: string, subscriptionId: string }
 */
router.post("/create-subscription", async (req: Request, res: Response) => {
  try {
    const { priceId, email } = req.body as { priceId: string; email?: string | null };
    if (!priceId) return res.status(400).json({ error: "Missing priceId" });

    // 1) Find or create customer. Replace with your user->customer mapping.
    let customer: Stripe.Customer;
    if (email) {
      const existing = await stripe.customers.list({ email, limit: 1 });
      customer = existing.data[0] ?? await stripe.customers.create({ email });
    } else {
      customer = await stripe.customers.create();
    }

    // 2) Create the subscription in incomplete state so we can collect payment
    const subscription = await stripe.subscriptions.create({
      customer: customer.id,
      items: [{ price: priceId }],
      payment_behavior: "default_incomplete",
      expand: ["latest_invoice.payment_intent"],
    });

    const paymentIntent = (subscription.latest_invoice as Stripe.Invoice).payment_intent as Stripe.PaymentIntent;
    if (!paymentIntent?.client_secret) {
      return res.status(500).json({ error: "No client secret on payment intent." });
    }

    return res.json({
      clientSecret: paymentIntent.client_secret,
      subscriptionId: subscription.id,
    });
  } catch (err: any) {
    console.error("create-subscription failed", err);
    return res.status(500).json({ error: err.message || "Stripe error" });
  }
});

export default router;