import type { Express, Request, Response, NextFunction } from "express";
import express from "express";
import { createServer, type Server } from "http";
import os from "os";
import { performance } from "perf_hooks";
import { Readable } from "stream";
import { EventEmitter } from "events";
import { storage } from "./storage";
import adminRoutes from "./admin/routes";
import iconsRoutes from "./routes/icons";
import iconLibraryRoutes from "./routes/iconLibraryRoutes";
import { requireAdmin } from "./middleware/adminGuard";
import { requireAnyAdmin } from "./middleware/authz";
import stockRoutes from "./routes/stock";
import stockLibraryRoutes from "./routes/stock-library";
import pptRoutes from "./routes/ppt";
import infographicsRoutes from "./routes/infographics";
import presentationTemplatesRoutes from "./routes/presentationTemplates";
import logoTemplatesRoutes from "./routes/logoTemplates";
import logoTemplatesPublicRoutes from "./routes/logoTemplatesPublic";
import bpTemplatesRoutes from "./routes/bpTemplates";
import bpImportCsv from "./routes/bpImportCsv";
import bpTemplatesFirebaseRoutes from "./routes/businessPlanTemplatesFirebase";
import bpTemplatesPublicRoutes from "./routes/bpTemplatesPublic";
import healthBpTemplates from "./routes/healthBpTemplates";
import { ObjectStorageService, ObjectNotFoundError, objectStorageClient } from "./objectStorage";
import { getFirebaseAdmin } from "./admin/firebaseAdmin";
import { trackVisitor } from "./middleware/visitorTracking";
import multer from "multer";
import { authenticator } from "otplib";
import * as crypto from "crypto";
import QRCode from "qrcode";
import { 
  insertUserSchema, 
  insertBrandKitSchema, 
  insertBusinessNameSchema, 
  insertSocialMediaKitSchema, 
  insertWebsiteTemplateSchema,
  insertUserTemplateCustomizationSchema,
  insertCancellationSchema,
  insertPauseSchema,
  insertDomainOrderSchema,
  insertCreatorSchema,
  type DomainSearchResult,
  type DomainSearchHit,
  type DomainCredit,
  type InsertDomainCredit,
  type Creator,
  type InsertCreator,
  CREATOR_ONBOARDING_STATUSES,
  CREATOR_MIN_PRICE_CENTS,
  CREATOR_MAX_PRICE_CENTS,
  type Asset,
  type InsertAsset,
  type CreatorAsset
} from "@shared/schema";
import { generateBusinessNames } from "./ai-business-names";
import { generateSlogans } from "./ai-slogan";
import { generateBusinessPlan } from "./ai-plan";
import { processAiSectionJob } from "./ai-section";
import OpenAI from "openai";
import { exec } from "child_process";
import { promises as fs } from "fs";
import fsSync from "fs";
import path from "path";
import { renderPlanPDF, renderPlanDOCX, inferBusinessName, slugify, type PlanResponse } from "./services/exportTemplates";
import { z } from "zod";
import { notificationBroadcaster } from "./events/notifications";
import { streamTokenService } from "./services/streamTokens";
import Stripe from "stripe";
import { opensrs } from "./domains/providers/opensrs";
import { jobQueue, type DomainRegistrationJobData } from "./jobs";
import { getDomainPricing } from "./domains/pricing";
import { applyWatermark, toTenDigitCode } from "./services/watermark";
import sharp from "sharp";
import { BUILD } from "./version";
// Note: getQuotaByLookupKey and hasQuota are now handled by entitlements.js


// Note: Firebase Admin now handled by separate module

// Lazy Stripe initialization - only when key is available
let stripe: Stripe | null = null;
function getStripe(): Stripe | null {
  if (stripe) return stripe;
  
  const stripeKey = process.env.STRIPE_SECRET_KEY;
  if (!stripeKey) {
    console.warn('STRIPE_SECRET_KEY not found - Stripe functionality will be disabled');
    return null;
  }
  
  try {
    stripe = new Stripe(stripeKey, {
      apiVersion: "2023-10-16",
    });
    return stripe;
  } catch (error) {
    console.error('Failed to initialize Stripe:', error);
    return null;
  }
}

// Helper function to build secure redirect URLs from environment variables
function buildSecureRedirectUrl(path: string, origin?: string): string {
  // Use strict allowlist of base URLs from environment
  const allowedBaseUrls = [
    process.env.DOMAIN_URL,
    process.env.FRONTEND_URL,
    process.env.VITE_BASE_URL,
    origin // Use request origin as fallback if provided
  ].filter(Boolean);
  
  // CRITICAL: Always return absolute URL for Stripe compatibility
  // In production, if no env vars are set, use the origin header as last resort
  // In development, fallback to localhost
  let baseUrl = allowedBaseUrls[0];
  
  if (!baseUrl) {
    if (process.env.NODE_ENV === 'development') {
      baseUrl = 'http://localhost:5000';
    } else {
      console.warn('[buildSecureRedirectUrl] No DOMAIN_URL/FRONTEND_URL configured, using empty baseUrl - Stripe may fail');
      baseUrl = ''; // Will cause Stripe to fail, but better than exposing localhost in prod
    }
  }
  
  // Ensure path starts with /
  const cleanPath = path.startsWith('/') ? path : `/${path}`;
  
  return `${baseUrl}${cleanPath}`;
}

// Helper function to validate state transitions for creator onboarding
function isValidStateTransition(fromStatus: string, toStatus: string): boolean {
  const validTransitions: Record<string, string[]> = {
    [CREATOR_ONBOARDING_STATUSES.PENDING]: [CREATOR_ONBOARDING_STATUSES.IN_PROGRESS],
    [CREATOR_ONBOARDING_STATUSES.IN_PROGRESS]: [CREATOR_ONBOARDING_STATUSES.COMPLETED, CREATOR_ONBOARDING_STATUSES.PENDING], // Can go back to pending if issues
    [CREATOR_ONBOARDING_STATUSES.COMPLETED]: [], // Completed is final state
    [CREATOR_ONBOARDING_STATUSES.REJECTED]: [CREATOR_ONBOARDING_STATUSES.PENDING] // Can be re-approved
  };
  
  return validTransitions[fromStatus]?.includes(toStatus) ?? false;
}

// Helper function to return proper 503 error for missing Stripe configuration
function handleStripeUnavailable(res: Response, context: string = 'this operation'): void {
  res.status(503).json({
    error: 'Payment processing service temporarily unavailable',
    message: 'Stripe Connect is not properly configured. Please contact support or try again later.',
    details: `${context} requires Stripe Connect integration`,
    code: 'STRIPE_UNAVAILABLE',
    retryable: true
  });
}

// SuperNova Domain Credit Configuration
const DEFAULT_DOMAIN_CREDIT_CAP_CENTS = Number(process.env.FREE_DOMAIN_WHOLESALE_CAP || 1500); // $15.00
const DEFAULT_ELIGIBLE_TLDS = (process.env.FREE_DOMAIN_TLDS || ".com,.net,.org,.co")
  .split(",").map(s => s.trim().toLowerCase()).filter(Boolean);
const CREDIT_EXPIRY_DAYS = Number(process.env.FREE_DOMAIN_CREDIT_EXPIRY_DAYS || 60);

/**
 * Check if user has received a domain credit in the last 12 months
 * SuperNova's business rule: One credit per 12 months per user
 */
async function hasCreditInLast12Months(userId: string): Promise<boolean> {
  try {
    const twelveMonthsAgo = new Date();
    twelveMonthsAgo.setFullYear(twelveMonthsAgo.getFullYear() - 1);
    
    const credits = await storage.getDomainCreditsByUserId(userId);
    return credits.some(credit => new Date(credit.issuedAt) >= twelveMonthsAgo);
  } catch (error) {
    console.error('Error checking credit history:', error);
    return true; // Err on the side of caution
  }
}

/**
 * Check if a Stripe subscription is eligible for domain credit (Pro subscription)
 * Validates against configured Pro plan price IDs
 */
function isEligibleSubscription(subscription: Stripe.Subscription): boolean {
  if (!subscription.items?.data) return false;
  
  const priceIds = {
    monthly: process.env.STRIPE_MONTHLY_PRICE_ID,
    yearly: process.env.STRIPE_YEARLY_PRICE_ID
  };
  
  // Check if any subscription item has a Pro plan price ID
  return subscription.items.data.some(item => {
    const priceId = item.price.id;
    return priceId === priceIds.monthly || priceId === priceIds.yearly;
  });
}

// Note: updateUserQuotaFromSubscription removed - quota updates now handled by entitlements.js system

/**
 * Check if domain credit already exists for a given subscription to prevent duplicates
 * Idempotency safeguard for webhook retries and race conditions
 */
async function hasDomainCreditForSubscription(subscriptionId: string): Promise<boolean> {
  try {
    // Get all credits with this source subscription ID
    const credits = await storage.getDomainCreditsBySubscriptionId(subscriptionId);
    return credits.length > 0;
  } catch (error) {
    console.error('Error checking existing domain credits for subscription:', error);
    return true; // Err on the side of caution to prevent duplicates
  }
}

/**
 * Mint a new domain credit for user subscription with eligibility and idempotency checks
 * SuperNova's business logic: $15 cap, eligible TLDs, 60-day expiry
 */
async function mintDomainCreditSafely(
  userId: string, 
  subscriptionId: string, 
  subscription?: Stripe.Subscription,
  reason = 'Pro subscription activation'
): Promise<DomainCredit | null> {
  try {
    console.log(`🔍 Checking domain credit eligibility for user ${userId}, subscription ${subscriptionId}`);
    
    // 1. Idempotency check - prevent duplicate credits from webhook retries
    if (await hasDomainCreditForSubscription(subscriptionId)) {
      console.log(`⚠️ Domain credit already exists for subscription ${subscriptionId}, skipping`);
      return null;
    }
    
    // 2. Business rule - one credit per 12 months per user
    if (await hasCreditInLast12Months(userId)) {
      console.log(`⚠️ User ${userId} already has domain credit in last 12 months, skipping`);
      return null;
    }
    
    // 3. Subscription eligibility check (if subscription data provided)
    if (subscription && !isEligibleSubscription(subscription)) {
      console.log(`⚠️ Subscription ${subscriptionId} is not eligible for domain credit (not Pro plan), skipping`);
      return null;
    }
    
    const expiresAt = new Date(Date.now() + CREDIT_EXPIRY_DAYS * 24 * 60 * 60 * 1000);
    
    const creditData: InsertDomainCredit = {
      userId,
      status: 'available',
      capCents: DEFAULT_DOMAIN_CREDIT_CAP_CENTS,
      eligibleTlds: DEFAULT_ELIGIBLE_TLDS,
      expiresAt,
      sourceSubscriptionId: subscriptionId
    };
    
    const credit = await storage.createDomainCredit(creditData);
    
    console.log(`💳✅ Domain credit minted - User: ${userId}, Subscription: ${subscriptionId}, Cap: $${DEFAULT_DOMAIN_CREDIT_CAP_CENTS/100}, Expires: ${expiresAt.toISOString()}, Reason: ${reason}`);
    
    // Create notification for user about their free domain
    await storage.createNotification(
      userId,
      `🎉 Free domain included with your Pro subscription! Use it within ${CREDIT_EXPIRY_DAYS} days on eligible domains (.com, .net, .org, .co).`,
      'success'
    );
    
    return credit;
  } catch (error) {
    console.error(`❌ Error minting domain credit for user ${userId}, subscription ${subscriptionId}:`, error);
    return null;
  }
}

// Owner email configuration
const OWNER_EMAILS = (process.env.IBRANDBIZ_OWNER_EMAILS || 'jrichards@ibrandbiz.com')
  .split(',')
  .map(email => email.trim().toLowerCase())
  .filter(email => email.length > 0);

/**
 * Server-side owner email verification
 * Checks if the provided email is in the configured owner emails list
 */
function isOwnerEmailServerSide(email?: string): boolean {
  if (!email) return false;
  
  const normalizedEmail = email.trim().toLowerCase();
  const isOwner = OWNER_EMAILS.includes(normalizedEmail);
  
  console.log(`🔐 Server-side owner check - Email: ${email}, Owner: ${isOwner}, Owner list: [${OWNER_EMAILS.join(', ')}]`);
  
  return isOwner;
}

// Authentication middleware
interface AuthenticatedRequest extends Request {
  user?: {
    uid: string;
    email?: string;
    displayName?: string;
  };
}

async function authenticateToken(req: AuthenticatedRequest, res: Response, next: NextFunction) {
  try {
    // Explicit development bypass - ONLY if DEV_BYPASS=true AND NODE_ENV=development
    const isDevelopment = process.env.NODE_ENV === 'development';
    const devBypassEnabled = process.env.DEV_BYPASS === 'true';
    const devBypassHeader = req.headers['x-dev-bypass'];
    
    if (isDevelopment && devBypassEnabled && devBypassHeader === 'development') {
      console.warn('🚨 DEV AUTH BYPASS ACTIVE - Only use in development!');
      req.user = {
        uid: 'dev-user-123',
        email: 'dev@example.com',
        displayName: 'Development User',
      };
      return next();
    }

    const admin = await getFirebaseAdmin();
    
    if (!admin) {
      // In production, fail hard if Firebase is not available
      if (process.env.NODE_ENV === 'production') {
        console.error('🚨 CRITICAL: Firebase Admin not available in production');
        return res.status(500).json({ error: 'Authentication service unavailable' });
      }
      
      // In development, only bypass if explicitly enabled
      if (isDevelopment && devBypassEnabled) {
        console.warn('🚨 Firebase unavailable in development with DEV_BYPASS enabled, using mock user');
        req.user = {
          uid: 'dev-user-123',
          email: 'dev@example.com',
          displayName: 'Development User',
        };
        return next();
      }
      
      return res.status(500).json({ error: 'Authentication service unavailable' });
    }

    const authHeader = req.headers.authorization;
    const token = authHeader && authHeader.split(' ')[1]; // Bearer TOKEN

    if (!token) {
      return res.status(401).json({ error: 'Access token required' });
    }

    const decodedToken = await admin.auth().verifyIdToken(token);
    req.user = {
      uid: decodedToken.uid,
      email: decodedToken.email,
      displayName: decodedToken.name,
    };
    next();
  } catch (error) {
    console.error('Authentication error:', error);
    return res.status(403).json({ error: 'Invalid or expired token' });
  }
}

/**
 * Middleware to check if user has Pro access (paid subscription or owner access)
 * Must be used after authenticateToken middleware
 */
async function hasProAccess(req: AuthenticatedRequest, res: Response, next: NextFunction) {
  try {
    if (!req.user) {
      return res.status(401).json({ error: 'User not authenticated' });
    }

    // Get user from database to check paid status
    const user = await storage.getUserByFirebaseUid(req.user.uid);
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    // Check if user is owner (owners get automatic Pro access)
    const isOwner = user.role === 'owner' || isOwnerEmailServerSide(user.email);
    
    // Check if user has Pro access (paid subscription or owner)
    const hasPro = user.isPaid || isOwner;
    
    if (!hasPro) {
      return res.status(402).json({ 
        error: 'Pro subscription required',
        message: 'This feature requires a Pro subscription. Please upgrade to continue.',
        feature: 'premium_api_access'
      });
    }

    // User has Pro access, continue to the protected route
    next();
  } catch (error) {
    console.error('Pro access check error:', error);
    return res.status(500).json({ error: 'Failed to verify Pro access' });
  }
}

// Error handling helper
function handleError(res: Response, error: any, defaultMessage: string) {
  console.error(error);
  if (error instanceof z.ZodError) {
    return res.status(400).json({ 
      error: 'Validation failed', 
      details: error.errors 
    });
  }
  return res.status(500).json({ error: defaultMessage });
}

// --- Telemetry ring buffer (in-memory) ---
type TelemetryEvent = { t: string; path?: string; from?: string; ts: number; meta?: any };
const TELEMETRY: TelemetryEvent[] = [];
const pushEvent = (e: TelemetryEvent) => {
  TELEMETRY.push(e);
  if (TELEMETRY.length > 200) TELEMETRY.shift();
};

export async function registerRoutes(app: Express): Promise<Server> {
  const objectStorageService = new ObjectStorageService();

  // Initialize job system
  console.log('🚀 Initializing job system...');
  await import('./jobs'); // This will register all job workers

  // Create upload directories if they don't exist
  const uploadDirs = [
    path.join(process.cwd(), 'uploads', 'photos'),
    path.join(process.cwd(), 'uploads', 'mockups'),
    path.join(process.cwd(), 'tmp') // For database files
  ];

  for (const dir of uploadDirs) {
    try {
      await fs.mkdir(dir, { recursive: true });
      console.log(`📁 Created directory: ${dir}`);
    } catch (error) {
      console.warn(`Failed to create directory ${dir}:`, error);
    }
  }

  // SECURITY: Removed blanket static serving of /uploads directory
  // Asset serving now handled by secure endpoints with proper authorization
  console.log('🔒 Secure asset serving enabled - no public file exposure');

  // Add visitor tracking middleware for public pages only
  app.use(trackVisitor);
  console.log('📊 Visitor tracking middleware enabled');

  // Configure multer for avatar uploads
  const upload = multer({ 
    dest: "uploads/",
    limits: {
      fileSize: 5 * 1024 * 1024, // 5MB limit
    },
    fileFilter: (req, file, cb) => {
      // Only allow image files
      if (file.mimetype.startsWith('image/')) {
        cb(null, true);
      } else {
        cb(new Error('Only image files are allowed'));
      }
    }
  });

  // Serve stable logo URL for external integrations (WHMCS, etc.)
  app.get('/assets/logo.png', (req, res) => {
    const logoPath = path.resolve(import.meta.dirname, '..', 'dist', 'public', 'assets');
    
    fs.readdir(logoPath)
      .then(files => {
        // Look for the main logo file (hashed filename)
        const logoFile = files.find(file => file.startsWith('IBrandBiz Logo 2025-') && file.endsWith('.svg'));
        
        if (logoFile) {
          const fullPath = path.join(logoPath, logoFile);
          res.setHeader('Content-Type', 'image/svg+xml');
          res.setHeader('Cache-Control', 'public, max-age=31536000'); // Cache for 1 year
          res.sendFile(fullPath);
        } else {
          res.status(404).send('Logo not found');
        }
      })
      .catch(error => {
        console.error('Error serving logo:', error);
        res.status(500).send('Error serving logo');
      });
  });

  // Serve stable reverse logo URL for external integrations (dark backgrounds)
  app.get('/assets/logo-reverse.png', (req, res) => {
    const logoPath = path.resolve(import.meta.dirname, '..', 'client', 'public', 'assets', 'IBrandBiz_Logo_Reversed_2025.png');
    
    res.setHeader('Content-Type', 'image/png');
    res.setHeader('Cache-Control', 'public, max-age=31536000'); // Cache for 1 year
    res.sendFile(logoPath, (err) => {
      if (err) {
        console.error('Error serving reverse logo:', err);
        res.status(404).send('Reverse logo not found');
      }
    });
  });

  // Proxy route to fetch Firebase Storage content (bypasses CORS issues)
  app.get("/api/firebase-proxy", async (req, res) => {
    try {
      const { url } = req.query;
      if (!url || typeof url !== 'string') {
        return res.status(400).json({ error: "URL parameter required" });
      }

      // Only allow Firebase Storage URLs for security
      if (!url.includes('firebasestorage.googleapis.com')) {
        return res.status(403).json({ error: "Only Firebase Storage URLs allowed" });
      }

      const response = await fetch(url);
      if (!response.ok) {
        return res.status(response.status).json({ error: `Firebase fetch failed: ${response.statusText}` });
      }

      const content = await response.text();
      res.setHeader('Content-Type', 'image/svg+xml');
      res.send(content);
    } catch (error) {
      console.error("Firebase proxy error:", error);
      res.status(500).json({ error: "Failed to fetch Firebase content" });
    }
  });

  // Wire admin routes
  app.use(adminRoutes);

  // Wire icons routes
  app.use('/api/icons', iconsRoutes);
  app.use('/api/icons', iconLibraryRoutes);

  // FE compatibility for existing fetch('/api/icons/imported') - redirect to new icon library
  app.get("/api/icons/imported", (req, res) => {
    req.url = "/list";
    iconLibraryRoutes.handle(req, res);
  });
  
  // Wire external stock photo search routes
  app.use('/api/external-stock', stockRoutes);
  
  // Wire stock library routes (admin uploads and public access)  
  const { default: stockLibraryRoutesNew } = await import('./routes/stockLibraryRoutes.js');
  app.use('/api/stock', stockLibraryRoutesNew);
  
  // Wire mockup library routes (mirrors stock photos)
  const { default: mockupLibraryRoutes } = await import('./routes/mockupLibraryRoutes.js');
  app.use('/api/mockups', mockupLibraryRoutes);
  
  // Wire presentation template routes
  app.use('/api/ppt', pptRoutes);
  app.use('/api/infographics', infographicsRoutes);
  app.use('/api/presentations', presentationTemplatesRoutes);
  app.use('/api/logo-templates', logoTemplatesRoutes);
  app.use('/api/bp-templates', bpTemplatesRoutes);
  app.use('/api/bp-templates', bpImportCsv);
  // PUBLIC (no auth) - free template downloads
  app.use('/api/bp-templates-firebase', bpTemplatesPublicRoutes);
  // ADMIN (upload/manage only)
  app.use('/api/bp-templates-firebase/admin', requireAdmin, bpTemplatesFirebaseRoutes);
  // HEALTH CHECK
  app.use('/health', healthBpTemplates);
  
  // Logo Templates - PUBLIC (no auth)
  app.use('/api/logo-templates-firebase', logoTemplatesPublicRoutes);

  // PDF PROXY - Download and cache Firebase PDFs temporarily for reliable rendering
  const pdfCacheDir = path.join(process.cwd(), 'tmp', 'pdf-cache');
  await fs.mkdir(pdfCacheDir, { recursive: true });
  
  // Cleanup old cached PDFs (older than 1 hour)
  const cleanupCache = async () => {
    try {
      const files = await fs.readdir(pdfCacheDir);
      const now = Date.now();
      for (const file of files) {
        const filePath = path.join(pdfCacheDir, file);
        const stat = await fs.stat(filePath);
        if (now - stat.mtimeMs > 60 * 60 * 1000) { // 1 hour
          await fs.unlink(filePath);
          console.log(`[PDF Cache] Cleaned up old file: ${file}`);
        }
      }
    } catch (e) {
      console.error('[PDF Cache] Cleanup error:', e);
    }
  };
  
  // Run cleanup every 30 minutes
  setInterval(cleanupCache, 30 * 60 * 1000);
  cleanupCache(); // Initial cleanup
  
  // Live PDF proxy with Range support (forwards to Firebase)
  app.get('/api/preview/pdf', async (req: Request, res: Response) => {
    try {
      const u = req.query.u;
      if (!u) return res.status(400).send('Missing ?u');
      const url = decodeURIComponent(String(u));
      
      // Allow Firebase Storage and Google Cloud Storage
      const allowed = /^(https:\/\/(?:storage\.googleapis\.com|firebasestorage\.googleapis\.com|[^/]+\.firebasestorage\.app)\/)/i;
      if (!allowed.test(url)) return res.status(400).send('Blocked');

      const headers: Record<string, string> = {};
      if (req.headers.range) headers.Range = req.headers.range;
      if (req.headers['if-none-match']) headers['If-None-Match'] = req.headers['if-none-match'];
      if (req.headers['if-modified-since']) headers['If-Modified-Since'] = req.headers['if-modified-since'];

      const upstream = await fetch(url, { headers, redirect: 'follow' });

      // Mirror status (206/200/304/etc.)
      res.status(upstream.status);

      // Mirror relevant headers so the viewer can seek
      for (const h of [
        'content-type', 'content-length', 'accept-ranges', 'content-range',
        'etag', 'last-modified', 'cache-control', 'vary', 'date'
      ]) {
        const v = upstream.headers.get(h);
        if (v) res.setHeader(h, v);
      }

      // Ensure inline
      const filename = (req.query.filename || 'document.pdf').toString();
      if (!res.getHeader('content-disposition')) {
        res.setHeader('Content-Disposition', `inline; filename="${filename}"`);
      }
      if (!res.getHeader('content-type')) res.setHeader('Content-Type', 'application/pdf');

      if (upstream.status === 304) return res.end();
      if (!upstream.body) return res.end();

      const { Readable } = await import('node:stream');
      Readable.fromWeb(upstream.body).pipe(res);
    } catch (e) {
      console.error('PDF proxy error', e);
      res.status(502).send('Proxy error');
    }
  });

  // SECURE Asset Serving Routes
  
  // Serve public preview assets (watermarked only)
  app.get("/public-objects/:filePath(*)", async (req, res) => {
    const filePath = req.params.filePath;
    try {
      // Only allow access to public preview directories
      if (!filePath.startsWith('assets/previews/')) {
        return res.status(403).json({ error: "Access denied - previews only" });
      }
      
      const file = await objectStorageService.searchPublicObject(filePath);
      if (!file) {
        return res.status(404).json({ error: "File not found" });
      }
      
      // Add security headers for all file types
      res.set({
        'X-Content-Type-Options': 'nosniff',
        'X-Frame-Options': 'DENY',
        'Cache-Control': 'public, max-age=3600'
      });
      
      objectStorageService.downloadObject(file, res);
    } catch (error) {
      console.error("Error searching for public object:", error);
      return res.status(500).json({ error: "Internal server error" });
    }
  });
  
  // Secure asset download endpoint with authorization
  app.get('/api/assets/:assetId/download', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'Authentication required' });
      }

      const assetId = req.params.assetId;
      
      // Get asset and verify it exists
      const asset = await storage.getAsset(assetId);
      if (!asset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Get creator asset for marketplace context
      const creatorAssets = await storage.getCreatorAssetsByAssetId(assetId);
      const creatorAsset = creatorAssets[0];
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Marketplace asset not found' });
      }

      // Only approved assets can be downloaded
      if (creatorAsset.approvalStatus !== 'approved') {
        return res.status(403).json({ error: 'Asset not available for download' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Check if user owns the asset (creator bypass)
      const creator = await storage.getCreatorByUserId(user.id);
      const isCreatorOwner = creator && creator.id === creatorAsset.creatorId;

      if (!isCreatorOwner) {
        // For non-owners, check if they have purchased/licensed this asset
        const hasEntitlement = await storage.getUserEntitlementByUserAndItem(
          user.id, 
          'creator_asset', 
          creatorAsset.id
        );

        if (!hasEntitlement || hasEntitlement.status !== 'active') {
          return res.status(402).json({ 
            error: 'Purchase required',
            message: 'This asset requires a purchase to download the original file.',
            assetId: creatorAsset.id,
            previewUrl: asset.previewUrl // Provide preview URL as alternative
          });
        }
      }

      // Generate secure download from private storage
      const privateDir = objectStorageService.getPrivateObjectDir();
      const filePath = asset.fileUrl.startsWith(privateDir) ? asset.fileUrl : `${privateDir}/${asset.fileUrl}`;
      
      try {
        // Parse the object path to get bucket and object name
        const parseObjectPath = (path: string): { bucketName: string; objectName: string } => {
          if (!path.startsWith("/")) {
            path = `/${path}`;
          }
          const pathParts = path.split("/");
          if (pathParts.length < 3) {
            throw new Error("Invalid path: must contain at least a bucket name");
          }
        
          const bucketName = pathParts[1];
          const objectName = pathParts.slice(2).join("/");
        
          return { bucketName, objectName };
        };
        
        const { bucketName, objectName } = parseObjectPath(filePath);
        const bucket = objectStorageClient.bucket(bucketName);
        const file = bucket.file(objectName);

        // Check if file exists
        const [exists] = await file.exists();
        if (!exists) {
          console.error(`Asset file not found in storage: ${filePath}`);
          return res.status(404).json({ error: 'Asset file not found' });
        }

        // Set secure download headers
        const isSvgOrPdf = asset.mimeType === 'image/svg+xml' || asset.mimeType === 'application/pdf';
        const filename = asset.originalFileName || asset.fileName;
        
        res.set({
          'Content-Type': asset.mimeType,
          'Content-Disposition': isSvgOrPdf ? `attachment; filename="${filename}"` : `inline; filename="${filename}"`,
          'X-Content-Type-Options': 'nosniff',
          'X-Frame-Options': 'DENY',
          'Cache-Control': 'private, no-cache, no-store, must-revalidate'
        });

        // Stream the file securely
        const stream = file.createReadStream();
        
        stream.on('error', (err) => {
          console.error('Download stream error:', err);
          if (!res.headersSent) {
            res.status(500).json({ error: 'Error streaming file' });
          }
        });

        // Track download
        await storage.incrementAssetDownloadCount(assetId);
        
        // Log download for audit
        console.log(`🔒 Secure download: User ${user.email} downloaded asset ${creatorAsset.title} (${assetId})`);
        
        stream.pipe(res);
        
      } catch (storageError) {
        console.error('Storage error during secure download:', storageError);
        return res.status(500).json({ error: 'Failed to retrieve asset file' });
      }

    } catch (error) {
      console.error('Secure asset download error:', error);
      handleError(res, error, 'Failed to download asset');
    }
  });
  
  // Asset preview endpoint (serves watermarked previews)
  app.get('/api/assets/:assetId/preview', async (req: Request, res: Response) => {
    try {
      const assetId = req.params.assetId;
      
      // Get asset and verify it exists
      const asset = await storage.getAsset(assetId);
      if (!asset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Only serve previews for approved assets
      const creatorAssets = await storage.getCreatorAssetsByAssetId(assetId);
      const creatorAsset = creatorAssets[0];
      if (!creatorAsset || creatorAsset.approvalStatus !== 'approved') {
        return res.status(404).json({ error: 'Asset preview not available' });
      }

      // Serve from public preview path if available
      if (asset.previewUrl) {
        try {
          const previewPath = asset.previewUrl.replace('/public-objects/', '');
          const file = await objectStorageService.searchPublicObject(previewPath);
          if (file) {
            res.set({
              'Content-Type': asset.mimeType,
              'Cache-Control': 'public, max-age=3600',
              'X-Content-Type-Options': 'nosniff'
            });
            return objectStorageService.downloadObject(file, res);
          }
        } catch (previewError) {
          console.warn('Preview file not found, falling back to original:', previewError);
        }
      }

      // Fallback: generate preview on-the-fly (for backwards compatibility)
      // This should only happen for old assets without previews
      return res.status(404).json({ 
        error: 'Preview not available',
        message: 'Asset preview has not been generated yet'
      });
      
    } catch (error) {
      console.error('Asset preview error:', error);
      res.status(500).json({ error: 'Failed to serve asset preview' });
    }
  });

  // Get upload URL for file uploads
  app.post("/api/upload-url", authenticateToken, async (req: AuthenticatedRequest, res) => {
    try {
      const { fileName } = req.body;
      if (!fileName) {
        return res.status(400).json({ error: "fileName is required" });
      }

      const uploadURL = await objectStorageService.getUploadURL(fileName, true);
      res.json({ uploadURL });
    } catch (error) {
      console.error("Error getting upload URL:", error);
      res.status(500).json({ error: "Internal server error" });
    }
  });
  // Auth routes
  app.get('/api/auth/me', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Check if user has owner privileges based on email
      const isOwner = isOwnerEmailServerSide(user.email);

      // Development mode: simulate Pro user for testing
      const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
      const simulateProUser = isDevelopment && req.headers['x-simulate-pro'] === 'true';
      
      res.json({
        authenticated: true,
        email: user.email,
        displayName: user.displayName,
        company: user.company,
        avatarUrl: user.avatarUrl,
        isPaid: simulateProUser || user.isPaid || isOwner, // Owner gets paid access
        isOwner: isOwner,
        proActivatedAt: simulateProUser ? (user.proActivatedAt || new Date().toISOString()) : user.proActivatedAt,
        proWelcomeDismissed: !!user.proWelcomeDismissed
      });
    } catch (error) {
      handleError(res, error, 'Failed to get user info');
    }
  });

  // Owner verification endpoint
  app.get('/api/auth/verify-owner', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Check if user has owner privileges based on email
      const isOwner = isOwnerEmailServerSide(user.email);
      
      res.json({
        isOwner,
        email: user.email,
        verifiedAt: new Date().toISOString()
      });
    } catch (error) {
      handleError(res, error, 'Failed to verify owner status');
    }
  });

  // Session revocation endpoint - revoke all refresh tokens for the authenticated user
  // Apply password rate limiting since this is called after password changes
  app.post('/api/auth/revoke-sessions', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Apply password rate limiting for enhanced security
      const clientIp = req.ip || req.connection.remoteAddress || 'unknown';
      if (!checkPasswordRateLimit(req.user.uid, clientIp)) {
        return res.status(429).json({ 
          error: 'Too many password-related requests. Please wait before trying again.',
          message: 'Rate limit exceeded: Maximum 5 password changes per minute per user.',
          retryAfter: Math.ceil(PASSWORD_RATE_WINDOW / 1000)
        });
      }

      const admin = await getFirebaseAdmin();
      
      if (!admin) {
        return res.status(500).json({ error: 'Authentication service unavailable' });
      }

      // Revoke all refresh tokens for the authenticated user
      // This will force sign-out on all devices except the current session
      await admin.auth().revokeRefreshTokens(req.user.uid);
      
      // Log the action for security audit
      console.log(`🔒 Session tokens revoked for user: ${req.user.email} (${req.user.uid})`);
      
      res.json({ 
        success: true,
        message: 'All refresh tokens have been revoked',
        revokedAt: new Date().toISOString()
      });
    } catch (error) {
      console.error('Session revocation error:', error);
      return res.status(500).json({ 
        error: 'Failed to revoke sessions',
        message: 'Unable to revoke refresh tokens. Please try again.'
      });
    }
  });

  // Profile routes
  app.post('/api/profile/dismiss-pro-welcome', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      await storage.updateUser(user.id, { proWelcomeDismissed: true });
      res.json({ ok: true });
    } catch (error) {
      handleError(res, error, 'Failed to dismiss Pro welcome');
    }
  });

  // Profile management routes
  app.post('/api/profile/update', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const { name, company } = req.body || {};
      
      // Update user profile
      await storage.updateUserByFirebaseUid(req.user.uid, { 
        displayName: name, 
        company 
      });
      
      res.json({ name, company });
    } catch (error) {
      handleError(res, error, 'Failed to update profile');
    }
  });

  app.get('/api/profile/preferences', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const preferences = await storage.getUserPreferences(req.user.uid);
      res.json(preferences);
    } catch (error) {
      console.error('Get preferences error:', error);
      res.json({ 
        emailNews: true, 
        marketingEmails: false, 
        productUpdates: true 
      }); // Default preferences
    }
  });

  app.post('/api/profile/preferences', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const { emailNews, marketingEmails, productUpdates, systemNotifications, billingNotifications, projectNotifications } = req.body || {};
      
      await storage.setUserPreferences(req.user.uid, {
        emailNews: !!emailNews,
        marketingEmails: !!marketingEmails,
        productUpdates: !!productUpdates,
        systemNotifications: !!systemNotifications,
        billingNotifications: !!billingNotifications,
        projectNotifications: !!projectNotifications
      });
      
      res.json({ ok: true });
    } catch (error) {
      handleError(res, error, 'Failed to save preferences');
    }
  });

  // Settings routes
  app.get('/api/user/settings', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // For now, return default settings since we don't have settings storage yet
      const defaultSettings = {
        theme: 'system',
        timezone: '',
        fontScale: 1.0,
        notifications: {
          system: true,
          billing: true,
          project: true
        }
      };

      res.json(defaultSettings);
    } catch (error) {
      handleError(res, error, 'Failed to get settings');
    }
  });

  app.post('/api/user/settings', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Validate fontScale if provided
      let fontScale = req.body.fontScale ?? 1.0;
      if (typeof fontScale === 'number') {
        // Clamp to valid range (85%-130%)
        fontScale = Math.max(0.85, Math.min(1.30, fontScale));
      } else {
        fontScale = 1.0;
      }

      // For now, just echo back the settings since we don't have storage yet
      const settings = {
        theme: req.body.theme || 'system',
        timezone: req.body.timezone || '',
        fontScale: fontScale,
        notifications: {
          system: req.body.notifications?.system ?? true,
          billing: req.body.notifications?.billing ?? true,
          project: req.body.notifications?.project ?? true
        }
      };

      res.json(settings);
    } catch (error) {
      handleError(res, error, 'Failed to save settings');
    }
  });

  // Avatar upload endpoint
  app.post('/api/profile/avatar', authenticateToken, upload.single("file"), async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      if (!req.file) {
        return res.status(400).json({ error: 'No file uploaded' });
      }

      // In a real app, you'd upload to S3/CloudStorage and get a permanent URL
      // For now, just store the local path
      const url = `/uploads/${req.file.filename}`;
      
      // Update user avatar URL
      await storage.updateUserByFirebaseUid(req.user.uid, { avatarUrl: url });
      
      res.json({ url });
    } catch (error) {
      handleError(res, error, 'Avatar upload failed');
    }
  });

  // Simplified password change endpoint (placeholder for Firebase Auth)
  app.post('/api/profile/change-password', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const { current, next } = req.body || {};
      
      if (!current || !next) {
        return res.status(400).json({ error: 'Current and new passwords are required' });
      }

      if (next.length < 6) {
        return res.status(400).json({ error: 'New password must be at least 6 characters long' });
      }

      // Note: Firebase Auth handles password changes client-side
      // This is a placeholder endpoint that would integrate with Firebase Admin SDK
      res.json({ ok: true });
    } catch (error) {
      handleError(res, error, 'Failed to update password');
    }
  });

  // Billing routes
  app.post('/api/billing/checkout', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const stripeClient = getStripe();
      if (!stripeClient) {
        return res.status(500).json({ error: 'Stripe not configured' });
      }

      // Validate request body
      const { priceType, successUrl, cancelUrl } = req.body;
      
      if (!['monthly', 'yearly'].includes(priceType)) {
        return res.status(400).json({ error: 'Invalid price type. Must be "monthly" or "yearly"' });
      }

      // Get user from database, create if doesn't exist
      let user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        console.log(`🔄 Auto-creating database record for Firebase user: ${req.user.email}`);
        
        // Create user record for authenticated Firebase user
        const userData = insertUserSchema.parse({
          firebaseUid: req.user.uid,
          email: req.user.email || '',
          displayName: req.user.displayName || 'User',
          role: 'user'
        });
        
        user = await storage.createUser(userData);
        console.log(`✅ Created database user record for: ${user.email}`);
      }

      // Check if user already has an active subscription
      if (user.isPaid) {
        return res.status(409).json({ error: 'User already has an active subscription' });
      }

      // Define price IDs - these must be set up in your Stripe Dashboard
      const priceIds = {
        monthly: process.env.STRIPE_MONTHLY_PRICE_ID,
        yearly: process.env.STRIPE_YEARLY_PRICE_ID
      };

      const priceId = priceIds[priceType as keyof typeof priceIds];
      
      if (!priceId) {
        console.error(`Missing Stripe Price ID for ${priceType}. Please set STRIPE_${priceType.toUpperCase()}_PRICE_ID environment variable.`);
        return res.status(500).json({ 
          error: 'Stripe configuration required', 
          message: `Missing Stripe Price ID for ${priceType} subscription. You need to create subscription products in your Stripe Dashboard and set the Price IDs as environment variables.`,
          setup: {
            steps: [
              "1. Go to your Stripe Dashboard → Products → Create Product",
              "2. Create a 'Pro Monthly' product with recurring billing at $19/month",
              "3. Create a 'Pro Yearly' product with recurring billing at $190/year", 
              "4. Copy the Price IDs (they start with 'price_') from each product",
              "5. Set environment variables: STRIPE_MONTHLY_PRICE_ID and STRIPE_YEARLY_PRICE_ID"
            ],
            required: {
              STRIPE_MONTHLY_PRICE_ID: "Your monthly subscription Price ID from Stripe Dashboard",
              STRIPE_YEARLY_PRICE_ID: "Your yearly subscription Price ID from Stripe Dashboard"
            }
          }
        });
      }

      console.log(`🔍 Using Stripe Price ID: ${priceId} for ${priceType} subscription`);
      
      // Validate that the Price ID exists in Stripe before proceeding
      try {
        const price = await stripeClient.prices.retrieve(priceId);
        console.log(`✅ Price ID validated: ${price.id} - ${price.currency} ${price.unit_amount/100}/${price.recurring?.interval}`);
        
        if (!price.active) {
          console.error(`❌ Price ID ${priceId} is not active in Stripe`);
          return res.status(500).json({ 
            error: 'Invalid Stripe configuration', 
            message: `The Price ID ${priceId} exists but is not active. Please activate it in your Stripe Dashboard or create a new one.`,
            details: {
              priceId: priceId,
              status: 'inactive',
              helpUrl: 'https://dashboard.stripe.com/products'
            }
          });
        }
      } catch (stripeError: any) {
        console.error(`❌ Stripe Price ID validation failed:`, stripeError.message);
        
        if (stripeError.code === 'resource_missing') {
          return res.status(500).json({ 
            error: 'Invalid Stripe Price ID', 
            message: `The Price ID "${priceId}" does not exist in your Stripe account. This commonly happens when:`,
            possibleCauses: [
              "• You're using test mode Price IDs with live mode Stripe keys (or vice versa)",
              "• The Price ID was copied incorrectly", 
              "• The Price was deleted or archived in Stripe Dashboard",
              "• You're connected to a different Stripe account"
            ],
            solution: {
              immediate: "Check your Stripe Dashboard to verify your Price IDs match your current mode (test/live)",
              longTerm: "Create new subscription products in Stripe Dashboard and update your environment variables"
            },
            details: {
              currentPriceId: priceId,
              priceType: priceType,
              stripeError: stripeError.message,
              dashboardUrl: 'https://dashboard.stripe.com/products'
            }
          });
        }
        
        return res.status(500).json({ 
          error: 'Stripe validation error', 
          message: stripeError.message,
          details: { priceId, priceType }
        });
      }

      // Create or get Stripe customer
      let customer;
      const existingCustomers = await stripeClient.customers.list({
        email: user.email,
        limit: 1,
      });

      if (existingCustomers.data.length > 0) {
        customer = existingCustomers.data[0];
      } else {
        customer = await stripeClient.customers.create({
          email: user.email,
          name: user.displayName || undefined,
          metadata: {
            uid: req.user.uid,
            user_email: user.email
          }
        });
      }

      // Restore proper URLs now that iframe issue is fixed
      const baseUrl = successUrl ? new URL(successUrl).origin : req.headers.origin;
      const finalSuccessUrl = successUrl || `${baseUrl}/dashboard?welcome=pro`;
      const finalCancelUrl = cancelUrl || `${baseUrl}/pricing?cancelled=true`;
      
      console.log(`🔗 Stripe URLs - Success: ${finalSuccessUrl}, Cancel: ${finalCancelUrl}`);

      // Create checkout session
      const session = await stripeClient.checkout.sessions.create({
        customer: customer.id,
        payment_method_types: ['card'],
        line_items: [
          {
            price: priceId,
            quantity: 1,
          },
        ],
        mode: 'subscription',
        subscription_data: {
          trial_end: 'now', // No trial period - immediate payment required
          metadata: {
            uid: req.user.uid,
            email: user.email,
            user_email: user.email
          }
        },
        success_url: finalSuccessUrl,
        cancel_url: finalCancelUrl,
        allow_promotion_codes: true,
        billing_address_collection: 'auto',
      });

      console.log(`✅ Created Stripe checkout session for ${user.email} - ${priceType} subscription`);
      
      res.json({ 
        checkoutUrl: session.url,
        sessionId: session.id,
        priceId: priceId
      });
    } catch (e: any) {
      // Surface Stripe's real error to the UI and logs (SN's patch)
      const msg = e?.raw?.message || e?.message || 'Stripe checkout error';
      const code = e?.raw?.code || e?.code || 'unknown_error';
      console.error('checkout error:', { code, msg, fullError: e });
      return res.status(400).json({ error: msg, code });
    }
  });

  // New embedded payment subscription creation route
  app.post('/api/billing/create-subscription', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const stripeClient = getStripe();
      if (!stripeClient) {
        return res.status(500).json({ error: 'Stripe not configured' });
      }

      const { billingType, email } = req.body as { billingType: "monthly" | "yearly"; email?: string | null };
      if (!billingType || !["monthly", "yearly"].includes(billingType)) {
        return res.status(400).json({ error: "Missing or invalid billingType. Must be 'monthly' or 'yearly'" });
      }

      // Map billing type to Stripe price IDs (configure these in your environment)
      const priceId = billingType === "monthly" 
        ? process.env.STRIPE_MONTHLY_PRICE_ID 
        : process.env.STRIPE_YEARLY_PRICE_ID;

      if (!priceId) {
        console.error(`Missing Stripe price ID for ${billingType} billing`);
        return res.status(500).json({ error: `${billingType} subscription not configured` });
      }

      // Use provided email or authenticated user's email
      const customerEmail = email || req.user.email;

      // 1) Find or create customer
      let customer: any;
      if (customerEmail) {
        const existing = await stripeClient.customers.list({ email: customerEmail, limit: 1 });
        customer = existing.data[0] ?? await stripeClient.customers.create({ email: customerEmail });
      } else {
        customer = await stripeClient.customers.create();
      }

      // 2) Create the subscription in incomplete state so we can collect payment
      const subscription = await stripeClient.subscriptions.create({
        customer: customer.id,
        items: [{ price: priceId }],
        payment_behavior: "default_incomplete",
        expand: ["latest_invoice.payment_intent"],
      });

      const paymentIntent = (subscription.latest_invoice as any).payment_intent;
      if (!paymentIntent?.client_secret) {
        return res.status(500).json({ error: "No client secret on payment intent." });
      }

      return res.json({
        clientSecret: paymentIntent.client_secret,
        subscriptionId: subscription.id,
      });
    } catch (err: any) {
      console.error("create-subscription failed", err);
      return res.status(500).json({ error: err.message || "Stripe error" });
    }
  });

  app.post('/api/billing/cancel', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Validate request body
      const cancellationData = insertCancellationSchema.parse(req.body);
      
      // Get user from database
      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Save cancellation data
      const cancellation = await storage.createCancellation({
        userId: user.id,
        firebaseUid: req.user.uid,
        email: req.user.email || user.email,
        reason: cancellationData.reason,
        note: cancellationData.note || null,
      });

      try {
        // Cancel Stripe subscription if the user has an active subscription and Stripe is available
        if (user.isPaid) {
          const stripeClient = getStripe();
          if (!stripeClient) {
            console.warn('Stripe not configured - skipping subscription cancellation, but updating user status');
            // Still update user's paid status in database even without Stripe
            await storage.updateUser(user.id, { isPaid: false });
          } else {
            // Find active subscriptions for this customer
            const customers = await stripeClient.customers.list({
              email: user.email,
              limit: 1,
            });

            if (customers.data.length > 0) {
              const customerId = customers.data[0].id;
              
              // Get active subscriptions
              const subscriptions = await stripeClient.subscriptions.list({
                customer: customerId,
                status: 'active',
                limit: 10,
              });

              // Cancel all active subscriptions with cancel_at_period_end for better UX
              for (const subscription of subscriptions.data) {
                await stripeClient.subscriptions.update(subscription.id, {
                  cancel_at_period_end: true
                });
                console.log(`✅ Scheduled Stripe subscription cancellation at period end: ${subscription.id} for user: ${user.email}`);
              }
            }

            // Update user's paid status in database
            await storage.updateUser(user.id, { isPaid: false });
          }
        }
      } catch (stripeError) {
        console.error('Stripe cancellation error:', stripeError);
        // Don't fail the request if Stripe fails - the survey data is still valuable
        console.warn('Failed to cancel Stripe subscription, but survey data saved');
      }

      res.json({ 
        success: true, 
        message: 'Subscription cancelled successfully',
        cancellationId: cancellation.id 
      });
    } catch (error) {
      handleError(res, error, 'Failed to cancel subscription');
    }
  });

  // Helper function to get or create Stripe customer
  async function getOrCreateCustomer(stripeClient: Stripe, userEmail: string, displayName?: string, uid?: string) {
    const existingCustomers = await stripeClient.customers.list({
      email: userEmail,
      limit: 1,
    });

    if (existingCustomers.data.length > 0) {
      return existingCustomers.data[0];
    } else {
      return await stripeClient.customers.create({
        email: userEmail,
        name: displayName || undefined,
        metadata: {
          uid: uid || '',
          user_email: userEmail
        }
      });
    }
  }

  // Stripe Elements - Create subscription with payment element
  app.post('/api/stripe/elements/create-subscription', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const stripeClient = getStripe();
      if (!stripeClient) {
        return res.status(500).json({ error: 'Stripe not configured' });
      }

      // Validate request body
      const { priceType } = req.body;
      
      if (!['monthly', 'yearly'].includes(priceType)) {
        return res.status(400).json({ error: 'Invalid price type. Must be "monthly" or "yearly"' });
      }

      // Get user from database, create if doesn't exist
      let user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        console.log(`🔄 Auto-creating database record for Firebase user: ${req.user.email}`);
        
        const userData = insertUserSchema.parse({
          firebaseUid: req.user.uid,
          email: req.user.email || '',
          displayName: req.user.displayName || 'User',
          role: 'user'
        });
        
        user = await storage.createUser(userData);
        console.log(`✅ Created database user record for: ${user.email}`);
      }

      // Check if user already has an active subscription
      if (user.isPaid) {
        return res.status(409).json({ error: 'User already has an active subscription' });
      }

      // Get price IDs from environment variables
      const priceIds = {
        monthly: process.env.STRIPE_MONTHLY_PRICE_ID,
        yearly: process.env.STRIPE_YEARLY_PRICE_ID
      };

      const priceId = priceIds[priceType as keyof typeof priceIds];
      
      if (!priceId) {
        return res.status(500).json({ 
          error: 'Stripe Price ID not configured', 
          message: `The ${priceType} Price ID is not set in environment variables. Please set STRIPE_${priceType.toUpperCase()}_PRICE_ID.`,
          priceType: priceType
        });
      }

      console.log(`🔍 Using Stripe Price ID: ${priceId} for ${priceType} subscription`);

      // Validate the price ID exists in Stripe
      try {
        const price = await stripeClient.prices.retrieve(priceId);
        console.log(`✅ Price ID validated: ${priceId} - ${price.currency} ${(price.unit_amount || 0) / 100}/${price.recurring?.interval}`);
      } catch (stripeError: any) {
        console.error(`❌ Stripe Price ID validation failed:`, stripeError.message);
        return res.status(500).json({ 
          error: 'Invalid Stripe Price ID', 
          message: `The Price ID "${priceId}" does not exist in your Stripe account.`,
          priceType: priceType
        });
      }

      // Create or get Stripe customer
      const customer = await getOrCreateCustomer(stripeClient, user.email, user.displayName, req.user.uid);

      // Create subscription with incomplete status for payment confirmation
      const subscription = await stripeClient.subscriptions.create({
        customer: customer.id,
        items: [{
          price: priceId,
        }],
        payment_behavior: 'default_incomplete',
        payment_settings: { save_default_payment_method: 'on_subscription' },
        expand: ['latest_invoice.payment_intent'],
        trial_end: 'now', // No trial period - immediate payment required
        metadata: {
          uid: req.user.uid,
          email: user.email,
          user_email: user.email
        }
      });

      const invoice = subscription.latest_invoice as Stripe.Invoice;
      const paymentIntent = invoice.payment_intent as Stripe.PaymentIntent;

      console.log(`✅ Created Stripe subscription for ${user.email} - ${priceType} subscription (status: ${subscription.status})`);
      
      // Payment intent is required for immediate payment - no trial period
      if (!paymentIntent) {
        console.error(`❌ No payment intent found - subscription creation failed`);
        return res.status(500).json({ 
          error: 'Subscription creation failed - no payment intent found',
          message: 'Unable to process payment. Please try again.'
        });
      }
      
      res.json({
        subscriptionId: subscription.id,
        clientSecret: paymentIntent.client_secret,
        customerId: customer.id,
        priceId: priceId
      });
    } catch (e: any) {
      const msg = e?.raw?.message || e?.message || 'Stripe subscription creation error';
      const code = e?.raw?.code || e?.code || 'unknown_error';
      console.error('subscription creation error:', { code, msg, fullError: e });
      return res.status(400).json({ error: msg, code });
    }
  });

  /**
   * Creator Marketplace Checkout Route
   * POST /api/checkout/creator-marketplace
   * Handles single-seller cart validation and Stripe Connect destination charges
   */
  app.post('/api/checkout/creator-marketplace', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const stripeClient = getStripe();
      if (!stripeClient) {
        return handleStripeUnavailable(res, 'Creator marketplace checkout');
      }

      // Get authenticated user
      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Parse and validate request body - SECURITY: Ignore client-supplied URLs
      const { cartItems } = req.body;
      
      if (!cartItems || !Array.isArray(cartItems) || cartItems.length === 0) {
        return res.status(400).json({ error: 'Cart is empty or invalid' });
      }

      // Define platform fee (30% = 3000 basis points)
      const PLATFORM_FEE_BPS = Number(process.env.PLATFORM_FEE_BPS) || 3000;

      console.log(`🛒 Processing creator marketplace checkout for ${cartItems.length} items`);

      // Validate cart items structure - SECURITY: Ignore client-supplied price/quantity
      const cartItemSchema = z.object({
        id: z.string(),
        itemType: z.literal('creator_asset'),
        itemId: z.string(), // creatorAssetId
        quantity: z.number().optional(), // We'll validate this explicitly below
        metadata: z.object({
          creatorId: z.string().optional(),
          assetId: z.string().optional()
        }).optional()
      });

      let validatedCartItems;
      try {
        validatedCartItems = cartItems.map(item => cartItemSchema.parse(item));
      } catch (validationError) {
        return res.status(400).json({ 
          error: 'Invalid cart item structure', 
          details: validationError instanceof z.ZodError ? validationError.errors : validationError 
        });
      }

      // CRITICAL: Validate client-supplied cart items respect digital license constraints
      // 1. Check for quantity > 1 violations
      const invalidQuantityItems = validatedCartItems.filter(item => item.quantity && item.quantity > 1);
      if (invalidQuantityItems.length > 0) {
        return res.status(400).json({
          error: 'Invalid quantity for digital licenses',
          message: 'Creator assets are digital licenses limited to one copy per asset',
          violatingItems: invalidQuantityItems.map(item => ({ id: item.id, quantity: item.quantity }))
        });
      }

      // 2. Check for duplicate creator assets (same itemId)
      const itemIds = validatedCartItems.map(item => item.itemId);
      const duplicateItemIds = itemIds.filter((id, index) => itemIds.indexOf(id) !== index);
      if (duplicateItemIds.length > 0) {
        return res.status(400).json({
          error: 'Duplicate creator assets detected',
          message: 'Each creator asset can only appear once in the cart',
          duplicateItemIds: [...new Set(duplicateItemIds)]
        });
      }

      // 3. Check for mixed creators in client metadata (preliminary check before DB validation)
      const clientCreatorIds = validatedCartItems
        .map(item => item.metadata?.creatorId)
        .filter(Boolean);
      if (clientCreatorIds.length > 0) {
        const uniqueClientCreatorIds = [...new Set(clientCreatorIds)];
        if (uniqueClientCreatorIds.length > 1) {
          return res.status(400).json({
            error: 'Mixed creator cart detected in client data',
            message: 'Cart contains assets from multiple creators according to client metadata',
            clientCreatorIds: uniqueClientCreatorIds
          });
        }
      }

      // Get creator assets and validate they exist, approved, and have valid pricing
      const creatorAssets = await Promise.all(
        validatedCartItems.map(async (item) => {
          const creatorAsset = await storage.getCreatorAsset(item.itemId);
          if (!creatorAsset) {
            throw new Error(`Creator asset not found: ${item.itemId}`);
          }
          if (creatorAsset.approvalStatus !== 'approved') {
            throw new Error(`Asset not available: ${creatorAsset.title} (status: ${creatorAsset.approvalStatus})`);
          }
          
          // SECURITY: Validate authoritative DB price against min/max limits
          const priceCents = creatorAsset.price;
          if (priceCents < CREATOR_MIN_PRICE_CENTS || priceCents > CREATOR_MAX_PRICE_CENTS) {
            throw new Error(`Asset price $${priceCents/100} is outside allowed range ($${CREATOR_MIN_PRICE_CENTS/100} - $${CREATOR_MAX_PRICE_CENTS/100})`);
          }
          
          return creatorAsset;
        })
      );

      // Validate single-seller cart - all assets must belong to same creator
      const creatorIds = [...new Set(creatorAssets.map(asset => asset.creatorId))];
      if (creatorIds.length > 1) {
        return res.status(400).json({
          error: 'Mixed creator cart not allowed',
          message: 'All items in cart must be from the same creator',
          creatorIds
        });
      }

      const creatorId = creatorIds[0];

      // Get creator profile and validate Stripe Connect status
      const creator = await storage.getCreator(creatorId);
      if (!creator) {
        return res.status(404).json({ error: 'Creator not found' });
      }

      if (!creator.stripeConnectAccountId) {
        return res.status(400).json({
          error: 'Creator payment setup incomplete',
          message: 'This creator has not completed their payment setup yet'
        });
      }

      if (!creator.payoutEnabled) {
        return res.status(400).json({
          error: 'Creator payouts disabled',
          message: 'This creator is not currently accepting payments'
        });
      }

      // Validate Stripe Connect account status
      let connectAccount;
      try {
        connectAccount = await stripeClient.accounts.retrieve(creator.stripeConnectAccountId);
      } catch (stripeError: any) {
        console.error('Failed to retrieve Stripe Connect account:', stripeError);
        return res.status(502).json({
          error: 'Creator payment account unavailable',
          message: 'Unable to process payments to this creator at the moment'
        });
      }

      if (!connectAccount.charges_enabled || !connectAccount.payouts_enabled) {
        return res.status(400).json({
          error: 'Creator account not ready',
          message: 'This creator\'s payment account is not fully set up for receiving payments',
          details: {
            charges_enabled: connectAccount.charges_enabled,
            payouts_enabled: connectAccount.payouts_enabled,
            requirements: connectAccount.requirements?.currently_due || []
          }
        });
      }

      // SECURITY: Calculate totals using only authoritative DB prices, force quantity=1
      let totalAmount = 0;
      const lineItems = validatedCartItems.map((item, index) => {
        const creatorAsset = creatorAssets[index];
        
        // SECURITY: Use only DB price, force quantity=1 for digital licenses
        const authoritativePrice = creatorAsset.price; // priceCents from DB
        const quantity = 1; // Force quantity=1 for digital licenses
        const itemTotal = authoritativePrice * quantity;
        totalAmount += itemTotal;

        return {
          price_data: {
            currency: 'usd',
            unit_amount: authoritativePrice,
            product_data: {
              name: creatorAsset.title, // Use DB title, not client name
              description: creatorAsset.description || undefined,
              metadata: {
                creator_asset_id: creatorAsset.id,
                asset_id: creatorAsset.assetId,
                creator_id: creatorId
              }
            }
          },
          quantity: quantity
        };
      });

      // Calculate platform fee (application_fee_amount is what the platform keeps)
      const platformFeeAmount = Math.round(totalAmount * PLATFORM_FEE_BPS / 10000);
      const creatorNetAmount = totalAmount - platformFeeAmount;

      console.log(`💰 Checkout totals - Total: $${totalAmount/100}, Platform Fee: $${platformFeeAmount/100}, Creator Net: $${creatorNetAmount/100}`);

      // Create or get Stripe customer
      const customer = await getOrCreateCustomer(stripeClient, user.email, user.displayName, req.user.uid);

      // SECURITY: Build secure redirect URLs using environment allowlist only
      const finalSuccessUrl = buildSecureRedirectUrl('/marketplace/success');
      const finalCancelUrl = buildSecureRedirectUrl('/marketplace/cart');

      // Create checkout session with destination charges
      const sessionData: any = {
        customer: customer.id,
        payment_method_types: ['card'],
        line_items: lineItems,
        mode: 'payment',
        success_url: `${finalSuccessUrl}?session_id={CHECKOUT_SESSION_ID}`,
        cancel_url: finalCancelUrl,
        allow_promotion_codes: false, // Disabled for creator marketplace
        billing_address_collection: 'required',
        metadata: {
          userId: user.id,
          creatorId,
          buyerUserId: user.id, // SECURITY: Add buyerUserId for fulfillment
          assetIds: creatorAssets.map(a => a.assetId).join(','), // SECURITY: Add assetIds array
          feeBps: PLATFORM_FEE_BPS.toString(), // SECURITY: Add feeBps for fulfillment
          totalItems: validatedCartItems.length.toString(),
          totalAmount: totalAmount.toString(),
          platformFeeAmount: platformFeeAmount.toString(),
          checkoutType: 'creator_marketplace'
        }
      };

      // Add Stripe Connect destination charges configuration
      if (platformFeeAmount > 0) {
        sessionData.payment_intent_data = {
          application_fee_amount: platformFeeAmount,
          transfer_data: {
            destination: creator.stripeConnectAccountId
          },
          metadata: {
            creator_id: creatorId,
            platform_fee_bps: PLATFORM_FEE_BPS.toString(),
            asset_ids: creatorAssets.map(a => a.assetId).join(',')
          }
        };
      }

      const session = await stripeClient.checkout.sessions.create(sessionData);

      // Store purchase record for tracking with authoritative data only
      const purchaseData = {
        userId: user.id,
        stripeSessionId: session.id,
        totalAmount,
        currency: 'USD',
        status: 'pending',
        // SECURITY: Store authoritative data from DB, not client input
        itemsData: validatedCartItems.map((item, index) => ({
          id: item.id,
          itemType: item.itemType,
          itemId: item.itemId,
          itemName: creatorAssets[index].title, // Use DB title
          itemPrice: creatorAssets[index].price, // Use DB price
          quantity: 1, // Force quantity=1
          metadata: {
            creatorId: creatorId,
            assetId: creatorAssets[index].assetId
          }
        })),
        metadata: {
          checkoutType: 'creator_marketplace',
          creatorId,
          platformFeeAmount,
          creatorNetAmount,
          assetIds: creatorAssets.map(a => a.assetId)
        }
      };

      await storage.createPurchase(purchaseData);

      console.log(`✅ Created creator marketplace checkout session: ${session.id} for creator: ${creatorId}`);

      res.json({
        sessionId: session.id,
        checkoutUrl: session.url,
        totalAmount,
        platformFeeAmount,
        creatorNetAmount,
        creatorId,
        itemCount: validatedCartItems.length
      });

    } catch (error: any) {
      console.error('Creator marketplace checkout error:', error);
      
      // Handle specific error types
      if (error.message.includes('Creator asset not found')) {
        return res.status(404).json({
          error: 'Asset not found',
          message: error.message
        });
      }
      
      if (error.message.includes('Asset not available')) {
        return res.status(400).json({
          error: 'Asset not available',
          message: error.message
        });
      }

      if (error.type === 'StripeCardError') {
        return res.status(400).json({
          error: 'Payment error',
          message: error.message,
          code: error.code
        });
      }

      return res.status(500).json({
        error: 'Checkout failed',
        message: 'An unexpected error occurred during checkout',
        details: error.message
      });
    }
  });

  // Stripe Customer Portal - for account management
  app.post('/api/stripe/customer-portal', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const stripeClient = getStripe();
      if (!stripeClient) {
        return res.status(500).json({ error: 'Stripe not configured' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Get Stripe customer
      const customer = await getOrCreateCustomer(stripeClient, user.email, user.displayName, req.user.uid);

      // Create customer portal session
      const { returnUrl } = req.body;
      const baseUrl = returnUrl ? new URL(returnUrl).origin : req.headers.origin;
      const finalReturnUrl = returnUrl || `${baseUrl}/dashboard`;

      const portalSession = await stripeClient.billingPortal.sessions.create({
        customer: customer.id,
        return_url: finalReturnUrl,
      });

      res.json({ 
        portalUrl: portalSession.url 
      });
    } catch (e: any) {
      const msg = e?.raw?.message || e?.message || 'Customer portal error';
      console.error('customer portal error:', e);
      return res.status(400).json({ error: msg });
    }
  });

  // Billing pause endpoint
  app.post('/api/billing/pause', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Validate request body
      const pauseData = insertPauseSchema.parse({
        ...req.body,
        uid: req.user.uid,
        email: req.user.email,
        resumeAt: new Date(Date.now() + (req.body.months || 1) * 30 * 24 * 60 * 60 * 1000) // Calculate resume date
      });
      
      // Get user from database
      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      if (!user.isPaid) {
        return res.status(400).json({ error: 'User does not have an active subscription' });
      }

      try {
        // Pause Stripe subscription if Stripe is available
        const stripeClient = getStripe();
        if (!stripeClient) {
          return res.status(500).json({ error: 'Stripe not configured - unable to pause subscription' });
        }

        // Find active subscriptions for this customer
        const customers = await stripeClient.customers.list({
          email: user.email,
          limit: 1,
        });

        if (customers.data.length === 0) {
          return res.status(404).json({ error: 'Stripe customer not found' });
        }

        const customerId = customers.data[0].id;
        
        // Get active subscriptions
        const subscriptions = await stripeClient.subscriptions.list({
          customer: customerId,
          status: 'active',
          limit: 10,
        });

        if (subscriptions.data.length === 0) {
          return res.status(404).json({ error: 'No active subscription found' });
        }

        // Pause the first active subscription (assuming one subscription per user)
        const subscription = subscriptions.data[0];
        const pauseEndDate = Math.floor(pauseData.resumeAt.getTime() / 1000); // Convert to Unix timestamp

        await stripeClient.subscriptions.update(subscription.id, {
          pause_collection: {
            behavior: 'mark_uncollectible',
            resumes_at: pauseEndDate,
          },
          metadata: {
            uid: req.user.uid,
            email: req.user.email || user.email,
            pause_reason: pauseData.reason || '',
          }
        });

        console.log(`✅ Paused Stripe subscription: ${subscription.id} for user: ${user.email} until ${pauseData.resumeAt}`);

        // Save pause data to database
        const pause = await storage.createPause({
          uid: req.user.uid,
          email: req.user.email || user.email,
          stripeSubId: subscription.id,
          months: pauseData.months,
          reason: pauseData.reason || null,
          note: pauseData.note || null,
          resumeAt: pauseData.resumeAt,
        });

        // Update user status in database
        await storage.updateUser(user.id, { 
          subscriptionStatus: 'paused',
          pausedAt: new Date(),
          resumeAt: pauseData.resumeAt
        });

        res.json({ 
          ok: true, 
          resumeAt: pauseData.resumeAt,
          pauseId: pause.id,
          message: `Subscription paused for ${pauseData.months} month${pauseData.months > 1 ? 's' : ''}`
        });

      } catch (stripeError) {
        console.error('Stripe pause error:', stripeError);
        return res.status(500).json({ error: 'Failed to pause Stripe subscription' });
      }
    } catch (error) {
      handleError(res, error, 'Failed to pause subscription');
    }
  });

  // Wire checkout routes
  const { default: checkoutRoutes } = await import('./routes/checkoutRoutes.js');
  app.use('/api/checkout', checkoutRoutes);



  // Keep the old checkout endpoint as fallback (remove this later)
  /*
  app.post('/api/checkout/create-session', async (req: Request, res: Response) => {
    try {
      const stripeClient = getStripe();
      if (!stripeClient) {
        return res.status(500).json({ error: "Stripe not configured" });
      }

      const user = req.user;
      const { cartItems: clientCartItems, sessionId } = req.body;

      if (!clientCartItems || !Array.isArray(clientCartItems) || clientCartItems.length === 0) {
        return res.status(400).json({ error: "Cart is empty" });
      }

      // Get cart items from storage to verify integrity
      const cartItems = await storage.getCartItems(user?.id, sessionId);
      if (cartItems.length === 0) {
        return res.status(400).json({ error: "Cart is empty" });
      }

      // Validate that cart doesn't mix subscription and one-time items
      const hasSubscription = cartItems.some(item => item.itemType === 'subscription');
      const hasStockPhoto = cartItems.some(item => item.itemType === 'stock_photo');
      
      if (hasSubscription && hasStockPhoto) {
        return res.status(400).json({ error: "Cannot mix subscription and stock photo items in the same cart" });
      }

      // Create line items for Stripe
      const lineItems = cartItems.map(item => ({
        price_data: {
          currency: 'usd',
          product_data: {
            name: item.itemName,
            metadata: {
              itemType: item.itemType,
              itemId: item.itemId
            }
          },
          unit_amount: item.itemPrice
        },
        quantity: item.quantity || 1
      }));

      const totalAmountCents = cartItems.reduce((total, item) => total + (item.itemPrice * (item.quantity || 1)), 0);

      // Create purchase record in pending status
      const purchase = await storage.createPurchase({
        userId: user?.id,
        stripeSessionId: '', // Will be updated after session creation
        totalAmountCents,
        items: cartItems,
        status: 'pending'
      });

      // Create Stripe checkout session
      const session = await stripeClient.checkout.sessions.create({
        payment_method_types: ['card'],
        line_items: lineItems,
        mode: hasSubscription ? 'subscription' : 'payment',
        success_url: buildSecureRedirectUrl('/checkout/success?session_id={CHECKOUT_SESSION_ID}', req.headers.origin as string),
        cancel_url: buildSecureRedirectUrl('/checkout/cancel', req.headers.origin as string),
        client_reference_id: user?.id,
        metadata: {
          purchaseId: purchase.id,
          cartIdentifier: user?.id || sessionId
        }
      });

      // Update purchase with Stripe session ID
      await storage.updatePurchase(purchase.id, {
        stripeSessionId: session.id
      });

      res.json({ url: session.url, sessionId: session.id });
    } catch (error) {
      console.error('Checkout session creation error:', error);
      res.status(500).json({ error: 'Failed to create checkout session' });
    }
  });
  */

  // Note: Stripe webhook now handled in index.ts BEFORE express.json() for proper signature verification
  // Old webhook endpoint (remove this later)
  /*
  app.post('/api/stripe/webhook', express.raw({ type: 'application/json' }), async (req: Request, res: Response) => {
    try {
      const stripeClient = getStripe();
      if (!stripeClient) {
        console.warn('Stripe not configured - webhook ignored');
        return res.status(200).json({ received: false, message: 'Stripe not configured' });
      }

      const sig = req.headers['stripe-signature'] as string;
      const endpointSecret = process.env.STRIPE_WEBHOOK_SECRET;

      if (!endpointSecret) {
        console.error('🚨 CRITICAL: STRIPE_WEBHOOK_SECRET not configured - webhook rejected for security');
        return res.status(400).json({ error: 'Webhook signature verification required - STRIPE_WEBHOOK_SECRET must be configured' });
      }

      let event: Stripe.Event;

      // SECURITY: Always require signature verification - no bypasses
      if (!sig) {
        console.error('🚨 CRITICAL: Missing Stripe signature header');
        return res.status(400).json({ error: 'Missing Stripe signature header' });
      }

      try {
        // Always verify webhook signature for security
        event = stripeClient.webhooks.constructEvent(req.body, sig, endpointSecret);
        console.log(`✅ Webhook signature verified for event: ${JSON.parse(req.body.toString()).type}`);
      } catch (err) {
        console.error('🚨 CRITICAL: Webhook signature verification failed:', err);
        return res.status(400).json({ error: 'Webhook signature verification failed' });
      }

      console.log(`🔔 Stripe webhook received: ${event.type}`);

      // Handle different event types
      switch (event.type) {
        case 'customer.subscription.updated': {
          const subscription = event.data.object as Stripe.Subscription;
          console.log(`🔄 Processing subscription update: ${subscription.id} (status: ${subscription.status}, customer: ${subscription.customer})`);

          // Extract user identification from metadata
          const uid = subscription.metadata?.uid;
          const email = subscription.metadata?.email || subscription.metadata?.user_email;

          if (!uid && !email) {
            console.warn(`⚠️ No user identification in subscription metadata for ${subscription.id}`);
            break;
          }

          // Get user from database
          let user;
          if (uid) {
            user = await storage.getUserByFirebaseUid(uid);
          } else if (email) {
            user = await storage.getUserByEmail(email);
          }

          if (!user) {
            console.warn(`⚠️ User not found for subscription ${subscription.id} (uid: ${uid}, email: ${email})`);
            break;
          }

          // Check if subscription pause was cleared (resumed)
          if (subscription.pause_collection === null && subscription.status === 'active') {
            console.log(`🚀 Subscription resumed: ${subscription.id} for user: ${user.email}`);
            
            // Update user status to active
            await storage.updateUser(user.id, {
              subscriptionStatus: 'active',
              pausedAt: null,
              resumeAt: null,
              isPaid: true
            });

            // Note: Quota updates now handled by entitlements.js system via webhook

            // Create notification for user
            await storage.createNotification(
              user.id,
              'Your subscription has been automatically resumed.',
              'info'
            );

            // NOTE: Domain credits are now only minted on customer.subscription.created to prevent duplicates
            // Resuming a subscription does not grant a new domain credit
            console.log(`ℹ️ Subscription resumed - domain credits are only granted on initial subscription creation`);
          }
          // Check if subscription was paused
          else if (subscription.pause_collection && subscription.status === 'active') {
            console.log(`Subscription paused: ${subscription.id} for user: ${user.email}`);
            
            const resumeAt = subscription.pause_collection.resumes_at 
              ? new Date(subscription.pause_collection.resumes_at * 1000) 
              : null;

            await storage.updateUser(user.id, {
              subscriptionStatus: 'paused',
              pausedAt: new Date(),
              resumeAt
            });

            // Create notification for user
            await storage.createNotification(
              user.id,
              `Your subscription has been paused${resumeAt ? ` until ${resumeAt.toLocaleDateString()}` : ''}.`,
              'info'
            );
          }
          break;
        }

        case 'customer.subscription.deleted':
        case 'customer.subscription.canceled': {
          const subscription = event.data.object as Stripe.Subscription;
          console.log(`Processing subscription cancellation: ${subscription.id}`);

          // Extract user identification from metadata
          const uid = subscription.metadata?.uid;
          const email = subscription.metadata?.email || subscription.metadata?.user_email;

          if (!uid && !email) {
            console.warn('No user identification in subscription metadata');
            break;
          }

          // Get user from database
          let user;
          if (uid) {
            user = await storage.getUserByFirebaseUid(uid);
          } else if (email) {
            user = await storage.getUserByEmail(email);
          }

          if (!user) {
            console.warn(`User not found for subscription ${subscription.id}`);
            break;
          }

          // Check if cancellation record already exists
          const existingCancellations = await storage.getCancellationsByUserId(user.id);
          
          if (existingCancellations.length === 0) {
            // Insert portal cancellation if no existing record
            await storage.createCancellation({
              userId: user.id,
              firebaseUid: user.firebaseUid,
              email: user.email,
              reason: 'Portal cancellation',
              note: `Cancelled via Stripe customer portal. Subscription ID: ${subscription.id}`,
            });

            console.log(`Created portal cancellation record for user: ${user.email}`);
          }

          // Update user status
          await storage.updateUser(user.id, {
            isPaid: false,
            subscriptionStatus: 'cancelled',
            pausedAt: null,
            resumeAt: null
          });

          // Create notification for user
          await storage.createNotification(
            user.id,
            'Your subscription has been cancelled.',
            'info'
          );

          console.log(`Updated user status for cancelled subscription: ${user.email}`);
          break;
        }

        case 'customer.subscription.created': {
          const subscription = event.data.object as Stripe.Subscription;
          console.log(`🎆 Processing new subscription: ${subscription.id} (status: ${subscription.status}, customer: ${subscription.customer})`);

          // Extract user identification from metadata
          const uid = subscription.metadata?.uid;
          const email = subscription.metadata?.email || subscription.metadata?.user_email;

          if (!uid && !email) {
            console.warn(`⚠️ No user identification in subscription metadata for ${subscription.id}`);
            break;
          }

          // Get user from database
          let user;
          if (uid) {
            user = await storage.getUserByFirebaseUid(uid);
          } else if (email) {
            user = await storage.getUserByEmail(email);
          }

          if (!user) {
            console.warn(`⚠️ User not found for subscription ${subscription.id} (uid: ${uid}, email: ${email})`);
            break;
          }

          // Only process active subscriptions for domain credit minting
          if (subscription.status !== 'active') {
            console.log(`ℹ️ Subscription ${subscription.id} status is '${subscription.status}', not processing domain credit`);
          }

          // Update user to paid status
          await storage.updateUser(user.id, {
            isPaid: true,
            subscriptionStatus: 'active',
            proActivatedAt: user.proActivatedAt || new Date(),
            pausedAt: null,
            resumeAt: null
          });

          // Note: Quota updates now handled by entitlements.js system via webhook

          // Create welcome notification for user
          await storage.createNotification(
            user.id,
            'Welcome to Pro! Your subscription is now active.',
            'success'
          );

          // SINGLE SOURCE OF TRUTH: Only mint domain credits here for active Pro subscriptions
          if (subscription.status === 'active') {
            const credit = await mintDomainCreditSafely(
              user.id, 
              subscription.id, 
              subscription,
              'New Pro subscription activation'
            );
            if (credit) {
              console.log(`🎉✅ Domain credit successfully minted for new Pro subscriber: ${user.email}`);
            } else {
              console.log(`ℹ️ No domain credit minted for user: ${user.email} (eligibility/idempotency checks failed)`);
            }
          }

          console.log(`✅ Activated Pro subscription for user: ${user.email}`);
          break;
        }

        case 'checkout.session.completed': {
          const session = event.data.object as Stripe.Checkout.Session;
          console.log(`💳 Processing checkout session completed: ${session.id} (mode: ${session.mode}, customer: ${session.customer})`);

          // Check session type
          const isDomainOrder = session.metadata?.type === 'domain_order';
          const isCartPurchase = session.metadata?.purchaseId;
          const isCreatorMarketplaceCheckout = session.metadata?.checkoutType === 'creator_marketplace';
          const isSubscriptionCheckout = session.mode === 'subscription';
          const isPresentationPurchase = session.metadata?.type === 'presentation_regular' || session.metadata?.type === 'presentation_premium';
          const isInfographicPurchase = session.metadata?.type === 'infographic_bundle';
          
          // Handle cart purchases
          if (isCartPurchase) {
            console.log(`🛒 Processing cart purchase checkout: ${session.id}`);
            
            try {
              const purchaseId = session.metadata.purchaseId;
              const cartIdentifier = session.metadata.cartIdentifier;
              
              // Get the purchase from database
              const purchase = await storage.getPurchase(purchaseId);
              if (!purchase) {
                console.error(`Purchase not found: ${purchaseId}`);
                break;
              }

              // Update purchase status to completed
              await storage.updatePurchase(purchaseId, {
                status: 'completed',
                stripePaymentIntentId: session.payment_intent as string,
                completedAt: new Date()
              });

              // Create entitlements for purchased items
              if (purchase.userId) {
                for (const item of purchase.items as any[]) {
                  await storage.createUserEntitlement({
                    userId: purchase.userId,
                    entitlementType: item.itemType,
                    entitlementId: item.itemId,
                    purchaseId: purchaseId,
                    status: 'active'
                  });
                }

                // Clear user's cart after successful purchase
                await storage.clearCart(purchase.userId);
              } else if (cartIdentifier) {
                // Clear session cart for unauthenticated users
                await storage.clearCart(undefined, cartIdentifier);
              }

              console.log(`✅ Cart purchase completed for purchase: ${purchaseId}`);
            } catch (error) {
              console.error(`Failed to process cart purchase for session ${session.id}:`, error);
            }
            break;
          }
          // Handle presentation purchases
          else if (isPresentationPurchase) {
            console.log(`📊 Processing presentation purchase checkout: ${session.id}`);
            
            try {
              const purchaseId = session.metadata.purchaseId;
              if (!purchaseId) {
                console.error(`Missing purchaseId in session metadata: ${session.id}`);
                break;
              }

              // Idempotency check - verify purchase exists and isn't already paid
              const existingPurchase = await storage.getPresentationPurchase(purchaseId);
              if (!existingPurchase) {
                console.error(`Presentation purchase not found: ${purchaseId}`);
                break;
              }

              if (existingPurchase.status === 'paid') {
                console.log(`✅ Session ${session.id} already processed successfully, skipping (idempotency protection)`);
                break;
              }

              // Validate amount and currency match expected values
              const expectedAmount = existingPurchase.type === 'regular' ? 2999 : 4999;
              const amountPaid = session.amount_total || 0;
              const currency = session.currency || 'usd';
              
              if (amountPaid !== expectedAmount) {
                console.error(`Amount mismatch for session ${session.id}: expected ${expectedAmount}, got ${amountPaid}`);
                break;
              }

              if (currency !== 'usd') {
                console.error(`Currency mismatch for session ${session.id}: expected usd, got ${currency}`);
                break;
              }
              
              // Mark purchase as paid and create download URL (placeholder for now)
              const downloadUrl = `https://ibrandbiz.com/downloads/presentations/${purchaseId}.zip`;
              
              await storage.markPresentationPurchasePaidBySession(
                session.id,
                session.payment_intent as string,
                downloadUrl,
                amountPaid,
                currency
              );
              
              console.log(`✅ Presentation purchase completed for session: ${session.id}`);
            } catch (error) {
              console.error(`Failed to process presentation purchase for session ${session.id}:`, error);
            }
            break;
          }
          // Handle infographic purchases  
          else if (isInfographicPurchase) {
            console.log(`📈 Processing infographic purchase checkout: ${session.id}`);
            
            try {
              const purchaseId = session.metadata.purchaseId;
              if (!purchaseId) {
                console.error(`Missing purchaseId in session metadata: ${session.id}`);
                break;
              }

              // Idempotency check - verify purchase exists and isn't already paid
              const existingPurchase = await storage.getInfographicPurchase(purchaseId);
              if (!existingPurchase) {
                console.error(`Infographic purchase not found: ${purchaseId}`);
                break;
              }

              if (existingPurchase.status === 'paid') {
                console.log(`✅ Session ${session.id} already processed successfully, skipping (idempotency protection)`);
                break;
              }

              // Validate amount and currency
              const expectedAmount = 1499; // $14.99 for infographic bundle
              const amountPaid = session.amount_total || 0;
              const currency = session.currency || 'usd';
              
              if (amountPaid !== expectedAmount) {
                console.error(`Amount mismatch for session ${session.id}: expected ${expectedAmount}, got ${amountPaid}`);
                break;
              }

              if (currency !== 'usd') {
                console.error(`Currency mismatch for session ${session.id}: expected usd, got ${currency}`);
                break;
              }
              
              // Create download URL (placeholder for now)
              const downloadUrl = `https://ibrandbiz.com/downloads/infographics/${purchaseId}.zip`;
              
              await storage.markInfographicPurchasePaidBySession(
                session.id,
                session.payment_intent as string,
                downloadUrl,
                amountPaid,
                currency
              );
              
              console.log(`✅ Infographic purchase completed for session: ${session.id}`);
            } catch (error) {
              console.error(`Failed to process infographic purchase for session ${session.id}:`, error);
            }
            break;
          }
          // Handle creator marketplace purchases
          else if (isCreatorMarketplaceCheckout) {
            console.log(`🎨 Processing creator marketplace checkout: ${session.id}`);
            
            try {
              // SECURITY: Idempotency check - prevent processing same session multiple times
              const existingPurchase = await storage.getPurchaseByStripeSessionId(session.id);
              if (!existingPurchase) {
                console.error(`Purchase not found for session ${session.id}`);
                break;
              }
              
              // SECURITY: Skip if already processed (idempotency protection)
              if (existingPurchase.status === 'completed') {
                console.log(`✅ Session ${session.id} already processed successfully, skipping (idempotency protection)`);
                break;
              }
              
              const userId = session.metadata?.userId;
              const creatorId = session.metadata?.creatorId;
              const totalAmount = parseInt(session.metadata?.totalAmount || '0');
              const platformFeeAmount = parseInt(session.metadata?.platformFeeAmount || '0');
              
              if (!userId || !creatorId) {
                console.error(`Missing required metadata for creator marketplace checkout ${session.id}`);
                break;
              }

              // Update purchase status
              await storage.updatePurchase(existingPurchase.id, {
                status: 'completed',
                stripePaymentIntentId: session.payment_intent as string,
                completedAt: new Date()
              });

              // Get creator for earning records
              const creator = await storage.getCreator(creatorId);
              if (!creator) {
                console.error(`Creator not found: ${creatorId}`);
                break;
              }

              // Process purchased items and create licenses
              const purchasedItems = existingPurchase.itemsData as any[];
              const licenses = [];
              
              for (const item of purchasedItems) {
                const creatorAsset = await storage.getCreatorAsset(item.itemId);
                if (!creatorAsset) {
                  console.warn(`Creator asset not found: ${item.itemId}`);
                  continue;
                }

                // Create license for each asset
                const license = await storage.createUserEntitlement({
                  userId: userId,
                  entitlementType: 'creator_asset',
                  entitlementId: creatorAsset.assetId, // Link to the actual asset
                  purchaseId: existingPurchase.id,
                  status: 'active',
                  metadata: {
                    creatorAssetId: creatorAsset.id,
                    creatorId: creatorId,
                    price: item.itemPrice,
                    purchasedAt: new Date().toISOString()
                  }
                });

                licenses.push(license);

                // Update creator asset sales stats
                const itemRevenue = item.itemPrice * item.quantity;
                await storage.incrementCreatorAssetSales(creatorAsset.id, itemRevenue);

                console.log(`📝 Created license for asset ${creatorAsset.assetId} for user ${userId}`);
              }

              // Create creator earning record
              const creatorNetAmount = totalAmount - platformFeeAmount;
              
              const earning = await storage.createCreatorEarning({
                creatorId: creatorId,
                type: 'sale',
                amountCents: creatorNetAmount,
                platformFeeCents: platformFeeAmount,
                grossAmountCents: totalAmount,
                stripePaymentIntentId: session.payment_intent as string,
                payoutStatus: 'pending',
                metadata: {
                  sessionId: session.id,
                  purchaseId: existingPurchase.id,
                  assetIds: purchasedItems.map(item => item.itemId),
                  itemCount: purchasedItems.length,
                  platformFeePercentage: Math.round((platformFeeAmount / totalAmount) * 100)
                }
              });

              // Update creator's total earnings
              await storage.updateCreatorEarnings(creatorId, creatorNetAmount);

              // Create success notification for buyer
              await storage.createNotification(
                userId,
                `🎉 Purchase successful! You now have access to ${purchasedItems.length} creator asset${purchasedItems.length > 1 ? 's' : ''}.`,
                'success'
              );

              // Create earning notification for creator
              const creatorUser = await storage.getUserByFirebaseUid((await storage.getCreator(creatorId))?.userId || '');
              if (creatorUser) {
                await storage.createNotification(
                  creatorUser.id,
                  `💰 You earned $${(creatorNetAmount / 100).toFixed(2)} from a new sale! (Platform fee: $${(platformFeeAmount / 100).toFixed(2)})`,
                  'success'
                );
              }

              console.log(`✅ Creator marketplace purchase completed: ${session.id}, Creator net: $${creatorNetAmount/100}, Platform fee: $${platformFeeAmount/100}`);
              
            } catch (error) {
              console.error(`Failed to process creator marketplace checkout for session ${session.id}:`, error);
            }
            break;
          }
          // Handle domain orders (existing logic)
          else if (isDomainOrder) {
            console.log(`🌐 Processing domain order checkout: ${session.id}`);
            // Domain order processing continues below...
          }
          // NOTE: Domain credits are now only minted on customer.subscription.created to prevent duplicates
          // We no longer mint credits on checkout.session.completed to eliminate race conditions
          else if (isSubscriptionCheckout) {
            console.log(`💵 Subscription checkout completed: ${session.id} - domain credits handled by subscription.created webhook`);
            break;
          }
          // Skip if not recognized type
          else {
            console.log(`ℹ️ Checkout session ${session.id} mode '${session.mode}' - not processing (no recognized type)`);
            break;
          }

          // Extract domain order data from session metadata
          const domainOrderId = session.metadata?.domain_order_id;
          const userId = session.metadata?.user_id;
          const domain = session.metadata?.domain;

          if (!domainOrderId || !userId || !domain) {
            console.error(`Missing required metadata in checkout session ${session.id}:`, session.metadata);
            break;
          }

          try {
            // Get the domain order from database
            const domainOrder = await storage.getDomainOrder(domainOrderId);
            if (!domainOrder) {
              console.error(`Domain order not found: ${domainOrderId}`);
              break;
            }

            // Check if already processed to ensure idempotency
            if (domainOrder.status === 'registering' || domainOrder.status === 'active') {
              console.log(`Domain order ${domainOrderId} already processed (status: ${domainOrder.status}), skipping`);
              break;
            }

            // Validate that the session matches the order
            if (domainOrder.stripeSessionId && domainOrder.stripeSessionId !== session.id) {
              console.error(`Session ID mismatch for domain order ${domainOrderId}: expected ${domainOrder.stripeSessionId}, got ${session.id}`);
              break;
            }

            // Update domain order status from payment_pending to registering
            await storage.updateDomainOrder(domainOrderId, {
              stripeSessionId: session.id,
              status: 'registering'
            });

            // Transform contact data to match DomainRegistrationJobData interface
            const registrantContact = domainOrder.registrantContact as any;
            const transformedContact = {
              firstName: registrantContact.firstName || registrantContact.first,
              lastName: registrantContact.lastName || registrantContact.last,
              email: registrantContact.email,
              phone: registrantContact.phone,
              address: registrantContact.address || registrantContact.address1,
              city: registrantContact.city,
              state: registrantContact.state,
              zip: registrantContact.zip || registrantContact.postal,
              country: registrantContact.country,
              organization: registrantContact.organization || registrantContact.org
            };

            // Enqueue domain registration job
            const jobData: DomainRegistrationJobData = {
              domainOrderId: domainOrder.id,
              domain: domainOrder.domain,
              years: domainOrder.years,
              contact: transformedContact,
              privacy: domainOrder.privacy,
              nameservers: domainOrder.nameservers,
              userId: domainOrder.userId,
              stripeSessionId: session.id
            };

            const jobId = jobQueue.enqueue('registerDomain', jobData, { maxRetries: 3 });
            
            console.log(`✅ Domain registration job enqueued: ${jobId} for domain: ${domain}`);

            // Create notification for user about payment success
            await storage.createNotification(
              userId,
              `Payment received for domain ${domain}. Registration is being processed.`,
              'info'
            );

          } catch (error) {
            console.error(`Error processing domain order checkout session ${session.id}:`, error);
            
            // Update order status to failed if there was an error
            try {
              await storage.updateDomainOrder(domainOrderId, {
                status: 'failed',
                errorMessage: 'Failed to process payment and enqueue registration job'
              });
            } catch (updateError) {
              console.error(`Failed to update domain order status to failed:`, updateError);
            }
            
            // Create error notification for user
            if (userId) {
              await storage.createNotification(
                userId,
                `Payment received for domain ${domain}, but there was an error processing the order. Please contact support.`,
                'error'
              );
            }
          }
          break;
        }

        default:
          console.log(`ℹ️ Unhandled webhook event type: ${event.type} (id: ${event.id})`);
          // Log for debugging - helps identify new webhook events we might want to handle
          if (process.env.NODE_ENV === 'development') {
            console.debug('Event data:', JSON.stringify(event.data, null, 2));
          }
      }

      res.json({ received: true, eventType: event.type });
    } catch (error) {
      console.error('Webhook error:', error);
      res.status(500).json({ error: 'Webhook processing failed' });
    }
  });
  */

  // Notification routes
  app.get('/api/notifications', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const notifications = await storage.getNotifications(req.user.uid);
      res.json(notifications);
    } catch (error) {
      handleError(res, error, 'Failed to get notifications');
    }
  });

  app.post('/api/notifications/mark-read/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const { id } = req.params;
      if (!id) {
        return res.status(400).json({ error: 'Notification ID is required' });
      }

      const success = await storage.markNotificationRead(id, req.user.uid);
      if (!success) {
        return res.status(404).json({ error: 'Notification not found or access denied' });
      }

      res.json({ ok: true });
    } catch (error) {
      handleError(res, error, 'Failed to mark notification as read');
    }
  });

  // SECURE: Generate stream token for SSE connection
  app.post('/api/notifications/stream-token', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Get user from database to ensure they exist
      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Generate secure stream token
      const streamToken = streamTokenService.generateToken(req.user.uid);
      
      res.json({ 
        streamToken,
        expiresIn: 300, // 5 minutes
        endpoint: '/api/notifications/stream'
      });
    } catch (error) {
      handleError(res, error, 'Failed to generate stream token');
    }
  });

  // SECURE SSE streaming endpoint using opaque tokens
  app.get('/api/notifications/stream', async (req: Request, res: Response) => {
    try {
      // Use secure stream token instead of Firebase token
      const streamToken = req.query.token as string;
      
      if (!streamToken) {
        res.status(401).json({ error: 'Stream token required' });
        return;
      }

      // Validate and consume the secure token (single-use)
      const tokenResult = streamTokenService.validateAndConsumeToken(streamToken);
      if (!tokenResult.valid) {
        console.warn(`SSE connection denied: ${tokenResult.reason}`);
        res.status(401).json({ error: 'Invalid or expired stream token' });
        return;
      }

      const userId = tokenResult.userId!;
      console.log(`SSE connection authenticated for user ${userId}`);
    
      // SECURITY: Set secure SSE headers (remove CORS wildcard)
      res.writeHead(200, {
        'Content-Type': 'text/event-stream',
        'Cache-Control': 'no-cache',
        'Connection': 'keep-alive',
        'X-Accel-Buffering': 'no' // Disable nginx buffering
      });

      // Add connection to the broadcaster with the stream token as ID
      notificationBroadcaster.addConnection(streamToken, userId, res);
      
    } catch (error) {
      console.error('SSE endpoint error:', error);
      res.status(500).json({ error: 'Internal server error' });
    }
  });

  // Test notification endpoint (development only)
  app.post('/api/notifications/test', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const { type = 'info', message = 'Test notification from the enhanced SSE system! 🎉' } = req.body;
      
      // Get user from database to get internal user ID
      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Create notification which will automatically broadcast via SSE
      const notification = await storage.createNotification(user.id, message, type);
      
      res.json({ 
        success: true, 
        message: 'Test notification created and broadcast via SSE',
        notification 
      });
    } catch (error) {
      handleError(res, error, 'Failed to create test notification');
    }
  });

  // Serve uploaded files
  app.use('/uploads', express.static('uploads'));
  
  // Serve public static files (includes site data and templates)
  app.use(express.static('public'));

  // Logo template upload endpoint
  const logoUpload = multer({ 
    storage: multer.memoryStorage(), 
    limits: { fileSize: 10 * 1024 * 1024 } // 10MB limit
  });

  app.post("/api/logo-templates/upload", requireAdmin, logoUpload.fields([
    { name: "svg", maxCount: 1 },
    { name: "preview", maxCount: 1 },
  ]), (req, res) => {
    try {
      const id = String(req.body.id || "").trim();
      const name = String(req.body.name || "").trim();
      const brandName = String(req.body.brandName || "");
      const tagline = String(req.body.tagline || "");
      const estYear = String(req.body.estYear || "");
      const primary = String(req.body.primary || "#231f20");
      const secondary = String(req.body.secondary || "#978752");
      const accent = String(req.body.accent || "#6dc282");
      const tags = String(req.body.tags || "").split(",").map(s=>s.trim()).filter(Boolean);

      if (!id || !/^[a-z0-9\-]+$/.test(id)) {
        return res.status(400).json({ error: "Invalid id (use kebab-case letters/numbers)" });
      }
      if (!name) {
        return res.status(400).json({ error: "Name is required" });
      }

      const svgFile = (req.files as any)?.svg?.[0];
      const previewFile = (req.files as any)?.preview?.[0];
      if (!svgFile) return res.status(400).json({ error: "SVG file is required" });
      if (!previewFile) return res.status(400).json({ error: "Preview file is required" });

      const svgText = svgFile.buffer.toString("utf-8");
      const tokenIssues: string[] = [];
      ["{Brand_Name}", "{Tagline}", "{Est_Year}"].forEach(tok => {
        if (!svgText.includes(tok)) tokenIssues.push(`Missing ${tok}`);
      });
      ["--primary","--secondary","--accent"].forEach(v => {
        if (!svgText.includes(v)) tokenIssues.push(`Missing ${v}`);
      });
      if (!/^\s*<svg[\s\S]*<\/svg>\s*$/i.test(svgText)) {
        tokenIssues.push("Not a valid <svg>…</svg>");
      }

      if (tokenIssues.length) {
        return res.status(400).json({ error: "SVG validation failed", details: tokenIssues });
      }

      // Ensure directories exist
      const PUBLIC_DIR = path.resolve("public");
      const MANIFEST_PATH = path.join(PUBLIC_DIR, "site", "data", "manifest.logo.json");
      const SRC_DIR = path.join(PUBLIC_DIR, "templates", "logo", "source");
      const PREV_DIR = path.join(PUBLIC_DIR, "templates", "logo", "previews");

      [PUBLIC_DIR, path.dirname(MANIFEST_PATH), SRC_DIR, PREV_DIR].forEach(p => {
        if (!fsSync.existsSync(p)) fsSync.mkdirSync(p, { recursive: true });
      });

      // Save SVG
      const svgOut = path.join(SRC_DIR, `${id}.svg`);
      fsSync.writeFileSync(svgOut, svgFile.buffer);

      // Save Preview
      const ext = previewFile.mimetype === "image/png" ? "png" : "jpg";
      const prevOut = path.join(PREV_DIR, `${id}.${ext}`);
      fsSync.writeFileSync(prevOut, previewFile.buffer);

      // Load or init manifest
      let manifest: any = { collection: "logo", version: 1, items: [] as any[] };
      if (fsSync.existsSync(MANIFEST_PATH)) {
        try { 
          manifest = JSON.parse(fsSync.readFileSync(MANIFEST_PATH, "utf-8")); 
        } catch (e) {
          console.error("Failed to parse existing manifest:", e);
        }
      }
      manifest.collection = "logo";
      manifest.version = Number(manifest.version || 1);

      const previewUrl = `/templates/logo/previews/${id}.${ext}`;
      const svgUrl = `/templates/logo/source/${id}.svg`;

      const item = {
        id,
        name,
        previewUrl,
        svgUrl,
        defaultFields: {
          Brand_Name: brandName,
          Tagline: tagline,
          Est_Year: estYear
        },
        fieldOrder: ["Brand_Name", "Tagline", "Est_Year"],
        defaultColors: { primary, secondary, accent },
        tags
      };

      const idx = (manifest.items || []).findIndex((x:any)=>x.id===id);
      if (idx >= 0) {
        manifest.items[idx] = item;
      } else {
        manifest.items.push(item);
      }

      fsSync.writeFileSync(MANIFEST_PATH, JSON.stringify(manifest, null, 2));
      return res.json({ ok: true, item });
    } catch (e: any) {
      console.error("Logo template upload error:", e);
      return res.status(500).json({ error: "Server error", detail: String(e?.message || e) });
    }
  });

  // User routes
  app.post('/api/users', async (req: Request, res: Response) => {
    try {
      const userData = insertUserSchema.parse(req.body);
      
      // Check if user already exists
      const existingUser = await storage.getUserByFirebaseUid(userData.firebaseUid);
      if (existingUser) {
        return res.status(200).json(existingUser);
      }

      const user = await storage.createUser(userData);
      res.status(201).json(user);
    } catch (error) {
      handleError(res, error, 'Failed to create user');
    }
  });

  app.get('/api/users/profile', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      res.json(user);
    } catch (error) {
      handleError(res, error, 'Failed to get user profile');
    }
  });

  // Brand Kits routes
  app.post('/api/brand-kits', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const brandKitData = insertBrandKitSchema.parse({
        ...req.body,
        userId: user.id
      });

      const brandKit = await storage.createBrandKit(brandKitData);
      res.status(201).json(brandKit);
    } catch (error) {
      handleError(res, error, 'Failed to create brand kit');
    }
  });

  app.get('/api/brand-kits', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const brandKits = await storage.getBrandKitsByUserId(user.id);
      res.json(brandKits);
    } catch (error) {
      handleError(res, error, 'Failed to get brand kits');
    }
  });

  // Creator Marketplace Routes

  /**
   * POST /api/creators/apply
   * Creator Application Route - Accept creator application with profile data
   */
  app.post('/api/creators/apply', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Check if user already has a creator profile - IDEMPOTENT BEHAVIOR
      const existingCreator = await storage.getCreatorByUserId(user.id);
      if (existingCreator) {
        // Return 200 with existing creator data for idempotent behavior
        return res.status(200).json({ 
          success: true,
          message: 'Creator application already exists',
          creator: existingCreator,
          idempotent: true // Flag to indicate this was an idempotent response
        });
      }

      // Validate profile data
      const profileDataSchema = z.object({
        name: z.string().min(1, 'Name is required'),
        bio: z.string().min(10, 'Bio must be at least 10 characters'),
        portfolio: z.string().url('Portfolio must be a valid URL').optional(),
        socialLinks: z.array(z.string().url()).optional(),
        skills: z.array(z.string()).optional()
      });

      const profileData = profileDataSchema.parse(req.body);

      // Create creator record with pending status
      const creatorData: InsertCreator = {
        userId: user.id,
        onboardingStatus: CREATOR_ONBOARDING_STATUSES.PENDING,
        profileData,
        isActive: true
      };

      const creator = await storage.createCreator(creatorData);
      
      // Create notification for successful application
      await storage.createNotification(
        user.id,
        '🎨 Your creator application has been submitted! We\'ll review it and get back to you soon.',
        'success'
      );

      res.status(201).json({
        success: true,
        message: 'Creator application submitted successfully',
        creator
      });
    } catch (error) {
      console.error('Creator application error:', error);
      handleError(res, error, 'Failed to submit creator application');
    }
  });

  /**
   * POST /api/creators/stripe-connect/create
   * Stripe Connect Account Creation - Create Stripe Connect Express account
   */
  app.post('/api/creators/stripe-connect/create', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Get creator profile
      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found. Please apply first.' });
      }

      // Check if creator is in pending status (approved for onboarding)
      if (creator.onboardingStatus !== CREATOR_ONBOARDING_STATUSES.PENDING) {
        return res.status(400).json({ 
          error: 'Creator onboarding not available', 
          status: creator.onboardingStatus 
        });
      }

      // IDEMPOTENCY CHECK: Return existing account if stripeConnectAccountId already exists
      if (creator.stripeConnectAccountId) {
        // Return the existing account instead of creating duplicate
        return res.json({
          success: true,
          message: 'Stripe Connect account already exists',
          accountId: creator.stripeConnectAccountId,
          creator: creator,
          idempotent: true // Flag to indicate this was an idempotent response
        });
      }

      const stripeInstance = getStripe();
      if (!stripeInstance) {
        return handleStripeUnavailable(res, 'Creating Stripe Connect accounts');
      }

      // VALIDATE STATE TRANSITION before creating account
      if (!isValidStateTransition(creator.onboardingStatus, CREATOR_ONBOARDING_STATUSES.IN_PROGRESS)) {
        return res.status(400).json({
          error: 'Invalid state transition',
          message: `Cannot transition from '${creator.onboardingStatus}' to 'in_progress'`,
          currentStatus: creator.onboardingStatus
        });
      }

      // Create Stripe Connect Express account
      const account = await stripeInstance.accounts.create({
        type: 'express',
        country: 'US', // Default to US, could be made configurable
        email: user.email,
        metadata: {
          userId: user.id,
          creatorId: creator.id,
          createdBy: 'creator-marketplace',
          createdAt: new Date().toISOString()
        }
      });

      // Update creator with Stripe Connect account ID and status
      const updatedCreator = await storage.updateCreator(creator.id, {
        stripeConnectAccountId: account.id,
        onboardingStatus: CREATOR_ONBOARDING_STATUSES.IN_PROGRESS
      });

      // Create notification
      await storage.createNotification(
        user.id,
        '💳 Stripe Connect account created! Complete your onboarding to start receiving payments.',
        'success'
      );

      res.json({
        success: true,
        message: 'Stripe Connect account created successfully',
        accountId: account.id,
        creator: updatedCreator
      });
    } catch (error) {
      console.error('Stripe Connect account creation error:', error);
      handleError(res, error, 'Failed to create Stripe Connect account');
    }
  });

  /**
   * POST /api/creators/stripe-connect/onboarding-link
   * Generate Stripe Connect onboarding link
   */
  app.post('/api/creators/stripe-connect/onboarding-link', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found. Please apply first.' });
      }

      // ACCOUNT EXISTENCE CHECK: Ensure Stripe Connect account exists
      if (!creator.stripeConnectAccountId) {
        return res.status(404).json({ 
          error: 'Stripe Connect account not found', 
          message: 'Please create your Stripe Connect account first before generating onboarding link.',
          action: 'create_stripe_account'
        });
      }

      // STATE VALIDATION: Only allow onboarding link for appropriate statuses
      const allowedStatuses = [CREATOR_ONBOARDING_STATUSES.PENDING, CREATOR_ONBOARDING_STATUSES.IN_PROGRESS];
      if (!allowedStatuses.includes(creator.onboardingStatus)) {
        return res.status(400).json({
          error: 'Onboarding link not available',
          message: `Onboarding links can only be generated for creators in 'pending' or 'in_progress' status`,
          currentStatus: creator.onboardingStatus,
          allowedStatuses
        });
      }

      const stripeInstance = getStripe();
      if (!stripeInstance) {
        return handleStripeUnavailable(res, 'Generating onboarding links');
      }

      // VERIFY ACCOUNT EXISTS IN STRIPE before creating link
      try {
        await stripeInstance.accounts.retrieve(creator.stripeConnectAccountId);
      } catch (stripeError: any) {
        console.error('Stripe account verification failed:', stripeError);
        if (stripeError.code === 'resource_missing') {
          return res.status(404).json({
            error: 'Stripe account not found',
            message: 'The Stripe Connect account appears to have been deleted. Please create a new account.',
            action: 'recreate_stripe_account'
          });
        }
        throw stripeError; // Re-throw other errors
      }

      // SECURE REDIRECT URLS: Build URLs from environment variables only
      const refreshUrl = buildSecureRedirectUrl('/creator/dashboard?refresh=true');
      const returnUrl = buildSecureRedirectUrl('/creator/dashboard?onboarding=complete');
      
      const accountLink = await stripeInstance.accountLinks.create({
        account: creator.stripeConnectAccountId,
        refresh_url: refreshUrl,
        return_url: returnUrl,
        type: 'account_onboarding'
      });

      res.json({
        success: true,
        onboardingUrl: accountLink.url,
        expiresAt: accountLink.expires_at,
        refreshUrl,
        returnUrl,
        accountId: creator.stripeConnectAccountId
      });
    } catch (error) {
      console.error('Stripe Connect onboarding link error:', error);
      handleError(res, error, 'Failed to generate onboarding link');
    }
  });

  /**
   * GET /api/creators/stripe-connect/status
   * Check Stripe Connect account status and capabilities
   */
  app.get('/api/creators/stripe-connect/status', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found' });
      }

      if (!creator.stripeConnectAccountId) {
        return res.status(404).json({ 
          error: 'Stripe Connect account not found',
          message: 'No Stripe Connect account associated with this creator profile',
          onboardingStatus: creator.onboardingStatus
        });
      }

      const stripeInstance = getStripe();
      if (!stripeInstance) {
        return handleStripeUnavailable(res, 'Checking account status');
      }

      // Get account details from Stripe with enhanced error handling
      let account;
      try {
        account = await stripeInstance.accounts.retrieve(creator.stripeConnectAccountId);
      } catch (stripeError: any) {
        console.error('Failed to retrieve Stripe account:', stripeError);
        if (stripeError.code === 'resource_missing') {
          return res.status(404).json({
            error: 'Stripe account not found',
            message: 'The Stripe Connect account appears to have been deleted',
            code: 'STRIPE_ACCOUNT_MISSING'
          });
        }
        return res.status(502).json({
          error: 'Stripe service error',
          message: 'Unable to retrieve account status from Stripe',
          code: 'STRIPE_SERVICE_ERROR'
        });
      }
      
      // Check account capabilities and requirements
      const charges_enabled = account.charges_enabled;
      const payouts_enabled = account.payouts_enabled;
      const requirements = account.requirements;
      const currently_due = requirements?.currently_due || [];
      const eventually_due = requirements?.eventually_due || [];

      // Determine new status based on Stripe account state
      let newOnboardingStatus = creator.onboardingStatus;
      let payoutEnabled = creator.payoutEnabled;
      let onboardingCompletedAt = creator.onboardingCompletedAt;

      if (charges_enabled && payouts_enabled && currently_due.length === 0) {
        newOnboardingStatus = CREATOR_ONBOARDING_STATUSES.COMPLETED;
        payoutEnabled = true;
        // SET ONBOARDING COMPLETED TIMESTAMP when enabling payouts
        if (!creator.onboardingCompletedAt) {
          onboardingCompletedAt = new Date();
        }
      } else if (currently_due.length > 0) {
        newOnboardingStatus = CREATOR_ONBOARDING_STATUSES.IN_PROGRESS;
        payoutEnabled = false;
      }

      // VALIDATE STATE TRANSITION before updating
      if (newOnboardingStatus !== creator.onboardingStatus) {
        if (!isValidStateTransition(creator.onboardingStatus, newOnboardingStatus)) {
          console.warn(`Invalid state transition attempted: ${creator.onboardingStatus} -> ${newOnboardingStatus} for creator ${creator.id}`);
          // Log the issue but don't fail - this might be a legitimate Stripe-driven state change
          // Instead, we'll allow the transition but log it for monitoring
        }
      }

      // Update creator record if any status changed
      const hasStatusChanged = newOnboardingStatus !== creator.onboardingStatus;
      const hasPayoutChanged = payoutEnabled !== creator.payoutEnabled;
      const hasCompletionChanged = onboardingCompletedAt !== creator.onboardingCompletedAt;
      
      if (hasStatusChanged || hasPayoutChanged || hasCompletionChanged) {
        await storage.updateCreator(creator.id, {
          onboardingStatus: newOnboardingStatus,
          payoutEnabled,
          onboardingCompletedAt
        });

        // Create notification if onboarding completed for the first time
        if (newOnboardingStatus === CREATOR_ONBOARDING_STATUSES.COMPLETED && creator.onboardingStatus !== CREATOR_ONBOARDING_STATUSES.COMPLETED) {
          await storage.createNotification(
            user.id,
            '🎉 Creator onboarding completed! You can now receive payments for your assets.',
            'success'
          );
        }
      }

      res.json({
        onboardingStatus: newOnboardingStatus,
        stripeAccountStatus: {
          charges_enabled,
          payouts_enabled,
          currently_due,
          eventually_due,
          disabled_reason: account.requirements?.disabled_reason,
          details_submitted: account.details_submitted,
          payouts_enabled: account.payouts_enabled
        },
        payoutEnabled,
        onboardingCompletedAt,
        requiresAction: currently_due.length > 0,
        statusChanged: hasStatusChanged,
        lastUpdated: new Date().toISOString()
      });
    } catch (error) {
      console.error('Stripe Connect status check error:', error);
      handleError(res, error, 'Failed to check Stripe Connect status');
    }
  });

  /**
   * GET /api/creators/profile
   * Get creator profile with onboarding status and earnings summary
   */
  app.get('/api/creators/profile', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found' });
      }

      // Get creator's assets and earnings
      const creatorAssets = await storage.getCreatorAssetsByCreatorId(creator.id);
      const creatorEarnings = await storage.getCreatorEarningsByCreatorId(creator.id);

      // Calculate earnings summary
      const totalEarnings = creatorEarnings.reduce((sum, earning) => sum + earning.amountCents, 0);
      const pendingPayouts = creatorEarnings
        .filter(earning => earning.payoutStatus === 'pending')
        .reduce((sum, earning) => sum + earning.amountCents, 0);
      
      const approvedAssets = creatorAssets.filter(asset => asset.approvalStatus === 'approved').length;
      const totalSales = creatorAssets.reduce((sum, asset) => sum + (asset.salesCount || 0), 0);

      // Get Stripe account status if available with proper error handling
      let stripeAccountStatus = null;
      if (creator.stripeConnectAccountId) {
        const stripeInstance = getStripe();
        if (stripeInstance) {
          try {
            const account = await stripeInstance.accounts.retrieve(creator.stripeConnectAccountId);
            stripeAccountStatus = {
              charges_enabled: account.charges_enabled,
              payouts_enabled: account.payouts_enabled,
              currently_due: account.requirements?.currently_due || [],
              eventually_due: account.requirements?.eventually_due || [],
              disabled_reason: account.requirements?.disabled_reason,
              details_submitted: account.details_submitted
            };
          } catch (stripeError: any) {
            console.warn('Failed to fetch Stripe account status for profile:', stripeError);
            // Don't expose Stripe errors to client, but provide helpful status
            stripeAccountStatus = {
              error: 'Unable to fetch current status',
              charges_enabled: false,
              payouts_enabled: false,
              currently_due: ['status_check_failed']
            };
          }
        } else {
          stripeAccountStatus = {
            error: 'Stripe service unavailable',
            charges_enabled: false,
            payouts_enabled: false,
            currently_due: ['stripe_service_unavailable']
          };
        }
      }

      res.json({
        creator,
        profileData: creator.profileData,
        onboardingStatus: creator.onboardingStatus,
        stripeAccountStatus,
        earningsSummary: {
          totalEarnings,
          pendingPayouts,
          approvedAssets,
          totalSales,
          payoutEnabled: creator.payoutEnabled
        },
        stats: {
          assetsCount: creatorAssets.length,
          approvedAssetsCount: approvedAssets,
          pendingAssetsCount: creatorAssets.filter(asset => asset.approvalStatus === 'pending').length,
          rejectedAssetsCount: creatorAssets.filter(asset => asset.approvalStatus === 'rejected').length
        }
      });
    } catch (error) {
      console.error('Creator profile error:', error);
      handleError(res, error, 'Failed to get creator profile');
    }
  });

  // Creator Asset Upload Configuration - Support multiple file types for digital assets
  const assetUpload = multer({
    dest: "uploads/assets/",
    limits: {
      fileSize: 25 * 1024 * 1024, // 25MB limit for digital assets
    },
    fileFilter: (req, file, cb) => {
      // Allow images, SVGs, and other digital asset types
      const allowedTypes = [
        'image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp',
        'image/svg+xml', 
        'application/pdf',
        'application/zip',
        'text/plain'
      ];
      
      if (allowedTypes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error(`File type ${file.mimetype} not allowed. Supported types: images, SVGs, PDFs, ZIP files, and text files.`));
      }
    }
  });

  /**
   * Creator Asset Upload Routes
   */

  /**
   * GET /api/creator/my-assets
   * Get creator's own assets filtered by status (for MyMarketplacePage)
   */
  app.get('/api/creator/my-assets', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Get creator profile
      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found' });
      }

      // Parse status filter
      const status = req.query.status as string;
      if (status && !['active', 'cancelled', 'pending', 'rejected'].includes(status)) {
        return res.status(400).json({ error: 'Invalid status filter. Must be: active, cancelled, pending, or rejected' });
      }

      // Get creator assets
      const creatorAssets = await storage.getCreatorAssetsByCreatorId(creator.id);
      
      // Map approval status to user-friendly status
      let filteredAssets = creatorAssets;
      if (status === 'active') {
        filteredAssets = creatorAssets.filter(asset => asset.approvalStatus === 'approved');
      } else if (status === 'cancelled') {
        filteredAssets = creatorAssets.filter(asset => asset.approvalStatus === 'cancelled');
      } else if (status) {
        filteredAssets = creatorAssets.filter(asset => asset.approvalStatus === status);
      }

      // Enrich with asset details
      const enrichedAssets = await Promise.all(
        filteredAssets.map(async (creatorAsset) => {
          const asset = await storage.getAsset(creatorAsset.assetId);
          if (!asset) return null;

          return {
            id: creatorAsset.id,
            assetId: creatorAsset.assetId,
            title: creatorAsset.title,
            description: creatorAsset.description,
            price: creatorAsset.price,
            isExclusive: creatorAsset.isExclusive,
            salesCount: creatorAsset.salesCount,
            status: creatorAsset.approvalStatus === 'approved' ? 'active' : creatorAsset.approvalStatus,
            createdAt: creatorAsset.createdAt,
            asset: {
              fileName: asset.originalFileName,
              fileUrl: asset.cloudUrl,
              previewUrl: asset.previewUrl,
              fileSize: asset.fileSize,
              mimeType: asset.mimeType,
              assetType: asset.assetType,
              category: asset.category,
              tags: asset.tags,
              downloadCount: asset.downloadCount
            }
          };
        })
      );

      const validAssets = enrichedAssets.filter(asset => asset !== null);
      
      // Sort by newest first
      validAssets.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());

      res.json(validAssets);

    } catch (error) {
      console.error('Creator my-assets error:', error);
      handleError(res, error, 'Failed to get creator assets');
    }
  });

  /**
   * POST /api/creator/assets/:id/cancel
   * Cancel/remove an asset from marketplace (for MyMarketplacePage)
   */
  app.post('/api/creator/assets/:id/cancel', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Get creator profile
      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found' });
      }

      // Get the creator asset
      const creatorAsset = await storage.getCreatorAsset(req.params.id);
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Verify ownership
      if (creatorAsset.creatorId !== creator.id) {
        return res.status(403).json({ error: 'Access denied. You can only cancel your own assets.' });
      }

      // Check if asset is already cancelled
      if (creatorAsset.approvalStatus === 'cancelled') {
        return res.status(400).json({ error: 'Asset is already cancelled' });
      }

      // Update asset status to cancelled
      await storage.updateCreatorAsset(creatorAsset.id, {
        approvalStatus: 'cancelled',
        updatedAt: new Date().toISOString()
      });

      console.log(`✅ Asset ${creatorAsset.id} cancelled by creator ${creator.id}`);

      res.json({
        success: true,
        message: 'Asset cancelled successfully',
        assetId: creatorAsset.id
      });

    } catch (error) {
      console.error('Creator asset cancellation error:', error);
      handleError(res, error, 'Failed to cancel asset');
    }
  });

  /**
   * GET /api/creator/payouts/summary
   * Get creator's payout summary (for MyMarketplacePage)
   */
  app.get('/api/creator/payouts/summary', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Get creator profile
      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found' });
      }

      // Get creator earnings
      const creatorEarnings = await storage.getCreatorEarningsByCreatorId(creator.id);

      // Calculate lifetime earnings
      const lifetimeEarningsCents = creatorEarnings.reduce((sum, earning) => sum + earning.amountCents, 0);
      
      // Calculate pending payouts
      const pendingPayoutCents = creatorEarnings
        .filter(earning => earning.payoutStatus === 'pending')
        .reduce((sum, earning) => sum + earning.amountCents, 0);

      // Get last payout date
      const paidEarnings = creatorEarnings
        .filter(earning => earning.payoutStatus === 'paid')
        .sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
      
      const lastPayoutAt = paidEarnings.length > 0 ? paidEarnings[0].createdAt : undefined;

      res.json({
        lifetimeEarningsCents,
        pendingPayoutCents,
        lastPayoutAt
      });

    } catch (error) {
      console.error('Creator payouts summary error:', error);
      handleError(res, error, 'Failed to get payouts summary');
    }
  });

  /**
   * GET /api/creator/me/marketplace
   * SuperNova's tabbed interface endpoint - returns grouped assets and stats
   */
  app.get('/api/creator/me/marketplace', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Get creator profile
      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found' });
      }

      // Get all creator assets
      const creatorAssets = await storage.getCreatorAssetsByCreatorId(creator.id);

      // Group assets by approval status
      const groupByStatus = (status: string) => creatorAssets.filter(asset => asset.approvalStatus === status);

      // Enrich assets with asset details
      const enrichAssets = async (assets: any[]) => {
        const enriched = await Promise.all(
          assets.map(async (creatorAsset) => {
            const asset = await storage.getAsset(creatorAsset.assetId);
            if (!asset) return null;

            return {
              id: creatorAsset.id,
              title: creatorAsset.title,
              description: creatorAsset.description,
              price: creatorAsset.price,
              salesCount: creatorAsset.salesCount || 0,
              createdAt: creatorAsset.createdAt,
              previewUrl: asset.previewUrl,
              approvalStatus: creatorAsset.approvalStatus
            };
          })
        );
        return enriched.filter(Boolean);
      };

      const [approved, pending, rejected, cancelled] = await Promise.all([
        enrichAssets(groupByStatus('approved')),
        enrichAssets(groupByStatus('pending')),
        enrichAssets(groupByStatus('rejected')),
        enrichAssets(groupByStatus('cancelled'))
      ]);

      // Calculate stats
      const totalSales = creatorAssets.reduce((sum, asset) => sum + (asset.salesCount || 0), 0);
      const totalRevenueCents = creatorAssets.reduce((sum, asset) => sum + (asset.totalRevenue || 0), 0);

      res.json({
        assets: {
          approved,
          pending,
          rejected,
          cancelled
        },
        stats: {
          totalSales,
          totalRevenueCents
        }
      });

    } catch (error) {
      console.error('Creator marketplace overview error:', error);
      handleError(res, error, 'Failed to load creator marketplace');
    }
  });

  /**
   * GET /api/creator/me/payouts/summary  
   * SuperNova's tabbed interface endpoint (alias to existing payouts endpoint)
   */
  app.get('/api/creator/me/payouts/summary', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Get creator profile
      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(404).json({ error: 'Creator profile not found' });
      }

      // Get creator earnings (reuse existing logic)
      const creatorEarnings = await storage.getCreatorEarningsByCreatorId(creator.id);

      // Calculate lifetime earnings
      const lifetimeEarningsCents = creatorEarnings.reduce((sum, earning) => sum + earning.amountCents, 0);
      
      // Calculate pending payouts
      const pendingPayoutCents = creatorEarnings
        .filter(earning => earning.payoutStatus === 'pending')
        .reduce((sum, earning) => sum + earning.amountCents, 0);

      // Get last payout date
      const paidEarnings = creatorEarnings
        .filter(earning => earning.payoutStatus === 'paid')
        .sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
      
      const lastPayoutAt = paidEarnings.length > 0 ? paidEarnings[0].createdAt : undefined;

      res.json({
        lifetimeEarningsCents,
        pendingPayoutCents,
        lastPayoutAt
      });

    } catch (error) {
      console.error('Creator payouts summary error:', error);
      handleError(res, error, 'Failed to get payouts summary');
    }
  });

  /**
   * POST /api/creators/assets/upload
   * Upload digital asset with metadata for marketplace
   */
  app.post('/api/creators/assets/upload', authenticateToken, assetUpload.single("file"), async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      if (!req.file) {
        return res.status(400).json({ error: 'No file uploaded' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Verify user has creator profile
      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(403).json({ error: 'Creator profile required. Please apply first.' });
      }

      // Validate creator is active and approved
      if (!creator.isActive || creator.onboardingStatus !== CREATOR_ONBOARDING_STATUSES.COMPLETED) {
        return res.status(403).json({ 
          error: 'Creator account not active or onboarding not completed',
          status: creator.onboardingStatus
        });
      }

      // Parse and validate metadata from request body
      const metadataSchema = z.object({
        title: z.string().min(1, 'Title is required').max(100, 'Title too long'),
        description: z.string().optional(),
        tags: z.array(z.string()).default([]),
        category: z.string().optional(),
        price: z.number().int().min(CREATOR_MIN_PRICE_CENTS).max(CREATOR_MAX_PRICE_CENTS)
      });

      const metadata = metadataSchema.parse({
        title: req.body.title,
        description: req.body.description,
        tags: req.body.tags ? (Array.isArray(req.body.tags) ? req.body.tags : JSON.parse(req.body.tags)) : [],
        category: req.body.category,
        price: parseInt(req.body.price)
      });

      // Generate unique filename and upload to object storage
      const fileExtension = req.file.originalname.split('.').pop() || '';
      const uniqueFileName = `${Date.now()}-${Math.random().toString(36).substr(2, 9)}.${fileExtension}`;
      
      // SECURE UPLOAD: Store original in private directory and generate public preview
      const objectStorageService = new ObjectStorageService();
      const privateDir = objectStorageService.getPrivateObjectDir();
      const publicPaths = objectStorageService.getPublicObjectSearchPaths();
      const publicDir = publicPaths[0]; // Use first public directory
      
      // Private path for original file (secure)
      const originalPath = `${privateDir}/assets/originals/${creator.id}/${uniqueFileName}`;
      
      // Public path for watermarked preview
      const previewFileName = `preview_${uniqueFileName}`;
      const previewPath = `${publicDir}/assets/previews/${creator.id}/${previewFileName}`;
      
      // Read file buffer
      const fileBuffer = await fs.readFile(req.file.path);
      
      // Upload original to private storage
      await objectStorageService.uploadFile(originalPath, fileBuffer, req.file.mimetype);
      console.log(`🔒 Original asset stored privately: ${originalPath}`);
      
      // Generate and upload watermarked preview for supported image types
      let previewUrl = null;
      if (req.file.mimetype.startsWith('image/') && req.file.mimetype !== 'image/svg+xml') {
        try {
          // Generate watermarked preview using the existing watermark service
          const assetCode = toTenDigitCode(uniqueFileName);
          const watermarkedBuffer = await applyWatermark(fileBuffer, {
            opacity: 0.15,
            stockCode: assetCode
          });
          
          // Upload watermarked preview to public storage
          await objectStorageService.uploadFile(previewPath, watermarkedBuffer, req.file.mimetype);
          previewUrl = `/public-objects/${previewPath.replace(publicDir + '/', '')}`;
          console.log(`📷 Watermarked preview generated: ${previewPath}`);
        } catch (watermarkError) {
          console.warn('Failed to generate watermarked preview:', watermarkError);
          // Continue without preview - asset can still be uploaded
        }
      }
      
      // Clean up temporary file
      await fs.unlink(req.file.path);

      // Determine asset type from mime type
      let assetType = 'document';
      if (req.file.mimetype.startsWith('image/')) {
        assetType = req.file.mimetype === 'image/svg+xml' ? 'svg' : 'image';
      } else if (req.file.mimetype === 'application/pdf') {
        assetType = 'pdf';
      } else if (req.file.mimetype === 'application/zip') {
        assetType = 'archive';
      }

      // Create asset record with original and preview paths
      const assetData = {
        creatorId: creator.id,
        fileName: uniqueFileName,
        originalFileName: req.file.originalname,
        fileUrl: originalPath, // Private original path
        previewUrl: previewUrl, // Public preview path (if generated)
        fileSize: req.file.size,
        mimeType: req.file.mimetype,
        assetType,
        category: metadata.category,
        tags: metadata.tags,
        metadata: {
          uploadedBy: user.id,
          uploadedAt: new Date().toISOString(),
          hasWatermarkedPreview: !!previewUrl
        },
        isPublic: false // Assets start as private until approved
      };

      const asset = await storage.createAsset(assetData);

      // Create creator asset marketplace record with pending status
      const creatorAssetData = {
        creatorId: creator.id,
        assetId: asset.id,
        title: metadata.title,
        description: metadata.description,
        price: metadata.price,
        approvalStatus: 'pending' as const,
        isExclusive: false
      };

      const creatorAsset = await storage.createCreatorAsset(creatorAssetData);

      // Create notification for successful upload
      await storage.createNotification(
        user.id,
        `🎨 Asset "${metadata.title}" uploaded successfully! It's now pending approval for the marketplace.`,
        'success'
      );

      res.status(201).json({
        success: true,
        message: 'Asset uploaded successfully',
        asset: {
          id: creatorAsset.id,
          assetId: asset.id,
          title: creatorAsset.title,
          description: creatorAsset.description,
          price: creatorAsset.price,
          approvalStatus: creatorAsset.approvalStatus,
          fileName: asset.fileName,
          originalFileName: asset.originalFileName,
          fileSize: asset.fileSize,
          assetType: asset.assetType,
          category: asset.category,
          tags: asset.tags,
          createdAt: creatorAsset.createdAt
        }
      });

    } catch (error) {
      // Clean up temporary file if it exists
      if (req.file?.path) {
        try {
          await fs.unlink(req.file.path);
        } catch (cleanupError) {
          console.warn('Failed to clean up temporary file:', cleanupError);
        }
      }
      
      console.error('Asset upload error:', error);
      handleError(res, error, 'Failed to upload asset');
    }
  });

  /**
   * GET /api/creators/assets
   * List creator's uploaded assets with status and stats
   */
  app.get('/api/creators/assets', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(403).json({ error: 'Creator profile not found' });
      }

      // Get creator assets with their associated asset details
      const creatorAssets = await storage.getCreatorAssetsByCreatorId(creator.id);
      
      // Enrich with asset file details
      const enrichedAssets = await Promise.all(
        creatorAssets.map(async (creatorAsset) => {
          const asset = await storage.getAsset(creatorAsset.assetId);
          return {
            id: creatorAsset.id,
            assetId: creatorAsset.assetId,
            title: creatorAsset.title,
            description: creatorAsset.description,
            price: creatorAsset.price,
            approvalStatus: creatorAsset.approvalStatus,
            rejectionReason: creatorAsset.rejectionReason,
            isExclusive: creatorAsset.isExclusive,
            salesCount: creatorAsset.salesCount,
            totalEarnings: creatorAsset.totalEarnings,
            createdAt: creatorAsset.createdAt,
            updatedAt: creatorAsset.updatedAt,
            // Asset file details
            fileName: asset?.fileName,
            originalFileName: asset?.originalFileName,
            fileSize: asset?.fileSize,
            assetType: asset?.assetType,
            category: asset?.category,
            tags: asset?.tags,
            downloadCount: asset?.downloadCount
          };
        })
      );

      // Calculate stats
      const stats = {
        total: enrichedAssets.length,
        pending: enrichedAssets.filter(a => a.approvalStatus === 'pending').length,
        approved: enrichedAssets.filter(a => a.approvalStatus === 'approved').length,
        rejected: enrichedAssets.filter(a => a.approvalStatus === 'rejected').length,
        totalSales: enrichedAssets.reduce((sum, a) => sum + (a.salesCount || 0), 0),
        totalEarnings: enrichedAssets.reduce((sum, a) => sum + (a.totalEarnings || 0), 0)
      };

      res.json({
        assets: enrichedAssets,
        stats
      });

    } catch (error) {
      console.error('Creator assets list error:', error);
      handleError(res, error, 'Failed to get creator assets');
    }
  });

  /**
   * GET /api/creators/assets/:id
   * Get specific asset details for creator
   */
  app.get('/api/creators/assets/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(403).json({ error: 'Creator profile not found' });
      }

      const creatorAsset = await storage.getCreatorAsset(req.params.id);
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Verify ownership
      if (creatorAsset.creatorId !== creator.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      // Get associated asset file details
      const asset = await storage.getAsset(creatorAsset.assetId);
      if (!asset) {
        return res.status(404).json({ error: 'Asset file not found' });
      }

      const enrichedAsset = {
        id: creatorAsset.id,
        assetId: creatorAsset.assetId,
        title: creatorAsset.title,
        description: creatorAsset.description,
        price: creatorAsset.price,
        approvalStatus: creatorAsset.approvalStatus,
        rejectionReason: creatorAsset.rejectionReason,
        isExclusive: creatorAsset.isExclusive,
        salesCount: creatorAsset.salesCount,
        totalEarnings: creatorAsset.totalEarnings,
        createdAt: creatorAsset.createdAt,
        updatedAt: creatorAsset.updatedAt,
        // Asset file details
        fileName: asset.fileName,
        originalFileName: asset.originalFileName,
        fileSize: asset.fileSize,
        assetType: asset.assetType,
        category: asset.category,
        tags: asset.tags,
        downloadCount: asset.downloadCount,
        metadata: asset.metadata
      };

      res.json(enrichedAsset);

    } catch (error) {
      console.error('Creator asset details error:', error);
      handleError(res, error, 'Failed to get asset details');
    }
  });

  /**
   * PATCH /api/creators/assets/:id
   * Update asset metadata (title, description, price, tags)
   */
  app.patch('/api/creators/assets/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(403).json({ error: 'Creator profile not found' });
      }

      const creatorAsset = await storage.getCreatorAsset(req.params.id);
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Verify ownership
      if (creatorAsset.creatorId !== creator.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      // Don't allow editing approved assets (would require re-approval)
      if (creatorAsset.approvalStatus === 'approved') {
        return res.status(400).json({ 
          error: 'Cannot edit approved assets. Contact support if changes are needed.' 
        });
      }

      // Validate update data
      const updateSchema = z.object({
        title: z.string().min(1, 'Title is required').max(100, 'Title too long').optional(),
        description: z.string().optional(),
        price: z.number().int().min(CREATOR_MIN_PRICE_CENTS).max(CREATOR_MAX_PRICE_CENTS).optional(),
        tags: z.array(z.string()).optional(),
        category: z.string().optional()
      });

      const updateData = updateSchema.parse(req.body);

      // Update creator asset
      const updatedCreatorAsset = await storage.updateCreatorAsset(req.params.id, updateData);
      if (!updatedCreatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // If tags or category were updated, also update the associated asset
      if (updateData.tags !== undefined || updateData.category !== undefined) {
        const assetUpdates: any = {};
        if (updateData.tags !== undefined) assetUpdates.tags = updateData.tags;
        if (updateData.category !== undefined) assetUpdates.category = updateData.category;
        
        if (Object.keys(assetUpdates).length > 0) {
          await storage.updateAsset(updatedCreatorAsset.assetId, assetUpdates);
        }
      }

      res.json({
        success: true,
        message: 'Asset updated successfully',
        asset: updatedCreatorAsset
      });

    } catch (error) {
      console.error('Creator asset update error:', error);
      handleError(res, error, 'Failed to update asset');
    }
  });

  /**
   * DELETE /api/creators/assets/:id
   * Soft delete asset (removes from marketplace but keeps record)
   */
  app.delete('/api/creators/assets/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const creator = await storage.getCreatorByUserId(user.id);
      if (!creator) {
        return res.status(403).json({ error: 'Creator profile not found' });
      }

      const creatorAsset = await storage.getCreatorAsset(req.params.id);
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Verify ownership
      if (creatorAsset.creatorId !== creator.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      // Soft delete by updating approval status to a special "deleted" state
      // This keeps the record for sales history but removes from marketplace
      const deleted = await storage.deleteCreatorAsset(req.params.id);
      if (!deleted) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Also mark the asset as not public
      await storage.updateAsset(creatorAsset.assetId, { isPublic: false });

      res.status(204).send();

    } catch (error) {
      console.error('Creator asset delete error:', error);
      handleError(res, error, 'Failed to delete asset');
    }
  });

  /**
   * Marketplace API Routes (Public)
   */

  /**
   * GET /api/marketplace/assets
   * Browse approved assets publicly with filtering and search
   */
  app.get('/api/marketplace/assets', async (req: Request, res: Response) => {
    try {
      // Parse query parameters for filtering and pagination
      const page = parseInt(req.query.page as string) || 1;
      const limit = Math.min(parseInt(req.query.limit as string) || 20, 100); // Max 100 per page
      const category = req.query.category as string;
      const search = req.query.search as string;
      const minPrice = req.query.minPrice ? parseInt(req.query.minPrice as string) : undefined;
      const maxPrice = req.query.maxPrice ? parseInt(req.query.maxPrice as string) : undefined;
      const assetType = req.query.assetType as string;
      const sortBy = req.query.sortBy as string || 'newest'; // newest, oldest, price_low, price_high, popular

      // Get approved creator assets
      const approvedAssets = await storage.getCreatorAssetsByStatus('approved');
      
      // Filter based on query parameters
      let filteredAssets = approvedAssets;

      if (search) {
        const searchLower = search.toLowerCase();
        filteredAssets = filteredAssets.filter(asset => 
          asset.title.toLowerCase().includes(searchLower) ||
          (asset.description && asset.description.toLowerCase().includes(searchLower))
        );
      }

      if (minPrice !== undefined) {
        filteredAssets = filteredAssets.filter(asset => asset.price >= minPrice);
      }

      if (maxPrice !== undefined) {
        filteredAssets = filteredAssets.filter(asset => asset.price <= maxPrice);
      }

      // Get asset details and filter by category/type
      const enrichedAssets = await Promise.all(
        filteredAssets.map(async (creatorAsset) => {
          const asset = await storage.getAsset(creatorAsset.assetId);
          const creator = await storage.getCreator(creatorAsset.creatorId);
          
          if (!asset || !creator) return null;

          // Filter by category and asset type
          if (category && asset.category !== category) return null;
          if (assetType && asset.assetType !== assetType) return null;

          return {
            id: creatorAsset.id,
            assetId: creatorAsset.assetId,
            title: creatorAsset.title,
            description: creatorAsset.description,
            price: creatorAsset.price,
            isExclusive: creatorAsset.isExclusive,
            salesCount: creatorAsset.salesCount,
            createdAt: creatorAsset.createdAt,
            // Asset details
            originalFileName: asset.originalFileName,
            fileSize: asset.fileSize,
            assetType: asset.assetType,
            category: asset.category,
            tags: asset.tags,
            downloadCount: asset.downloadCount,
            // Creator details (minimal for privacy)
            creator: {
              id: creator.id,
              name: creator.profileData?.name || 'Anonymous Creator',
              // Don't expose sensitive creator data
            }
          };
        })
      );

      // Remove null entries and sort
      const validAssets = enrichedAssets.filter(asset => asset !== null);

      // Sort assets
      validAssets.sort((a, b) => {
        switch (sortBy) {
          case 'oldest':
            return new Date(a.createdAt).getTime() - new Date(b.createdAt).getTime();
          case 'price_low':
            return a.price - b.price;
          case 'price_high':
            return b.price - a.price;
          case 'popular':
            return (b.salesCount + b.downloadCount) - (a.salesCount + a.downloadCount);
          case 'newest':
          default:
            return new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime();
        }
      });

      // Paginate
      const startIndex = (page - 1) * limit;
      const endIndex = startIndex + limit;
      const paginatedAssets = validAssets.slice(startIndex, endIndex);

      // Get unique categories and asset types for filtering metadata
      const categories = [...new Set(validAssets.map(a => a.category).filter(Boolean))];
      const assetTypes = [...new Set(validAssets.map(a => a.assetType))];

      res.json({
        assets: paginatedAssets,
        pagination: {
          page,
          limit,
          total: validAssets.length,
          pages: Math.ceil(validAssets.length / limit),
          hasNext: endIndex < validAssets.length,
          hasPrev: page > 1
        },
        filters: {
          categories: categories.sort(),
          assetTypes: assetTypes.sort(),
          priceRange: {
            min: Math.min(...validAssets.map(a => a.price)),
            max: Math.max(...validAssets.map(a => a.price))
          }
        }
      });

    } catch (error) {
      console.error('Marketplace assets browse error:', error);
      handleError(res, error, 'Failed to browse marketplace assets');
    }
  });

  /**
   * GET /api/marketplace/assets/:id
   * Get public asset details for purchase
   */
  app.get('/api/marketplace/assets/:id', async (req: Request, res: Response) => {
    try {
      const creatorAsset = await storage.getCreatorAsset(req.params.id);
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Only show approved assets in marketplace
      if (creatorAsset.approvalStatus !== 'approved') {
        return res.status(404).json({ error: 'Asset not available' });
      }

      const asset = await storage.getAsset(creatorAsset.assetId);
      if (!asset) {
        return res.status(404).json({ error: 'Asset file not found' });
      }

      const creator = await storage.getCreator(creatorAsset.creatorId);
      if (!creator) {
        return res.status(404).json({ error: 'Creator not found' });
      }

      const publicAssetDetails = {
        id: creatorAsset.id,
        assetId: creatorAsset.assetId,
        title: creatorAsset.title,
        description: creatorAsset.description,
        price: creatorAsset.price,
        isExclusive: creatorAsset.isExclusive,
        salesCount: creatorAsset.salesCount,
        createdAt: creatorAsset.createdAt,
        // Asset details
        originalFileName: asset.originalFileName,
        fileSize: asset.fileSize,
        assetType: asset.assetType,
        category: asset.category,
        tags: asset.tags,
        downloadCount: asset.downloadCount,
        // Creator details (minimal for privacy)
        creator: {
          id: creator.id,
          name: creator.profileData?.name || 'Anonymous Creator',
          bio: creator.profileData?.bio || null,
          // Don't expose sensitive creator data like earnings, etc.
        }
      };

      res.json(publicAssetDetails);

    } catch (error) {
      console.error('Marketplace asset details error:', error);
      handleError(res, error, 'Failed to get asset details');
    }
  });

  /**
   * GET /api/marketplace/creators/:creatorId/assets
   * Browse specific creator's approved assets
   */
  app.get('/api/marketplace/creators/:creatorId/assets', async (req: Request, res: Response) => {
    try {
      const creator = await storage.getCreator(req.params.creatorId);
      if (!creator) {
        return res.status(404).json({ error: 'Creator not found' });
      }

      // Only show active creators
      if (!creator.isActive) {
        return res.status(404).json({ error: 'Creator not available' });
      }

      // Get approved assets for this creator
      const creatorAssets = await storage.getCreatorAssetsByCreatorId(creator.id);
      const approvedAssets = creatorAssets.filter(asset => asset.approvalStatus === 'approved');

      // Enrich with asset details
      const enrichedAssets = await Promise.all(
        approvedAssets.map(async (creatorAsset) => {
          const asset = await storage.getAsset(creatorAsset.assetId);
          if (!asset) return null;

          return {
            id: creatorAsset.id,
            assetId: creatorAsset.assetId,
            title: creatorAsset.title,
            description: creatorAsset.description,
            price: creatorAsset.price,
            isExclusive: creatorAsset.isExclusive,
            salesCount: creatorAsset.salesCount,
            createdAt: creatorAsset.createdAt,
            // Asset details
            originalFileName: asset.originalFileName,
            fileSize: asset.fileSize,
            assetType: asset.assetType,
            category: asset.category,
            tags: asset.tags,
            downloadCount: asset.downloadCount
          };
        })
      );

      const validAssets = enrichedAssets.filter(asset => asset !== null);

      // Sort by newest first
      validAssets.sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());

      res.json({
        creator: {
          id: creator.id,
          name: creator.profileData?.name || 'Anonymous Creator',
          bio: creator.profileData?.bio || null,
          portfolio: creator.profileData?.portfolio || null
        },
        assets: validAssets,
        stats: {
          totalAssets: validAssets.length,
          totalSales: validAssets.reduce((sum, a) => sum + (a.salesCount || 0), 0)
        }
      });

    } catch (error) {
      console.error('Creator marketplace assets error:', error);
      handleError(res, error, 'Failed to get creator assets');
    }
  });

  /**
   * Admin Approval Routes (for moderation)
   */

  /**
   * Helper function to build enriched asset response with pagination
   * Used by all admin asset listing endpoints
   */
  const buildEnrichedAssetResponse = async (
    creatorAssets: any[], 
    page: number = 1, 
    limit: number = 50,
    sortOrder: 'asc' | 'desc' = 'asc'
  ) => {
    // Enrich with asset and creator details
    const enrichedAssets = await Promise.all(
      creatorAssets.map(async (creatorAsset) => {
        const asset = await storage.getAsset(creatorAsset.assetId);
        const creator = await storage.getCreator(creatorAsset.creatorId);
        const user = creator ? await storage.getUser(creator.userId) : null;

        return {
          id: creatorAsset.id,
          assetId: creatorAsset.assetId,
          title: creatorAsset.title,
          description: creatorAsset.description,
          price: creatorAsset.price,
          isExclusive: creatorAsset.isExclusive,
          createdAt: creatorAsset.createdAt,
          approvalStatus: creatorAsset.approvalStatus,
          rejectionReason: creatorAsset.rejectionReason,
          // Asset details
          fileName: asset?.fileName,
          originalFileName: asset?.originalFileName,
          fileSize: asset?.fileSize,
          assetType: asset?.assetType,
          category: asset?.category,
          tags: asset?.tags,
          fileUrl: asset?.fileUrl, // Admin can see file URLs for review
          // Creator details (for admin review)
          creator: {
            id: creator?.id,
            userId: creator?.userId,
            name: creator?.profileData?.name,
            email: user?.email,
            onboardingStatus: creator?.onboardingStatus
          }
        };
      })
    );

    // Sort assets
    enrichedAssets.sort((a, b) => {
      const aTime = new Date(a.createdAt).getTime();
      const bTime = new Date(b.createdAt).getTime();
      return sortOrder === 'asc' ? aTime - bTime : bTime - aTime;
    });

    // Apply pagination
    const startIndex = (page - 1) * limit;
    const endIndex = startIndex + limit;
    const paginatedAssets = enrichedAssets.slice(startIndex, endIndex);

    return {
      assets: paginatedAssets,
      count: paginatedAssets.length,
      total: enrichedAssets.length,
      page,
      limit,
      totalPages: Math.ceil(enrichedAssets.length / limit),
      hasNextPage: endIndex < enrichedAssets.length,
      hasPrevPage: page > 1
    };
  };

  /**
   * GET /api/admin/assets/pending
   * List assets pending approval (admin only) with pagination
   */
  app.get('/api/admin/assets/pending', requireAdmin, async (req: AuthenticatedRequest, res: Response) => {
    try {
      // Parse pagination parameters
      const page = parseInt(req.query.page as string) || 1;
      const limit = Math.min(parseInt(req.query.limit as string) || 50, 100); // Cap at 100

      // Get all pending creator assets
      const pendingAssets = await storage.getCreatorAssetsByStatus('pending');

      // Build response with pagination (FIFO - oldest first for pending review)
      const response = await buildEnrichedAssetResponse(pendingAssets, page, limit, 'asc');

      res.json(response);

    } catch (error) {
      console.error('Admin pending assets error:', error);
      handleError(res, error, 'Failed to get pending assets');
    }
  });

  /**
   * GET /api/admin/assets/approved
   * List approved assets (admin only) with pagination
   */
  app.get('/api/admin/assets/approved', requireAdmin, async (req: AuthenticatedRequest, res: Response) => {
    try {
      // Parse pagination parameters
      const page = parseInt(req.query.page as string) || 1;
      const limit = Math.min(parseInt(req.query.limit as string) || 50, 100); // Cap at 100

      // Get all approved creator assets
      const approvedAssets = await storage.getCreatorAssetsByStatus('approved');

      // Build response with pagination (newest first for approved)
      const response = await buildEnrichedAssetResponse(approvedAssets, page, limit, 'desc');

      res.json(response);

    } catch (error) {
      console.error('Admin approved assets error:', error);
      handleError(res, error, 'Failed to get approved assets');
    }
  });

  /**
   * GET /api/admin/assets/rejected
   * List rejected assets (admin only) with pagination
   */
  app.get('/api/admin/assets/rejected', requireAdmin, async (req: AuthenticatedRequest, res: Response) => {
    try {
      // Parse pagination parameters
      const page = parseInt(req.query.page as string) || 1;
      const limit = Math.min(parseInt(req.query.limit as string) || 50, 100); // Cap at 100

      // Get all rejected creator assets
      const rejectedAssets = await storage.getCreatorAssetsByStatus('rejected');

      // Build response with pagination (newest first for rejected)
      const response = await buildEnrichedAssetResponse(rejectedAssets, page, limit, 'desc');

      res.json(response);

    } catch (error) {
      console.error('Admin rejected assets error:', error);
      handleError(res, error, 'Failed to get rejected assets');
    }
  });

  /**
   * POST /api/admin/assets/:id/approve
   * Approve asset for marketplace (admin only)
   */
  app.post('/api/admin/assets/:id/approve', requireAdmin, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'Admin not authenticated' });
      }

      const creatorAsset = await storage.getCreatorAsset(req.params.id);
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      if (creatorAsset.approvalStatus !== 'pending') {
        return res.status(400).json({ 
          error: 'Asset is not pending approval',
          currentStatus: creatorAsset.approvalStatus
        });
      }

      // Approve the creator asset
      const approvedAsset = await storage.approveCreatorAsset(req.params.id, req.user.uid);
      if (!approvedAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Make the associated asset public so it appears in marketplace
      await storage.updateAsset(approvedAsset.assetId, { isPublic: true });

      // Get creator to send notification
      const creator = await storage.getCreator(approvedAsset.creatorId);
      if (creator) {
        await storage.createNotification(
          creator.userId,
          `🎉 Your asset "${approvedAsset.title}" has been approved and is now live in the marketplace!`,
          'success'
        );
      }

      // Log the approval action
      console.log(`✅ Asset ${req.params.id} approved by admin ${req.user.uid}`);

      res.json({
        success: true,
        message: 'Asset approved successfully',
        asset: approvedAsset
      });

    } catch (error) {
      console.error('Admin asset approval error:', error);
      handleError(res, error, 'Failed to approve asset');
    }
  });

  /**
   * POST /api/admin/assets/:id/reject
   * Reject asset with reason (admin only)
   */
  app.post('/api/admin/assets/:id/reject', requireAdmin, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'Admin not authenticated' });
      }

      const creatorAsset = await storage.getCreatorAsset(req.params.id);
      if (!creatorAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      if (creatorAsset.approvalStatus !== 'pending') {
        return res.status(400).json({ 
          error: 'Asset is not pending approval',
          currentStatus: creatorAsset.approvalStatus
        });
      }

      // Validate rejection reason
      const rejectionSchema = z.object({
        reason: z.string().min(10, 'Rejection reason must be at least 10 characters')
      });

      const { reason } = rejectionSchema.parse(req.body);

      // Reject the creator asset
      const rejectedAsset = await storage.rejectCreatorAsset(req.params.id, reason);
      if (!rejectedAsset) {
        return res.status(404).json({ error: 'Asset not found' });
      }

      // Get creator to send notification
      const creator = await storage.getCreator(rejectedAsset.creatorId);
      if (creator) {
        await storage.createNotification(
          creator.userId,
          `❌ Your asset "${rejectedAsset.title}" was not approved. Reason: ${reason}`,
          'warning'
        );
      }

      // Log the rejection action
      console.log(`❌ Asset ${req.params.id} rejected by admin ${req.user.uid}: ${reason}`);

      res.json({
        success: true,
        message: 'Asset rejected',
        asset: rejectedAsset,
        rejectionReason: reason
      });

    } catch (error) {
      console.error('Admin asset rejection error:', error);
      handleError(res, error, 'Failed to reject asset');
    }
  });

  app.get('/api/brand-kits/:id', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const brandKit = await storage.getBrandKit(req.params.id);
      if (!brandKit) {
        return res.status(404).json({ error: 'Brand kit not found' });
      }

      // Check ownership
      if (brandKit.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      res.json(brandKit);
    } catch (error) {
      handleError(res, error, 'Failed to get brand kit');
    }
  });

  app.put('/api/brand-kits/:id', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingBrandKit = await storage.getBrandKit(req.params.id);
      if (!existingBrandKit) {
        return res.status(404).json({ error: 'Brand kit not found' });
      }

      // Check ownership
      if (existingBrandKit.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const updateData = insertBrandKitSchema.partial().parse(req.body);
      const updatedBrandKit = await storage.updateBrandKit(req.params.id, updateData);
      
      if (!updatedBrandKit) {
        return res.status(404).json({ error: 'Brand kit not found' });
      }

      res.json(updatedBrandKit);
    } catch (error) {
      handleError(res, error, 'Failed to update brand kit');
    }
  });

  app.delete('/api/brand-kits/:id', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingBrandKit = await storage.getBrandKit(req.params.id);
      if (!existingBrandKit) {
        return res.status(404).json({ error: 'Brand kit not found' });
      }

      // Check ownership
      if (existingBrandKit.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const deleted = await storage.deleteBrandKit(req.params.id);
      if (!deleted) {
        return res.status(404).json({ error: 'Brand kit not found' });
      }

      res.status(204).send();
    } catch (error) {
      handleError(res, error, 'Failed to delete brand kit');
    }
  });

  // Business Names routes
  app.post('/api/business-names', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const businessNameData = insertBusinessNameSchema.parse({
        ...req.body,
        userId: user.id
      });

      const businessName = await storage.createBusinessName(businessNameData);
      res.status(201).json(businessName);
    } catch (error) {
      handleError(res, error, 'Failed to create business name');
    }
  });

  app.get('/api/business-names', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const businessNames = await storage.getBusinessNamesByUserId(user.id);
      res.json(businessNames);
    } catch (error) {
      handleError(res, error, 'Failed to get business names');
    }
  });

  app.get('/api/business-names/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const businessName = await storage.getBusinessName(req.params.id);
      if (!businessName) {
        return res.status(404).json({ error: 'Business name not found' });
      }

      // Check ownership
      if (businessName.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      res.json(businessName);
    } catch (error) {
      handleError(res, error, 'Failed to get business name');
    }
  });

  app.put('/api/business-names/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingBusinessName = await storage.getBusinessName(req.params.id);
      if (!existingBusinessName) {
        return res.status(404).json({ error: 'Business name not found' });
      }

      // Check ownership
      if (existingBusinessName.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const updateData = insertBusinessNameSchema.partial().parse(req.body);
      const updatedBusinessName = await storage.updateBusinessName(req.params.id, updateData);
      
      if (!updatedBusinessName) {
        return res.status(404).json({ error: 'Business name not found' });
      }

      res.json(updatedBusinessName);
    } catch (error) {
      handleError(res, error, 'Failed to update business name');
    }
  });

  app.delete('/api/business-names/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingBusinessName = await storage.getBusinessName(req.params.id);
      if (!existingBusinessName) {
        return res.status(404).json({ error: 'Business name not found' });
      }

      // Check ownership
      if (existingBusinessName.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const deleted = await storage.deleteBusinessName(req.params.id);
      if (!deleted) {
        return res.status(404).json({ error: 'Business name not found' });
      }

      res.status(204).send();
    } catch (error) {
      handleError(res, error, 'Failed to delete business name');
    }
  });

  // Saved business names routes
  app.get('/api/business-names/saved', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const savedBusinessNames = await storage.getSavedBusinessNamesByUserId(user.id);
      res.json(savedBusinessNames);
    } catch (error) {
      handleError(res, error, 'Failed to get saved business names');
    }
  });

  app.post('/api/business-names/saved', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const businessNameData = insertBusinessNameSchema.parse({
        ...req.body,
        userId: user.id,
        isSaved: true
      });

      const businessName = await storage.createBusinessName(businessNameData);
      res.status(201).json(businessName);
    } catch (error) {
      handleError(res, error, 'Failed to save business name');
    }
  });

  app.delete('/api/business-names/saved/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingBusinessName = await storage.getBusinessName(req.params.id);
      if (!existingBusinessName) {
        return res.status(404).json({ error: 'Business name not found' });
      }

      // Check ownership
      if (existingBusinessName.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      // Check if it's a saved business name
      if (!existingBusinessName.isSaved) {
        return res.status(400).json({ error: 'Business name is not saved' });
      }

      const deleted = await storage.deleteBusinessName(req.params.id);
      if (!deleted) {
        return res.status(404).json({ error: 'Business name not found' });
      }

      res.status(204).send();
    } catch (error) {
      handleError(res, error, 'Failed to delete saved business name');
    }
  });

  // Generate business names with AI
  app.post('/api/business-names/generate', async (req: Request, res: Response) => {
    try {
      const { 
        industry, 
        keywords, 
        style, 
        count,
        creativity,
        description,
        vibe,
        length_range,
        starts_with,
        must_include,
        must_exclude
      } = req.body;
      
      if (!industry && !keywords) {
        return res.status(400).json({ error: 'Either industry or keywords is required' });
      }

      const result = await generateBusinessNames({ 
        industry, 
        keywords, 
        style, 
        count,
        creativity,
        description,
        vibe,
        length_range,
        starts_with,
        must_include,
        must_exclude
      });
      
      res.json(result);
    } catch (error) {
      handleError(res, error, 'Failed to generate business names');
    }
  });

  // Rate limiting store for slogan generation (simple in-memory store)
  const sloganRateLimit = new Map<string, { count: number; resetTime: number }>();
  const SLOGAN_RATE_LIMIT = 30; // 30 requests per minute per IP
  const SLOGAN_RATE_WINDOW = 60 * 1000; // 1 minute in milliseconds

  // Rate limiting store for business plan generation
  const planRateLimit = new Map<string, { count: number; resetTime: number }>();
  const PLAN_RATE_LIMIT = 20; // 20 requests per minute per IP
  const PLAN_RATE_WINDOW = 60 * 1000; // 1 minute in milliseconds

  // Rate limiting store for password changes (enhanced security)
  const passwordRateLimit = new Map<string, { count: number; resetTime: number }>();
  const PASSWORD_RATE_LIMIT = 5; // 5 password changes per minute per user/IP
  const PASSWORD_RATE_WINDOW = 60 * 1000; // 1 minute in milliseconds

  // Helper function to check rate limit
  function checkSloganRateLimit(ip: string): boolean {
    const now = Date.now();
    const clientData = sloganRateLimit.get(ip);

    if (!clientData || now > clientData.resetTime) {
      // Reset or initialize
      sloganRateLimit.set(ip, { count: 1, resetTime: now + SLOGAN_RATE_WINDOW });
      return true;
    }

    if (clientData.count >= SLOGAN_RATE_LIMIT) {
      return false;
    }

    clientData.count++;
    return true;
  }

  // Helper function to check business plan rate limit
  function checkPlanRateLimit(ip: string): boolean {
    const now = Date.now();
    const clientData = planRateLimit.get(ip);

    if (!clientData || now > clientData.resetTime) {
      // Reset or initialize
      planRateLimit.set(ip, { count: 1, resetTime: now + PLAN_RATE_WINDOW });
      return true;
    }

    if (clientData.count >= PLAN_RATE_LIMIT) {
      return false;
    }

    clientData.count++;
    return true;
  }

  // Helper function to check password change rate limit (enhanced security)
  // Tracks both by user ID and IP address for comprehensive protection
  function checkPasswordRateLimit(userIdentifier: string, ip: string): boolean {
    const now = Date.now();
    
    // Create a composite key using both user identifier and IP
    const compositeKey = `${userIdentifier}:${ip}`;
    const clientData = passwordRateLimit.get(compositeKey);

    if (!clientData || now > clientData.resetTime) {
      // Reset or initialize
      passwordRateLimit.set(compositeKey, { count: 1, resetTime: now + PASSWORD_RATE_WINDOW });
      return true;
    }

    if (clientData.count >= PASSWORD_RATE_LIMIT) {
      return false;
    }

    clientData.count++;
    return true;
  }

  // TOTP Security Endpoints
  
  // Secure TOTP encryption using AES-256-GCM
  let TOTP_ENCRYPTION_KEY = process.env.TOTP_ENCRYPTION_KEY;
  
  // Development fallback: generate a temporary key if missing
  if (!TOTP_ENCRYPTION_KEY) {
    if (process.env.NODE_ENV === 'production') {
      console.error('🚨 CRITICAL: TOTP_ENCRYPTION_KEY environment variable is required for TOTP functionality');
      throw new Error('TOTP_ENCRYPTION_KEY is required in production');
    } else {
      // Development mode: use a temporary key (not secure for production)
      TOTP_ENCRYPTION_KEY = '1d58cc60fc25367aaf564abd6051ad5f37cf9f52727bc1a726cbfb3990e3118a';
      console.warn('⚠️ Using temporary TOTP encryption key for development. Set TOTP_ENCRYPTION_KEY environment variable for production use.');
    }
  }
  
  function encryptTotpSecret(secret: string): string {
    if (!TOTP_ENCRYPTION_KEY) {
      throw new Error('TOTP encryption key not configured');
    }
    
    // Generate random IV for each encryption
    const iv = crypto.randomBytes(16);
    const cipher = crypto.createCipherGCM('aes-256-gcm', Buffer.from(TOTP_ENCRYPTION_KEY, 'hex'));
    cipher.setIVLength(16);
    
    let encrypted = cipher.update(secret, 'utf8', 'hex');
    encrypted += cipher.final('hex');
    
    const authTag = cipher.getAuthTag();
    
    // Return IV + authTag + encrypted data (all hex encoded)
    return iv.toString('hex') + ':' + authTag.toString('hex') + ':' + encrypted;
  }
  
  function decryptTotpSecret(encryptedSecret: string): string {
    if (!TOTP_ENCRYPTION_KEY) {
      throw new Error('TOTP encryption key not configured');
    }
    
    const parts = encryptedSecret.split(':');
    if (parts.length !== 3) {
      throw new Error('Invalid encrypted secret format');
    }
    
    const iv = Buffer.from(parts[0], 'hex');
    const authTag = Buffer.from(parts[1], 'hex');
    const encrypted = parts[2];
    
    const decipher = crypto.createDecipherGCM('aes-256-gcm', Buffer.from(TOTP_ENCRYPTION_KEY, 'hex'));
    decipher.setAuthTag(authTag);
    
    let decrypted = decipher.update(encrypted, 'hex', 'utf8');
    decrypted += decipher.final('utf8');
    
    return decrypted;
  }
  
  function generateBackupCodes(): string[] {
    const codes = [];
    for (let i = 0; i < 10; i++) {
      // Generate cryptographically secure random backup codes
      const code = crypto.randomBytes(4).toString('hex').toUpperCase();
      codes.push(code);
    }
    return codes;
  }
  
  async function hashBackupCodes(codes: string[]): Promise<string[]> {
    const hashedCodes = [];
    for (const code of codes) {
      const hashed = await bcrypt.hash(code, 12); // Use bcrypt with cost factor 12
      hashedCodes.push(hashed);
    }
    return hashedCodes;
  }
  
  async function verifyBackupCode(plainCode: string, hashedCodes: string[]): Promise<{ isValid: boolean; remainingCodes: string[] }> {
    for (let i = 0; i < hashedCodes.length; i++) {
      const isMatch = await bcrypt.compare(plainCode, hashedCodes[i]);
      if (isMatch) {
        // Remove the used code and return remaining codes
        const remainingCodes = hashedCodes.filter((_, index) => index !== i);
        return { isValid: true, remainingCodes };
      }
    }
    return { isValid: false, remainingCodes: hashedCodes };
  }

  // Setup TOTP - Generate QR code for authenticator app
  app.post('/api/security/totp/setup', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Rate limiting check
      const clientIp = req.ip || req.connection.remoteAddress || 'unknown';
      if (!checkPasswordRateLimit(req.user.uid, clientIp)) {
        return res.status(429).json({ 
          error: 'Too many security requests. Please wait before trying again.',
          retryAfter: Math.ceil(PASSWORD_RATE_WINDOW / 1000)
        });
      }

      // Generate TOTP secret
      const secret = authenticator.generateSecret();
      const serviceName = 'IBrandBiz';
      const accountName = req.user.email || req.user.uid;
      
      // Create otpauth URL for QR code
      const otpauthURL = authenticator.keyuri(accountName, serviceName, secret);
      
      // Generate QR code
      const qrCodeDataUrl = await QRCode.toDataURL(otpauthURL);
      
      // Store encrypted secret temporarily (not enabled yet)
      const encryptedSecret = encryptTotpSecret(secret);
      await storage.setTotpSecret(req.user.uid, encryptedSecret);
      
      res.json({
        qrCode: qrCodeDataUrl,
        secret: secret, // Send plaintext for manual entry
        backupCodes: [], // Will be generated after verification
      });
    } catch (error) {
      console.error('TOTP setup error:', error);
      handleError(res, error, 'Failed to setup TOTP');
    }
  });

  // Verify TOTP setup and enable 2FA
  app.post('/api/security/totp/verify', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Rate limiting check
      const clientIp = req.ip || req.connection.remoteAddress || 'unknown';
      if (!checkPasswordRateLimit(req.user.uid, clientIp)) {
        return res.status(429).json({ 
          error: 'Too many security requests. Please wait before trying again.',
          retryAfter: Math.ceil(PASSWORD_RATE_WINDOW / 1000)
        });
      }

      const { token } = req.body;
      
      if (!token) {
        return res.status(400).json({ error: 'TOTP token is required' });
      }

      // Get stored secret
      const encryptedSecret = await storage.getTotpSecret(req.user.uid);
      if (!encryptedSecret) {
        return res.status(400).json({ error: 'TOTP setup not initiated. Please start setup first.' });
      }

      const secret = decryptTotpSecret(encryptedSecret);
      
      // Verify the token
      const isValid = authenticator.verify({
        token: token.toString(),
        secret: secret,
        window: 2, // Allow 2 steps tolerance
      });

      if (!isValid) {
        return res.status(400).json({ error: 'Invalid TOTP token. Please try again.' });
      }

      // Generate backup codes
      const plaintextBackupCodes = generateBackupCodes();
      const hashedBackupCodes = await hashBackupCodes(plaintextBackupCodes);
      
      // Enable TOTP with hashed backup codes
      await storage.enableTotp(req.user.uid, hashedBackupCodes);
      
      res.json({
        success: true,
        backupCodes: plaintextBackupCodes, // Return plaintext codes to user (one-time display)
        message: 'Two-factor authentication has been successfully enabled. Save these backup codes securely!'
      });
    } catch (error) {
      console.error('TOTP verification error:', error);
      handleError(res, error, 'Failed to verify TOTP');
    }
  });

  // Disable TOTP
  app.post('/api/security/totp/disable', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Rate limiting check
      const clientIp = req.ip || req.connection.remoteAddress || 'unknown';
      if (!checkPasswordRateLimit(req.user.uid, clientIp)) {
        return res.status(429).json({ 
          error: 'Too many security requests. Please wait before trying again.',
          retryAfter: Math.ceil(PASSWORD_RATE_WINDOW / 1000)
        });
      }

      const { currentPassword, totpToken } = req.body;
      
      // For security, require either current password OR valid TOTP token
      if (!currentPassword && !totpToken) {
        return res.status(400).json({ 
          error: 'Current password or TOTP token required to disable two-factor authentication' 
        });
      }

      // If TOTP token provided, verify it
      if (totpToken) {
        const encryptedSecret = await storage.getTotpSecret(req.user.uid);
        if (encryptedSecret) {
          const secret = decryptTotpSecret(encryptedSecret);
          const isValid = authenticator.verify({
            token: totpToken.toString(),
            secret: secret,
            window: 2,
          });
          
          if (!isValid) {
            return res.status(400).json({ error: 'Invalid TOTP token' });
          }
        }
      }

      // Disable TOTP
      await storage.disableTotp(req.user.uid);
      
      res.json({
        success: true,
        message: 'Two-factor authentication has been disabled.'
      });
    } catch (error) {
      console.error('TOTP disable error:', error);
      handleError(res, error, 'Failed to disable TOTP');
    }
  });

  // Get TOTP status
  app.get('/api/security/totp/status', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      res.json({
        enabled: !!user.totpEnabled,
        hasBackupCodes: !!(user.totpBackupCodes && user.totpBackupCodes.length > 0),
        backupCodesCount: user.totpBackupCodes?.length || 0
      });
    } catch (error) {
      console.error('TOTP status error:', error);
      handleError(res, error, 'Failed to get TOTP status');
    }
  });

  // Verify backup code for authentication
  app.post('/api/security/totp/verify-backup', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Rate limiting check
      const clientIp = req.ip || req.connection.remoteAddress || 'unknown';
      if (!checkPasswordRateLimit(req.user.uid, clientIp)) {
        return res.status(429).json({ 
          error: 'Too many security requests. Please wait before trying again.',
          retryAfter: Math.ceil(PASSWORD_RATE_WINDOW / 1000)
        });
      }

      const { backupCode } = req.body;
      
      if (!backupCode) {
        return res.status(400).json({ error: 'Backup code is required' });
      }

      // Get user's hashed backup codes
      const hashedBackupCodes = await storage.getTotpBackupCodes(req.user.uid);
      if (!hashedBackupCodes || hashedBackupCodes.length === 0) {
        return res.status(400).json({ error: 'No backup codes available' });
      }

      // Verify the backup code
      const { isValid, remainingCodes } = await verifyBackupCode(backupCode, hashedBackupCodes);
      
      if (!isValid) {
        return res.status(400).json({ error: 'Invalid backup code' });
      }

      // Update the user's backup codes (remove the used one)
      await storage.updateTotpBackupCodes(req.user.uid, remainingCodes);
      
      res.json({
        success: true,
        message: 'Backup code verified successfully',
        remainingBackupCodes: remainingCodes.length
      });
    } catch (error) {
      console.error('Backup code verification error:', error);
      handleError(res, error, 'Failed to verify backup code');
    }
  });

  // Generate slogans with AI - authenticated endpoint
  app.post('/api/ai/slogan', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Rate limiting check
      const clientIp = req.ip || req.connection.remoteAddress || 'unknown';
      if (!checkSloganRateLimit(clientIp)) {
        return res.status(429).json({ 
          error: 'Rate limit exceeded. Maximum 30 requests per minute.' 
        });
      }

      const { brandName, description, tone, industry, audience, maxResults } = req.body;
      
      // Validation: brandName is required
      if (!brandName || typeof brandName !== 'string' || brandName.trim().length === 0) {
        return res.status(400).json({ error: 'brandName is required and must be a non-empty string' });
      }

      // Get user from database to check paid status server-side
      // In development mode, bypass database lookup if there's a connection issue
      let user;
      try {
        user = await storage.getUserByFirebaseUid(req.user.uid);
      } catch (dbError) {
        // Development fallback - create mock user if database is unavailable
        const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
        if (isDevelopment) {
          console.warn('Database unavailable, using mock user for testing:', dbError);
          user = { 
            id: 'dev-user-123', 
            firebaseUid: req.user.uid,
            email: req.user.email || 'dev@example.com',
            subscriptionTier: 'free' // Default to free tier for testing
          };
        } else {
          throw dbError; // Re-throw in production
        }
      }
      
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Clamp maxResults to [1, 12], default 8
      let clampedMaxResults = 8;
      if (maxResults !== undefined) {
        if (typeof maxResults === 'number' && !isNaN(maxResults)) {
          clampedMaxResults = Math.max(1, Math.min(12, Math.floor(maxResults)));
        }
      }

      // Server-side paywall enforcement: For Phase 1, simple free tier logic
      // TODO: In Phase 2, implement proper subscription checking with user.subscriptionTier
      const isPaid = false; // All users are free tier in Phase 1

      if (!isPaid) {
        // Free tier: limit to 1 result
        clampedMaxResults = 1;
      } else {
        // Paid tier: limit to 8 results maximum
        clampedMaxResults = Math.min(clampedMaxResults, 8);
      }

      const slogans = await generateSlogans({ 
        brandName: brandName.trim(),
        description,
        tone,
        industry,
        audience,
        maxResults: clampedMaxResults
      });
      
      res.json(slogans);
    } catch (error) {
      handleError(res, error, 'Failed to generate slogans');
    }
  });

  // New slogan endpoint - SuperNova's enhanced version (no auth required)
  app.post('/api/slogan', async (req: Request, res: Response) => {
    try {
      // Rate limiting check
      const clientIp = req.ip || req.connection.remoteAddress || 'unknown';
      if (!checkSloganRateLimit(clientIp)) {
        return res.status(429).json({ 
          error: 'Rate limit exceeded. Maximum 30 requests per minute.' 
        });
      }

      const { brandName, description = "", tone = "Professional", industry = "", audience = "", maxResults = 8 } = req.body || {};
      
      // Validation: brandName is required
      if (!brandName || typeof brandName !== 'string' || brandName.trim().length === 0) {
        return res.status(400).json({ error: 'brandName is required and must be a non-empty string' });
      }
      
      const n = Math.min(Math.max(Number(maxResults) || 8, 3), 20);

      const system = `You are a branding copywriter. Return only short, punchy, original slogans (max ~8 words). No clichés.`;
      const user = `Brand: ${brandName}
Tone: ${tone}
Industry: ${industry || "General"}
Audience: ${audience || "General"}
About: ${description || "N/A"}

Write ${n} distinct, catchy slogans. Vary structure and rhythm. Return as a simple numbered list.`;

      const openai = new OpenAI({
        apiKey: process.env.OPENAI_API_KEY,
      });

      const r = await openai.chat.completions.create({
        model: "gpt-4o-mini",
        temperature: 0.9,
        top_p: 0.9,
        frequency_penalty: 0.4,
        presence_penalty: 0.3,
        max_tokens: 300,
        messages: [{ role: "system", content: system }, { role: "user", content: user }],
      });

      if (!r.choices?.[0]?.message?.content) {
        return res.status(502).json({ error: "Upstream error", detail: "No content from AI" });
      }

      const raw = r.choices[0].message.content.trim();
      const suggestions = raw
        .split(/\n+/)
        .map((l: string) => l.replace(/^\d+[\).\s-]?\s*/, "").trim())
        .filter(Boolean);

      res.json({ suggestions });
    } catch (e: any) {
      res.status(500).json({ error: "Server error", detail: e?.message || e });
    }
  });

  // Generate business plan with AI - authenticated endpoint
  app.post('/api/ai/plan', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Rate limiting check
      const clientIP = req.ip || req.socket.remoteAddress || 'unknown';
      if (!checkPlanRateLimit(clientIP)) {
        return res.status(429).json({ 
          error: 'Too many requests. Please try again later.',
          retryAfter: 60 
        });
      }

      // Get user from database to check paid status server-side
      // In development mode, bypass database lookup if there's a connection issue
      let user;
      try {
        user = await storage.getUserByFirebaseUid(req.user.uid);
      } catch (dbError) {
        // Development fallback - create mock user if database is unavailable
        const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
        if (isDevelopment) {
          console.warn('Database unavailable, using mock user for testing:', dbError);
          user = { 
            id: 'dev-user-123', 
            firebaseUid: req.user.uid,
            email: req.user.email || 'dev@example.com',
            subscriptionTier: 'free' // Default to free tier for testing
          };
        } else {
          throw dbError; // Re-throw in production
        }
      }
      
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Request validation
      const {
        mode = "lean",
        businessName,
        description,
        industry,
        audience,
        problem,
        solution,
        advantage,
        pricing,
        timeframeMonths,
        tone = "Professional"
      } = req.body;

      // Validate required fields
      if (!businessName || typeof businessName !== 'string' || businessName.trim().length === 0) {
        return res.status(400).json({ error: 'businessName is required and must be a non-empty string' });
      }

      if (!description || typeof description !== 'string' || description.trim().length === 0) {
        return res.status(400).json({ error: 'description is required and must be a non-empty string' });
      }

      // Validate mode
      if (!['lean', 'full'].includes(mode)) {
        return res.status(400).json({ error: 'mode must be either "lean" or "full"' });
      }

      // Validate tone
      if (!['Professional', 'Friendly', 'Bold', 'Minimal'].includes(tone)) {
        return res.status(400).json({ error: 'tone must be one of: Professional, Friendly, Bold, Minimal' });
      }

      // Paywall enforcement - check if user is trying to access full plan without paid subscription
      // For now, we'll use a simple check. In production, this would check the user's subscription status
      const isPaidUser = false; // TODO: Replace with actual subscription check
      
      if (mode === 'full' && !isPaidUser) {
        return res.status(402).json({ 
          error: 'Full business plans require a Pro subscription',
          code: 'PAYWALL',
          message: 'Upgrade to Pro to access comprehensive business plans with financial projections'
        });
      }

      // Prepare request for business plan generation
      const planRequest = {
        mode: mode as "lean" | "full",
        businessName: businessName.trim(),
        description: description.trim(),
        industry: industry?.trim(),
        audience: audience?.trim(),
        problem: problem?.trim(),
        solution: solution?.trim(),
        advantage: advantage?.trim(),
        pricing: pricing?.trim(),
        timeframeMonths: timeframeMonths ? Math.max(1, Math.min(60, parseInt(timeframeMonths))) : 12,
        tone: tone as "Professional" | "Friendly" | "Bold" | "Minimal"
      };

      // Generate the business plan
      const businessPlan = await generateBusinessPlan(planRequest);
      
      // Log successful generation for analytics
      console.log(`Business plan generated successfully for user ${user.id}, mode: ${mode}, businessName: ${businessName}`);
      
      res.json(businessPlan);
    } catch (error) {
      console.error('Business plan generation error:', error);
      
      // Handle specific error types
      if (error instanceof Error) {
        if (error.message.includes('required')) {
          return res.status(400).json({ error: error.message });
        }
        if (error.message.includes('Invalid JSON')) {
          return res.status(500).json({ error: 'AI response formatting error. Please try again.' });
        }
      }
      
      handleError(res, error, 'Failed to generate business plan');
    }
  });

  // AI Writing Assistant for Business Plan Sections - authenticated endpoint
  app.post('/api/ai/section', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Rate limiting check - reuse the plan rate limiter for now
      const clientIP = req.ip || req.socket.remoteAddress || 'unknown';
      if (!checkPlanRateLimit(clientIP)) {
        return res.status(429).json({ 
          error: 'Too many requests. Please try again later.',
          retryAfter: 60 
        });
      }

      // Get user from database to check paid status server-side
      let user;
      try {
        user = await storage.getUserByFirebaseUid(req.user.uid);
      } catch (dbError) {
        // Development fallback - create mock user if database is unavailable
        const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
        if (isDevelopment) {
          console.warn('Database unavailable, using mock user for testing:', dbError);
          user = { 
            id: 'dev-user-123', 
            firebaseUid: req.user.uid,
            email: req.user.email || 'dev@example.com',
            isPaid: true // Allow AI features in development
          };
        } else {
          throw dbError; // Re-throw in production
        }
      }
      
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Request validation using AiJob interface
      const aiJob = req.body;
      
      if (!aiJob.action || !aiJob.tone || !aiJob.sectionKind || !aiJob.sectionTitle) {
        return res.status(400).json({ 
          error: 'Missing required fields: action, tone, sectionKind, and sectionTitle are required' 
        });
      }

      // Validate action type
      if (!['generate', 'rephrase', 'expand', 'summarize'].includes(aiJob.action)) {
        return res.status(400).json({ 
          error: `Invalid action: ${aiJob.action}. Must be one of: generate, rephrase, expand, summarize` 
        });
      }

      // Validate tone
      if (!['Professional', 'Friendly', 'Bold', 'Minimal'].includes(aiJob.tone)) {
        return res.status(400).json({ 
          error: `Invalid tone: ${aiJob.tone}. Must be one of: Professional, Friendly, Bold, Minimal` 
        });
      }

      // Check for existing content when required
      if (['rephrase', 'expand', 'summarize'].includes(aiJob.action) && (!aiJob.existingContent || aiJob.existingContent.trim().length === 0)) {
        return res.status(400).json({ 
          error: `Action "${aiJob.action}" requires existing content to work with` 
        });
      }

      console.log(`🤖 Processing AI section request: ${aiJob.action} for "${aiJob.sectionTitle}" by user ${user.email}`);

      // Process the AI job
      const response = await processAiSectionJob(aiJob);

      // Return the response
      res.json(response);

    } catch (error) {
      console.error('AI section endpoint error:', error);
      handleError(res, error, 'Failed to process AI section request');
    }
  });

  // AI Structured Template Generator - authenticated endpoint
  app.post('/api/ai/structured-template', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Get user from database
      let user;
      try {
        user = await storage.getUserByFirebaseUid(req.user.uid);
      } catch (dbError) {
        const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
        if (isDevelopment) {
          user = { 
            id: 'dev-user-123', 
            firebaseUid: req.user.uid,
            email: req.user.email || 'dev@example.com',
            isPaid: true
          };
        } else {
          throw dbError;
        }
      }
      
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const { templateKey, businessBrief } = req.body;
      
      if (!templateKey) {
        return res.status(400).json({ error: 'templateKey is required' });
      }

      console.log(`🎯 Generating structured template: ${templateKey} for user ${user.email}`);

      const { generateStructuredTemplate } = await import('./ai-section');
      const result = await generateStructuredTemplate(templateKey, businessBrief);

      res.json({ success: true, data: result.data, isSeedDraft: result.isSeedDraft });

    } catch (error) {
      console.error('Structured template generation error:', error);
      handleError(res, error, 'Failed to generate structured template');
    }
  });

  // Export business plan to PDF/DOCX - Pro-only endpoint
  app.post('/api/ai/plan/export', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Get user from database to check paid status server-side
      // In development mode, bypass database lookup if there's a connection issue
      let user;
      try {
        user = await storage.getUserByFirebaseUid(req.user.uid);
      } catch (dbError) {
        // Development fallback - create mock user if database is unavailable
        const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
        if (isDevelopment) {
          console.warn('Database unavailable, using mock user for testing:', dbError);
          user = { 
            id: 'dev-user-123', 
            firebaseUid: req.user.uid,
            email: req.user.email || 'dev@example.com',
            subscriptionTier: 'free' // Default to free tier for testing
          };
        } else {
          throw dbError; // Re-throw in production
        }
      }
      
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Request validation
      const { plan, format, businessName } = req.body as { plan: PlanResponse; format: "pdf"|"docx"; businessName?: string };
      
      if (!plan || !format) {
        return res.status(400).json({ error: 'Missing plan or format' });
      }

      if (!['pdf', 'docx'].includes(format)) {
        return res.status(400).json({ error: 'Invalid format. Must be "pdf" or "docx"' });
      }

      if (!plan.sections || !Array.isArray(plan.sections) || plan.sections.length === 0) {
        return res.status(400).json({ error: 'Plan must contain sections' });
      }

      // Paywall enforcement - Phase 1: All users are free tier
      const isPaidUser = false; // TODO: Replace with actual subscription check in Phase 2
      
      if (!isPaidUser) {
        return res.status(402).json({ 
          error: 'Export functionality requires a Pro subscription',
          code: 'PAYWALL',
          message: 'Upgrade to Pro to export your business plans as PDF or DOCX files'
        });
      }

      // Prepare export metadata
      const meta = {
        businessName: businessName || inferBusinessName(plan) || "Your Business",
        subtitle: "Business Plan",
        generatedAt: new Date(),
      };

      // Generate and return the export
      if (format === "pdf") {
        const doc = renderPlanPDF(plan, meta);
        const filename = slugify(`${meta.businessName}-Business-Plan.pdf`);
        res.setHeader("Content-Type", "application/pdf");
        res.setHeader("Content-Disposition", `attachment; filename="${filename}"`);
        doc.pipe(res);
        doc.end();
        
        // Log successful export for analytics
        console.log(`Business plan PDF exported successfully for user ${user.id}, businessName: ${meta.businessName}`);
        return;
      }

      if (format === "docx") {
        const buffer = await renderPlanDOCX(plan, meta);
        const filename = slugify(`${meta.businessName}-Business-Plan.docx`);
        res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.wordprocessingml.document");
        res.setHeader("Content-Disposition", `attachment; filename="${filename}"`);
        
        // Log successful export for analytics
        console.log(`Business plan DOCX exported successfully for user ${user.id}, businessName: ${meta.businessName}`);
        return res.send(buffer);
      }

      return res.status(400).json({ error: 'Invalid format' });
    } catch (error) {
      console.error('Business plan export error:', error);
      
      // Handle specific error types
      if (error instanceof Error) {
        if (error.message.includes('required') || error.message.includes('Missing')) {
          return res.status(400).json({ error: error.message });
        }
        if (error.message.includes('PDF') || error.message.includes('DOCX')) {
          return res.status(500).json({ error: 'Document generation error. Please try again.' });
        }
      }
      
      handleError(res, error, 'Failed to export business plan');
    }
  });

  // Export business plan to Google Docs - Pro-only endpoint
  app.post('/api/ai/plan/export/google-doc', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Get user from database to check paid status server-side
      // In development mode, bypass database lookup if there's a connection issue
      let user;
      try {
        user = await storage.getUserByFirebaseUid(req.user.uid);
      } catch (dbError) {
        // Development fallback - create mock user if database is unavailable
        const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
        if (isDevelopment) {
          console.warn('Database unavailable, using mock user for testing:', dbError);
          user = { 
            id: 'dev-user-123', 
            firebaseUid: req.user.uid,
            email: req.user.email || 'dev@example.com',
            subscriptionTier: 'free' // Default to free tier for testing
          };
        } else {
          throw dbError; // Re-throw in production
        }
      }
      
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Request validation
      const { plan, businessName, accessToken } = req.body as { 
        plan: PlanResponse; 
        businessName: string; 
        accessToken: string 
      };
      
      if (!plan || !businessName || !accessToken) {
        return res.status(400).json({ error: 'Missing plan, businessName, or accessToken' });
      }

      if (!plan.sections || !Array.isArray(plan.sections) || plan.sections.length === 0) {
        return res.status(400).json({ error: 'Plan must contain sections' });
      }

      // Paywall enforcement - Phase 1: All users are free tier
      const isPaidUser = false; // TODO: Replace with actual subscription check in Phase 2
      
      // Development bypass for testing Google Docs functionality
      const allowGoogleDocsDev = process.env.ALLOW_GOOGLE_DOCS_DEV === 'true';
      const isDevelopment = process.env.NODE_ENV === 'development' || process.env.NODE_ENV !== 'production';
      
      if (!isPaidUser && !(isDevelopment && allowGoogleDocsDev)) {
        return res.status(402).json({ 
          error: 'Google Docs export requires a Pro subscription',
          code: 'PAYWALL',
          message: 'Upgrade to Pro to save your business plans to Google Docs'
        });
      }

      // Import googleapis dynamically
      const { google } = await import('googleapis');
      
      // Initialize Google Drive API with user's access token
      const auth = new google.auth.OAuth2();
      auth.setCredentials({ access_token: accessToken });
      const drive = google.drive({ version: 'v3', auth });

      // Prepare export metadata
      const meta = {
        businessName: businessName || inferBusinessName(plan) || "Your Business",
        subtitle: "Business Plan",
        generatedAt: new Date(),
      };

      // Generate DOCX buffer using existing function
      const docxBuffer = await renderPlanDOCX(plan, meta);
      
      // Create filename for Google Doc
      const sanitizedName = meta.businessName.replace(/[^a-zA-Z0-9\s-]/g, '').trim();
      const modeLabel = plan.mode === 'full' ? 'Full' : 'Lean';
      const fileName = `${sanitizedName} - ${modeLabel} Business Plan`;

      // Upload DOCX to Google Drive and convert to Google Docs format
      const fileMetadata = {
        name: fileName,
        mimeType: 'application/vnd.google-apps.document' // Convert to Google Doc
      };

      const media = {
        mimeType: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        body: Readable.from(docxBuffer)
      };

      const response = await drive.files.create({
        requestBody: fileMetadata,
        media: media,
        fields: 'id,name,webViewLink'
      });

      const fileId = response.data.id;
      const fileUrl = response.data.webViewLink;
      const createdFileName = response.data.name;

      if (!fileId || !fileUrl) {
        throw new Error('Failed to create Google Doc - missing file ID or URL');
      }

      // Log successful export for analytics
      console.log(`Business plan exported to Google Docs successfully for user ${user.id}, businessName: ${meta.businessName}, fileId: ${fileId}`);

      // Return file details
      res.json({
        fileId,
        fileUrl,
        fileName: createdFileName
      });

    } catch (error) {
      console.error('Google Docs export error:', error);
      
      // Handle specific error types
      if (error instanceof Error) {
        if (error.message.includes('required') || error.message.includes('Missing')) {
          return res.status(400).json({ error: error.message });
        }
        if (error.message.includes('Google') || error.message.includes('Drive')) {
          return res.status(500).json({ error: 'Google Drive API error. Please check your Google account permissions and try again.' });
        }
        if (error.message.includes('access_token') || error.message.includes('authentication')) {
          return res.status(401).json({ error: 'Google authentication failed. Please sign in to Google again.' });
        }
        if (error.message.includes('scope') || error.message.includes('permission')) {
          return res.status(403).json({ error: 'Insufficient Google Drive permissions. Please grant access to create files.' });
        }
      }
      
      handleError(res, error, 'Failed to save business plan to Google Docs');
    }
  });

  // Helper function to get latest Recraft V3 SVG version
  const getLatestVersion = async (token: string): Promise<string> => {
    const resp = await fetch("https://api.replicate.com/v1/models/recraft-ai/recraft-v3-svg", {
      headers: { Authorization: `Token ${token}` },
      cache: "no-store" as RequestCache,
    });

    if (!resp.ok) {
      const text = await resp.text();
      throw new Error(`Failed to read model info (${resp.status}): ${text}`);
    }

    const json = await resp.json();
    const id = json?.latest_version?.id;
    if (!id) throw new Error("No latest_version.id on model payload");
    return id as string;
  };

  // Helper function to pick SVG URL from various output shapes (Nova's robust parsing)
  const pickSvgUrl = (output: unknown): string | null => {
    // DEBUG: Log the actual output structure
    console.log("[DEBUG] pickSvgUrl - output type:", typeof output);
    console.log("[DEBUG] pickSvgUrl - output structure:", JSON.stringify(output, null, 2));
    
    // Case 1: a plain string URL
    if (typeof output === "string") {
      console.log("[DEBUG] Case 1: string URL:", output);
      if (isValidImageUrl(output)) return output;
      return null;
    }

    // Case 2: array of URLs
    if (Array.isArray(output)) {
      console.log("[DEBUG] Case 2: array of URLs, length:", output.length);
      // Prefer explicit .svg
      const svg = output.find((u: any) => typeof u === "string" && isValidImageUrl(u));
      if (typeof svg === "string") {
        console.log("[DEBUG] Found SVG in array:", svg);
        return svg;
      }

      // Some models return objects with url/uri fields
      for (const item of output) {
        if (item && typeof item === "object") {
          const maybe = (item as any).url || (item as any).uri || (item as any).href;
          if (typeof maybe === "string" && isValidImageUrl(maybe)) {
            console.log("[DEBUG] Found image in object field:", maybe);
            return maybe;
          }
        }
      }
      
      // Also check for any string URL in array (even if not .svg)
      for (const item of output) {
        if (typeof item === "string") {
          console.log("[DEBUG] Found string in array:", item);
          // Maybe it's an SVG URL without .svg extension?
          if (item.includes("replicate") || item.includes("svg")) {
            console.log("[DEBUG] Returning suspected SVG URL:", item);
            return item;
          }
        }
      }
      
      return null;
    }

    // Case 3: object with a field containing the URL
    if (output && typeof output === "object") {
      console.log("[DEBUG] Case 3: object");
      const obj = output as any;
      const candidates = [obj.svg, obj.svg_url, obj.url, obj.uri, obj.href];
      console.log("[DEBUG] Candidates:", candidates);
      
      for (const c of candidates) {
        if (typeof c === "string") {
          console.log("[DEBUG] Checking candidate:", c);
          if (isValidImageUrl(c)) {
            console.log("[DEBUG] Found image in object candidate:", c);
            return c;
          }
        }
      }
      
      // Check all string values in the object
      for (const [key, value] of Object.entries(obj)) {
        if (typeof value === "string" && (value.includes("replicate") || value.includes("svg"))) {
          console.log("[DEBUG] Found potential URL in object key", key, ":", value);
          return value;
        }
      }
    }

    console.log("[DEBUG] No valid image URL found in output");
    return null;
  };

  const isSvgUrl = (u: string): boolean => {
    const lower = u.toLowerCase();
    return lower.endsWith(".svg") || lower.includes(".svg?");
  };

  const isValidImageUrl = (u: string): boolean => {
    const lower = u.toLowerCase();
    return lower.endsWith(".svg") || lower.includes(".svg?") || 
           lower.endsWith(".webp") || lower.includes(".webp") ||
           lower.endsWith(".png") || lower.includes(".png") ||
           lower.includes("replicate.delivery");
  };

  // Helper function to sanitize Replicate API token
  const sanitizeReplicateToken = (): { token: string; authHeader: string } | { error: string } => {
    const raw = process.env.REPLICATE_API_TOKEN || "";
    const token = raw
      .normalize('NFKC')
      .replace(/[''‚‛]/g, "'")
      .replace(/[""„‟]/g, '"')
      .replace(/^(Token|Bearer)\s+(?!r8_)/i, '') // Don't remove if followed by r8_
      .replace(/^['"]+|['"]+$/g, '')
      .trim();
    
    if (!token || /[^\x20-\x7E]/.test(token)) {
      return { 
        error: 'Invalid REPLICATE_API_TOKEN format. Please re-enter the token with plain ASCII (no quotes).' 
      };
    }
    
    return { token, authHeader: `Token ${token}` };
  };

  // Recraft Route 1: Create prediction and return predictionId
  app.post('/api/recraft/create', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      const tokenResult = sanitizeReplicateToken();
      if ('error' in tokenResult) {
        return res.status(500).json({ error: tokenResult.error });
      }
      
      const { token, authHeader } = tokenResult;

      const { prompt } = req.body;
      if (!prompt || typeof prompt !== 'string' || !prompt.trim()) {
        return res.status(400).json({ error: 'Missing prompt' });
      }

      // Enhance prompt to ensure transparent background
      const enhancedPrompt = `${prompt.trim()}, transparent background, no background fill, isolated logo icon, vector graphic`;
      console.log("[recraft] enhanced prompt:", enhancedPrompt);

      // Force resolve version every time to ensure we get the latest
      let version = "";
      try {
        version = await getLatestVersion(token);
        console.log("[recraft] resolved latest version:", version);
      } catch (e: any) {
        console.error("[recraft] Failed to resolve version:", e.message);
        return res.status(502).json({ error: `Could not resolve version: ${e.message}` });
      }

      // Start prediction
      const createRes = await fetch("https://api.replicate.com/v1/predictions", {
        method: "POST",
        headers: {
          Authorization: authHeader,
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          version: version,
          input: {
            prompt: enhancedPrompt,
            output_format: "svg", // ask explicitly for SVG
            background: "transparent",   // first try
            remove_background: true,     // fallback param some models use
            vectorize: true,            // Ensure vector output
            no_text: true,              // Remove text elements that could create backgrounds
          },
        }),
      });

      const rawCreate = await createRes.text();
      if (!createRes.ok) {
        console.error("[recraft] create failed:", createRes.status, rawCreate);
        return res.status(createRes.status || 502).json({ error: `Create failed: ${rawCreate}` });
      }

      const start = JSON.parse(rawCreate);
      const predictionId: string | undefined = start?.id;
      if (!predictionId) {
        console.error("[recraft] no prediction id. payload:", start);
        return res.status(502).json({ error: "No prediction id" });
      }

      console.log("[recraft] created prediction:", predictionId);
      return res.json({ predictionId, version });
    } catch (error: any) {
      console.error("[recraft] create route error:", error);
      handleError(res, error, 'Failed to create Recraft prediction');
    }
  });

  // Recraft Route 2: Check prediction status and extract SVG URL
  app.get('/api/recraft/status/:predictionId', async (req: Request, res: Response) => {
    try {
      const tokenResult = sanitizeReplicateToken();
      if ('error' in tokenResult) {
        return res.status(500).json({ error: tokenResult.error });
      }
      
      const { authHeader } = tokenResult;
      const { predictionId } = req.params;

      if (!predictionId) {
        return res.status(400).json({ error: 'Missing predictionId' });
      }

      // Fetch prediction status
      const stRes = await fetch(`https://api.replicate.com/v1/predictions/${predictionId}`, {
        headers: { Authorization: authHeader },
        cache: "no-store" as RequestCache,
      });

      const rawStatus = await stRes.text();
      if (!stRes.ok) {
        console.error("[recraft] status failed:", stRes.status, rawStatus);
        return res.status(stRes.status || 502).json({ error: `Status failed: ${rawStatus}` });
      }

      const data = JSON.parse(rawStatus);
      console.log(`[recraft] status for ${predictionId}:`, data.status);

      // Return detailed status info for client polling
      const response: any = {
        status: data.status,
        predictionId,
        rawOutputShape: data.output ? typeof data.output : null,
      };

      if (data.status === "succeeded") {
        const svgUrl = pickSvgUrl(data.output);
        if (svgUrl) {
          response.svgUrl = svgUrl;
          console.log("[recraft] extracted SVG URL:", svgUrl);
        } else {
          console.error("[recraft] no svgUrl resolved from output:", JSON.stringify(data.output));
          response.error = "No valid SVG URL found in output";
          response.debugOutput = data.output;
        }
      } else if (data.status === "failed") {
        console.error("[recraft] generation failed:", rawStatus);
        response.error = data.error || "Generation failed";
      }

      return res.json(response);
    } catch (error: any) {
      console.error("[recraft] status route error:", error);
      handleError(res, error, 'Failed to check Recraft status');
    }
  });

  // Font Generation Route using Replicate API - Pro-only endpoint
  app.post('/api/generate-fonts', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      const tokenResult = sanitizeReplicateToken();
      if ('error' in tokenResult) {
        return res.status(500).json({ error: tokenResult.error });
      }
      
      const { token, authHeader } = tokenResult;

      const prompt = "Generate 120 distinct, professional font options for branding kits. Each font should be listed by name only (no extra description). Return them as a clean JSON array of strings.";
      
      console.log("[fonts] generating fonts with prompt:", prompt);

      // Use meta/llama-2-70b-chat for text generation
      const createRes = await fetch("https://api.replicate.com/v1/predictions", {
        method: "POST",
        headers: {
          Authorization: authHeader,
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          version: "02e509c789964a7ea8736978a43525956ef40397be9033abf9fd2badfe68c9e3", // meta/llama-2-70b-chat
          input: {
            prompt: prompt,
            max_tokens: 1000,
            temperature: 0.2,
            top_p: 1,
            system_prompt: "You are a typography expert. Generate exactly 120 professional, real font names that are commonly available for branding projects. Return only a clean JSON array of strings with no additional text or explanation."
          },
        }),
      });

      const rawCreate = await createRes.text();
      if (!createRes.ok) {
        console.error("[fonts] create failed:", createRes.status, rawCreate);
        return res.status(createRes.status || 502).json({ error: `Font generation failed: ${rawCreate}` });
      }

      const start = JSON.parse(rawCreate);
      const predictionId: string | undefined = start?.id;
      if (!predictionId) {
        console.error("[fonts] no prediction id. payload:", start);
        return res.status(502).json({ error: "No prediction id" });
      }

      console.log("[fonts] created prediction:", predictionId);
      
      // Poll for completion
      const timeoutAt = Date.now() + 120_000; // 2 minutes
      let pollCount = 0;

      while (Date.now() < timeoutAt) {
        pollCount++;
        
        // Wait before polling
        if (pollCount > 1) {
          await new Promise(resolve => setTimeout(resolve, 2000));
        }

        const statusRes = await fetch(`https://api.replicate.com/v1/predictions/${predictionId}`, {
          headers: { Authorization: authHeader },
          cache: "no-store" as RequestCache,
        });

        const rawStatus = await statusRes.text();
        if (!statusRes.ok) {
          console.error("[fonts] status failed:", statusRes.status, rawStatus);
          return res.status(statusRes.status || 502).json({ error: `Status check failed: ${rawStatus}` });
        }

        const data = JSON.parse(rawStatus);
        console.log(`[fonts] poll #${pollCount} for ${predictionId}: ${data.status}`);

        if (data.status === "succeeded") {
          try {
            // Extract text from output
            let outputText = '';
            if (typeof data.output === 'string') {
              outputText = data.output;
            } else if (Array.isArray(data.output)) {
              outputText = data.output.join('');
            } else {
              throw new Error("Unexpected output format");
            }

            console.log("[fonts] raw output:", outputText.substring(0, 200) + "...");

            // Parse JSON from the output
            let fontArray: string[] = [];
            
            // Try to find and parse JSON array from the output
            const jsonMatch = outputText.match(/\[[\s\S]*?\]/);
            if (jsonMatch) {
              fontArray = JSON.parse(jsonMatch[0]);
            } else {
              // Fallback: try to parse the entire output
              fontArray = JSON.parse(outputText.trim());
            }

            // Validate and clean the array
            if (!Array.isArray(fontArray)) {
              throw new Error("Output is not an array");
            }

            // Clean and filter fonts
            const cleanedFonts = fontArray
              .filter(font => typeof font === 'string' && font.trim().length > 0)
              .map(font => font.trim())
              .slice(0, 120); // Ensure max 120 fonts

            if (cleanedFonts.length === 0) {
              throw new Error("No valid fonts found in output");
            }

            console.log(`[fonts] successfully generated ${cleanedFonts.length} fonts`);
            return res.json({ fonts: cleanedFonts });

          } catch (parseError: any) {
            console.error("[fonts] failed to parse output:", parseError.message);
            console.error("[fonts] raw output for debugging:", data.output);
            
            // Return fallback fonts if parsing fails
            const fallbackFonts = [
              "Inter", "Poppins", "Roboto", "Lato", "Open Sans", "Montserrat", "Nunito", "Raleway", 
              "Playfair Display", "Merriweather", "Lora", "PT Serif", "Crimson Text", "Spectral",
              "Arvo", "Roboto Slab", "Noto Serif Display", "Zilla Slab", "Alegreya",
              "Bebas Neue", "Oswald", "Orbitron", "Anton", "Fjalla One", "Righteous",
              "Pacifico", "Dancing Script", "Great Vibes", "Sacramento", "Amatic SC"
            ];
            
            return res.json({ 
              fonts: fallbackFonts, 
              fallback: true, 
              error: "Used fallback fonts due to parsing error" 
            });
          }
        } else if (data.status === "failed") {
          console.error("[fonts] generation failed:", rawStatus);
          return res.status(500).json({ error: data.error || "Font generation failed" });
        }
      }

      // Timeout
      console.error("[fonts] timeout waiting for completion");
      return res.status(408).json({ error: "Font generation timed out" });

    } catch (error: any) {
      console.error("[fonts] route error:", error);
      handleError(res, error, 'Failed to generate fonts');
    }
  });

  // Social Media Kits routes
  app.post('/api/social-media-kits', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const socialMediaKitData = insertSocialMediaKitSchema.parse({
        ...req.body,
        userId: user.id
      });

      const socialMediaKit = await storage.createSocialMediaKit(socialMediaKitData);
      res.status(201).json(socialMediaKit);
    } catch (error) {
      handleError(res, error, 'Failed to create social media kit');
    }
  });

  app.get('/api/social-media-kits', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const socialMediaKits = await storage.getSocialMediaKitsByUserId(user.id);
      res.json(socialMediaKits);
    } catch (error) {
      handleError(res, error, 'Failed to get social media kits');
    }
  });

  app.get('/api/social-media-kits/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const socialMediaKit = await storage.getSocialMediaKit(req.params.id);
      if (!socialMediaKit) {
        return res.status(404).json({ error: 'Social media kit not found' });
      }

      // Check ownership
      if (socialMediaKit.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      res.json(socialMediaKit);
    } catch (error) {
      handleError(res, error, 'Failed to get social media kit');
    }
  });

  app.put('/api/social-media-kits/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingSocialMediaKit = await storage.getSocialMediaKit(req.params.id);
      if (!existingSocialMediaKit) {
        return res.status(404).json({ error: 'Social media kit not found' });
      }

      // Check ownership
      if (existingSocialMediaKit.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const updateData = insertSocialMediaKitSchema.partial().parse(req.body);
      const updatedSocialMediaKit = await storage.updateSocialMediaKit(req.params.id, updateData);
      
      if (!updatedSocialMediaKit) {
        return res.status(404).json({ error: 'Social media kit not found' });
      }

      res.json(updatedSocialMediaKit);
    } catch (error) {
      handleError(res, error, 'Failed to update social media kit');
    }
  });

  app.delete('/api/social-media-kits/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingSocialMediaKit = await storage.getSocialMediaKit(req.params.id);
      if (!existingSocialMediaKit) {
        return res.status(404).json({ error: 'Social media kit not found' });
      }

      // Check ownership
      if (existingSocialMediaKit.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const deleted = await storage.deleteSocialMediaKit(req.params.id);
      if (!deleted) {
        return res.status(404).json({ error: 'Social media kit not found' });
      }

      res.status(204).send();
    } catch (error) {
      handleError(res, error, 'Failed to delete social media kit');
    }
  });

  // Website Templates routes
  app.post('/api/website-templates', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const websiteTemplateData = insertWebsiteTemplateSchema.parse({
        ...req.body,
        userId: user.id
      });

      const websiteTemplate = await storage.createWebsiteTemplate(websiteTemplateData);
      res.status(201).json(websiteTemplate);
    } catch (error) {
      handleError(res, error, 'Failed to create website template');
    }
  });

  app.get('/api/website-templates', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const websiteTemplates = await storage.getWebsiteTemplatesByUserId(user.id);
      res.json(websiteTemplates);
    } catch (error) {
      handleError(res, error, 'Failed to get website templates');
    }
  });

  app.get('/api/website-templates/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const websiteTemplate = await storage.getWebsiteTemplate(req.params.id);
      if (!websiteTemplate) {
        return res.status(404).json({ error: 'Website template not found' });
      }

      // Check ownership
      if (websiteTemplate.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      res.json(websiteTemplate);
    } catch (error) {
      handleError(res, error, 'Failed to get website template');
    }
  });

  app.put('/api/website-templates/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingWebsiteTemplate = await storage.getWebsiteTemplate(req.params.id);
      if (!existingWebsiteTemplate) {
        return res.status(404).json({ error: 'Website template not found' });
      }

      // Check ownership
      if (existingWebsiteTemplate.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const updateData = insertWebsiteTemplateSchema.partial().parse(req.body);
      const updatedWebsiteTemplate = await storage.updateWebsiteTemplate(req.params.id, updateData);
      
      if (!updatedWebsiteTemplate) {
        return res.status(404).json({ error: 'Website template not found' });
      }

      res.json(updatedWebsiteTemplate);
    } catch (error) {
      handleError(res, error, 'Failed to update website template');
    }
  });

  app.delete('/api/website-templates/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingWebsiteTemplate = await storage.getWebsiteTemplate(req.params.id);
      if (!existingWebsiteTemplate) {
        return res.status(404).json({ error: 'Website template not found' });
      }

      // Check ownership
      if (existingWebsiteTemplate.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const deleted = await storage.deleteWebsiteTemplate(req.params.id);
      if (!deleted) {
        return res.status(404).json({ error: 'Website template not found' });
      }

      res.status(204).send();
    } catch (error) {
      handleError(res, error, 'Failed to delete website template');
    }
  });

  // User Template Customization routes
  app.post('/api/template-customizations', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const customizationData = insertUserTemplateCustomizationSchema.parse({
        ...req.body,
        userId: user.id
      });

      const customization = await storage.createUserTemplateCustomization(customizationData);
      res.status(201).json(customization);
    } catch (error) {
      handleError(res, error, 'Failed to create template customization');
    }
  });

  app.get('/api/template-customizations', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const customizations = await storage.getUserTemplateCustomizationsByUserId(user.id);
      res.json(customizations);
    } catch (error) {
      handleError(res, error, 'Failed to get template customizations');
    }
  });

  app.get('/api/template-customizations/template/:templateId', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const customization = await storage.getUserTemplateCustomizationByTemplateId(user.id, req.params.templateId);
      if (!customization) {
        return res.status(404).json({ error: 'Template customization not found' });
      }

      res.json(customization);
    } catch (error) {
      handleError(res, error, 'Failed to get template customization');
    }
  });

  app.get('/api/template-customizations/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const customization = await storage.getUserTemplateCustomization(req.params.id);
      if (!customization) {
        return res.status(404).json({ error: 'Template customization not found' });
      }

      // Check ownership
      if (customization.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      res.json(customization);
    } catch (error) {
      handleError(res, error, 'Failed to get template customization');
    }
  });

  app.put('/api/template-customizations/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingCustomization = await storage.getUserTemplateCustomization(req.params.id);
      if (!existingCustomization) {
        return res.status(404).json({ error: 'Template customization not found' });
      }

      // Check ownership
      if (existingCustomization.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const updateData = insertUserTemplateCustomizationSchema.partial().parse(req.body);
      const updatedCustomization = await storage.updateUserTemplateCustomization(req.params.id, updateData);
      
      if (!updatedCustomization) {
        return res.status(404).json({ error: 'Template customization not found' });
      }

      res.json(updatedCustomization);
    } catch (error) {
      handleError(res, error, 'Failed to update template customization');
    }
  });

  app.delete('/api/template-customizations/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingCustomization = await storage.getUserTemplateCustomization(req.params.id);
      if (!existingCustomization) {
        return res.status(404).json({ error: 'Template customization not found' });
      }

      // Check ownership
      if (existingCustomization.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const deleted = await storage.deleteUserTemplateCustomization(req.params.id);
      if (!deleted) {
        return res.status(404).json({ error: 'Template customization not found' });
      }

      res.status(204).send();
    } catch (error) {
      handleError(res, error, 'Failed to delete template customization');
    }
  });

  app.post('/api/template-customizations/:id/duplicate', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const existingCustomization = await storage.getUserTemplateCustomization(req.params.id);
      if (!existingCustomization) {
        return res.status(404).json({ error: 'Template customization not found' });
      }

      // Check ownership
      if (existingCustomization.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      const { name } = req.body;
      if (!name || typeof name !== 'string') {
        return res.status(400).json({ error: 'Name is required for duplication' });
      }

      const duplicatedCustomization = await storage.duplicateUserTemplateCustomization(req.params.id, name);
      if (!duplicatedCustomization) {
        return res.status(500).json({ error: 'Failed to duplicate template customization' });
      }

      res.status(201).json(duplicatedCustomization);
    } catch (error) {
      handleError(res, error, 'Failed to duplicate template customization');
    }
  });

  // Enhanced website template route for better integration
  app.put('/api/website-templates/:id/customization', authenticateToken, hasProAccess, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const template = await storage.getWebsiteTemplate(req.params.id);
      if (!template) {
        return res.status(404).json({ error: 'Website template not found' });
      }

      // Check if user has permission to customize this template
      if (template.userId && template.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      // Check if customization already exists for this user/template
      let customization = await storage.getUserTemplateCustomizationByTemplateId(user.id, req.params.id);
      
      const customizationData = {
        configurations: req.body.configurations,
        contentOverrides: req.body.contentOverrides,
        styleOverrides: req.body.styleOverrides
      };

      if (customization) {
        // Update existing customization
        const updatedCustomization = await storage.updateUserTemplateCustomization(customization.id, customizationData);
        res.json(updatedCustomization);
      } else {
        // Create new customization
        const newCustomizationData = insertUserTemplateCustomizationSchema.parse({
          userId: user.id,
          templateId: req.params.id,
          customizationName: `${template.name || 'Template'} Customization`,
          ...customizationData
        });

        const newCustomization = await storage.createUserTemplateCustomization(newCustomizationData);
        res.status(201).json(newCustomization);
      }
    } catch (error) {
      handleError(res, error, 'Failed to save template customization');
    }
  });





  // Build Version Endpoint - Returns build fingerprint for deployment verification
  app.get('/api/version', (_req: Request, res: Response) => {
    res.json(BUILD);
  });

  // General Health Check - Overall system status with full diagnostics
  app.get('/api/health', async (_req: Request, res: Response) => {
    const corsOrigins = (process.env.CORS_ORIGINS || "").split(",").map(s => s.trim()).filter(Boolean);
    
    // Determine domain provider (OpenSRS direct or WHMCS proxy)
    const whmcsUrl = process.env.WHMCS_API_URL || null;
    const domainProvider = process.env.DOMAIN_PROVIDER || (whmcsUrl ? "whmcs" : "opensrs");
    let whmcsUrlHost = null;
    if (whmcsUrl) {
      try {
        whmcsUrlHost = new URL(whmcsUrl).host;
      } catch {
        whmcsUrlHost = whmcsUrl;
      }
    }
    
    // Fetch server public IP for OpenSRS whitelisting
    let serverPublicIp = null;
    try {
      const ipResponse = await fetch('https://api.ipify.org?format=json');
      if (ipResponse.ok) {
        const ipData = await ipResponse.json() as { ip: string };
        serverPublicIp = ipData.ip;
      }
    } catch {
      // Silently fail - IP is optional diagnostic info
    }
    
    res.json({ 
      ok: true,
      env: process.env.NODE_ENV || "development",
      hostname: os.hostname(),
      serverPublicIp,
      uptimeSec: Math.floor(process.uptime()),
      corsOrigins,
      opensrsConfigured: opensrs.isConfigured(),
      opensrsMode: process.env.OPENSRS_MODE || null,
      domainProvider,
      whmcsUrlHost,
      domainUrl: process.env.DOMAIN_URL || process.env.FRONTEND_URL || null,
      time: new Date().toISOString()
    });
  });

  // Domain Health Check - Check if OpenSRS credentials are configured
  app.get('/api/domain/health', (_req: Request, res: Response) => {
    const configured = opensrs.isConfigured();
    res.json({ configured });
  });

  // Domain Ping - Test OpenSRS provider latency and status
  app.get('/api/domain/ping', async (req: Request, res: Response) => {
    const user = process.env.OPENSRS_USER;
    const key = process.env.OPENSRS_KEY;
    const mode = process.env.OPENSRS_MODE || "ote";
    const configured = !!user && !!key;

    if (!configured) {
      return res.json({
        ok: false,
        provider: "OpenSRS",
        configured: false,
        mode,
        message: "OpenSRS not configured",
      });
    }

    const t0 = performance.now();
    try {
      // Lightweight sample: check availability of example.com
      const sampleDomain = "example.com";
      const available = await opensrs.checkDomainAvailability(sampleDomain);
      const latencyMs = Math.round(performance.now() - t0);
      
      return res.status(200).json({
        ok: true,
        provider: "OpenSRS",
        configured: true,
        mode,
        latencyMs,
        sample: sampleDomain,
        available,
      });
    } catch (err: any) {
      const latencyMs = Math.round(performance.now() - t0);
      console.error("[domain.ping] error", err?.message);
      return res.status(200).json({
        ok: false,
        provider: "OpenSRS",
        configured: true,
        mode,
        latencyMs,
        sample: "example.com",
        message: err?.message || "Ping failed",
      });
    }
  });

  // Stripe Ping - Test Stripe URLs reachability and configuration
  app.get('/api/stripe/ping', async (req: Request, res: Response) => {
    const domainUrl = process.env.DOMAIN_URL || process.env.FRONTEND_URL || null;
    const stripeKeyPresent = !!(process.env.STRIPE_SECRET_KEY || process.env.STRIPE_KEY);
    const configured = !!domainUrl && stripeKeyPresent;

    if (!configured) {
      return res.json({
        ok: false,
        configured: false,
        stripeKeyPresent,
        domainUrl,
        message: "Stripe not fully configured (missing STRIPE key and/or DOMAIN_URL/FRONTEND_URL).",
      });
    }

    // Build typical return URLs (HEAD check for reachability)
    const successUrl = `${domainUrl.replace(/\/$/, "")}/billing/success`;
    const cancelUrl = `${domainUrl.replace(/\/$/, "")}/billing/cancel`;
    const urls = [successUrl, cancelUrl];
    const results: any[] = [];
    let okOverall = true;

    for (const u of urls) {
      const t0 = performance.now();
      try {
        const r = await fetch(u, { method: "HEAD" });
        const latencyMs = Math.round(performance.now() - t0);
        const ok = r.ok || (r.status >= 200 && r.status < 400); // allow redirects
        results.push({ url: u, status: r.status, ok, latencyMs });
        if (!ok) okOverall = false;
      } catch (e: any) {
        const latencyMs = Math.round(performance.now() - t0);
        results.push({ url: u, ok: false, error: e?.message, latencyMs });
        okOverall = false;
      }
    }

    return res.json({
      ok: okOverall,
      configured: true,
      stripeKeyPresent,
      domainUrl,
      checks: results,
    });
  });

  // Telemetry Endpoint - Track client-side events (404s, redirects, errors, etc.)
  app.post('/api/telemetry', express.json(), (req: Request, res: Response) => {
    try {
      const body = req.body || {};
      const event: TelemetryEvent = {
        t: String(body.t || "unknown"),
        path: body.path,
        from: body.from,
        meta: body.meta,
        ts: body.ts || Date.now()
      };
      
      // Store in ring buffer
      pushEvent(event);
      
      // Log the event
      if (event.t === 'spa-404') {
        console.log(`[telemetry] 404 → ${event.path}`);
      } else if (event.t === 'redirect') {
        console.log(`[telemetry] redirect → ${event.from} → ${event.path}`);
      } else {
        console.log(`[telemetry] ${event.t} →`, event);
      }
      
      res.json({ ok: true });
    } catch (error) {
      // Silent fail - don't break client if telemetry fails
      res.json({ ok: true });
    }
  });

  // Telemetry Recent Events - Get last 200 events from ring buffer
  app.get('/api/telemetry/recent', (_req: Request, res: Response) => {
    res.json({ items: TELEMETRY.slice(-200).reverse() });
  });

  // Domain Search API route - OpenSRS based
  app.post('/api/domains/search', async (req: Request, res: Response) => {
    try {
      const searchInputSchema = z.object({
        query: z.string().min(1, 'Domain query is required').max(100),
        tlds: z.array(z.string()).optional()
      });

      const { query, tlds } = searchInputSchema.parse(req.body);
      
      console.log(`Domain search requested for: ${query}`);
      
      const hits = await opensrs.search({ query, tlds });
      const result: DomainSearchResult = { hits };
      
      res.json(result);
    } catch (error) {
      console.error('Domain search error:', error);
      handleError(res, error, 'Failed to search domains');
    }
  });

  // Domain Price Check API route - PUBLIC - Check domain pricing and credit eligibility
  app.get('/api/domain/price', async (req: Request, res: Response) => {
    const startTime = Date.now();
    const requestId = Math.random().toString(36).substring(7);
    
    try {
      // Simple validation: require domain with TLD
      const name = String(req.query.q || '').trim().toLowerCase();
      if (!name.includes('.')) {
        return res.status(400).json({ 
          error: 'Unsupported TLD', 
          message: 'Add a TLD like .com, .net, .org, or .co.' 
        });
      }

      const domain = name;
      
      // Extract and validate TLD
      const tldMatch = domain.match(/\.(\w+)$/);
      if (!tldMatch) {
        return res.status(400).json({ 
          error: 'Invalid domain format', 
          message: 'Add a TLD like .com, .net, .org, or .co.'
        });
      }
      
      const tld = '.' + tldMatch[1].toLowerCase();
      
      // Check if TLD is supported
      const supportedTlds = await opensrs.getSupportedTlds();
      const supportedTldList = supportedTlds.map(t => t.tld);
      if (!supportedTldList.includes(tld)) {
        return res.status(400).json({ 
          error: 'Unsupported TLD', 
          message: `TLD '${tld}' is not supported. Supported TLDs: ${supportedTldList.join(', ')}`,
          supportedTlds: supportedTldList
        });
      }
      
      // Get user if authenticated (for credit checking)
      let user = null;
      const authHeader = req.headers.authorization;
      if (authHeader?.startsWith('Bearer ')) {
        try {
          const token = authHeader.substring(7);
          const decodedToken = await getFirebaseAdmin().auth().verifyIdToken(token);
          user = await storage.getUserByFirebaseUid(decodedToken.uid);
        } catch (err) {
          // Ignore auth errors - route is public
        }
      }

      // Check domain availability using existing OpenSRS integration with enhanced error handling
      let searchResults;
      let domainResult;
      
      try {
        searchResults = await opensrs.search({ query: domain });
        domainResult = searchResults.find(hit => hit.domain === domain);
      } catch (opensrsError: any) {
        console.error('OpenSRS search error:', opensrsError);
        
        // Map OpenSRS validation errors to 400 status codes
        const errorMessage = opensrsError.message || 'Domain lookup failed';
        
        // Check for common OpenSRS validation errors that should return 400
        if (
          errorMessage.includes('invalid domain') ||
          errorMessage.includes('Invalid domain format') ||
          errorMessage.includes('domain syntax') ||
          errorMessage.includes('malformed domain') ||
          errorMessage.toLowerCase().includes('validation')
        ) {
          return res.status(400).json({ 
            error: 'Invalid domain', 
            message: errorMessage,
            domain: domain
          });
        }
        
        // For other OpenSRS errors, return 500
        return res.status(500).json({ 
          error: 'Domain lookup service error', 
          message: 'Unable to check domain availability at this time. Please try again later.'
        });
      }
      
      if (!domainResult) {
        return res.status(400).json({ 
          error: 'Domain lookup failed', 
          message: 'Unable to retrieve domain information',
          domain: domain
        });
      }

      // Get comprehensive pricing (wholesale + retail) - ensure we use wholesale for coverage calculation
      const pricing = getDomainPricing(domain);
      
      // Check user's available domain credits if authenticated
      let creditInfo = null;
      if (user) {
        const availableCredits = await storage.getAvailableDomainCredits(user.id);
        
        // Sort credits by expiration date (earliest first) to select the most urgent one
        const sortedCredits = availableCredits.sort((a, b) => 
          new Date(a.expiresAt).getTime() - new Date(b.expiresAt).getTime()
        );
        
        // Determine credit eligibility and coverage
        if (sortedCredits.length > 0 && domainResult.available) {
        const credit = sortedCredits[0]; // Use earliest-expiring credit
        const eligibleTlds = Array.isArray(credit.eligibleTlds) 
          ? credit.eligibleTlds 
          : DEFAULT_ELIGIBLE_TLDS;
        
        // Check if TLD is eligible and domain is not premium
        const isTldEligible = eligibleTlds.includes(tld);
        const isPremium = domainResult.premium || domainResult.type === 'premium';
        
        if (isTldEligible && !isPremium) {
          // Calculate coverage using wholesale price per SuperNova business rules
          const wholesaleCents = pricing.wholesale.priceCents;
          const coveredCents = Math.min(wholesaleCents, credit.capCents);
          
          creditInfo = {
            credit_id: credit.id,
            eligible: true,
            cap_cents: credit.capCents,
            covered_cents: coveredCents,
            expires_at: credit.expiresAt
          };
        } else {
          // Structured error for ineligible credits
          creditInfo = {
            credit_id: credit.id,
            eligible: false,
            cap_cents: credit.capCents,
            covered_cents: 0,
            expires_at: credit.expiresAt,
            ineligible_reason: isPremium 
              ? 'Premium domains are not eligible for domain credits'
              : `TLD '${tld}' is not eligible for domain credits. Eligible TLDs: ${eligibleTlds.join(', ')}`
          };
        }
        }
      }

      // Calculate out-of-pocket amount for frontend convenience
      const retailCents = pricing.retail.priceCents;
      const coveredCents = creditInfo?.covered_cents || 0;
      const outOfPocketCents = Math.max(0, retailCents - coveredCents);

      // Format response according to specification with enhanced fields
      const response = {
        domain: domainResult.domain,
        tld: tld,
        available: domainResult.available,
        premium: domainResult.premium || domainResult.type === 'premium' || false,
        pricing: {
          wholesale: {
            registration: pricing.wholesale.priceCents / 100,
            currency: pricing.wholesale.currency
          },
          retail: {
            registration: pricing.retail.priceCents / 100,
            currency: pricing.retail.currency
          }
        },
        credit: creditInfo,
        // Convenience field for frontend cart calculations
        out_of_pocket_cents: outOfPocketCents
      };
      
      const duration = Date.now() - startTime;
      console.log(`[domain-price:${requestId}] ✅ Success in ${duration}ms - ${domain}: available=${domainResult.available}, retail=$${pricing.retail.priceCents / 100}, credit_eligible=${creditInfo?.eligible || false}, out_of_pocket=$${outOfPocketCents / 100}`);
      
      res.json(response);
    } catch (error: any) {
      const duration = Date.now() - startTime;
      console.error(`[domain-price:${requestId}] ❌ Error after ${duration}ms:`, {
        message: error.message,
        stack: error.stack?.split('\n').slice(0, 3).join('\n'),
        type: error.constructor.name
      });
      
      // Enhanced error handling with more specific error responses
      if (error instanceof z.ZodError) {
        return res.status(400).json({ 
          error: 'Validation failed', 
          details: error.errors,
          message: 'Invalid request parameters'
        });
      }
      
      // Generic error handler for unexpected errors
      handleError(res, error, 'Failed to check domain price');
    }
  });

  // Domain Order API route - Creates Stripe checkout session
  app.post('/api/domains/order', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const orderInputSchema = insertDomainOrderSchema.omit({
        userId: true,
        status: true,
        priceCents: true,
        stripeSessionId: true,
        providerRegId: true,
        expiresAt: true,
        errorMessage: true
      });

      const orderData = orderInputSchema.parse(req.body);
      
      // Get user from database to ensure they exist
      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Verify domain is available
      const searchResult = await opensrs.search({ query: orderData.domain });
      const domainResult = searchResult.find(hit => hit.domain === orderData.domain);
      
      if (!domainResult || !domainResult.available) {
        return res.status(400).json({ error: 'Domain is not available for registration' });
      }

      if (!domainResult.priceCents) {
        return res.status(400).json({ error: 'Domain pricing not available' });
      }

      const stripe = getStripe();
      if (!stripe) {
        return res.status(500).json({ error: 'Payment system not available' });
      }

      // Create domain order in database with pending status
      const domainOrder = await storage.createDomainOrder({
        ...orderData,
        userId: user.id,
        priceCents: domainResult.priceCents,
        status: 'pending'
      });

      // Create Stripe checkout session
      const session = await stripe.checkout.sessions.create({
        mode: 'payment',
        payment_method_types: ['card'],
        line_items: [{
          price_data: {
            currency: 'usd',
            product_data: {
              name: `Domain Registration: ${orderData.domain}`,
              description: `${orderData.years} year${orderData.years > 1 ? 's' : ''} domain registration${orderData.privacy ? ' with privacy protection' : ''}`
            },
            unit_amount: domainResult.priceCents
          },
          quantity: 1
        }],
        customer_email: user.email,
        metadata: {
          type: 'domain_order',
          domainOrderId: domainOrder.id,
          userId: user.id,
          domain: orderData.domain
        },
        success_url: `${req.protocol}://${req.get('host')}/domains/orders/${domainOrder.id}?success=true`,
        cancel_url: `${req.protocol}://${req.get('host')}/domains/orders/${domainOrder.id}?canceled=true`
      });

      // Update domain order with Stripe session ID
      await storage.updateDomainOrder(domainOrder.id, {
        stripeSessionId: session.id
      });

      res.json({
        checkoutUrl: session.url,
        orderId: domainOrder.id
      });
    } catch (error) {
      console.error('Domain order error:', error);
      handleError(res, error, 'Failed to create domain order');
    }
  });

  // Domain Order Status API route
  app.get('/api/domains/orders/:id', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      const domainOrder = await storage.getDomainOrder(req.params.id);
      if (!domainOrder) {
        return res.status(404).json({ error: 'Domain order not found' });
      }

      // Ensure user owns this order
      if (domainOrder.userId !== user.id) {
        return res.status(403).json({ error: 'Access denied' });
      }

      res.json(domainOrder);
    } catch (error) {
      console.error('Domain order fetch error:', error);
      handleError(res, error, 'Failed to fetch domain order');
    }
  });

  // Domain Registration API route - Applies credits and handles payments
  app.post('/api/domain/register', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      // Input validation schema
      const registerInputSchema = z.object({
        domain: z.string()
          .min(1, 'Domain is required')
          .max(253, 'Domain name too long')
          .regex(
            /^[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*\.[a-zA-Z]{2,}$/,
            'Invalid domain format'
          )
          .transform((val) => val.toLowerCase().trim()),
        contact: z.object({
          firstName: z.string().min(1, 'First name is required'),
          lastName: z.string().min(1, 'Last name is required'),
          email: z.string().email('Valid email is required'),
          phone: z.string().min(1, 'Phone number is required'),
          address: z.string().min(1, 'Address is required'),
          city: z.string().min(1, 'City is required'),
          state: z.string().min(1, 'State is required'),
          zip: z.string().min(1, 'ZIP code is required'),
          country: z.string().min(2, 'Country code is required').max(2),
          organization: z.string().optional()
        }),
        nameservers: z.array(z.string()).optional(),
        payment_method_id: z.string().optional() // Stripe payment method ID
      });

      const registrationData = registerInputSchema.parse(req.body);
      const { domain, contact, nameservers, payment_method_id } = registrationData;
      
      console.log(`Domain registration requested for: ${domain} by user ${req.user.uid}`);

      // Get user from database
      const user = await storage.getUserByFirebaseUid(req.user.uid);
      if (!user) {
        return res.status(404).json({ error: 'User not found' });
      }

      // Check domain availability via OpenSRS
      const searchResults = await opensrs.search({ query: domain });
      const domainResult = searchResults.find(hit => hit.domain === domain);
      
      if (!domainResult || !domainResult.available) {
        return res.status(400).json({ 
          error: 'Domain not available', 
          message: `Domain ${domain} is not available for registration` 
        });
      }

      if (!domainResult.priceCents) {
        return res.status(400).json({ error: 'Domain pricing not available' });
      }

      // Get comprehensive pricing and apply credit coverage logic (reused from /api/domain/price)
      const pricing = getDomainPricing(domain);
      const tld = '.' + domain.split('.').pop()?.toLowerCase();
      
      // Get user's available domain credits (earliest-expiring first)
      const availableCredits = await storage.getAvailableDomainCredits(user.id);
      const sortedCredits = availableCredits.sort((a, b) => 
        new Date(a.expiresAt).getTime() - new Date(b.expiresAt).getTime()
      );
      
      // Determine credit eligibility and coverage
      let creditInfo = null;
      let applicableCredit = null;
      let creditCoverageCents = 0;
      
      for (const credit of sortedCredits) {
        if (credit.eligibleTlds.includes(tld) && pricing.wholesale.priceCents <= credit.capCents) {
          applicableCredit = credit;
          creditCoverageCents = pricing.wholesale.priceCents;
          creditInfo = {
            id: credit.id,
            eligible: true,
            coverage_cents: creditCoverageCents,
            cap_cents: credit.capCents,
            expires_at: credit.expiresAt,
            reason: `Credit covers full wholesale cost ($${creditCoverageCents / 100})`
          };
          break;
        } else if (credit.eligibleTlds.includes(tld)) {
          applicableCredit = credit;
          creditCoverageCents = credit.capCents;
          creditInfo = {
            id: credit.id,
            eligible: true,
            coverage_cents: creditCoverageCents,
            cap_cents: credit.capCents,
            expires_at: credit.expiresAt,
            reason: `Credit provides partial coverage ($${creditCoverageCents / 100} of $${pricing.wholesale.priceCents / 100} wholesale)`
          };
          break;
        }
      }
      
      // Calculate payment delta
      const outOfPocketCents = Math.max(0, pricing.wholesale.priceCents - creditCoverageCents);
      
      console.log(`Domain ${domain} pricing: wholesale=$${pricing.wholesale.priceCents / 100}, credit_coverage=$${creditCoverageCents / 100}, out_of_pocket=$${outOfPocketCents / 100}`);

      // Handle payment if there's an out-of-pocket amount
      let paymentIntent = null;
      const stripe = getStripe();
      
      if (outOfPocketCents > 0) {
        if (!payment_method_id) {
          return res.status(400).json({ 
            error: 'Payment required', 
            message: `Payment of $${outOfPocketCents / 100} is required. Please provide payment_method_id.`,
            out_of_pocket_cents: outOfPocketCents
          });
        }
        
        if (!stripe) {
          return res.status(500).json({ error: 'Payment system not available' });
        }

        // Create Stripe customer if needed
        let customerId = user.stripeCustomerId;
        if (!customerId) {
          const customer = await stripe.customers.create({
            email: user.email,
            name: user.displayName || `${contact.firstName} ${contact.lastName}`,
            metadata: { user_id: user.id, firebase_uid: user.firebaseUid }
          });
          customerId = customer.id;
          
          // Update user with Stripe customer ID
          await storage.updateUser(user.id, { stripeCustomerId: customerId });
        }

        // Create PaymentIntent for the delta amount
        paymentIntent = await stripe.paymentIntents.create({
          amount: outOfPocketCents,
          currency: 'usd',
          customer: customerId,
          payment_method: payment_method_id,
          confirmation_method: 'manual',
          confirm: true,
          return_url: `${req.protocol}://${req.get('host')}/domains`,
          metadata: {
            type: 'domain_registration',
            domain,
            user_id: user.id,
            credit_used: applicableCredit?.id || 'none'
          },
          description: `Domain registration payment for ${domain}`
        });

        // Handle payment confirmation
        if (paymentIntent.status === 'requires_action' || paymentIntent.status === 'requires_source_action') {
          return res.json({
            requires_action: true,
            payment_intent: {
              id: paymentIntent.id,
              client_secret: paymentIntent.client_secret,
              status: paymentIntent.status
            }
          });
        }

        if (paymentIntent.status !== 'succeeded') {
          return res.status(400).json({ 
            error: 'Payment failed', 
            message: `Payment status: ${paymentIntent.status}`,
            payment_intent_id: paymentIntent.id
          });
        }
        
        console.log(`Payment successful for ${domain}: ${paymentIntent.id}, amount=$${outOfPocketCents / 100}`);
      }

      // Prepare contact data for OpenSRS registration
      const opensrsContact = {
        firstName: contact.firstName,
        lastName: contact.lastName,
        email: contact.email,
        phone: contact.phone,
        address: contact.address,
        city: contact.city,
        state: contact.state,
        zip: contact.zip,
        country: contact.country,
        organization: contact.organization
      };

      // Register domain via OpenSRS
      console.log(`Registering domain ${domain} via OpenSRS...`);
      const registrationResult = await opensrs.register({
        domain,
        years: 1, // Default to 1 year
        contact: opensrsContact,
        privacy: false, // Default to no privacy for now
        nameservers: nameservers?.length ? nameservers : undefined
      });

      if (!registrationResult.success) {
        // If registration failed and we charged payment, we should refund it
        if (paymentIntent && paymentIntent.status === 'succeeded') {
          try {
            await stripe.refunds.create({
              payment_intent: paymentIntent.id,
              reason: 'requested_by_customer'
            });
            console.log(`Refunded payment ${paymentIntent.id} due to failed domain registration`);
          } catch (refundError) {
            console.error(`Failed to refund payment ${paymentIntent.id}:`, refundError);
          }
        }
        
        return res.status(400).json({ 
          error: 'Domain registration failed',
          message: registrationResult.message || 'Registration failed with registrar',
          opensrs_error: registrationResult.message
        });
      }

      console.log(`Domain registration successful: ${domain}, OpenSRS ID: ${registrationResult.providerRegId}`);

      // Create domain record in database
      const domainRecord = await storage.createDomain({
        userId: user.id,
        domain,
        expiresAt: new Date(Date.now() + 365 * 24 * 60 * 60 * 1000), // 1 year from now
        autorenew: true,
        registrar: 'opensrs',
        providerRegId: registrationResult.providerRegId,
        nameservers: nameservers || [],
        status: 'active'
      });

      // Redeem the credit if one was used
      let creditUsed = null;
      if (applicableCredit && creditCoverageCents > 0) {
        await storage.updateDomainCreditStatus(
          applicableCredit.id, 
          'used', 
          domain, 
          new Date()
        );
        
        creditUsed = {
          credit_id: applicableCredit.id,
          amount_cents: creditCoverageCents
        };
        
        console.log(`Domain credit redeemed: ${applicableCredit.id}, amount=$${creditCoverageCents / 100} for domain ${domain}`);
        
        // Create success notification
        await storage.createNotification(
          user.id,
          `Domain credit of $${creditCoverageCents / 100} successfully applied to ${domain} registration!`,
          'success'
        );
      }

      // Create success notification for domain registration
      await storage.createNotification(
        user.id,
        `Domain ${domain} has been successfully registered and is now active!`,
        'success'
      );

      // Successful response
      const response = {
        success: true,
        domain: domainRecord,
        payment_intent: paymentIntent?.id || null,
        opensrs_order: registrationResult.providerRegId || null,
        credit_used: creditUsed,
        pricing: {
          wholesale_cents: pricing.wholesale.priceCents,
          credit_coverage_cents: creditCoverageCents,
          out_of_pocket_cents: outOfPocketCents
        }
      };

      console.log(`Domain registration completed successfully: ${domain}, user: ${user.id}, credit_used: ${creditUsed?.credit_id || 'none'}, payment: ${paymentIntent?.id || 'none'}`);
      
      res.json(response);
    } catch (error) {
      console.error('Domain registration error:', error);
      
      if (error instanceof z.ZodError) {
        return res.status(400).json({ 
          error: 'Validation failed', 
          details: error.errors,
          message: 'Invalid request parameters'
        });
      }
      
      // Enhanced error handling for specific error types
      if (error instanceof Error) {
        if (error.message.includes('OpenSRS')) {
          return res.status(500).json({ 
            error: 'Domain registration service error',
            message: 'Failed to communicate with domain registrar'
          });
        }
        
        if (error.message.includes('Stripe') || error.message.includes('payment')) {
          return res.status(500).json({ 
            error: 'Payment processing error',
            message: 'Failed to process payment'
          });
        }
      }
      
      handleError(res, error, 'Failed to register domain');
    }
  });

  // ===============================
  // MCP (Model Context Protocol) Endpoints for SuperNova Integration
  // ===============================
  
  // MCP audit logger
  async function mcpAudit(event: any) {
    const line = `[${new Date().toISOString()}] ${JSON.stringify(event)}\n`;
    await fs.appendFile("./mcp-audit.log", line).catch(() => {});
  }

  // MCP Bearer authentication middleware
  const mcpAuth = (req: Request, res: Response, next: NextFunction) => {
    const auth = (req.headers.authorization || "").split(" ")[1];
    const bearerToken = process.env.MCP_BEARER_TOKEN || "changeme";
    
    if (!auth || auth !== bearerToken) {
      mcpAudit({ type: "unauth", ip: req.ip, path: req.path, headers: { ua: req.headers["user-agent"] } }).catch(() => {});
      return res.status(401).json({ error: "Unauthorized" });
    }
    next();
  };

  // Command whitelist for MCP /run endpoint
  const MCP_COMMAND_WHITELIST = {
    build: "npm run build",
    test: "npm test",
    lint: "npm run lint",
    typecheck: "npm run check",
    dev: "npm run dev",
    "seed:db": "node scripts/seed.js"
  };

  // MCP: Ping endpoint
  app.get('/mcp/ping', mcpAuth, async (req, res) => {
    await mcpAudit({ type: "ping", requester: req.headers["x-requester"] || "unknown" });
    res.json({ ok: true, ts: Date.now() });
  });

  // MCP: Run whitelisted commands
  app.post('/mcp/run', mcpAuth, async (req, res) => {
    const { cmdKey, args = [] } = req.body || {};
    const requester = req.headers["x-requester"] || "unknown";

    if (!cmdKey || !(cmdKey in MCP_COMMAND_WHITELIST)) {
      await mcpAudit({ type: "run_rejected", cmdKey, requester, ip: req.ip });
      return res.status(400).json({ error: "Invalid or non-whitelisted command key" });
    }

    const fullCmd = `${MCP_COMMAND_WHITELIST[cmdKey]} ${args.map(a => `"${String(a).replace(/"/g, '\\"')}"`).join(" ")}`.trim();

    // Additional safety: destructive commands require confirmation
    const requiresConfirm = ["seed:db"].includes(cmdKey);
    if (requiresConfirm && req.headers["x-confirm"] !== "yes") {
      await mcpAudit({ type: "run_needs_confirm", cmdKey, requester, fullCmd, ip: req.ip });
      return res.status(409).json({ error: "Command requires explicit confirmation header x-confirm: yes" });
    }

    await mcpAudit({ type: "run_start", cmdKey, requester, fullCmd, ip: req.ip });

    exec(fullCmd, { maxBuffer: 10 * 1024 * 1024 }, async (err, stdout, stderr) => {
      await mcpAudit({
        type: "run_finish",
        cmdKey,
        requester,
        fullCmd,
        exitCode: err ? (err.code || "err") : 0,
        stdout: String(stdout).slice(0, 10000),
        stderr: String(stderr).slice(0, 10000),
      }).catch(() => {});

      res.json({
        cmdKey,
        fullCmd,
        success: !err,
        stdout: String(stdout),
        stderr: String(stderr),
        error: err ? err.message : undefined
      });
    });
  });

  // MCP: Read files
  app.post('/mcp/files/read', mcpAuth, async (req, res) => {
    const { path: relPath } = req.body || {};
    if (!relPath) return res.status(400).json({ error: "path required" });

    // Prevent path traversal
    const safePath = path.resolve(process.cwd(), relPath);
    if (!safePath.startsWith(process.cwd())) return res.status(403).json({ error: "Forbidden" });

    try {
      const content = await fs.readFile(safePath, "utf8");
      await mcpAudit({ type: "files_read", path: relPath, requester: req.headers["x-requester"] || "unknown" });
      res.json({ path: relPath, content });
    } catch (e) {
      await mcpAudit({ type: "files_read_error", path: relPath, error: e.message });
      res.status(500).json({ error: e.message });
    }
  });

  // MCP: Write files (requires confirmation)
  app.post('/mcp/files/write', mcpAuth, async (req, res) => {
    const { path: relPath, content } = req.body || {};
    if (!relPath) return res.status(400).json({ error: "path required" });

    if (req.headers["x-confirm"] !== "yes") {
      await mcpAudit({ type: "files_write_needs_confirm", path: relPath, requester: req.headers["x-requester"] || "unknown" });
      return res.status(409).json({ error: "File writes require x-confirm: yes header" });
    }

    const safePath = path.resolve(process.cwd(), relPath);
    if (!safePath.startsWith(process.cwd())) return res.status(403).json({ error: "Forbidden" });

    try {
      await fs.mkdir(path.dirname(safePath), { recursive: true });
      await fs.writeFile(safePath, String(content || ""), "utf8");
      await mcpAudit({ type: "files_write", path: relPath, requester: req.headers["x-requester"] || "unknown" });
      res.json({ path: relPath, success: true });
    } catch (e) {
      await mcpAudit({ type: "files_write_error", path: relPath, error: e.message });
      res.status(500).json({ error: e.message });
    }
  });

  // MCP: Tail logs
  app.get('/mcp/logs/tail', mcpAuth, async (req, res) => {
    const lines = parseInt(req.query.lines as string || "200", 10);
    const logfile = process.env.APP_LOG || "logs/app.log";
    
    try {
      const content = await fs.readFile(logfile, "utf8");
      const all = content.split(/\r?\n/).filter(Boolean);
      const tail = all.slice(-lines).join("\n");
      await mcpAudit({ type: "logs_tail", lines, requester: req.headers["x-requester"] || "unknown" });
      res.type("text/plain").send(tail);
    } catch (e) {
      await mcpAudit({ type: "logs_error", error: e.message, path: logfile });
      res.status(500).json({ error: "Unable to read log file", details: e.message });
    }
  });

  // MCP: Environment variables
  app.post('/mcp/env/get', mcpAuth, async (req, res) => {
    const { key } = req.body || {};
    if (!key) return res.status(400).json({ error: "key required" });
    await mcpAudit({ type: "env_get", key, requester: req.headers["x-requester"] || "unknown" });
    res.json({ key, value: process.env[key] || null });
  });

  app.post('/mcp/env/set', mcpAuth, async (req, res) => {
    const { key, value } = req.body || {};
    if (!key) return res.status(400).json({ error: "key required" });
    if (req.headers["x-confirm"] !== "yes") {
      await mcpAudit({ type: "env_set_needs_confirm", key, requester: req.headers["x-requester"] || "unknown" });
      return res.status(409).json({ error: "Setting env requires x-confirm: yes" });
    }
    process.env[key] = String(value || "");
    await mcpAudit({ type: "env_set", key, requester: req.headers["x-requester"] || "unknown" });
    res.json({ success: true, key });
  });

  // MCP: Create GitHub PR (optional - requires GitHub token)
  app.post('/mcp/repo/pr', mcpAuth, async (req, res) => {
    const GITHUB_TOKEN = process.env.GITHUB_TOKEN;
    const REPO_OWNER = process.env.REPO_OWNER;
    const REPO_NAME = process.env.REPO_NAME;

    if (!GITHUB_TOKEN || !REPO_OWNER || !REPO_NAME) {
      return res.status(500).json({ error: "GitHub integration not configured" });
    }

    const { branchName, title, body } = req.body || {};
    if (!branchName || !title) return res.status(400).json({ error: "branchName and title required" });

    try {
      const api = `https://api.github.com/repos/${REPO_OWNER}/${REPO_NAME}/pulls`;
      const response = await fetch(api, {
        method: "POST",
        headers: { "Authorization": `token ${GITHUB_TOKEN}`, "Accept": "application/vnd.github.v3+json" },
        body: JSON.stringify({ title, head: branchName, base: "main", body: body || "" })
      });
      const data = await response.json();
      await mcpAudit({ type: "repo_pr", branchName, title, result: { status: response.status, id: data.number } });
      res.json({ ok: response.ok, status: response.status, data });
    } catch (e) {
      await mcpAudit({ type: "repo_pr_error", error: e.message });
      res.status(500).json({ error: e.message });
    }
  });

  // Brand Kit Queue Endpoint
  app.post('/api/brand-kit/queue', authenticateToken, async (req: AuthenticatedRequest, res: Response) => {
    try {
      if (!req.user) {
        return res.status(401).json({ error: 'User not authenticated' });
      }

      const { svgMarkup, brandName, colors, fonts } = req.body;

      if (!svgMarkup || !brandName) {
        return res.status(400).json({ error: 'Missing required fields: svgMarkup, brandName' });
      }

      // TODO: Implement actual queueing system (e.g., Redis queue, database job table)
      // For now, log the request and return success
      console.log('Brand kit requested:', {
        userId: req.user.uid,
        brandName,
        colorsCount: colors?.length || 0,
        fontsCount: fonts?.length || 0,
        svgLength: svgMarkup.length,
      });

      // Future: Queue job for background processing
      // await jobQueue.add('send-to-brand-kit', {
      //   userId: req.user.uid,
      //   svgMarkup,
      //   brandName,
      //   colors,
      //   fonts,
      //   });

      res.json({
        success: true,
        message: 'Logo sent to brand kit successfully',
      });
    } catch (error) {
      console.error('Brand kit queue error:', error);
      handleError(res, error, 'Failed to send to brand kit');
    }
  });

  const httpServer = createServer(app);
  return httpServer;
}

// GoDaddy API Integration
async function searchDomainsWithGoDaddy(searchDomain: string) {
  const apiKey = process.env.API_KEY;
  const apiSecret = process.env.API_SECRET;
  
  if (!apiKey || !apiSecret) {
    throw new Error('GoDaddy API credentials not configured');
  }

  // Use test environment to avoid 50+ domain restriction
  const baseUrl = 'https://api.ote-godaddy.com';
  console.log(`Using GoDaddy TEST environment: ${baseUrl}`);

  const authHeader = `sso-key ${apiKey}:${apiSecret}`;
  const results = [];

  try {
    // First, try the suggestions API which might have looser restrictions
    const suggestionsResponse = await fetch(
      `${baseUrl}/v1/domains/suggest?query=${encodeURIComponent(searchDomain)}&tlds=com,net,org,io,co,app`,
      {
        method: 'GET',
        headers: {
          'Authorization': authHeader,
          'Accept': 'application/json',
        },
      }
    );

    // Helper to safely parse JSON responses from GoDaddy
    const parseGoDaddyResponse = async (response: Response) => {
      const raw = await response.text();
      const ct = response.headers.get('content-type') || '';
      const isJson = ct.includes('application/json');
      
      if (!response.ok) {
        if (isJson) {
          try {
            const err = JSON.parse(raw);
            throw new Error(err.message || err.code || `GoDaddy API error ${response.status}`);
          } catch {
            throw new Error(`GoDaddy API error ${response.status}`);
          }
        }
        const snippet = raw.slice(0, 160).replace(/\s+/g, ' ');
        throw new Error(`GoDaddy API error ${response.status}: ${snippet}`);
      }
      
      if (isJson) {
        try {
          return JSON.parse(raw);
        } catch {
          throw new Error('Invalid JSON from GoDaddy API');
        }
      }
      throw new Error('Expected JSON but received non-JSON from GoDaddy API');
    };

    if (suggestionsResponse.ok) {
      try {
        const suggestionsData = await parseGoDaddyResponse(suggestionsResponse);
        console.log(`GoDaddy suggestions returned ${suggestionsData.length} results`);
        
        // Convert suggestions to our format
        for (const suggestion of suggestionsData.slice(0, 6)) {
          results.push({
            domain: suggestion.domain,
            available: true,
            price: suggestion.price || 1200000, // Default $12.00 if no price
            currency: 'USD',
            period: 1,
            definitive: false,
          });
        }
      } catch (parseError) {
        console.error('Failed to parse GoDaddy suggestions response:', parseError);
      }
    } else {
      const errorText = await suggestionsResponse.text();
      console.error('GoDaddy suggestions API error:', suggestionsResponse.status, errorText);
      
      // Fallback to availability check for the exact domain
      const cleanDomain = searchDomain.includes('.') ? searchDomain : `${searchDomain}.com`;
      const availabilityResponse = await fetch(
        `${baseUrl}/v1/domains/available?domain=${encodeURIComponent(cleanDomain)}`,
        {
          method: 'GET',
          headers: {
            'Authorization': authHeader,
            'Accept': 'application/json',
          },
        }
      );

      if (availabilityResponse.ok) {
        try {
          const availabilityData = await parseGoDaddyResponse(availabilityResponse);
          console.log(`GoDaddy availability check result:`, availabilityData);
          
          if (availabilityData.available) {
            results.push({
              domain: cleanDomain,
              available: true,
              price: availabilityData.price || 1200000,
              currency: 'USD',
              period: 1,
              definitive: availabilityData.definitive || false,
            });
          }
        } catch (parseError) {
          console.error('Failed to parse GoDaddy availability response:', parseError);
        }
      } else {
        const errorText = await availabilityResponse.text();
        console.error('GoDaddy availability API error:', availabilityResponse.status, errorText);
      }
    }
  } catch (error) {
    console.error('GoDaddy API request failed:', error);
  }

  // If no results from API, generate some realistic suggestions
  if (results.length === 0) {
    console.log('No API results, generating suggestions...');
    const domainsToSearch = generateDomainVariations(searchDomain);
    
    for (const domain of domainsToSearch.slice(0, 4)) {
      const tld = getTld(domain);
      let price = 1200000; // Default $12.00
      
      switch (tld) {
        case 'com': price = 1200000; break;
        case 'net': price = 1400000; break;
        case 'org': price = 1300000; break;
        case 'io': price = 4900000; break;
        case 'co': price = 2900000; break;
        case 'app': price = 1800000; break;
      }

      results.push({
        domain,
        available: true,
        price,
        currency: 'USD',
        period: 1,
        definitive: false,
      });
    }
  }

  console.log(`Returning ${results.length} domain results`);
  return results;
}

function generateDomainVariations(domain: string): string[] {
  const variations = [];
  
  // Remove any existing TLD to get the base name
  const baseName = domain.split('.')[0];
  
  // Popular TLD variations
  const popularTlds = ['.com', '.net', '.org', '.io', '.co', '.app', '.dev'];
  
  // If the original domain already has a TLD, include it first
  if (domain.includes('.')) {
    variations.push(domain);
  } else {
    // If no TLD provided, default to .com first
    variations.push(`${baseName}.com`);
  }
  
  // Add other popular TLD variations
  for (const tld of popularTlds) {
    const variant = `${baseName}${tld}`;
    if (!variations.includes(variant)) {
      variations.push(variant);
    }
  }
  
  // Limit to first 6 variations to avoid too many API calls
  return variations.slice(0, 6);
}

function getTld(domain: string): string {
  const parts = domain.split('.');
  if (parts.length < 2) {
    return 'com'; // Default TLD
  }
  return parts[parts.length - 1];
}
