import { Storage, File } from "@google-cloud/storage";
import { Response } from "express";
import { randomUUID } from "crypto";

const REPLIT_SIDECAR_ENDPOINT = "http://127.0.0.1:1106";

// The object storage client is used to interact with the object storage service.
export const objectStorageClient = new Storage({
  credentials: {
    audience: "replit",
    subject_token_type: "access_token",
    token_url: `${REPLIT_SIDECAR_ENDPOINT}/token`,
    type: "external_account",
    credential_source: {
      url: `${REPLIT_SIDECAR_ENDPOINT}/credential`,
      format: {
        type: "json",
        subject_token_field_name: "access_token",
      },
    },
    universe_domain: "googleapis.com",
  },
  projectId: "",
});

export class ObjectNotFoundError extends Error {
  constructor() {
    super("Object not found");
    this.name = "ObjectNotFoundError";
    Object.setPrototypeOf(this, ObjectNotFoundError.prototype);
  }
}

// The object storage service is used to interact with the object storage service.
export class ObjectStorageService {
  constructor() {}

  // Gets the public object search paths.
  getPublicObjectSearchPaths(): Array<string> {
    const pathsStr = process.env.PUBLIC_OBJECT_SEARCH_PATHS || "";
    const paths = Array.from(
      new Set(
        pathsStr
          .split(",")
          .map((path) => path.trim())
          .filter((path) => path.length > 0)
      )
    );
    if (paths.length === 0) {
      throw new Error(
        "PUBLIC_OBJECT_SEARCH_PATHS not set. Create a bucket in 'Object Storage' " +
          "tool and set PUBLIC_OBJECT_SEARCH_PATHS env var (comma-separated paths)."
      );
    }
    return paths;
  }

  // Gets the private object directory.
  getPrivateObjectDir(): string {
    const dir = process.env.PRIVATE_OBJECT_DIR || "";
    if (!dir) {
      throw new Error(
        "PRIVATE_OBJECT_DIR not set. Create a bucket in 'Object Storage' " +
          "tool and set PRIVATE_OBJECT_DIR env var."
      );
    }
    return dir;
  }

  // Search for a public object from the search paths.
  async searchPublicObject(filePath: string): Promise<File | null> {
    for (const searchPath of this.getPublicObjectSearchPaths()) {
      const fullPath = `${searchPath}/${filePath}`;

      // Full path format: /<bucket_name>/<object_name>
      const { bucketName, objectName } = parseObjectPath(fullPath);
      const bucket = objectStorageClient.bucket(bucketName);
      const file = bucket.file(objectName);

      // Check if file exists
      const [exists] = await file.exists();
      if (exists) {
        return file;
      }
    }

    return null;
  }

  // Downloads an object to the response.
  async downloadObject(file: File, res: Response, cacheTtlSec: number = 3600) {
    try {
      // Get file metadata
      const [metadata] = await file.getMetadata();
      
      // Set appropriate headers
      res.set({
        "Content-Type": metadata.contentType || "application/octet-stream",
        "Content-Length": metadata.size,
        "Cache-Control": `public, max-age=${cacheTtlSec}`,
      });

      // Stream the file to the response
      const stream = file.createReadStream();

      stream.on("error", (err) => {
        console.error("Stream error:", err);
        if (!res.headersSent) {
          res.status(500).json({ error: "Error streaming file" });
        }
      });

      stream.pipe(res);
    } catch (error) {
      console.error("Error downloading file:", error);
      if (!res.headersSent) {
        res.status(500).json({ error: "Error downloading file" });
      }
    }
  }

  // Gets the upload URL for an object entity.
  async getUploadURL(fileName: string, isPublic: boolean = true): Promise<string> {
    const objectId = randomUUID();
    const fileExtension = fileName.split('.').pop() || '';
    const sanitizedFileName = `${objectId}.${fileExtension}`;
    
    let fullPath: string;
    if (isPublic) {
      const publicPath = this.getPublicObjectSearchPaths()[0];
      fullPath = `${publicPath}/${sanitizedFileName}`;
    } else {
      const privateDir = this.getPrivateObjectDir();
      fullPath = `${privateDir}/${sanitizedFileName}`;
    }

    const { bucketName, objectName } = parseObjectPath(fullPath);

    // Sign URL for PUT method with TTL
    return signObjectURL({
      bucketName,
      objectName,
      method: "PUT",
      ttlSec: 900,
    });
  }

  // Get public URL for an uploaded file
  getPublicURL(objectPath: string): string {
    if (objectPath.startsWith('http')) {
      // Already a full URL
      return objectPath;
    }
    
    // Convert object path to public URL
    const { bucketName, objectName } = parseObjectPath(objectPath);
    return `/public-objects/${objectName}`;
  }

  // List files with a given prefix
  async listPrefix(prefix: string): Promise<{ key: string; size: number; lastModified: Date }[]> {
    try {
      const publicPaths = this.getPublicObjectSearchPaths();
      const allFiles: { key: string; size: number; lastModified: Date }[] = [];

      for (const searchPath of publicPaths) {
        const fullPath = `${searchPath}/${prefix}`;
        const { bucketName, objectName } = parseObjectPath(fullPath);
        const bucket = objectStorageClient.bucket(bucketName);
        
        // Use the full objectName (including directory from searchPath) as the prefix
        const [files] = await bucket.getFiles({ prefix: objectName });
        
        for (const file of files) {
          const [metadata] = await file.getMetadata();
          allFiles.push({
            key: file.name,
            size: parseInt(metadata.size || '0'),
            lastModified: new Date(metadata.timeCreated || Date.now())
          });
        }
      }

      return allFiles;
    } catch (error) {
      console.error("Error listing files with prefix:", error);
      return [];
    }
  }

  // Get a readable stream for a file by key
  async getStream(key: string): Promise<NodeJS.ReadableStream> {
    try {
      const publicPaths = this.getPublicObjectSearchPaths();
      
      for (const searchPath of publicPaths) {
        const { bucketName, objectName } = parseObjectPath(searchPath);
        const bucket = objectStorageClient.bucket(bucketName);
        // Include the directory portion from searchPath when building the file key
        const fullObjectKey = objectName ? `${objectName}/${key}` : key;
        const file = bucket.file(fullObjectKey);
        
        const [exists] = await file.exists();
        if (exists) {
          return file.createReadStream();
        }
      }
      
      throw new ObjectNotFoundError();
    } catch (error) {
      console.error("Error getting file stream:", error);
      throw error;
    }
  }

  // Get public URL from file key
  publicUrlFromKey(key: string): string {
    // For public objects, we can construct the URL directly
    return `/public-objects/${key}`;
  }

  // Upload a file to object storage
  async uploadFile(fullPath: string, buffer: Buffer, contentType?: string): Promise<void> {
    try {
      // Parse the bucket name and object name from the full path
      const { bucketName, objectName } = parseObjectPath(fullPath);
      
      // Get the bucket and file reference
      const bucket = objectStorageClient.bucket(bucketName);
      const file = bucket.file(objectName);
      
      // Upload options
      const options: any = {
        resumable: false,
      };
      
      // Set content type if provided
      if (contentType) {
        options.metadata = {
          contentType,
        };
      }
      
      // Upload the buffer to the file
      await file.save(buffer, options);
    } catch (error) {
      console.error("Error uploading file:", error);
      throw error;
    }
  }

  // Delete files from object storage by key pattern
  async deleteFilesByPattern(keyPattern: string): Promise<boolean> {
    try {
      const publicPaths = this.getPublicObjectSearchPaths();
      let deletedAny = false;

      for (const searchPath of publicPaths) {
        const { bucketName, objectName } = parseObjectPath(searchPath);
        const bucket = objectStorageClient.bucket(bucketName);
        
        // List all files that match the pattern
        const baseKey = objectName ? `${objectName}/${keyPattern}` : keyPattern;
        const [files] = await bucket.getFiles({ prefix: baseKey });
        
        // Delete each matching file
        for (const file of files) {
          try {
            await file.delete();
            console.log(`🗑️ Deleted object storage file: ${file.name}`);
            deletedAny = true;
          } catch (deleteError) {
            console.warn(`Failed to delete file ${file.name}:`, deleteError);
          }
        }
      }

      return deletedAny;
    } catch (error) {
      console.error("Error deleting files by pattern:", error);
      throw error;
    }
  }

  // Delete a single file from object storage
  async deleteFile(key: string): Promise<boolean> {
    try {
      const publicPaths = this.getPublicObjectSearchPaths();
      
      for (const searchPath of publicPaths) {
        const { bucketName, objectName } = parseObjectPath(searchPath);
        const bucket = objectStorageClient.bucket(bucketName);
        const fullObjectKey = objectName ? `${objectName}/${key}` : key;
        const file = bucket.file(fullObjectKey);
        
        const [exists] = await file.exists();
        if (exists) {
          await file.delete();
          console.log(`🗑️ Deleted object storage file: ${fullObjectKey}`);
          return true;
        }
      }
      
      return false;
    } catch (error) {
      console.error("Error deleting single file:", error);
      throw error;
    }
  }
}

function parseObjectPath(path: string): {
  bucketName: string;
  objectName: string;
} {
  if (!path.startsWith("/")) {
    path = `/${path}`;
  }
  const pathParts = path.split("/");
  if (pathParts.length < 3) {
    throw new Error("Invalid path: must contain at least a bucket name");
  }

  const bucketName = pathParts[1];
  const objectName = pathParts.slice(2).join("/");

  return {
    bucketName,
    objectName,
  };
}

async function signObjectURL({
  bucketName,
  objectName,
  method,
  ttlSec,
}: {
  bucketName: string;
  objectName: string;
  method: "GET" | "PUT" | "DELETE" | "HEAD";
  ttlSec: number;
}): Promise<string> {
  const request = {
    bucket_name: bucketName,
    object_name: objectName,
    method,
    expires_at: new Date(Date.now() + ttlSec * 1000).toISOString(),
  };
  const response = await fetch(
    `${REPLIT_SIDECAR_ENDPOINT}/object-storage/signed-object-url`,
    {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(request),
    }
  );
  if (!response.ok) {
    throw new Error(
      `Failed to sign object URL, errorcode: ${response.status}, ` +
        `make sure you're running on Replit`
    );
  }

  const { signed_url: signedURL } = await response.json();
  return signedURL;
}