import type { Request, Response, NextFunction } from "express";
import { storage } from "../storage";
import type { InsertVisitorSession, InsertDailyVisitorStats } from "../../shared/schema";
import crypto from "crypto";

// IP Geolocation using ipapi.co (free tier: 1000 requests/day)
interface IpApiResponse {
  ip: string;
  city?: string;
  region?: string;
  country?: string;
  country_name?: string;
  country_code?: string;
  latitude?: number;
  longitude?: number;
  error?: boolean;
  reason?: string;
}

// Cache for IP geolocation to avoid repeated API calls with expiry
interface CacheEntry {
  data: IpApiResponse;
  timestamp: number;
  requestCount: number;
}

const locationCache = new Map<string, CacheEntry>();
const cacheExpiry = 24 * 60 * 60 * 1000; // 24 hours
const maxRequestsPerMinute = 10; // Rate limiting
const requestCounts = new Map<string, { count: number; resetTime: number }>();

// Server secret for IP hashing (should be in env vars in production)
const getServerSecret = (): string => {
  return process.env.VISITOR_HASH_SECRET || 'default-dev-secret-change-in-production';
};

/**
 * Get geolocation data for an IP address using ipapi.co
 * Uses caching with expiry and rate limiting for privacy and performance
 */
async function getIpLocation(ip: string): Promise<IpApiResponse | null> {
  // Check cache first with expiry
  const cached = locationCache.get(ip);
  if (cached && (Date.now() - cached.timestamp) < cacheExpiry) {
    return cached.data;
  }

  // Remove expired cache entry
  if (cached && (Date.now() - cached.timestamp) >= cacheExpiry) {
    locationCache.delete(ip);
  }

  // Don't track local/private IPs
  if (isPrivateIP(ip)) {
    return null;
  }

  // Rate limiting check
  if (!checkRateLimit(ip)) {
    console.warn(`Rate limit exceeded for IP geolocation: ${ip}`);
    return null;
  }

  try {
    // Use AbortController for timeout
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 5000);

    const response = await fetch(`https://ipapi.co/${ip}/json/`, {
      signal: controller.signal,
      headers: {
        'User-Agent': 'IBrandBiz-Analytics/1.0'
      }
    });
    
    clearTimeout(timeoutId);

    if (!response.ok) {
      console.warn(`IP geolocation API error for ${ip}:`, response.status);
      return null;
    }

    const data: IpApiResponse = await response.json();
    
    if (data.error) {
      console.warn(`IP geolocation error for ${ip}:`, data.reason);
      return null;
    }

    // Cache successful results with timestamp
    locationCache.set(ip, {
      data,
      timestamp: Date.now(),
      requestCount: 1
    });
    
    // Clean up cache periodically with better strategy
    cleanupLocationCache();

    return data;
  } catch (error) {
    if (error.name === 'AbortError') {
      console.warn(`Geolocation API timeout for IP ${ip}`);
    } else {
      console.warn(`Failed to get geolocation for IP ${ip}:`, error);
    }
    return null;
  }
}

/**
 * Check rate limiting for geolocation API calls
 */
function checkRateLimit(ip: string): boolean {
  const now = Date.now();
  const minute = Math.floor(now / 60000);
  
  const current = requestCounts.get(ip);
  if (!current || current.resetTime !== minute) {
    requestCounts.set(ip, { count: 1, resetTime: minute });
    return true;
  }
  
  if (current.count >= maxRequestsPerMinute) {
    return false;
  }
  
  current.count++;
  return true;
}

/**
 * Clean up expired cache entries and rate limit records
 */
function cleanupLocationCache(): void {
  const now = Date.now();
  const maxSize = 1000;
  
  // Clean expired entries
  for (const [key, entry] of locationCache) {
    if (now - entry.timestamp >= cacheExpiry) {
      locationCache.delete(key);
    }
  }
  
  // If still too large, remove oldest entries
  if (locationCache.size > maxSize) {
    const entries = Array.from(locationCache.entries()).sort((a, b) => a[1].timestamp - b[1].timestamp);
    const toRemove = entries.slice(0, entries.length - maxSize);
    toRemove.forEach(([key]) => locationCache.delete(key));
  }
  
  // Clean up old rate limit records
  const currentMinute = Math.floor(now / 60000);
  for (const [ip, record] of requestCounts) {
    if (record.resetTime < currentMinute - 5) { // Keep last 5 minutes
      requestCounts.delete(ip);
    }
  }
}

/**
 * Check if an IP address is private/local (don't track these)
 */
function isPrivateIP(ip: string): boolean {
  // Common private IP ranges
  const privateRanges = [
    /^127\./, // 127.0.0.1 (localhost)
    /^192\.168\./, // 192.168.x.x (private)
    /^10\./, // 10.x.x.x (private)
    /^172\.1[6-9]\./, // 172.16-19.x.x (private)
    /^172\.2[0-9]\./, // 172.20-29.x.x (private) 
    /^172\.3[0-1]\./, // 172.30-31.x.x (private)
    /^::1$/, // IPv6 localhost
    /^fc00:/, // IPv6 private
    /^fe80:/, // IPv6 link-local
  ];

  return privateRanges.some(range => range.test(ip));
}

/**
 * Generate a salted hash of IP address for privacy compliance
 */
function hashIP(ip: string): string {
  const secret = getServerSecret();
  return crypto.createHash('sha256').update(`${ip}${secret}`).digest('hex');
}

/**
 * Generate a unique session identifier from IP + user agent + date
 * This creates a privacy-friendly identifier for daily unique tracking
 */
function generateSessionId(ip: string, date: string, userAgent?: string): string {
  const data = `${ip}-${date}-${userAgent || 'unknown'}`;
  return crypto.createHash('sha256').update(data).digest('hex').substring(0, 32);
}

/**
 * Extract real IP address from request (handles proxies)
 */
function getClientIP(req: Request): string {
  const xForwardedFor = req.headers['x-forwarded-for'] as string;
  const xRealIP = req.headers['x-real-ip'] as string;
  const connectionRemoteAddress = req.connection?.remoteAddress;
  const socketRemoteAddress = req.socket?.remoteAddress;
  
  // Check various headers in order of preference
  if (xForwardedFor) {
    // x-forwarded-for may contain multiple IPs, take the first one
    return xForwardedFor.split(',')[0].trim();
  }
  
  if (xRealIP) {
    return xRealIP;
  }
  
  return connectionRemoteAddress || socketRemoteAddress || 'unknown';
}

/**
 * Enhanced bot detection patterns
 */
function isBot(userAgent: string): boolean {
  const botPatterns = [
    // Search engine crawlers
    /googlebot/i, /bingbot/i, /slurp/i, /duckduckbot/i, /baiduspider/i,
    // Social media crawlers
    /facebookexternalhit/i, /twitterbot/i, /linkedinbot/i, /whatsapp/i,
    // Monitoring and security
    /pingdom/i, /uptimerobot/i, /statuscake/i, /site24x7/i,
    // Generic patterns
    /bot\b/i, /crawl/i, /spider/i, /scraper/i, /archiver/i,
    /curl/i, /wget/i, /python-requests/i, /node-fetch/i,
    // Headless browsers often used by bots
    /headlesschrome/i, /phantomjs/i, /puppeteer/i
  ];
  
  return botPatterns.some(pattern => pattern.test(userAgent));
}

/**
 * Check if the request is to a public page that should be tracked
 * Enhanced to handle query strings, trailing slashes, and better URL matching
 */
function isPublicPage(path: string): boolean {
  // Normalize path: remove query string, trailing slashes, fragments
  const normalizedPath = path.split('?')[0].split('#')[0].replace(/\/+$/, '') || '/';
  
  // Public pages to track
  const publicRoutes = [
    '/',
    '/about',
    '/pricing', 
    '/services',
    '/contact',
    '/privacy',
    '/terms',
    '/features',
    '/help',
    '/faq',
    '/blog',
    '/resources'
  ];

  // Paths to exclude (admin, API, assets, etc.)
  const excludePatterns = [
    /^\/api\//,
    /^\/admin/,
    /^\/auth/,
    /^\/assets/,
    /^\/static/,
    /^\/uploads/,
    /^\/_/,
    /^\/favicon/,
    /^\/robots/,
    /^\/sitemap/,
    /^\/\.well-known/,
    /\.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot|pdf|zip|txt|xml|json)$/i
  ];

  // First check if it's explicitly excluded
  if (excludePatterns.some(pattern => pattern.test(normalizedPath))) {
    return false;
  }

  // Then check if it's a public route (exact match or starts with public route)
  return publicRoutes.some(route => 
    normalizedPath === route || 
    (route !== '/' && normalizedPath.startsWith(route + '/'))
  );
}

/**
 * Check if user is authenticated admin (exclude from tracking)
 */
function isAuthenticatedAdmin(req: Request): boolean {
  // Check if user has admin session or bearer token
  const authHeader = req.headers.authorization;
  const sessionCookie = req.headers.cookie;
  
  // Basic check for admin authentication (could be enhanced based on your auth system)
  return !!(authHeader && authHeader.includes('admin')) || 
         !!(sessionCookie && sessionCookie.includes('admin'));
}

/**
 * Update or create daily visitor statistics with proper unique visitor counting
 */
async function updateDailyStats(date: string, isNewUniqueVisitor: boolean, countryCode?: string, pageUrl?: string): Promise<void> {
  try {
    // Get existing stats for the date
    let stats = await storage.getDailyVisitorStatsByDate(date);
    
    if (!stats) {
      // Create new stats entry
      const newStats: InsertDailyVisitorStats = {
        date,
        uniqueVisitors: isNewUniqueVisitor ? 1 : 0,
        totalPageViews: 1,
        countries: countryCode ? { [countryCode]: 1 } : {},
        topPages: pageUrl ? { [pageUrl]: 1 } : {},
      };
      await storage.createDailyVisitorStats(newStats);
    } else {
      // Update existing stats
      const countries = { ...(stats.countries as Record<string, number> || {}) };
      const topPages = { ...(stats.topPages as Record<string, number> || {}) };
      
      if (countryCode) {
        countries[countryCode] = (countries[countryCode] || 0) + 1;
      }
      
      if (pageUrl) {
        topPages[pageUrl] = (topPages[pageUrl] || 0) + 1;
      }

      await storage.updateDailyVisitorStats(stats.id, {
        uniqueVisitors: stats.uniqueVisitors + (isNewUniqueVisitor ? 1 : 0),
        totalPageViews: stats.totalPageViews + 1,
        countries,
        topPages,
      });
    }
  } catch (error) {
    console.error('Failed to update daily visitor stats:', error);
  }
}

/**
 * Visitor tracking middleware
 * Captures visitor sessions and location data for public pages only
 * Enhanced with privacy compliance, bot filtering, and proper unique visitor counting
 */
export async function trackVisitor(req: Request, res: Response, next: NextFunction) {
  // Skip tracking if not a public page
  if (!isPublicPage(req.path)) {
    return next();
  }

  const userAgent = req.headers['user-agent'] || '';
  
  // Skip tracking for bots/crawlers
  if (isBot(userAgent)) {
    return next();
  }
  
  // Skip tracking for authenticated admin users
  if (isAuthenticatedAdmin(req)) {
    return next();
  }

  try {
    const ip = getClientIP(req);
    const today = new Date().toISOString().split('T')[0]; // YYYY-MM-DD
    const sessionId = generateSessionId(ip, today, userAgent);
    const ipHash = hashIP(ip);
    const pageUrl = req.path;
    const referrer = req.headers.referer || req.headers.referrer as string;

    // Try to get existing session for today
    let session = await storage.getVisitorSessionBySessionId(sessionId);
    let isNewUniqueVisitor = false;

    if (session) {
      // Update existing session with new page view
      await storage.updateVisitorSession(session.id, {
        pageViews: session.pageViews + 1,
        lastPageUrl: pageUrl,
        updatedAt: new Date(),
      });
    } else {
      // Create new session - get geolocation data
      isNewUniqueVisitor = true;
      const locationData = await getIpLocation(ip);
      
      const newSession: InsertVisitorSession = {
        sessionId,
        ipHash, // Store hashed IP instead of raw IP
        country: locationData?.country_name,
        countryCode: locationData?.country_code,
        city: locationData?.city,
        latitude: locationData?.latitude, // Now numeric as per schema
        longitude: locationData?.longitude,
        userAgent,
        pageViews: 1,
        firstPageUrl: pageUrl,
        lastPageUrl: pageUrl,
        referrer,
        isPublicPage: true,
        date: today, // Track date for daily unique counting
      };

      await storage.createVisitorSession(newSession);
    }

    // Update daily stats (only increment unique visitors for new sessions)
    await updateDailyStats(
      today,
      isNewUniqueVisitor,
      session?.countryCode || (await getIpLocation(ip))?.country_code,
      pageUrl
    );

  } catch (error) {
    // Don't let tracking errors break the request
    console.error('Visitor tracking error:', error);
  }

  next();
}

/**
 * Background job to clean up old visitor sessions (optional)
 * Can be called periodically to remove sessions older than X days
 */
export async function cleanupOldSessions(daysOld: number = 90): Promise<void> {
  try {
    const cutoffDate = new Date();
    cutoffDate.setDate(cutoffDate.getDate() - daysOld);
    
    await storage.deleteVisitorSessionsOlderThan(cutoffDate);
    console.log(`Cleaned up visitor sessions older than ${daysOld} days`);
  } catch (error) {
    console.error('Failed to cleanup old visitor sessions:', error);
  }
}