/**
 * Role-based Authorization Middleware
 * 
 * Provides hierarchy-based role guards for backend endpoints
 * Role hierarchy: owner (4) > management (3) > staff (2) > analyst (1) > pro/user (0)
 */

import { Request, Response, NextFunction } from 'express';
import { verifyAdminBearer, hasRolePermission } from '../admin/firebaseAdmin';
import { logAuditEvent } from '../admin/auditLogger';
import type { Role } from '../../shared/schema';

interface AuthenticatedRequest extends Request {
  user?: {
    uid: string;
    role: Role;
    email?: string;
    [key: string]: any;
  };
}

/**
 * Generic role guard middleware factory
 */
function requireRole(minimumRole: Role) {
  return async (req: AuthenticatedRequest, res: Response, next: NextFunction) => {
    try {
      // If user is already set by devBypass, just check role
      if (req.user) {
        if (!hasRolePermission(req.user.role, minimumRole)) {
          return res.status(403).json({ 
            error: 'Insufficient permissions',
            userRole: req.user.role,
            requiredRole: minimumRole
          });
        }
        
        // Log access attempt
        logAuditEvent({
          adminUserId: req.user.uid,
          adminRole: req.user.role,
          action: 'API_ACCESS',
          details: `Accessed ${req.method} ${req.path} with role ${req.user.role}`
        });
        
        return next();
      }
      
      // Otherwise, verify with Firebase
      const authHeader = req.headers.authorization;
      const adminUser = await verifyAdminBearer(authHeader, minimumRole);
      
      // Attach user info to request for downstream use
      req.user = {
        uid: adminUser.uid,
        role: adminUser.role,
        email: adminUser.email,
        ...adminUser
      };
      
      // Log access attempt
      logAuditEvent({
        adminUserId: adminUser.uid,
        adminRole: adminUser.role,
        action: 'API_ACCESS',
        details: `Accessed ${req.method} ${req.path} with role ${adminUser.role}`
      });
      
      next();
    } catch (error) {
      console.error(`Role guard failed for ${minimumRole}:`, error);
      
      const statusCode = error.message === 'missing' ? 401 : 
                        error.message === 'not_admin' || error.message === 'insufficient_role' ? 403 : 500;
      
      res.status(statusCode).json({ 
        error: statusCode === 401 ? 'Authentication required' : 
               statusCode === 403 ? 'Insufficient permissions' : 
               'Internal server error',
        requiredRole: minimumRole
      });
    }
  };
}

/**
 * Owner-only access (level 4)
 * Full system control, user role management, critical operations
 */
export const requireOwner = requireRole('owner');

/**
 * Management or above access (level 3+)
 * Business management, user data access, financial data
 */
export const requireManagementOrAbove = requireRole('management');

/**
 * Staff or above access (level 2+)
 * Operations access, content management, user support
 */
export const requireStaffOrAbove = requireRole('staff');

/**
 * Analyst or above access (level 1+)
 * Read-only analytics, reports, dashboard data
 */
export const requireAnalystOrAbove = requireRole('analyst');

/**
 * Any admin role access (level 1+)
 * General admin endpoints that any admin can access
 */
export const requireAnyAdmin = requireAnalystOrAbove;

/**
 * Conditional role check middleware
 * Checks if user has at least the specified role level
 */
export function checkRole(minimumRole: Role) {
  return (req: AuthenticatedRequest, res: Response, next: NextFunction) => {
    if (!req.user) {
      return res.status(401).json({ error: 'Authentication required' });
    }
    
    if (!hasRolePermission(req.user.role, minimumRole)) {
      return res.status(403).json({ 
        error: 'Insufficient permissions',
        userRole: req.user.role,
        requiredRole: minimumRole
      });
    }
    
    next();
  };
}

/**
 * Role-specific conditional middleware
 * Only executes middleware if user has sufficient role
 */
export function ifRole(minimumRole: Role, middleware: any) {
  return (req: AuthenticatedRequest, res: Response, next: NextFunction) => {
    if (req.user && hasRolePermission(req.user.role, minimumRole)) {
      return middleware(req, res, next);
    }
    next();
  };
}

/**
 * Development bypass middleware
 * Bypasses auth in development mode for testing
 */
export function devBypass(fallbackRole: Role = 'owner') {
  return (req: AuthenticatedRequest, res: Response, next: NextFunction) => {
    if (process.env.NODE_ENV === 'development' && !process.env.FIREBASE_PROJECT_ID) {
      console.log(`🔓 Development mode: bypassing auth with role '${fallbackRole}'`);
      req.user = {
        uid: 'dev-user',
        role: fallbackRole,
        email: 'dev@localhost'
      };
    }
    next();
  };
}

// Export the request interface for use in route files
export type { AuthenticatedRequest };