import { Request, Response, NextFunction } from "express";
import { getFirebaseAdmin } from "../admin/firebaseAdmin";

// Owner email configuration
const OWNER_EMAILS = (process.env.IBRANDBIZ_OWNER_EMAILS || 'jrichards@ibrandbiz.com')
  .split(',')
  .map(email => email.trim().toLowerCase())
  .filter(email => email.length > 0);

/**
 * Server-side owner email verification for admin middleware
 */
function isOwnerEmailServerSide(email?: string): boolean {
  if (!email) return false;
  
  const normalizedEmail = email.trim().toLowerCase();
  const isOwner = OWNER_EMAILS.includes(normalizedEmail);
  
  console.log(`🔐 Admin Guard - Owner check for ${email}: ${isOwner}`);
  
  return isOwner;
}

/**
 * Dual authentication admin middleware
 * Allows EITHER:
 * 1. MANAGER_KEY for regular admins, OR
 * 2. Authenticated Firebase user with owner email (no key required)
 */
export async function requireAdmin(req: Request, res: Response, next: NextFunction) {
  // First, try MANAGER_KEY authentication (for regular admins)
  const adminKey = req.header("x-admin-key");
  const expectedKey = process.env.MANAGER_KEY;
  
  if (expectedKey && adminKey === expectedKey) {
    console.log('✅ Admin access granted via MANAGER_KEY');
    return next();
  }
  
  // Second, try Firebase owner authentication (for owner users)
  try {
    const authHeader = req.headers.authorization;
    const token = authHeader && authHeader.split(' ')[1]; // Bearer TOKEN
    
    if (token) {
      const admin = await getFirebaseAdmin();
      
      if (admin) {
        const decodedToken = await admin.auth().verifyIdToken(token);
        const userEmail = decodedToken.email;
        
        if (isOwnerEmailServerSide(userEmail)) {
          console.log(`✅ Admin access granted to owner: ${userEmail}`);
          // Populate req.user with decoded token for route handlers to use
          (req as any).user = decodedToken;
          return next();
        }
      }
    }
  } catch (error) {
    console.error('Firebase auth verification failed:', error);
  }
  
  // If neither authentication method worked, deny access
  if (!expectedKey) {
    console.error("MANAGER_KEY not configured in environment variables");
    return res.status(500).json({ error: "Admin authentication not configured" });
  }
  
  return res.status(403).json({ error: "Admin access denied. Provide valid admin key or authenticate as owner." });
}