/**
 * Pricing/markup utilities for domain sales.
 * - Supports flat + percent markup
 * - Per-TLD overrides (e.g., .ai, .com, etc.)
 * - Rounds to 2 decimals
 */

interface MarkupConfig {
  flat?: number;
  percent?: number;
  perTld?: Record<string, { flat?: number; percent?: number }>;
}

interface ApplyMarkupInput {
  wholesale: number;
  tld: string;
  config?: MarkupConfig;
}

// Auto-generated per-TLD markup config (percent then flat), 2025-09-16
// Aim: ~10% under GoDaddy where known; otherwise healthy margin over wholesale.
const DEFAULTS: Required<MarkupConfig> = {
  flat: 3.0,   // default fallback
  percent: 10,
  perTld: {
    ".com": { flat: 2.5, percent: 10 },
    ".net": { flat: 2.34, percent: 10 },
    ".org": { flat: 11.5, percent: 10 },
    ".biz": { flat: 0, percent: 10 },
    ".info": { flat: 0, percent: 10 },
    ".co": { flat: 12.49, percent: 10 },
    ".ai": { flat: 0, percent: 10 }
  },
};

function round2(n: number): number {
  return Math.round((Number(n) + Number.EPSILON) * 100) / 100;
}

export function applyMarkup({ wholesale, tld, config = {} }: ApplyMarkupInput): number {
  const cfg: Required<MarkupConfig> = {
    flat: DEFAULTS.flat,
    percent: DEFAULTS.percent,
    perTld: { ...DEFAULTS.perTld, ...(config.perTld || {}) },
  };

  const key = (tld || "").toLowerCase();
  const ovr = cfg.perTld[key] || {};
  const flat = (ovr.flat ?? cfg.flat);
  const percent = (ovr.percent ?? cfg.percent);

  let retail = Number(wholesale);
  if (!Number.isFinite(retail)) throw new Error("Invalid wholesale price");

  // percent first, then flat (order doesn't matter much, but keeps it consistent)
  retail = retail * (1 + (Number(percent) / 100));
  retail = retail + Number(flat);

  return round2(retail);
}

// Helper to extract TLD from domain
export function extractTld(domain: string): string {
  const m = String(domain).toLowerCase().match(/(\.[a-z0-9-]+)$/i);
  return m ? m[1] : "";
}