import { Job } from './jobQueue';
import { storage } from '../storage';
import { opensrs } from '../domains/providers/opensrs';
import { type DomainRegistrationResult } from '@shared/schema';

export interface DomainRegistrationJobData {
  domainOrderId: string;
  domain: string;
  years: number;
  contact: {
    firstName: string;
    lastName: string;
    email: string;
    phone: string;
    address: string;
    city: string;
    state: string;
    zip: string;
    country: string;
    organization?: string;
  };
  privacy?: boolean;
  nameservers?: string[];
  userId: string;
  stripeSessionId: string;
}

export async function registerDomainWorker(job: Job): Promise<void> {
  const data: DomainRegistrationJobData = job.data;
  
  console.log(`🌐 Starting domain registration for: ${data.domain} (attempt ${job.retries + 1}/${job.maxRetries + 1})`);
  
  try {
    // Get the domain order from database
    const domainOrder = await storage.getDomainOrder(data.domainOrderId);
    if (!domainOrder) {
      throw new Error(`Domain order not found: ${data.domainOrderId}`);
    }

    // Check if already registering or active
    if (domainOrder.status === 'active') {
      console.log(`Domain order already active: ${data.domainOrderId}`);
      return;
    }

    // Update status to registering
    await storage.updateDomainOrder(data.domainOrderId, {
      status: 'registering'
    });

    // Call OpenSRS to register the domain
    console.log(`🔗 Calling OpenSRS to register domain: ${data.domain}`);
    
    const registrationResult: DomainRegistrationResult = await opensrs.register({
      domain: data.domain,
      years: data.years,
      contact: data.contact,
      privacy: data.privacy,
      nameservers: data.nameservers
    });

    if (registrationResult.success) {
      // Registration successful
      console.log(`✅ Domain registration successful: ${data.domain}`);
      
      await storage.updateDomainOrder(data.domainOrderId, {
        status: 'active',
        providerRegId: registrationResult.providerRegId
      });

      // Create success notification for user
      await storage.createNotification(
        data.userId,
        `Domain ${data.domain} has been successfully registered!`,
        'success'
      );

      console.log(`🎉 Domain registration completed successfully: ${data.domain}`);
      
    } else {
      // Registration failed - determine if error is retryable
      const isRetryable = isRetryableRegistrationError(registrationResult.message);
      console.error(`❌ Domain registration failed: ${data.domain} - ${registrationResult.message} (retryable: ${isRetryable})`);
      
      // For retryable errors, let job queue handle retry
      if (isRetryable && job.retries < job.maxRetries) {
        throw new Error(`Domain registration failed (retryable): ${registrationResult.message}`);
      }
      
      // For permanent failures or max retries reached, mark as failed
      await storage.updateDomainOrder(data.domainOrderId, {
        status: 'failed',
        errorMessage: getUserFriendlyErrorMessage(registrationResult.message)
      });

      // Create error notification for user
      await storage.createNotification(
        data.userId,
        `Domain registration failed for ${data.domain}. ${getUserFriendlyErrorMessage(registrationResult.message)}`,
        'error'
      );

      throw new Error(`Domain registration failed: ${registrationResult.message}`);
    }
    
  } catch (error) {
    const errorMessage = error instanceof Error ? error.message : String(error);
    const isRetryable = isRetryableError(errorMessage);
    
    console.error(`💥 Domain registration job failed for ${data.domain} (attempt ${job.retries + 1}/${job.maxRetries + 1}):`, error, `(retryable: ${isRetryable})`);
    
    // For retryable errors, let job queue handle retry
    if (isRetryable && job.retries < job.maxRetries) {
      throw error; // Re-throw to trigger job retry
    }
    
    // For permanent failures or max retries reached, mark as failed
    await storage.updateDomainOrder(data.domainOrderId, {
      status: 'failed',
      errorMessage: getUserFriendlyErrorMessage(errorMessage)
    });

    // Create error notification for user
    await storage.createNotification(
      data.userId,
      `Domain registration failed for ${data.domain}. ${getUserFriendlyErrorMessage(errorMessage)} Please contact support if the issue persists.`,
      'error'
    );

    throw error; // Re-throw to mark job as permanently failed
  }
}

// Helper function to determine if an error is retryable
function isRetryableError(errorMessage: string): boolean {
  const retryablePatterns = [
    'Authentication Failed',
    'Connection timeout',
    'Network error',
    'Service temporarily unavailable',
    'Internal server error',
    'Gateway timeout',
    'Service unavailable',
    'Temporary failure',
    'Rate limit exceeded'
  ];
  
  const lowerError = errorMessage.toLowerCase();
  return retryablePatterns.some(pattern => 
    lowerError.includes(pattern.toLowerCase())
  );
}

// Helper function to determine if a registration result error is retryable
function isRetryableRegistrationError(errorMessage: string): boolean {
  const retryablePatterns = [
    'Authentication Failed',
    'Temporary failure',
    'Service temporarily unavailable',
    'Rate limit exceeded',
    'Connection timeout'
  ];
  
  const lowerError = errorMessage.toLowerCase();
  return retryablePatterns.some(pattern => 
    lowerError.includes(pattern.toLowerCase())
  );
}

// Helper function to convert technical errors to user-friendly messages
function getUserFriendlyErrorMessage(errorMessage: string): string {
  const lowerError = errorMessage.toLowerCase();
  
  if (lowerError.includes('authentication failed')) {
    return 'Registrar authentication is not configured. Please contact support.';
  }
  
  if (lowerError.includes('domain already exists') || lowerError.includes('domain not available')) {
    return 'This domain is no longer available for registration.';
  }
  
  if (lowerError.includes('invalid contact') || lowerError.includes('contact information')) {
    return 'There was an issue with the contact information provided.';
  }
  
  if (lowerError.includes('payment') || lowerError.includes('billing')) {
    return 'There was a payment processing issue.';
  }
  
  if (lowerError.includes('rate limit')) {
    return 'Too many requests. The registration will be retried automatically.';
  }
  
  // For unknown errors, return a generic message
  return 'An unexpected error occurred during registration.';
}