import { EventEmitter } from 'events';

export interface Job {
  id: string;
  type: string;
  data: any;
  retries: number;
  maxRetries: number;
  createdAt: Date;
  updatedAt: Date;
  status: 'pending' | 'processing' | 'completed' | 'failed';
  nextRunAt?: Date;
  error?: string;
}

export class JobQueue extends EventEmitter {
  private jobs: Map<string, Job> = new Map();
  private processingJobs: Set<string> = new Set();
  private workers: Map<string, (job: Job) => Promise<void>> = new Map();
  private isProcessing = false;
  private processingTimeouts: Map<string, NodeJS.Timeout> = new Map();

  constructor() {
    super();
    this.startProcessing();
  }

  // Register a job worker for a specific job type
  registerWorker(jobType: string, worker: (job: Job) => Promise<void>) {
    this.workers.set(jobType, worker);
    console.log(`📋 Job worker registered for type: ${jobType}`);
  }

  // Add a new job to the queue
  enqueue(type: string, data: any, options: { maxRetries?: number } = {}): string {
    const jobId = `${type}_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    
    const job: Job = {
      id: jobId,
      type,
      data,
      retries: 0,
      maxRetries: options.maxRetries || 3,
      createdAt: new Date(),
      updatedAt: new Date(),
      status: 'pending'
    };

    this.jobs.set(jobId, job);
    console.log(`📋 Job enqueued: ${jobId} (type: ${type})`);
    
    // Trigger processing
    this.processJobs();
    
    return jobId;
  }

  // Get job status
  getJob(jobId: string): Job | undefined {
    return this.jobs.get(jobId);
  }

  // Get all jobs (for debugging/monitoring)
  getAllJobs(): Job[] {
    return Array.from(this.jobs.values());
  }

  // Get jobs by status
  getJobsByStatus(status: Job['status']): Job[] {
    return Array.from(this.jobs.values()).filter(job => job.status === status);
  }

  // Start processing jobs continuously with exponential backoff
  private startProcessing() {
    // Initial check for jobs immediately
    this.processJobs();
    
    // Then check for jobs every 5 seconds (reduced frequency since we use exponential backoff for retries)
    setInterval(() => {
      this.processJobs();
    }, 5000);
  }

  // Process pending jobs
  private async processJobs() {
    if (this.isProcessing) return;
    
    this.isProcessing = true;

    try {
      const pendingJobs = this.getJobsByStatus('pending');
      const now = new Date();
      
      // Filter jobs that are ready to run (respect exponential backoff)
      const readyJobs = pendingJobs.filter(job => {
        // If no nextRunAt is set, job is ready immediately
        if (!job.nextRunAt) return true;
        // Otherwise, only run if nextRunAt has passed
        return job.nextRunAt <= now;
      });
      
      for (const job of readyJobs) {
        if (this.processingJobs.has(job.id)) continue;
        
        const worker = this.workers.get(job.type);
        if (!worker) {
          console.warn(`⚠️ No worker registered for job type: ${job.type}`);
          continue;
        }

        // Mark job as processing
        this.processingJobs.add(job.id);
        job.status = 'processing';
        job.updatedAt = new Date();
        // Clear nextRunAt since job is now processing
        job.nextRunAt = undefined;

        // Process job in background
        this.processJob(job, worker);
      }
    } finally {
      this.isProcessing = false;
    }
  }

  // Process individual job
  private async processJob(job: Job, worker: (job: Job) => Promise<void>) {
    try {
      console.log(`🔄 Processing job: ${job.id} (type: ${job.type})`);
      
      await worker(job);
      
      // Job completed successfully
      job.status = 'completed';
      job.updatedAt = new Date();
      this.processingJobs.delete(job.id);
      
      console.log(`✅ Job completed: ${job.id}`);
      this.emit('jobCompleted', job);
      
    } catch (error) {
      console.error(`❌ Job failed: ${job.id}`, error);
      
      job.retries++;
      job.error = error instanceof Error ? error.message : String(error);
      job.updatedAt = new Date();
      
      if (job.retries >= job.maxRetries) {
        job.status = 'failed';
        console.error(`💥 Job failed permanently after ${job.retries} retries: ${job.id}`);
        this.emit('jobFailed', job);
        this.processingJobs.delete(job.id);
      } else {
        job.status = 'pending';
        const delay = this.calculateRetryDelay(job.retries);
        const nextRunAt = new Date(Date.now() + delay);
        job.nextRunAt = nextRunAt;
        
        console.log(`🔄 Job will retry (${job.retries}/${job.maxRetries}) at ${nextRunAt.toISOString()}: ${job.id}`);
        
        this.processingJobs.delete(job.id);
      }
    }
  }

  // Clean up old completed/failed jobs (call periodically)
  cleanup(olderThanHours: number = 24) {
    const cutoffTime = new Date(Date.now() - olderThanHours * 60 * 60 * 1000);
    let cleaned = 0;

    for (const [jobId, job] of this.jobs.entries()) {
      if ((job.status === 'completed' || job.status === 'failed') && job.updatedAt < cutoffTime) {
        this.jobs.delete(jobId);
        cleaned++;
      }
    }

    if (cleaned > 0) {
      console.log(`🧹 Cleaned up ${cleaned} old jobs`);
    }
  }

  // Calculate exponential backoff delay with jitter
  private calculateRetryDelay(retryAttempt: number): number {
    // Base delay: 2^attempt seconds, capped at 5 minutes
    const baseDelay = Math.min(Math.pow(2, retryAttempt) * 1000, 5 * 60 * 1000);
    
    // Add jitter: ±25% of base delay to prevent thundering herd
    const jitterFactor = 0.5 + (Math.random() * 0.5); // Range: 0.5 to 1.0 (50% to 100% of base)
    const finalDelay = Math.floor(baseDelay * jitterFactor);
    
    return Math.max(finalDelay, 1000); // Minimum 1 second delay
  }
}

// Global job queue instance
export const jobQueue = new JobQueue();

// Clean up old jobs every hour
setInterval(() => {
  jobQueue.cleanup(24); // Keep jobs for 24 hours
}, 60 * 60 * 1000);