import express, { type Request, Response, NextFunction } from "express";
import cors from "cors";
import { registerRoutes } from "./routes";
import coverTemplatesRouter from "./routes/coverTemplates";
import infographicTemplatesRouter from "./routes/infographicTemplates";
import stripeWebhookRouter from "./routes/stripeWebhook";
import { BUILD } from "./version";

const app = express();

// Trust proxy for deployments behind reverse proxies (like Replit)
app.set("trust proxy", 1);

// CORS configuration for live deployment
const corsOrigins = (process.env.CORS_ORIGINS || "").split(",").filter(Boolean);
app.use(cors({
  origin: (origin, callback) => {
    // Allow requests with no origin (like mobile apps, curl, Postman)
    if (!origin) return callback(null, true);
    
    // In development, allow all origins
    if (app.get("env") === "development") {
      return callback(null, true);
    }
    
    // In production, allow configured origins or allow all if none configured
    if (corsOrigins.length === 0) {
      return callback(null, true);
    }
    
    callback(null, corsOrigins.includes(origin));
  },
  credentials: true,
}));

// 1) Stripe webhook BEFORE JSON parser (for signature verification)
app.use("/api/stripe/webhook", stripeWebhookRouter); // raw body route

// 2) Now enable JSON parsing for other routes
app.use(express.json());
app.use(express.urlencoded({ extended: false }));

// 3) Cover templates routes after JSON parsing (moved to after registerRoutes to avoid conflicts)

app.use((req, res, next) => {
  const start = Date.now();
  const path = req.path;
  let capturedJsonResponse: Record<string, any> | undefined = undefined;

  const originalResJson = res.json;
  res.json = function (bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };

  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path.startsWith("/api")) {
      let logLine = `${req.method} ${path} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }

      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "…";
      }

      const formattedTime = new Date().toLocaleTimeString("en-US", {
        hour: "numeric",
        minute: "2-digit",
        second: "2-digit",
        hour12: true,
      });
      console.log(`${formattedTime} [express] ${logLine}`);
    }
  });

  next();
});

(async () => {
  
  console.log("[build]", BUILD);
  
  // Firebase proxy to handle CORS issues with SVG loading
  app.get("/api/firebase-proxy", async (req, res) => {
    try {
      const url = String(req.query.url || "");
      if (!/^https:\/\/(firebasestorage|storage|firestore)\.googleapis\.com\//i.test(url)) return res.status(400).send("blocked");
      const r = await fetch(url);
      if (!r.ok) return res.status(r.status).send(await r.text());
      res.setHeader("Content-Type", r.headers.get("content-type") || "application/octet-stream");
      res.send(await r.text());
    } catch (e:any) { res.status(500).send(e?.message || "proxy error"); }
  });
  
  // 301 redirect from old path to correct path (preserves query strings)
  app.get("/business-templates*", (req, res) => {
    const newUrl = req.url.replace("/business-templates", "/business-assets/templates/business-plan");
    res.redirect(301, newUrl);
  });
  
  const server = await registerRoutes(app);
  
  // Register cover templates routes after main routes to avoid conflicts
  app.use(coverTemplatesRouter);
  
  // Register infographic templates routes
  app.use(infographicTemplatesRouter);

  // API guards to ensure all /api/* requests return JSON, never HTML
  // This prevents Vite from serving HTML for unmatched API routes
  app.all('/api/domains/search', (req, res, next) => {
    if (req.method !== 'POST') {
      return res.status(405).json({ error: 'Method Not Allowed. Use POST.' });
    }
    next();
  });

  app.use('/api', (req, res, next) => {
    res.type('application/json');
    next();
  });

  app.use('/api', (_req, res) => {
    res.status(404).json({ error: 'API endpoint not found' });
  });

  app.use((err: any, _req: Request, res: Response, _next: NextFunction) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";

    res.status(status).json({ message });
    throw err;
  });

  // Add no-cache and build headers to HTML responses
  app.use((_req, res, next) => {
    const originalSend = res.send;
    const originalSendFile = res.sendFile;
    
    res.send = function(body) {
      if (res.getHeader('Content-Type')?.toString().includes('text/html')) {
        res.set("Cache-Control", "no-store");
        res.set("X-IBrandBiz-Build", `${BUILD.sha}@${BUILD.time}`);
      }
      return originalSend.call(this, body);
    };
    
    res.sendFile = function(path: string, ...args: any[]) {
      if (path?.toString().endsWith('index.html')) {
        res.set("Cache-Control", "no-store");
        res.set("X-IBrandBiz-Build", `${BUILD.sha}@${BUILD.time}`);
      }
      // @ts-ignore - Express types for sendFile are overly strict
      return originalSendFile.call(this, path, ...args);
    };
    
    next();
  });

  // importantly only setup vite in development and after
  // setting up all the other routes so the catch-all route
  // doesn't interfere with the other routes
  if (app.get("env") === "development") {
    const { setupVite } = await import("./vite");
    await setupVite(app, server);
  } else {
    const { serveStatic } = await import("./vite");
    serveStatic(app);
  }

  // ALWAYS serve the app on the port specified in the environment variable PORT
  // Other ports are firewalled. Default to 5000 if not specified.
  // this serves both the API and the client.
  // It is the only port that is not firewalled.
  const port = parseInt(process.env.PORT || '5000', 10);
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true,
  }, () => {
    const formattedTime = new Date().toLocaleTimeString("en-US", {
      hour: "numeric",
      minute: "2-digit",
      second: "2-digit",
      hour12: true,
    });
    console.log(`${formattedTime} [express] serving on port ${port}`);
  });
})();
