import { EventEmitter } from 'events';
import type { Response } from 'express';

// Interface for notification data
export interface NotificationData {
  id: string;
  userId: string;
  message: string;
  type: 'info' | 'success' | 'warning' | 'error' | 'pro_welcome' | 'system';
  isRead: boolean;
  createdAt: string;
}

// SSE message types
export interface SSEMessage {
  type: 'connected' | 'notification' | 'keepalive' | 'error';
  message?: string;
  data?: NotificationData;
  timestamp?: number;
}

// Client connection tracking
interface SSEConnection {
  response: Response;
  userId: string;
  streamToken: string;
  connectedAt: Date;
}

class NotificationBroadcaster extends EventEmitter {
  private connections = new Map<string, SSEConnection>();
  private keepaliveInterval: NodeJS.Timeout | null = null;
  private readonly KEEPALIVE_INTERVAL = 30000; // 30 seconds

  constructor() {
    super();
    this.setMaxListeners(200); // Support many SSE connections
    this.startKeepalive();
  }

  // Start keepalive mechanism to detect dead connections
  private startKeepalive() {
    if (this.keepaliveInterval) {
      clearInterval(this.keepaliveInterval);
    }

    this.keepaliveInterval = setInterval(() => {
      this.broadcastKeepalive();
    }, this.KEEPALIVE_INTERVAL);
  }

  // Send keepalive to all connections to detect dead ones
  private broadcastKeepalive() {
    const message: SSEMessage = {
      type: 'keepalive',
      timestamp: Date.now()
    };

    const deadConnections: string[] = [];

    for (const [tokenId, connection] of this.connections) {
      try {
        connection.response.write(`data: ${JSON.stringify(message)}\n\n`);
      } catch (error) {
        console.log(`Connection ${tokenId} appears dead, marking for removal`);
        deadConnections.push(tokenId);
      }
    }

    // Clean up dead connections
    for (const tokenId of deadConnections) {
      this.removeConnection(tokenId);
    }
  }

  // Add a new SSE connection
  addConnection(streamToken: string, userId: string, response: Response): void {
    // Remove any existing connection with same token (shouldn't happen but be safe)
    this.removeConnection(streamToken);

    // Store the connection
    this.connections.set(streamToken, {
      response,
      userId,
      streamToken,
      connectedAt: new Date()
    });

    console.log(`SSE connection added for user ${userId} (token: ${streamToken.substring(0, 8)}...)`);

    // Set up connection close handlers
    response.on('close', () => {
      console.log(`SSE connection closed for user ${userId}`);
      this.removeConnection(streamToken);
    });

    response.on('error', (error) => {
      console.error(`SSE connection error for user ${userId}:`, error);
      this.removeConnection(streamToken);
    });

    // Send initial connected message
    const welcomeMessage: SSEMessage = {
      type: 'connected',
      message: 'Notifications stream connected',
      timestamp: Date.now()
    };

    try {
      response.write(`data: ${JSON.stringify(welcomeMessage)}\n\n`);
    } catch (error) {
      console.error('Failed to send welcome message:', error);
      this.removeConnection(streamToken);
    }
  }

  // Remove an SSE connection
  removeConnection(streamToken: string): void {
    const connection = this.connections.get(streamToken);
    if (connection) {
      try {
        if (!connection.response.headersSent) {
          connection.response.end();
        }
      } catch (error) {
        // Connection already closed, ignore
      }
      this.connections.delete(streamToken);
      console.log(`SSE connection removed (token: ${streamToken.substring(0, 8)}...)`);
    }
  }

  // Broadcast a notification to specific user
  sendNotificationToUser(userId: string, notification: NotificationData): void {
    const message: SSEMessage = {
      type: 'notification',
      data: notification,
      timestamp: Date.now()
    };

    const userConnections = Array.from(this.connections.values())
      .filter(conn => conn.userId === userId);

    if (userConnections.length === 0) {
      console.log(`No active SSE connections for user ${userId}`);
      return;
    }

    let sentCount = 0;
    const deadConnections: string[] = [];

    for (const connection of userConnections) {
      try {
        connection.response.write(`data: ${JSON.stringify(message)}\n\n`);
        sentCount++;
      } catch (error) {
        console.error(`Failed to send notification to user ${userId}:`, error);
        deadConnections.push(connection.streamToken);
      }
    }

    // Clean up dead connections
    for (const tokenId of deadConnections) {
      this.removeConnection(tokenId);
    }

    console.log(`Notification sent to ${sentCount} connections for user ${userId}`);
  }

  // Broadcast to all connected users (system-wide notifications)
  broadcastToAll(notification: NotificationData): void {
    const message: SSEMessage = {
      type: 'notification',
      data: notification,
      timestamp: Date.now()
    };

    let sentCount = 0;
    const deadConnections: string[] = [];

    for (const [tokenId, connection] of this.connections) {
      try {
        connection.response.write(`data: ${JSON.stringify(message)}\n\n`);
        sentCount++;
      } catch (error) {
        console.error(`Failed to send broadcast notification:`, error);
        deadConnections.push(tokenId);
      }
    }

    // Clean up dead connections
    for (const tokenId of deadConnections) {
      this.removeConnection(tokenId);
    }

    console.log(`Broadcast notification sent to ${sentCount} connections`);
  }

  // Get connection stats
  getStats() {
    const now = new Date();
    const connections = Array.from(this.connections.values());
    
    return {
      totalConnections: connections.length,
      uniqueUsers: new Set(connections.map(c => c.userId)).size,
      oldestConnection: connections.length > 0 
        ? Math.min(...connections.map(c => now.getTime() - c.connectedAt.getTime()))
        : 0,
      connectionsByUser: connections.reduce((acc, conn) => {
        acc[conn.userId] = (acc[conn.userId] || 0) + 1;
        return acc;
      }, {} as Record<string, number>)
    };
  }

  // Cleanup all connections (for shutdown)
  cleanup(): void {
    if (this.keepaliveInterval) {
      clearInterval(this.keepaliveInterval);
      this.keepaliveInterval = null;
    }

    for (const [tokenId] of this.connections) {
      this.removeConnection(tokenId);
    }

    this.connections.clear();
    console.log('Notification broadcaster cleanup completed');
  }
}

// Export singleton instance
export const notificationBroadcaster = new NotificationBroadcaster();

// Graceful shutdown handling
process.on('SIGTERM', () => {
  console.log('SIGTERM received, cleaning up notification broadcaster...');
  notificationBroadcaster.cleanup();
});

process.on('SIGINT', () => {
  console.log('SIGINT received, cleaning up notification broadcaster...');
  notificationBroadcaster.cleanup();
});