import { Registrar, RegisterInput, DomainSearchInput, RegistrarConfig } from "./types";
import { DomainSearchHit, DomainRegistrationResult } from "@shared/schema";
import { createHash } from 'crypto';
import { DOMParser } from '@xmldom/xmldom';
import { getRetailPriceForDomain } from "../pricing";

const DEFAULT_TLDS = ['.com', '.net', '.co', '.io', '.ai'];

export class OpenSRSRegistrar implements Registrar {
  private config: RegistrarConfig;

  constructor() {
    // Map OpenSRS mode properly: 'ote' -> test, 'live' -> live
    const envMode = process.env.OPENSRS_MODE || 'ote';
    const mode = (envMode === 'live') ? 'live' : 'test';
    
    // Map endpoints by mode: OTE uses horizon, live uses rr-n1-tor
    const baseUrl = mode === 'test' 
      ? 'https://horizon.opensrs.net:55443'     // OTE (testing) environment
      : 'https://rr-n1-tor.opensrs.net:55443';  // Production environment
    
    // Trim credentials to prevent whitespace signature issues
    const username = (process.env.OPENSRS_USER || '').trim();
    const key = (process.env.OPENSRS_KEY || '').trim();
    
    this.config = {
      mode,
      baseUrl,
      credentials: {
        username,
        key,
      },
      defaultNameservers: process.env.DOMAIN_DEFAULT_NAMESERVERS?.split(',') || ['ns1.opensrs.net', 'ns2.opensrs.net'],
      defaultYears: parseInt(process.env.DOMAIN_DEFAULT_YEARS || '1'),
    };
    
    // Log credential presence for debugging (secure, doesn't expose values)
    const hasUsername = !!username && username.length > 0;
    const hasKey = !!key && key.length > 0;
    const isFullyConfigured = hasUsername && hasKey;
    
    console.log(`[OpenSRS] Initialization - mode=${mode}, baseUrl=${baseUrl}`);
    console.log(`[OpenSRS] Credentials - OPENSRS_USER present: ${hasUsername}, OPENSRS_KEY present: ${hasKey}, configured: ${isFullyConfigured}`);
    
    if (!isFullyConfigured) {
      console.warn(`[OpenSRS] ⚠️  Missing credentials - domain search will use demo fallback mode`);
      console.warn(`[OpenSRS] Required env vars: OPENSRS_USER, OPENSRS_KEY, OPENSRS_MODE (optional, defaults to 'ote')`);
    }
  }

  isConfigured(): boolean {
    return !!(this.config.credentials.username && this.config.credentials.key);
  }

  async getSupportedTlds(): Promise<{ tld: string; priceCents: number; }[]> {
    return DEFAULT_TLDS.map(tld => {
      const domain = `example${tld}`;
      const pricing = getRetailPriceForDomain(domain);
      return {
        tld,
        priceCents: pricing.priceCents
      };
    });
  }

  private parseDomainQuery(query: string, availableTlds: string[]): {
    label: string;
    existingTld: string | null;
    fullDomain: string;
  } {
    // Create a combined list of known TLDs, sorted by length (descending) to handle multi-label TLDs
    const allKnownTlds = [...new Set([...availableTlds, ...DEFAULT_TLDS, '.co.uk', '.com.au', '.net.au', '.org.uk'])];
    const sortedTlds = allKnownTlds.sort((a, b) => b.length - a.length);

    // Check if the query already contains a known TLD
    for (const tld of sortedTlds) {
      if (query.endsWith(tld)) {
        const label = query.substring(0, query.length - tld.length);
        // Ensure there's actually a label before the TLD and it's a valid domain format
        if (label && label.length > 0 && !label.endsWith('.')) {
          return {
            label,
            existingTld: tld,
            fullDomain: query
          };
        }
      }
    }

    // No recognized TLD found, treat the entire query as a label
    return {
      label: query,
      existingTld: null,
      fullDomain: query
    };
  }

  async search(input: DomainSearchInput): Promise<DomainSearchHit[]> {
    const { query } = input;
    const tlds = input.tlds || DEFAULT_TLDS;

    // Parse the query to detect if it already contains a TLD
    const parsedQuery = this.parseDomainQuery(query.toLowerCase().trim(), tlds);
    
    // Generate domain variants based on whether the query already has a TLD
    const domains: string[] = [];
    
    if (parsedQuery.existingTld) {
      // Query already contains a TLD, only search that exact domain
      domains.push(parsedQuery.fullDomain);
    } else {
      // Query is just a label, generate variants with each TLD
      domains.push(...tlds.map(tld => `${parsedQuery.label}${tld}`));
    }

    const results: DomainSearchHit[] = [];

    for (const domain of domains) {
      try {
        const available = await this.checkDomainAvailability(domain);
        const isPremium = this.isPremiumDomain(domain);
        const pricing = getRetailPriceForDomain(domain);
        
        // For MVP: Block premium domains to avoid surprises
        if (isPremium && available) {
          results.push({
            domain,
            available: false, // Block purchase
            priceCents: undefined,
            type: 'premium',
            premium: true,
            errorMessage: 'Premium domains not supported yet'
          });
        } else {
          results.push({
            domain,
            available,
            priceCents: available ? pricing.priceCents : undefined,
            type: 'standard'
          });
        }
      } catch (error) {
        console.error(`[OpenSRS] ❌ Error checking domain ${domain}:`, error);
        console.error(`[OpenSRS] Error details:`, {
          message: error instanceof Error ? error.message : 'Unknown error',
          stack: error instanceof Error ? error.stack : undefined,
          raw: error
        });
        
        // Temporary fallback for demo purposes when OpenSRS auth fails
        // In real usage, this would be resolved with proper OpenSRS credentials
        const isLikelyAvailable = this.getDemoAvailability(domain);
        const isPremium = this.isPremiumDomain(domain);
        const pricing = getRetailPriceForDomain(domain);
        
        // For MVP: Block premium domains to avoid surprises
        if (isPremium && isLikelyAvailable) {
          results.push({
            domain,
            available: false, // Block purchase
            priceCents: undefined,
            type: 'premium',
            premium: true,
            errorMessage: 'Premium domains not supported yet'
          });
        } else {
          results.push({
            domain,
            available: isLikelyAvailable,
            priceCents: isLikelyAvailable ? pricing.priceCents : undefined,
            type: 'standard'
          });
        }
      }
    }

    return results;
  }

  async register(input: RegisterInput): Promise<DomainRegistrationResult> {
    const { domain, years, contact, privacy, nameservers } = input;

    if (!this.isConfigured()) {
      return {
        success: false,
        message: 'OpenSRS not configured - missing credentials'
      };
    }

    try {
      const privacySettings = this.mapPrivacySettings(domain, privacy);
      
      const result = await this.callOpenSRS('SW_REGISTER', 'DOMAIN', {
        domain,
        period: years,
        nameserver_list: nameservers?.length ? nameservers : this.config.defaultNameservers,
        contact_set: {
          owner: this.mapContactToOpenSRS(contact),
          admin: this.mapContactToOpenSRS(contact),
          billing: this.mapContactToOpenSRS(contact),
          tech: this.mapContactToOpenSRS(contact)
        },
        ...privacySettings
      });

      if (result.is_success) {
        return {
          success: true,
          providerRegId: result.attributes?.registration_id || result.attributes?.order_id,
          message: result.response_text || 'Domain registered successfully'
        };
      } else {
        return {
          success: false,
          message: result.response_text || result.attributes?.error || 'Domain registration failed'
        };
      }
    } catch (error) {
      console.error('OpenSRS registration error:', error);
      return {
        success: false,
        message: error instanceof Error ? error.message : 'Unknown registration error'
      };
    }
  }

  private async checkDomainAvailability(domain: string): Promise<boolean> {
    if (!this.isConfigured()) {
      throw new Error('OpenSRS not configured - missing credentials');
    }

    console.log(`[OpenSRS] Checking availability for: ${domain}`);
    const result = await this.callOpenSRS('LOOKUP', 'DOMAIN', {
      domain,
      search_type: 'domain'
    });

    // Parse availability from various OpenSRS response formats
    const available = this.parseAvailabilityFromResponse(result);
    console.log(`[OpenSRS] ${domain} availability: ${available ? 'AVAILABLE' : 'TAKEN'}`);
    return available;
  }

  private async callOpenSRS(action: string, object: string, attributes: Record<string, any>): Promise<OpenSRSResponse> {
    const bodyString = this.buildXMLRequest(action, object, attributes);
    
    // Log upstream provider for diagnostics
    try {
      const url = new URL(this.config.baseUrl);
      console.log(`[domain] OpenSRS upstream → host=${url.host} path=${url.pathname} action=${action}`);
    } catch {
      console.log(`[domain] OpenSRS upstream → ${this.config.baseUrl} action=${action}`);
    }
    
    // Security: Enhanced PII redaction for registration calls that contain sensitive contact information
    if (action === 'SW_REGISTER') {
      const redactedBody = this.redactPIIFromXML(bodyString);
      console.log(`OpenSRS Request (${action} ${object}) - PII redacted:`, redactedBody.substring(0, 1000));
    } else {
      console.log(`OpenSRS Request XML (${action} ${object}):`, bodyString.substring(0, 1000));
    }
    
    const signature = this.computeSignature(bodyString);

    const response = await fetch(this.config.baseUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'text/xml; charset=UTF-8',
        'Accept': 'text/xml',
        'X-Username': this.config.credentials.username,
        'X-Signature': signature
      },
      body: bodyString
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`OpenSRS API error: ${response.status} ${response.statusText} - ${errorText}`);
    }

    const responseText = await response.text();
    
    // Check if response is XML or JSON
    let result: OpenSRSResponse;
    if (responseText.trim().startsWith('<?xml')) {
      // Parse XML response
      result = this.parseXMLResponse(responseText);
    } else {
      // Parse JSON response (fallback)
      try {
        result = JSON.parse(responseText) as OpenSRSResponse;
      } catch (error) {
        throw new Error(`Failed to parse OpenSRS response: ${responseText.substring(0, 200)}...`);
      }
    }
    
    // Enhanced error handling for OpenSRS responses
    if (!result.is_success || result.is_success === 0) {
      const errorMsg = result.response_text || 
                      result.attributes?.error || 
                      `OpenSRS error ${result.response_code || 'unknown'}`;
      throw new Error(errorMsg);
    }

    return result;
  }

  private parseXMLResponse(xmlText: string): OpenSRSResponse {
    try {
      const parser = new DOMParser();
      const doc = parser.parseFromString(xmlText, 'text/xml');
      
      // Find the main OPS_envelope element
      const envelope = doc.getElementsByTagName('OPS_envelope')[0];
      if (!envelope) {
        throw new Error('Invalid OpenSRS XML format - missing OPS_envelope');
      }

      // Extract body information
      const body = envelope.getElementsByTagName('body')[0];
      if (!body) {
        throw new Error('Invalid OpenSRS XML format - missing body');
      }

      // Extract data_block
      const dataBlock = body.getElementsByTagName('data_block')[0];
      if (!dataBlock) {
        throw new Error('Invalid OpenSRS XML format - missing data_block');
      }

      // Parse the main response structure
      const result: OpenSRSResponse = {
        is_success: false,
        response_code: undefined,
        response_text: undefined,
        attributes: {}
      };

      // Find the top-level dt_assoc element under data_block
      const topLevelAssoc = dataBlock.getElementsByTagName('dt_assoc')[0];
      if (!topLevelAssoc) {
        throw new Error('Invalid OpenSRS XML format - missing top-level dt_assoc');
      }

      // Get all direct child <item> elements from the top-level dt_assoc
      const items = Array.from(topLevelAssoc.childNodes).filter(
        node => node.nodeType === 1 && node.nodeName === 'item'
      ) as Element[];

      // Parse each item based on its key attribute
      for (const item of items) {
        const key = item.getAttribute('key');
        
        if (key === 'is_success') {
          const value = item.textContent?.trim();
          result.is_success = value === '1' || value === 'true';
        } else if (key === 'response_code') {
          result.response_code = item.textContent?.trim();
        } else if (key === 'response_text') {
          result.response_text = item.textContent?.trim();
        } else if (key === 'attributes') {
          // Parse nested attributes - look for nested dt_assoc within this item
          const nestedAssoc = item.getElementsByTagName('dt_assoc')[0];
          if (nestedAssoc) {
            result.attributes = this.parseXMLAttributesFromAssoc(nestedAssoc);
          }
        }
      }

      // Add debugging for failed responses
      if (!result.is_success || result.is_success === 0) {
        console.log('OpenSRS XML Response Debug (first 1000 chars):', xmlText.substring(0, 1000));
      }

      return result;
    } catch (error) {
      console.error('XML parsing error:', error);
      console.log('Failed XML Response (first 1000 chars):', xmlText.substring(0, 1000));
      throw new Error(`Failed to parse OpenSRS XML response: ${error instanceof Error ? error.message : 'Unknown error'}`);
    }
  }

  private parseXMLAttributesFromAssoc(assocElement: Element): Record<string, any> {
    const attributes: Record<string, any> = {};
    
    // Get all direct child <item> elements from the dt_assoc
    const items = Array.from(assocElement.childNodes).filter(
      node => node.nodeType === 1 && node.nodeName === 'item'
    ) as Element[];

    for (const item of items) {
      const key = item.getAttribute('key');
      if (key) {
        attributes[key] = this.parseXMLValueFromItem(item);
      }
    }
    
    return attributes;
  }

  private parseXMLAttributes(element: Element): Record<string, any> {
    const attributes: Record<string, any> = {};
    
    // Look for nested dt_assoc elements
    const assocElements = element.getElementsByTagName('dt_assoc');
    for (let i = 0; i < assocElements.length; i++) {
      const item = assocElements[i];
      const key = item.getAttribute('key');
      if (key) {
        attributes[key] = this.parseXMLValue(item);
      }
    }

    // Also check for direct child dt_assoc elements
    const directChildren = Array.from(element.childNodes).filter(
      node => node.nodeType === 1 && node.nodeName === 'dt_assoc'
    ) as Element[];

    for (const child of directChildren) {
      const key = child.getAttribute('key');
      if (key) {
        attributes[key] = this.parseXMLValue(child);
      }
    }
    
    return attributes;
  }

  private parseXMLValueFromItem(item: Element): any {
    // Check if item has nested dt_assoc elements (object)
    const nestedAssocs = item.getElementsByTagName('dt_assoc');
    if (nestedAssocs.length > 0) {
      return this.parseXMLAttributesFromAssoc(nestedAssocs[0]);
    }

    // Check if item has dt_array elements (array)
    const arrays = item.getElementsByTagName('dt_array');
    if (arrays.length > 0) {
      const result: any[] = [];
      for (let i = 0; i < arrays.length; i++) {
        const arrayItem = arrays[i];
        // For arrays, look for nested items
        const arrayItems = Array.from(arrayItem.childNodes).filter(
          node => node.nodeType === 1 && node.nodeName === 'item'
        ) as Element[];
        for (const arrayItemElement of arrayItems) {
          result.push(this.parseXMLValueFromItem(arrayItemElement));
        }
      }
      return result;
    }

    // Simple text content
    const textContent = item.textContent?.trim();
    
    // Try to parse as number
    if (textContent && !isNaN(Number(textContent))) {
      return Number(textContent);
    }
    
    // Try to parse as boolean
    if (textContent === '1' || textContent === 'true') {
      return true;
    }
    if (textContent === '0' || textContent === 'false') {
      return false;
    }
    
    return textContent || '';
  }

  private parseXMLValue(element: Element): any {
    // Check if element has nested dt_assoc elements (object)
    const nestedAssocs = element.getElementsByTagName('dt_assoc');
    if (nestedAssocs.length > 0) {
      return this.parseXMLAttributes(element);
    }

    // Check if element has dt_array elements (array)
    const arrays = element.getElementsByTagName('dt_array');
    if (arrays.length > 0) {
      const result: any[] = [];
      for (let i = 0; i < arrays.length; i++) {
        const arrayItem = arrays[i];
        result.push(this.parseXMLValue(arrayItem));
      }
      return result;
    }

    // Simple text content
    const textContent = element.textContent?.trim();
    
    // Try to parse as number
    if (textContent && !isNaN(Number(textContent))) {
      return Number(textContent);
    }
    
    // Try to parse as boolean
    if (textContent === '1' || textContent === 'true') {
      return true;
    }
    if (textContent === '0' || textContent === 'false') {
      return false;
    }
    
    return textContent || '';
  }

  private buildXMLRequest(action: string, object: string, attributes: Record<string, any>): string {
    // Build the main dt_assoc items
    const mainItems: string[] = [
      `<item key="protocol">XCP</item>`,
      `<item key="action">${this.escapeXML(action)}</item>`,
      `<item key="object">${this.escapeXML(object)}</item>`
    ];
    
    // Add attribute items directly to the main structure
    for (const [key, value] of Object.entries(attributes)) {
      mainItems.push(this.valueToXMLItem(key, value));
    }
    
    const xmlBody = `<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<!DOCTYPE OPS_envelope SYSTEM "ops.dtd">
<OPS_envelope>
  <header>
    <version>0.9</version>
  </header>
  <body>
    <data_block>
      <dt_assoc>
        ${mainItems.join('\n        ')}
      </dt_assoc>
    </data_block>
  </body>
</OPS_envelope>`;
    
    return xmlBody;
  }

  private objectToXML(obj: Record<string, any>): string {
    const items: string[] = [];
    
    for (const [key, value] of Object.entries(obj)) {
      items.push(this.valueToXMLItem(key, value));
    }
    
    return `<dt_assoc>\n${items.join('\n')}\n</dt_assoc>`;
  }

  private valueToXMLItem(key: string, value: any): string {
    if (value === null || value === undefined) {
      return `<item key="${this.escapeXML(key)}"></item>`;
    }
    
    if (Array.isArray(value)) {
      const arrayItems = value.map((item, index) => {
        if (typeof item === 'object' && item !== null) {
          return `<item key="${index}">${this.objectToXML(item)}</item>`;
        } else {
          return `<item key="${index}">${this.escapeXML(String(item))}</item>`;
        }
      }).join('\n');
      
      return `<item key="${this.escapeXML(key)}"><dt_array>\n${arrayItems}\n</dt_array></item>`;
    }
    
    if (typeof value === 'object' && value !== null) {
      return `<item key="${this.escapeXML(key)}">${this.objectToXML(value)}</item>`;
    }
    
    return `<item key="${this.escapeXML(key)}">${this.escapeXML(String(value))}</item>`;
  }

  private escapeXML(text: string): string {
    return text
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#x27;');
  }

  private computeSignature(body: string): string {
    // OpenSRS signature computation: MD5(MD5(body + key) + key)
    // Use bytes for accurate signature computation
    const bodyBytes = Buffer.from(body, 'utf8');
    const keyBytes = Buffer.from(this.config.credentials.key, 'utf8');
    
    const innerHash = createHash('md5')
      .update(Buffer.concat([bodyBytes, keyBytes]))
      .digest('hex');
    
    const signature = createHash('md5')
      .update(innerHash + this.config.credentials.key)
      .digest('hex');
    
    return signature;
  }

  // Security: Redact PII from XML request logs for SW_REGISTER operations
  private redactPIIFromXML(xmlString: string): string {
    // List of PII fields to redact in OpenSRS XML
    const piiFields = [
      'first_name', 'last_name', 'email', 'phone', 'org_name',
      'address1', 'address2', 'city', 'state', 'postal_code'
    ];
    
    let redacted = xmlString;
    
    // Redact each PII field by replacing the value with [REDACTED]
    piiFields.forEach(field => {
      // Match pattern: <item key="field_name">value</item>
      const regex = new RegExp(`(<item key="${field}">)[^<]*(<\/item>)`, 'gi');
      redacted = redacted.replace(regex, `$1[REDACTED]$2`);
    });
    
    // Also redact any obvious email patterns that might be missed
    redacted = redacted.replace(/\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,}\b/g, '[EMAIL_REDACTED]');
    
    // Redact phone numbers (various formats)
    redacted = redacted.replace(/(\+?1?[-.\s]?)?\(?[0-9]{3}\)?[-.\s]?[0-9]{3}[-.\s]?[0-9]{4}/g, '[PHONE_REDACTED]');
    
    return redacted;
  }

  private mapContactToOpenSRS(contact: any) {
    return {
      first_name: contact.firstName,
      last_name: contact.lastName,
      email: contact.email,
      phone: contact.phone,
      org_name: contact.organization || '',
      address1: contact.address,
      address2: '',
      city: contact.city,
      state: contact.state || '',
      postal_code: contact.postalCode || '',
      country: contact.country
    };
  }

  private extractTldFromDomain(domain: string, tlds: string[]): string {
    // Sort TLDs by length (descending) to handle multi-label TLDs like .co.uk properly
    const sortedTlds = [...tlds].sort((a, b) => b.length - a.length);
    
    for (const tld of sortedTlds) {
      if (domain.endsWith(tld)) {
        return tld;
      }
    }
    
    // Fallback: extract TLD from domain directly
    const lastDotIndex = domain.lastIndexOf('.');
    return lastDotIndex >= 0 ? domain.substring(lastDotIndex) : '.com';
  }

  private getDemoAvailability(domain: string): boolean {
    // Demo logic: make most domains appear available for testing checkout flow
    // In production, this would be replaced by real OpenSRS responses
    
    // Make domains unavailable only if they contain common "taken" indicators
    const commonTakenWords = ['google', 'facebook', 'amazon', 'microsoft', 'apple'];
    const domainLower = domain.toLowerCase();
    
    // If domain contains a common brand name, mark as taken
    if (commonTakenWords.some(word => domainLower.includes(word))) {
      return false;
    }
    
    // Otherwise, make 80% of domains available for demo/testing
    const hash = domain.split('').reduce((a, b) => {
      a = ((a << 5) - a) + b.charCodeAt(0);
      return a & a;
    }, 0);
    
    return Math.abs(hash) % 10 < 8; // 80% available for testing
  }

  private isPremiumDomain(domain: string): boolean {
    // Demo logic: detect premium domains for MVP blocking
    // In production, this would be determined by OpenSRS response data
    
    // Simple heuristic: short domains, common words, numbers, or patterns
    const label = domain.split('.')[0].toLowerCase();
    
    // Short domains (3 chars or less)
    if (label.length <= 3) return true;
    
    // Common words/patterns that would typically be premium
    const premiumPatterns = [
      'ai', 'app', 'buy', 'car', 'cash', 'crypto', 'deal', 'fast', 'free',
      'game', 'golf', 'home', 'hot', 'news', 'online', 'pay', 'pro', 'real',
      'sale', 'shop', 'tax', 'tech', 'trade', 'web', 'win'
    ];
    
    return premiumPatterns.some(pattern => label.includes(pattern));
  }

  private parseAvailabilityFromResponse(result: OpenSRSResponse): boolean {
    const attrs = result.attributes;
    if (!attrs) return false;

    // Check various formats for availability
    // Handle 0/1/true/"1" variations
    if (attrs.is_available !== undefined) {
      return !!(attrs.is_available === true || attrs.is_available === 1 || attrs.is_available === "1");
    }
    
    if (attrs['is-available'] !== undefined) {
      return !!(attrs['is-available'] === true || attrs['is-available'] === 1 || attrs['is-available'] === "1");
    }
    
    if (attrs.status !== undefined) {
      return attrs.status === 'available' || attrs.status === 'AVAILABLE';
    }
    
    if (attrs.status_text !== undefined) {
      return attrs.status_text.toLowerCase().includes('available');
    }
    
    // Fallback: parse response_text
    if (result.response_text) {
      return result.response_text.toLowerCase().includes('available');
    }
    
    return false;
  }

  private mapPrivacySettings(domain: string, privacy?: boolean): Record<string, any> {
    if (!privacy) {
      return { whois_privacy: false };
    }

    // Basic privacy mapping - can be extended for TLD-specific requirements
    const settings: Record<string, any> = {
      whois_privacy: true
    };

    // Some TLDs may require specific privacy settings
    const tld = domain.substring(domain.lastIndexOf('.'));
    
    switch (tld) {
      case '.ca':
        // .ca domains may have specific privacy requirements
        settings.ca_privacy = true;
        break;
      case '.eu':
        // .eu domains may have GDPR-specific settings
        settings.eu_privacy = true;
        break;
      default:
        // Standard whois privacy for most TLDs
        break;
    }

    return settings;
  }
}

// Type definitions for OpenSRS responses
interface OpenSRSResponse {
  is_success: boolean | number;
  response_code?: string | number;
  response_text?: string;
  attributes?: {
    is_available?: boolean | number | string;
    'is-available'?: boolean | number | string;
    status?: string;
    status_text?: string;
    error?: string;
    registration_id?: string;
    order_id?: string;
    [key: string]: any;
  };
}

// Export singleton instance
export const opensrs = new OpenSRSRegistrar();